package PSP::Pile;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Pile.pm,v 1.4 2001/04/07 01:00:30 muaddie Exp $

use strict;

=head1 NAME

 PSP::Pile - an base object which maintains a collection of page methods.

=head1 SYNOPSIS

=head1 DESCRIPTION

=cut

use CGI;
use URI::Escape;
use PSP::Utils qw(dump_object 
		  path_to_page_name
		  page_name_to_path
		  reduce_url);
use PSP::Page;

use vars qw(@ISA);
@ISA = qw(PSP::Page);

=head1 METHODS

=head2 loader

  instance
  (Loader) loader ([Loader $loader])

DESCRIPTION:

Accessor method to possibly store, then retrieve the Loader object.

=cut

sub loader {
  my ($this,$loader) = @_;
  defined $loader and $this->{loader} = $loader;
  return $this->{loader};
}

sub basepile_loader { 
  my ($this) = @_;

  my $loader = $this->loader() or return;
  my $parent = $loader->parent() or return;

  while ($parent->parent()) {
    $loader = $parent;
    $parent = $parent->parent();
  }

  return $loader;
}

sub page_url {
  my ($this,$n_to_skip) = @_;
  my $loader = $this->loader();
  my $url = $loader->url($n_to_skip)."/".page_name_to_path($this->page_name());
  $url = reduce_url($url);
  return $url;
}

sub is_subpile {
  my ($this) = @_;

  if (my $loader = $this->loader()) {
    if (my $parent = $loader->parent()) {
      # if we have a grand-parent, we're a subpile.
      if ($parent->parent()) {
        return 1;
      }
    }
  }
  return 0;
}

sub page_not_found {
  my ($this,$page_name) = @_;

  $this->header(status => "404 Page Not Found");
  my $url = $this->page_url(0);
  my $cgi = $this->cgi();
  my $out = "";

  $out .=<<eohtml;
<html>
<head>
<title> Not Found </title>
</head>
<body>
<h1> Not Found </h1>
<p>

The URL you referenced:  $url  was not found.<br>
Someday, this default message will provide better help, but for right
now, unfortunately, you're on your own.

</p>
</body>
</html>

eohtml
# '
  return $out;
}

sub goto_page {
  my ($this,$page_url) = @_;

  my ($page_name,$pile);

  if (substr($page_url,0,1) eq '/') {

    # if an absolute path is given, it is with respect to the base pile

    # get the loader of the base pile.
    my $loader = $this->basepile_loader() or throw
      Error::Simple("Could not determine base loader.");

    # rebless $this to be its base pile.
    ref($this) ne $loader->class_name() and
      bless $this, $loader->class_name();

    # map the page_url to some pile and page_name.
    ($pile,$page_name) = $loader->map_page($this,$page_url);
    $pile or throw Error::Simple("'$page_url' does not map to a pile?!?\n");

  } else {

    # otherwise, it is with respect to the current pile.
    $pile = $this;

    # get the path to the current page.
    my $url = page_name_to_path($pile->page_name());
    # remove the trailing component from path.
    $url =~ s!(/?)[^/]+/?$!$1!;
    # append new page_url to this path.
    $url .= $page_url;
    # reduce ".."-related aspects of url.
    $url = reduce_url($url);
    # convert and set the pile's current page_name.
    $page_name = $pile->page_name($url);
  }

  # if this method is not available, return the output of page_not_found().
  $pile->can($page_name) or
    return $pile->page_not_found($page_name);

  # call the pile method.
  return $pile->$page_name();
}

sub link_page {
  my ($this,$page_url) = @_;

  my $url;
  if (substr($page_url,0,1) eq '/') {

    $url = $this->url().$page_url;

  } else {

    # get the path to the current page.
    $url = $this->page_url();
    # remove the trailing component from path.
    $url =~ s![^/]+/?$!$1!;
    # append new page_url to this path.
    $url .= $page_url;
  }

  # reduce the url.
  $url = reduce_url($url);

  # return the url.
  return $url;
}

sub redirect_page {
  my ($this,$page_url,$time) = @_;
  $time ||= 15;

  my $url;
  if (substr($page_url,0,1) eq '/') {

    $url = $this->url().$page_url;

  } else {

    # get the path to the current page.
    $url = $this->page_url();
    # remove the trailing component from path.
    $url =~ s!(/?)[^/]+/?$!$1!;
    # append new page_url to this path.
    $url = $page_url;
  }

  # reduce the url.
  $url = reduce_url($url);

  return $this->header("refresh", "$time;URL=$url");
}

sub error_page {
  my ($this) = @_;
  return("<h2>An unknown error has occurred.</h2>\n");
}

=head2 free_internals

  instance
  () free_internals ()

DESCRIPTION:

A method to remove internal references to avoid circular references.

=cut

sub free_internals {
  my ($this) = @_;
  delete $this->{loader};
  $this->SUPER::free_internals();
  return;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<CGI>, L<PSP::Utils>

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
