package AtomicData::USPhone;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: USPhone.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtmoicData::USPhone - for storing US phone numbers

=head1 SYNOPSIS

see AtomicData.pm

=head1 DESCRIPTION

Implements data encapsulation for US phone numbers

=cut

use AtomicData::Text;
@AtomicData::USPhone::ISA = qw (AtomicData::Text);

=head1 METHODS

see AtomicData.pm for further methods.

=head2 parameters

see parent class for additional documentation. This class depricates
the min and max_length parameters and handles length itself since the
length must be either 7 or 10 digits. This class supports
'area_code'/bool and 'numbers'/bool, which require that an area code
be included and that all interesting data be numeric respectively.

=head2 set_format

see parent class for additional documentation. Will accept 'dots'
bool, two mutually exclusive options which will replace hyphens with a
dot. 'parens'/bool will place parens around the area code.
'numbers'/bool will translate all alpha information into digits.

=head2 canonicalize

see parent class for additional documentation.

Will return data in the form of
/^([a-zA-Z\d]{3}-([a-zA-Z\d]{3}-)?[a-zA-Z\d]{4})?$/.

=head2 _canonicalize

 instance/private
 (int $value) _canonicalize (int $raw_value)

DESCRIPTION:

Will accept a value and return the cannonical form or undef and set
C<$this->{_can_failed}>.

=cut

sub _canonicalize {
  my ($this, $val) = @_;

  my $value = $this->SUPER::_canonicalize($val);
  defined $value or return;
  $value eq "" and return $value;

  #strip leading 1's
  $val =~ s/^1[- \.]?//;
  #strip parens around area code; strip - or ' ' from after
  $val =~ s/^\(?([\da-zA-Z]{3})\)?[- \.]?/$1/;
  #strip valid seperators form the second and third digit clusters
  $val =~ s/([\da-zA-Z]{3})[-\. ]([\da-zA-Z]{4})$/$1$2/;

  my @hold;
  unless ((@hold) = $val =~ /^(([a-zA-Z\d]{3})([a-zA-Z\d]{3})?([a-zA-Z\d]{4}))?$/) {
    $this->{_can_failed} = ['Invalid US phone number.'];
    return undef;
  }
  shift @hold;

  if ($hold[0] =~ /^(0|1)/) {
    $this->{_can_failed} = ['Phone number cannot start with 0 or 1.'];
  }
  if (($hold[1] && ($hold[1] =~ /555|[jkl]{3}/i)) ||
      ($hold[0] =~ /555|[jkl]{3}/i)){
    $this->{_can_failed} = ['555 is not a valid local prefix.'];
  }

  if (!$hold[1]) { #no area code
    $val = $hold[0].'-'.$hold[2];
  }
  else { #area code
    $val = $hold[0].'-'.$hold[1].'-'.$hold[2];
    $this->{_area_code} = 1;
  }

  return $val;
}

=head2 _verify

 instance/private
 (1) _verify ()

DESCRIPTION:

Verifies that the data is 9 digits if long is set.

=cut

sub _verify {
  my ($this) = @_;

  my ($bool,$problems) = $this->SUPER::_verify();
  $bool or return (0, $problems);

  my $value = $this->canonical_value();
  my $parameters = $this->parameters();

  if ($parameters->{numbers} && ($value =~ /[a-zA-Z]/)) {
    push @$problems, 'Alpha characters are not acceptable.';
  }
  if ($parameters->{area_code} && (length($value) < 10)) {
    push @$problems, 'Must include an area code.';
  }

  @$problems and return (0, $problems);
  return (1, []);
}

=head2 _format

  instance
  (string $zip_code) _format (int $can_value)

DESCRIPTION:

Returns a formatted phone number.

=cut

sub _format {
  my ($this, $value) = @_;

  if ($this->{_format}->{numbers}) {
    $value =~ s/[abc]/2/ig;
    $value =~ s/[def]/3/ig;
    $value =~ s/[ghi]/4/ig;
    $value =~ s/[jkl]/5/ig;
    $value =~ s/[mno]/6/ig;
    $value =~ s/[prsq]/7/ig;
    $value =~ s/[tuv]/8/ig;
    $value =~ s/[wxyz]/9/ig;
  }

  if ($this->{_area_code} && $this->{_format}->{parens}) {
    $value =~ s/(\d{3})/\($1\)/;
  }

  if ($this->{_format}->{dots}) {
    $value =~ s/-/\./g;
  }
  if ($this->{_format}->{dash}) {
     $value =~ s/\./-/g;
  }
  
  return $value;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
