package PSP::Parser::Group;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Group.pm,v 1.12 2001/01/23 05:41:01 muaddib Exp $

use strict;

use PSP::Parser;
@PSP::Parser::Group::ISA = qw(Exporter PSP::Parser);

use Exporter;
use Error qw(:try);
use PSP::Utils;

use vars qw(@ISA @EXPORT_OK);
BEGIN {
  @EXPORT_OK = qw(process_groups process_group);
}

use vars qw(@handled @handled_no_end @stacks @current @propagatable
	    $ERR_STAR);

BEGIN {
  @handled = qw(group dynamicdisplay do
		rollto rollover rollback refresh);
  @handled_no_end = qw(indexoverride);

  @stacks        = qw(group ddisplay);
  @current       = qw(gname obj numvar
		      form fsdef define fsuse submit verify);
  @propagatable = ((map { "stack_$_" } @stacks),
		   (map { "current_$_" } @current),
		   'fieldspaces','form_count');
  $ERR_STAR='<font color="#ff0000" size="+3" face="arial,helvetica">*</font>';
};

=head2 begin_pspgroup

 [private] instance
 () begin_pspgroup (string $tag, hash $attr)

DESCRIPTION:

A tag handler concerned with the GROUP TAG.

=cut

sub begin_pspgroup {
  my ($this, $tag, $attr) = @_;

  my $fieldspace = $this->fieldspace();
  my $gname = $attr->{name} or throw
    Error::Simple("<$tag> requires NAME attribute.");
  $fieldspace->{group_defs}->{$gname} and throw
    Error::Simple("A <$tag> with NAME '$gname' already exists.");
  my $fsname = $this->fsname(1);

  my $basepkg = "Group";
  $basepkg .= "::$this->{pile_name}" if $this->{pile_name};

  my $group = $fieldspace->{group_defs}->{$gname} =
    {
     setup    => "",
     name     => $gname,
     grpvar   => $attr->{grpvar}     || '$_'.$gname.'_group',
     numvar   => $attr->{numvar}     || '$_'.$gname.'_index',
     objvar   => $attr->{obj}        || '$_'.$gname.'_obj',
     number   => $attr->{numdisplay} || 20,
     maxnum   => $attr->{maxnum}     || "",
     package  => $basepkg."::${fsname}::${gname}",
     field_names => [],
     dummy_ok => bool_att($attr->{dummyok}) ? 1 : 0
    };

  $this->{stack_group} ||= [];
  push @{$this->{stack_group}}, $group;

  # Acquire
  my ($begin0,$end0) = $this->handlers();
  my $begin = { %$begin0 };
  my $end   = { %$end0   };

  $begin->{'psp:define'} = \&begin_pspdefine_group;
  $begin->{'psp:include'} = \&PSP::Parser::Control::begin_pspinclude;
  $end->{'psp:define'}   = \&end_pspdefine_group;
  $this->push_handlers($begin,$end);
  $this->script_mode();
  $this->push_code_sub(\&code_pspgroup);
  $this->push_decl();
}

=head2 code_pspgroup

 [private] instance
 () code_pspgroup (string @code)

DESCRIPTION:

See PSP specification.

=cut

sub code_pspgroup {
  my ($this,$text) = @_;
  my $fieldspace = $this->fieldspace();
  my $group = $this->{stack_group}->[-1] or throw
    Error::Simple("Internal error: text_pspgroup without current group?!?");
  $this->append_code($text,\$group->{setup});
}

=head2 end_pspgroup

 [private] instance
 () end_pspgroup (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspgroup {
  my ($this,$tag) = @_;

  # forget which group we have.
  my $group = pop @{$this->{stack_group}} or throw
    Error::Simple("<$tag> used outside of GROUP context");
  $this->pop_handlers();
  $this->script_mode(0);
  $this->pop_code_sub();

  # transfer any declarations into the fieldspace.
  if (my $decl = $this->pop_decl()) {
    $group->{declaration} = $decl;
  }

  # possibly print verbose message.
  $this->{verbose} and print " Group '$group->{name}' defined\n";
}

=head2 begin_pspdynamicdisplay

 [private] instance
 () begin_pspdynamicdisplay (string $tag, \%attrs, \@atrseq, string $orig)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspdynamicdisplay {
  my ($this,$tag,$attr,$attr_seq,$orig_txt) = @_;

  my $fieldspace = $this->fieldspace();
  my $gname = $attr->{name} or throw
    Error::Simple("<$tag> requires NAME attribute.");
  my $group_def = $fieldspace->{group_defs}->{$gname} or 
    $this->log_exception("There is no group defined for $gname.");
  $this->{dyndisp}->{$gname} and throw
    Error::Simple("<$tag> used in nested $gname \U$tag\E context.");

  push @{$this->{stack_ddisplay}}, $gname;

  my $dyndisp = $this->{dyndisp}->{$gname} = {}; 
  for my $p (qw(grp num obj)) {
    $dyndisp->{$p."var"} = 
      ($attr->{$p."var"} || $group_def->{$p."var"} || '$_'.$p.'_'.$gname);
  }

  # note, index effectively starts at 1

  $this->code("${gname}_setup:");
  $this->begin_pspblock("setup($gname)");
  $this->code("my (".$dyndisp->{numvar}.",".$dyndisp->{objvar}.");");
  $this->code("my ".$dyndisp->{grpvar}." = \$fs->group('$gname');");
  if (my $num = $attr->{numdisplay}) {
    $this->code($dyndisp->{grpvar}."->{propagated_controls} or");
    $this->code("  ".$dyndisp->{grpvar}."->n_items_per_page($num);");
  }
  $this->code($dyndisp->{grpvar}."->set_cursor(".
	      $dyndisp->{grpvar}."->first_item_n());");
  $this->code("\$fs->errors_p() or");
  $this->code("  ".$dyndisp->{grpvar}."->import_controls(\$cgi);");

  $this->code("${gname}_loop:");
  $this->code("while (".$dyndisp->{grpvar}."->more_to_come())");
  $this->begin_pspblock("while($gname->more_to_come)");
  $this->code($dyndisp->{numvar}." = ".$dyndisp->{grpvar}."->cursor();");
  $this->code($dyndisp->{objvar}." = ".$dyndisp->{grpvar}."->object();");
# $this->code("print STDERR \"top of $gname loop: ".
#	      "\\".$dyndisp->{numvar}."='".$dyndisp->{numvar}."'\\n\";");
}

=head2 end_pspdynamicdisplay

 [private] instance
 () end_pspdynamicdisplay (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspdynamicdisplay {
  my ($this,$tag) = @_;

  my $fieldspace = $this->fieldspace();
  my $gname = pop @{$this->{stack_ddisplay}} or throw
    Error::Simple("<$tag> used outside of GROUP context");
  my $group_def = $fieldspace->{group_defs}->{$gname} or 
    $this->log_exception("There is no group defined for $gname.");
  my $dyndisp = delete $this->{dyndisp}->{$gname} or throw
    Error::Simple("<$tag> used outside of $gname \U$tag\E context.");

  $this->end_pspblock("while($gname->more_to_come)");
  $this->code("continue");
  $this->begin_pspblock("continue($gname->more_to_come)");
  $this->code($dyndisp->{grpvar}."->advance_cursor();");
  $this->end_pspblock("continue($gname->more_to_come)");

  $this->end_pspblock("setup($gname)");
}

=head2 begin_pspdo

 [private] instance
 () begin_pspdo (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspdo {
  my ($this,$tag,$attr,$attr_seq,$orig_txt) = @_;
  $this->debug_line($orig_txt);

  # defer to dynamic display
  $this->begin_pspdynamicdisplay($tag,$attr,$attr_seq,$orig_txt);

  if (my $do_field = $attr->{field}) {
    $this->{current_do_field} = $do_field;
    $this->begin_pspfield($tag,{name => $do_field});
  }

  $this->script_mode();
  $this->push_handlers( {'psp:include' => \&begin_psp_include},
			{'psp:do' => \&end_pspdo});
  $this->{is_cdata}++;
}

=head2 end_pspdo

 [private] instance
 () end_pspdo (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_pspdo {
  my ($this,$tag) = @_;
  $this->debug_line($tag);

  undef $this->{is_cdata};
  $this->code("\n");
  $this->pop_handlers();
  $this->script_mode(0);

  if (my $do_field = delete $this->{current_do_field}) {
    $this->end_pspfield($tag);
  }

  $this->end_pspdynamicdisplay($tag);
}

sub begin_pspdefine_group {
  my ($this,$tag,$attr,$attr_seq,$orig_txt) = @_;
  $this->debug_line($orig_txt);
  my $group = $this->{stack_group}->[-1] or throw
    Error::Simple("Internal error: begin_pspdefine_group without a group?!?");
  my $name = $attr->{name} or throw
    Error::Simple("<$tag> requires NAME attribute.");
  my $fieldspace = $this->fieldspace();

  # defer to the normal DEFINE operation..
  $this->begin_pspdefine($tag,$attr,$attr_seq,$orig_txt);

  # the normal operation should have created a field def.
  my $field_def = $fieldspace->{field_defs}->{$name} or throw
    Error::Simple("Internal error: $name field not defined?!?");

  # note the group of this field in the field def.
  $field_def->{group} = $group->{name};
  push @{$group->{field_names}}, $name;
}

sub end_pspdefine_group {
  my ($this,$tag) = @_;
  $this->debug_line($tag);

  # simply defer to the normal DEFINE operation..
  $this->end_pspdefine($tag);
}

=head2 begin_pspindexoverride

 [private] instance
 () begin_pspindexxoverride (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspindexoverride {
  my ($this, $tag, $attr) = @_;

  my $fieldspace = $this->fieldspace();
  my $gname = $attr->{group} or throw
    Error::Simple("<$tag> requires GROUP attribute.");
  my $group = $fieldspace->{group_defs}->{$gname} or 
    $this->log_exception("There is no group defined for $gname.");

  my $test     = defined $attr->{test} ? $attr->{test} : '1';
  my $previous = bool_att($attr->{previous});
  my $steps    = $attr->{steps};
  my $number   = $group->{number};
  (!$previous and !($steps and $number)) and throw
    Error::Simple("<$tag>: PREVIOUS attribute requires STEP");

  $this->code("if ($test) {");
  if ($previous) {
    $this->code("  \$_dd_index_$gname = \$_prev_dd_index_$gname;");
  } elsif ($steps) {
    $this->code("  \$_dd_index_$gname += \$number * $steps;");
  }
  $this->code("  \$cgi->param('_dd_index_$gname', \$_dd_index_$gname);");

  #$this->fs_init_code($fsname);

  $this->code('}');
}

=head2 begin_psprollto

 [private] instance
 () begin_psprollto (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_psprollto  {
  my ($this,$tag,$attr,$attr_seq,$orig_txt) = @_;
  $this->debug_line($orig_txt);

  my $fieldspace = $this->fieldspace();
  my $gname = $attr->{group} or throw
    Error::Simple("<$tag> requires GROUP attribute.");
  my $group = $fieldspace->{group_defs}->{$gname} or throw
    Error::Simple("GROUP $gname does not exist in ".ref($fieldspace));
  my $name = $attr->{name} || "$gname:rollto";
  my $text = $attr->{text} || "Go to page -->";

  $attr->{style} and
    $this->log_exception("STYLE attribute not yet supported.");
  $attr->{param} and
    $this->log_exception("PARAM attribute deprecated.");

  # generate page code.
  $this->begin_pspblock("rollto($gname)");
  $this->code("my \$_grp = \$fs->group('$gname');");
  my $qname = quote_bareword($name);
  my $qtext = quote_bareword($text);
  $this->code("\$out->put(\$_grp->html_page_select($qtext,$qname));");
  $this->end_pspblock("rollto($gname)");

  # prepare the call for and call pspsubmit
  my ($submit_attr,$submit_seq) = ({},[]);
  $submit_attr->{name} = $name;
  push @$submit_seq, 'name';
  $submit_attr->{rlike} = "^\\d+\$";
  push @$submit_seq, 'rlike';
  for my $n (qw(verify errorpage goto status)) {
    next unless defined $attr->{$n};
    $submit_attr->{$n} = $attr->{$n};
    push @$submit_seq, $n;
  }
  $this->begin_pspsubmit($tag,$submit_attr,$submit_seq,$orig_txt);
}

=head2 end_psprollto

 [private] instance
 () end_psprollto (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_psprollto {
  my ($this,$orig_txt) = @_;
  return $this->end_pspsubmit($orig_txt);
}

=head2 begin_psprollover

 [private] instance
 () begin_psprollover (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_psprollover {
  my ($this,$tag,$attr,$attr_seq,$orig_txt) = @_;
  $this->debug_line($orig_txt);

  my $fieldspace = $this->fieldspace();
  my $gname = $attr->{group} or
    throw Error::Simple("<$tag> requires GROUP attribute.");
  my $group = $fieldspace->{group_defs}->{$gname} or throw
    Error::Simple("GROUP $gname does not exist in ".ref($fieldspace));
  my $name = $attr->{name} || "$gname:rollover";
  my $text = $attr->{text} || "Next >>";

  $attr->{style} and
    $this->log_exception("STYLE attribute not yet supported.");
  $attr->{param} and
    $this->log_exception("PARAM attribute deprecated.");

  $this->begin_pspblock("rollover($gname)");
  $this->code("my \$_grp = \$fs->group('$gname');");
  $this->code('if ($_grp->{dummy_ok} or $_grp->page_n() < $_grp->n_pages())');
  $this->begin_pspblock("index($gname)");
  my $qname = quote_bareword($name);
  my $qtext = quote_bareword($text);
  $this->code("\$out->put(\$_grp->html_next_page_button($qtext,$qname));");
  $this->end_pspblock("index($gname)");
  $this->end_pspblock("rollover($gname)");

  # prepare the call for and call pspsubmit
  my ($submit_attr,$submit_seq) = ({},[]);
  $submit_attr->{name} = $name;
  push @$submit_seq, 'name';
  $submit_attr->{value} = $text;
  push @$submit_seq, 'value';
  for my $n (qw(verify errorpage goto status)) {
    next unless defined $attr->{$n};
    $submit_attr->{$n} = $attr->{$n};
    push @$submit_seq, $n;
  }
  $this->begin_pspsubmit($tag,$submit_attr,$submit_seq,$orig_txt);
}

=head2 end_psprollover

 [private] instance
 () end_psprollover (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_psprollover {
  my ($this,$orig_txt) = @_;
  return $this->end_pspsubmit($orig_txt);
}

=head2 begin_psprollback

 [private] instance
 () begin_psprollback (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_psprollback {
  my ($this,$tag,$attr,$attr_seq,$orig_txt) = @_;
  $this->debug_line($orig_txt);

  my $fieldspace = $this->fieldspace() or throw
    Error::Simple("<$tag> called outside of FIELDSPACE context.");
  my $gname = $attr->{group} or throw
    Error::Simple("<$tag> requires GROUP attribute.");
  my $group = $fieldspace->{group_defs}->{$gname} or throw
    Error::Simple("GROUP $gname does not exist in ".ref($fieldspace));
  my $name = $attr->{name} || "$gname:rollback";
  my $text = $attr->{text} || "<< Previous";

  $attr->{style} and
    $this->log_exception("STYLE attribute not yet supported.");
  $attr->{param} and
    $this->log_exception("PARAM attribute deprecated.");

  $this->begin_pspblock("rollback($gname)");
  $this->code("my \$_grp = \$fs->group('$gname');");
  $this->code("if (\$_grp->page_n() > 1)");
  $this->begin_pspblock("index($gname)");
  my $qname = quote_bareword($name);
  my $qtext = quote_bareword($text);
  $this->code("\$out->put(\$_grp->html_prev_page_button($qtext,$qname));");
  $this->end_pspblock("index($gname)");
  $this->end_pspblock("rollback($gname)");

  # prepare the call for and call pspsubmit
  my ($submit_attr,$submit_seq) = ({},[]);
  $submit_attr->{name} = $name;
  push @$submit_seq, 'name';
  $submit_attr->{value} = $text;
  push @$submit_seq, 'value';
  for my $n (qw(verify errorpage goto status)) {
    next unless defined $attr->{$n};
    $submit_attr->{$n} = $attr->{$n};
    push @$submit_seq, $n;
  }
  $this->begin_pspsubmit($tag,$submit_attr,$submit_seq,$orig_txt);
}

=head2 end_psprollback

 [private] instance
 () end_psprollback (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_psprollback {
  my ($this,$orig_txt) = @_;
  return $this->end_pspsubmit($orig_txt);
}

=head2 begin_psprefresh

 [private] instance
 (void) begin_psprefresh ($tag, $attr)

DESCRIPTION:

This function handles the REFRESH tag. See PSP specification for
behavior and effect.

=cut

sub begin_psprefresh {
  my ($this,$tag,$attr,$attr_seq,$orig_txt) = @_;
  $this->debug_line($orig_txt);

  my $fieldspace = $this->fieldspace();
  my $gname = $attr->{group} or
    throw Error::Simple("<$tag> requires GROUP attribute.");
  my $group = $fieldspace->{group_defs}->{$gname} or throw
    Error::Simple("GROUP $gname does not exist in ".ref($fieldspace));
  my $name = $attr->{name} || "$gname:refresh";
  my $text = $attr->{text} || "Refresh";

  $attr->{style} and
    $this->log_exception("STYLE attribute not yet supported.");
  $attr->{param} and
    $this->log_exception("PARAM attribute deprecated.");

  $this->begin_pspblock("refresh($gname)");
  $this->code("my \$_grp = \$fs->group('$gname');");
  my $qname = quote_bareword($name);
  my $qtext = quote_bareword($text);
  $this->code("\$out->put(\$_grp->html_refresh_button($qtext,$qname));");
  $this->end_pspblock("refresh($gname)");

  # prepare the call for and call pspsubmit
  my ($submit_attr,$submit_seq) = ({},[]);
  $submit_attr->{name} = $name;
  push @$submit_seq, 'name';
  $submit_attr->{value} = $text;
  push @$submit_seq, 'value';
  for my $n (qw(verify errorpage goto status)) {
    next unless defined $attr->{$n};
    $submit_attr->{$n} = $attr->{$n};
    push @$submit_seq, $n;
  }
  $this->begin_pspsubmit($tag,$submit_attr,$submit_seq,$orig_txt);
}

=head2 end_psprefresh

 [private] instance
 () end_psprefresh (string $tag)

DESCRIPTION:

See PSP specification.

=cut

sub end_psprefresh {
  my ($this,$orig_txt) = @_;
  return $this->end_pspsubmit($orig_txt);
}

=head2 process_groups

 [private] instance
 () process_groups()

DESCRIPTION:

See PSP specification.

=cut

sub process_groups {
  my ($fieldspace) = @_;

  my $group_defs = $fieldspace->{group_defs} || {};

  my $out = "";
  for my $gname (sort keys %$group_defs) {
    my $group = $group_defs->{$gname};
    $out .= process_group($fieldspace,$group);
  }
  return $out;
}

sub process_group {
  my ($fieldspace,$group) = @_;
  my $fsname    = $fieldspace->{fullfsname};
  my $fspackage = $fieldspace->{package};
  my $gname     = $group->{name};
  my $gpackage  = $group->{package};

  my $out = "";

#  '  $this->{setup}    = "'.$group->{setup}.'";',

  $out .= ("#" x 78)."\n";
  $out .= "package $gpackage;\n\n";

  # collect any modules we need to use.
  my $isa = $group->{isa} || "PSP::FieldSpace::Group";
  my @modules = ("strict","Error=:try",$isa);
  push @modules, $group->{use} if $group->{use};

  # use those modules, and define inheritance relationships.
  for my $str1 (@modules) {
    for my $str2 (split /,+/, $str1) {
      my @s = split /=/, $str2 or next;
      $out .= "use ".(shift @s);
      $out .= " qw(@s)" if @s;
      $out .= ";\n";
    }
  }
  $out .= "use vars qw(\@ISA);\n";
  $out .= "BEGIN { \@ISA = qw($isa) }\n\n";

  # initialize the list of shared varables.
  (my $share_vars = $fieldspace->{share_vars}||"") =~ s/,/ /g;
  $share_vars and $share_vars .= " ";
  $share_vars .= '$cgi';
  $out .= "# shared variables:\n";
  $out .= "use vars qw($share_vars);\n\n";

  # print any declaration code.
  if ($group->{declaration}) {
    $out .= ("#" x 78)."\n";
    $out .= $group->{declaration}."\n\n";
  }

  # Begin the constructor.
  $out .= join("\n",
	(("#" x 78),
	 "sub new {",
	 '  my ($proto,@args) = @_;',
	 '  my $this = $proto->SUPER::new(@args);',
	 "  \$this->share([qw($share_vars)]);",
	 '  return $this;',
	 '}'
	))."\n\n";

  # generate the group setup.
  $out .= join("\n",
	(("#" x 78),
	 "sub setup {",
	 "  my (\$this,$group->{numvar}) = \@_;",
	 "  my $group->{objvar};",
	 "  ".$group->{setup},
	 "  return $group->{objvar};",
	 "}"
	))."\n\n";

  return $out;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
