#ifndef VISUAL_MOUSEOBJECT_H
#define VISUAL_MOUSEOBJECT_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "cvisual.h"
#include "displaylist.h"
#include <queue>
#include <utility>
#include <bitset>
#include <boost/shared_ptr.hpp>

namespace visual {

/* This common base class implements common functionality for clickObject
 * and mouseObject.  It should never be used directly.
 * 
 */
class mousebase
{
 protected:
 	std::string button_name( int);
 	std::bitset<3> buttonstate;
	std::bitset<5> eventtype;
 
 public:
	mousebase() : buttons(0) {}
	int buttons;
	vector position;
	vector cam;
	vector ray;
	// The object nearest the cursor when this event happened.
	boost::shared_ptr<DisplayObject> pick;
	vector pickpos;
	
	/* 'buttonstate' contains the following state flags as defined by 'button'.
	 */
	enum button { ctrl, alt, shift };
	
	/* 'eventtype' contains state flags as defined by 'event'.
	 */
	enum event { press, release, click, drag, drop };	


	inline bool is_press() { return eventtype.test( press); }
	inline bool is_release() { return eventtype.test( release); }
	inline bool is_click() { return eventtype.test( click); }
	inline bool is_drag() { return eventtype.test( drag); }
	inline bool is_drop() { return eventtype.test( drop); }
	std::string get_buttons();
	inline bool is_alt() { return buttonstate.test( alt); }
	inline bool is_shift() { return buttonstate.test( shift); }
	inline bool is_ctrl(){ return buttonstate.test( ctrl); }
	inline vector get_pos() { return position; }
	inline vector get_camera() { return cam; }
	inline vector get_ray(){ return ray; }
	inline vector get_pickpos() { return pickpos; }
	boost::python::object get_pick();
	
	inline void set_shift( bool _shift) { buttonstate.set( shift, _shift); }
	inline void set_ctrl( bool _ctrl) { buttonstate.set( ctrl, _ctrl); }
	inline void set_alt( bool _alt) { buttonstate.set( alt,  _alt); }
	inline void set_press( bool _press) { eventtype.set( press, _press); }
	inline void set_release( bool _release) { eventtype.set( release, _release); }
	inline void set_click( bool _click) { eventtype.set( click, _click); }
	inline void set_drag( bool _drag) { eventtype.set( drag, _drag); }
	inline void set_drop( bool _drop) { eventtype.set( drop, _drop); }
	
	vector py_project1( vector normal, double dist);
	vector py_project2( vector normal, vector point = vector(0,0,0));

	// These functions will return an object constructed from std::string, or None.
	boost::python::object get_press();
	boost::python::object get_release();
	boost::python::object get_click();
	boost::python::object get_drag();
	boost::python::object get_drop();
};

/* Objects of this class represent the state of the mouse at a distinct event:
 * 	either press, release, click, drag, or drop.
 */
class clickObject: public mousebase 
{
 public:
	clickObject(){}
};

/* A class exported to python as the single object display.mouse.
 * All of the python access for data within this class get the present value of
 * the data.
 */
class mouseObject : public mousebase
{
 public:
	mutex mtx;
	// The bool tells whether or not the click was a left click or not.
	std::queue< std::pair< boost::shared_ptr<clickObject>, bool> > clicks;
	int clickCount; // number of queued events which are left clicks

	mouseObject() : clickCount(0) {}
	// Python access data accesses.
	int get_clicked() { mutex::lock L( mtx); return clickCount; }
	int get_events() { mutex::lock L(mtx); return clicks.size(); }
	void clear_events( int i);
    
	// Python access member functions.
	boost::shared_ptr<clickObject> py_getevent();
	boost::shared_ptr<clickObject> py_getclick();
	vector py_project( vector normal, double dist);
	vector py_project( vector normal, vector point = vector(0,0,0));
};

/*
 * A thin wrapper for buffering cursor visibility information between the python loop
 * and the rendering loop.
 */
class cursorObject  
{
 public:
	mutex mtx;

	bool visible; // whether cursor should be visible
	bool last_visible; // previous state of cursor visibility

	inline cursorObject() : visible(true), last_visible(true) {}
	void set_visible( bool vis) { visible = vis; }
	bool get_visible() { mutex::lock L(mtx); return visible; }
};

void ui_object_init_type( void);

} // !namespace visual

#endif // !VISUAL_MOUSEOBJECT_H
