from visual import *
from string import find
from visual.ui import display

# A graph package for plotting a curve, with labeled axes and autoscaling
# Bruce Sherwood, Carnegie Mellon University, begun April 2000

# minmax[xaxis][negaxis], minmax[yaxis][negaxis] refer to absolute values of minimum negative values;
# minmax[xaxis][posaxis], minmax[yaxis][posaxis] to maximum positive values.
# This graph package forces the origin (0,0) to be on screen, at the axes intersection.

usefrac = 0.8 # graph fills this fraction of the window
grey = (0.7,0.7,0.7) # color of axes
tmajor = 10. # length of major tick marks in pixels
tminor = 5. # length of minor tick marks in pixels
border = 10. # border around graph
minorticks = 5 # number of minor tick intervals between major ticks
maxmajorticks = 3 # max number of major ticks (not including 0)
maxminorticks = (maxmajorticks+1)*minorticks # max number of minor ticks (4 between major ticks)
lastgdisplay = None # the most recently created gdisplay
gdotsize = 6.0 # width (diameter if gdot were round) of gdot in pixels
zdepth = 1.0 # z range of scene
dz = 0.02 # offset for plots relative to axes and labels
xaxis = 0
yaxis = 1
negaxis = 0
posaxis = 1
graphfont = 'system'
fontheight = 12 # font point size
charwidth = 8 # approximate character width
     
def labelnum(x): # determine what labels to show, in what format
    if x >= 0.: sign = 1.
    else: sign = -1.
    mantissa, exponent = modf(log10(abs(x)))
    number = 10.**mantissa
    if number < 1.:
        number = 10.*number
        exponent = exponent-1.
    if number >= 7.5: 
        number = 7.5
        marks = [2.5, 5.0, 7.5]
        extra = 1
    elif number >= 5.: 
        number = 5.
        marks = [2.5, 5.0]
        extra = 1
    elif number >= 4.:
        number = 4.
        marks = [2.0, 4.0]
        extra = 0
    elif number >= 3.:
        number = 3.
        marks = [1.0, 2.0, 3.0]
        extra = 0
    elif number >= 2.:
        number = 3.
        marks = [1.0, 2.0]
        extra = 0
    elif number >= 1.5:
        number = 1.5
        marks = [0.5, 1.0, 1.5]
        extra = 1
    else: 
        number = 1.
        marks = [0.5, 1.0]
        extra = 1
    if exponent > 0:
        digits = 0
    else:
        digits = int(-exponent)+extra
    if digits < 3. and exponent <= 3.:
        format = '%0.'+('%s' % digits)+'f'
    else:
        format = '%0.1E'
    return (sign*(array(marks)*10.**exponent)).tolist(), format

def cleaneformat(string): # convert 2.5E-006 to 2.5E-6; 2.5E+006 to 2.5E6
    index = find(string, 'E')
    if index == -1: return string # not E format
    index = index+1
    if string[index] == '-':
        index = index+1
    elif string[index] == '+':
        string = string[:index]+string[index+1:]
    while index < len(string) and string[index] == '0':
        string = string[:index]+string[index+1:]
    if string[-1] == '-':
	string = string[:-1]
    if string[-1] == 'E':
	string = string[:-1]
    return string

class gdisplay:
    def __init__(self, x=0, y=0, width=800, height=400,
                 title=None, xtitle=None, ytitle=None,
                 xmax=None, xmin=None, ymax=None, ymin=None,
                 foreground=None, background=None):
        global lastgdisplay
        lastgdisplay = self
        test = sphere(visible=0)
        currentdisplay = test.display
        test = None # discard test sphere
        if title is None:
            title = 'Graph'
        self.width = width
        self.height = height
        if foreground is not None:
            self.foreground = foreground
        else:
            self.foreground = color.white
        if background is not None:
            self.background = background
        else:
            self.background = color.black
        self.display = display(title=title, x=x, y=y,
                               width=self.width, height=self.height,
                               uniform=0, fov=0.001, autoscale=0,
                               foreground=self.foreground, background=self.background,
                               userzoom=1, userspin=0)
        self.autoscale = [1, 1]
        self.title = [[xtitle, 0], [ytitle, 0]] # titles not displayed yet
        if xtitle is not None:
            self.xtitlewidth = len(xtitle)*charwidth
        else:
            self.xtitlewidth = 0
        if ytitle is not None:
            self.ytitlewidth = len(ytitle)*charwidth
        else:
            self.ytitlewidth = 0
        # For all axis-related quantities: [x axis 0 or y axis 1][neg axis 0 or pos axis 1]
        self.axis = [[None, None], [None, None]]
        self.zero = [None, None]
        self.lastlabel = [[0., 0.], [0., 0.]]
        self.format = [None, None]
        self.majormarks = [[None, None], [None, None]]
        self.lastminmax = [[0., 0.], [0., 0.]]
        self.minmax = [[0., 0.], [0., 0.]] # [x or y][negative 0 or positive 1]
        if xmax is not None:
            if xmax > 0:
                self.minmax[xaxis][posaxis] = abs(xmax)
                self.autoscale[xaxis] = 0
        if xmin is not None:
            if xmin < 0:
                self.minmax[xaxis][negaxis] = abs(xmin)
                self.autoscale[xaxis] = 0
        if ymax is not None:
            if ymax > 0:
                self.minmax[yaxis][posaxis] = abs(ymax)
                self.autoscale[yaxis] = 0
        if ymin is not None:
            if ymin < 0:
                self.minmax[yaxis][negaxis] = abs(ymin)
                self.autoscale[yaxis] = 0

        self.minorticks = [ [ [], [] ], [ [],[] ] ] # all the minor ticks we'll ever use
        for axis in range(2):
            for axissign in range(2):
                for nn in range(maxminorticks):
                    if axis == xaxis:
                        self.minorticks[axis][axissign].append(label(display=self.display, yoffset=-tminor,
                            linecolor=grey, visible=0, box=0, opacity=0))
                    else:
                        self.minorticks[axis][axissign].append(label(display=self.display, xoffset=-tminor,
                            linecolor=grey, visible=0, box=0, opacity=0))

        self.majorticks = [ [ [], [] ], [ [],[] ] ] # all the major ticks we'll ever use
        for axis in range(2):
            for axissign in range(2):
                for nn in range(maxmajorticks):
                    if axis == xaxis:
                        self.majorticks[axis][axissign].append(label(display=self.display, yoffset=-tmajor, 
                            text='', font=graphfont,
                            height=fontheight, border=0, 
                            linecolor=grey, visible=0, box=0, opacity=0))
                    else:
                        self.majorticks[axis][axissign].append(label(display=self.display, xoffset=-tmajor, 
                            text='', font=graphfont,
                            height=fontheight, border=2,
                            linecolor=grey, visible=0, box=0, opacity=0))
                        
        if self.minmax[xaxis][posaxis]+self.minmax[xaxis][negaxis] == 0:
            self.minmax[xaxis][posaxis] = self.lastminmax[xaxis][posaxis] = 1E-35
        if self.minmax[yaxis][posaxis]+self.minmax[yaxis][negaxis] == 0:
            self.minmax[yaxis][posaxis] = self.lastminmax[yaxis][posaxis] = 1E-35
        for axis in range(2):
            for axissign in range(2):
                self.axisdisplay(axis, axissign)
        self.setrangex()
        self.setrangey()

        currentdisplay.select()

    def setxyparams(self):
        global x0, y0
        if self.minmax[xaxis][posaxis] >= 0 and self.minmax[xaxis][negaxis] >= 0:
            x0 = 0.
        else:
            x0 = self.minmax[xaxis][negaxis]
        if self.minmax[yaxis][posaxis] >= 0 and self.minmax[yaxis][negaxis] >= 0:
            y0 = 0.
        else:
            y0 = self.minmax[yaxis][negaxis]
        
    def axisdisplay(self, axis, axissign):
        if self.minmax[axis][axissign] == 0.: return
        self.setxyparams()
        if axissign == posaxis: sign = 1.
        else: sign = -1.
        if self.axis[axis][axissign] is None: # new; no axis displayed up till now
            if axis == xaxis:
                axispos = [(0,y0,0), (sign*self.minmax[axis][axissign],y0,0)]
                titlepos=(self.minmax[axis][posaxis],y0,0)
            else:
                axispos = [(x0,0,0), (x0,sign*self.minmax[axis][axissign],0)]            
                titlepos=(x0,self.minmax[axis][posaxis],0)
            self.axis[axis][axissign] = curve(pos=axispos, color=grey, display=self.display)
            if (self.title[axis][0] is not None) and (not self.title[axis][1]):
                title = self.title[axis][0]
                self.title[axis][0] = label(display=self.display, pos=titlepos, text=title,
                            font=graphfont, height=fontheight, xoffset=tminor, opacity=0, box=0, line=0)                    
                self.title[axis][1] = 1

            if self.minmax[axis][posaxis] >= self.minmax[axis][negaxis]:
                newmajormarks, format = labelnum(self.minmax[axis][posaxis])
            else:
                newmajormarks, format = labelnum(self.minmax[axis][negaxis])
            if self.format[axis] is None:
                self.format[axis] = format
                
            if self.zero[axis] is None:
                if axis == xaxis:
                    self.zero[axis] = label(display=self.display, pos=(0,0,0), text='0',
                                color=self.foreground,
                                font=graphfont, height=fontheight, border=0,
                                yoffset=-tmajor, linecolor=grey, box=0, opacity=0)
                elif self.minmax[xaxis][negaxis] == 0:
                    self.zero[axis] = label(display=self.display, pos=(0,0,0), text='0',
                                color=self.foreground,
                                font=graphfont, height=fontheight, border=2,
                                xoffset=-tmajor, linecolor=grey, box=0, opacity=0)

            d1 = newmajormarks[0]
            d2 = d1/minorticks
            nminor = 0
            nmajor = 0
            marks = []
            for x1 in newmajormarks:
                if x1 > self.minmax[axis][axissign]: break # newmajormarks can refer to opposite half-axis
                marks.append(x1)
                obj = self.majorticks[axis][axissign][nmajor]
                obj.text = cleaneformat(self.format[axis] % (sign*x1))
                obj.color = self.foreground
                obj.visible = 1
                if axis == xaxis:
                    obj.pos = (sign*x1,y0,0)
                    obj.yoffset = -tmajor 
                else:
                    obj.pos= (x0,sign*x1,0)
                    obj.xoffset = -tmajor
                nmajor = nmajor+1

            nminor = 0
            for x2 in arange(0.+d2, self.minmax[axis][axissign]+1.01*d2, d2):
                if x2 > self.minmax[axis][axissign]+0.01*d2: break
                if x2 % d1 < d2/2.: continue # don't put minor tick where there is a major one
                obj = self.minorticks[axis][axissign][nminor]
                obj.visible = 1
                if axis == xaxis:
                    obj.pos = (sign*x2,y0,0)
                else:
                    obj.pos = (x0,sign*x2,0)
                nminor = nminor+1
                    
            if marks <> []:
                self.majormarks[axis][axissign] = marks
                self.lastlabel[axis][axissign] = self.majormarks[axis][axissign][-1]
            else:
                self.lastlabel[axis][axissign] = 0

        else:
            if (self.zero[axis] is not None) and (axis == yaxis) and (self.minmax[xaxis][negaxis] <> 0):
                    self.zero[axis].visible = 0 # don't show y=0 label if there is a negative-x axis
                    self.zero[axis] = None

            if axis == xaxis:
                self.axis[axis][axissign].pos[1] = vector(sign*self.minmax[axis][axissign], y0, 0)
            else:
                self.axis[axis][axissign].pos[1] = vector(x0,sign*self.minmax[axis][axissign],0)
                
            if self.title[axis][1] and axissign == posaxis:
                if axis == xaxis:
                    titlepos=(self.minmax[axis][posaxis],y0,0)
                else:
                    titlepos=(x0,self.minmax[axis][axissign],0) 
                self.title[axis][0].pos = titlepos

            if (self.majormarks[axis][axissign] is not None) and (len(self.majormarks[axis][axissign]) > 0):
                # this axis already has major tick marks/labels
                oldd1 = self.majormarks[axis][axissign][0]
            else:
                oldd1 = 0.
            oldd2 = oldd1/minorticks
            if self.minmax[axis][posaxis] >= self.minmax[axis][negaxis]:
                newmajormarks, format = labelnum(self.minmax[axis][posaxis])
            else:
                newmajormarks, format = labelnum(self.minmax[axis][negaxis])
            while newmajormarks[-1] > self.minmax[axis][axissign]:
                if len(newmajormarks) == 1: break
                del newmajormarks[-1]
            d1 = newmajormarks[0]
            d2 = d1/minorticks
                
            newformat = (format <> self.format[axis])
            self.format[axis] = format
            needminor = (self.minmax[axis][axissign] >= self.lastlabel[axis][axissign]+d2) or (d2 <> oldd2)
            needmajor = ((self.majormarks[axis][axissign] is None)
                        or (newmajormarks[-1] <> self.majormarks[axis][axissign][-1]) or newformat)
                    
            if needmajor: # need new labels
                start = 0
                if (self.majormarks[axis][axissign] is None) or newformat or (d1 <> oldd1):
                    marks = []
                else:
                    for num in newmajormarks:
                        if num > self.majormarks[axis][axissign][-1]:
                            start = num
                            break
                    marks = self.majormarks[axis][axissign]
                for nmajor in range(maxmajorticks):
                    x1 = (nmajor+1)*d1
                    obj = self.majorticks[axis][axissign][nmajor]
                    if x1 > self.minmax[axis][axissign]:
                        obj.visible = 0
                        continue
                    if x1 < start: continue
                    marks.append(x1)
                    obj.text = cleaneformat(self.format[axis] % (sign*x1))
                    obj.color = self.foreground
                    obj.visible = 1
                    if axis == xaxis:
                        obj.pos = (sign*x1,y0,0)
                    else:
                        obj.pos =(x0,sign*x1,0)

                if marks <> []:
                    self.majormarks[axis][axissign] = marks
                        
            if needminor: # adjust minor tick marks
                nminor = 0
                for x2 in arange(0.+d2, self.minmax[axis][axissign]+1.01*d2, d2):
                    if x2 > self.minmax[axis][axissign]+0.01*d2: break
                    if x2 % d1 < d2/2.: continue # don't put minor tick where there is a major one
                    if (d2 <> oldd2) or x2 > self.lastlabel[axis][axissign]-0.01*d2:
                        if axis == xaxis:
                            self.minorticks[axis][axissign][nminor].pos = (sign*x2,y0,0)
                        else:
                            self.minorticks[axis][axissign][nminor].pos = (x0,sign*x2,0)
                        self.minorticks[axis][axissign][nminor].visible = 1
                    nminor = nminor+1
                while nminor < maxminorticks:
                    self.minorticks[axis][axissign][nminor].visible = 0
                    nminor = nminor+1

            self.lastlabel[axis][axissign] = d2*int(self.minmax[axis][axissign]/d2)
            
    def setrangexy(self):
        # initial estimate of per-pixel horizontal scaling
        x1 = -self.minmax[xaxis][negaxis] # kept as positive number, need actual negative value here
        x2 = self.minmax[xaxis][posaxis]
        xperpixel = (x2-x1)/(0.9*self.width) # approximate
        maxright = max(self.ytitlewidth, x2/xperpixel+self.xtitlewidth,
                     x2/xperpixel+charwidth*len(self.majorticks[xaxis][posaxis][0].text)/2.)
        xpixright = maxright-x2/xperpixel+2.*tmajor
        ylabelwidth = tmajor+charwidth*max(len(self.majorticks[yaxis][negaxis][0].text),
                                    len(self.majorticks[yaxis][posaxis][0].text))
        maxleft = max(ylabelwidth, -x1/xperpixel+charwidth*len(self.majorticks[xaxis][negaxis][0].text)/2.)
        xpixleft = maxleft+x1/xperpixel+2.*tmajor
        xpixels = self.width-xpixleft-xpixright
        xperpixel = (x2-x1)/xpixels
        
        # initial estimate of per-pixel vertical scaling
        y1 = -self.minmax[yaxis][negaxis] # kept as positive number, need actual negative value here
        y2 = self.minmax[yaxis][posaxis]
        yperpixel = (y2-y1)/(0.9*self.height) # approximate
        ypixtop = 2.*tmajor
        if self.ytitlewidth > 0:
            ypixtop = ypixtop+fontheight
        if 3.*tmajor+fontheight > -y1/yperpixel:
            ypixbottom = 3.*tmajor+fontheight
        else:
            ypixbottom = 2.*tmajor
        ypixels = self.height-ypixtop-ypixbottom
        yperpixel = (y2-y1)/ypixels
        
        self.display.range = (xperpixel*self.width/2.,yperpixel*self.height/2.,1.)
        self.display.center = ((x1+x2+(xpixright-xpixleft)*xperpixel)/2.,
                               (y1+y2+(ypixtop-ypixbottom)*yperpixel)/2.)
        # visual criterion for resizing -- change bounds by more than tmajor, size of major tick mark
        xleft = -x1+tmajor*xperpixel # if to the left of xleft more than tmajor, resize
        xright = x2+tmajor*xperpixel # if to the right of xright more than tmajor, resize
        ybottom = -y1+tmajor*yperpixel # if below ybottom more than tmajor, resize
        ytop = y2+tmajor*yperpixel # if above ytop more than tmajor, resize
        return (xleft, xright, ybottom, ytop)

    def setrangex(self):
        (xleft, xright, ybottom, ytop) = self.setrangexy()
        self.xleft = xleft
        self.xright = xright
        
    def setrangey(self):
        (xleft, xright, ybottom, ytop) = self.setrangexy()
        self.ybottom = ybottom
        self.ytop = ytop
       
    def resize(self, x, y):
        if self.autoscale[xaxis]:
            redo = 0
            x = float(x) # avoid integer division problems
            if x > self.lastminmax[xaxis][posaxis]:
                self.minmax[xaxis][posaxis] = x
                if (self.lastminmax[xaxis][posaxis] == 0 or
                        (x >= self.xright)):
                    redo = 1
            elif -x > self.lastminmax[xaxis][negaxis]:
                self.minmax[xaxis][negaxis] = -x
                if (self.lastminmax[xaxis][negaxis] == 0 or
                        (-x >= self.xleft)):
                    redo = 1
            if redo:
                self.setrangex()
                for axissign in range(2):
                    self.axisdisplay(xaxis,axissign)
                    self.lastminmax[xaxis][axissign] = self.minmax[xaxis][axissign]
                    
        if self.autoscale[yaxis]:
            redo = 0
            y = float(y) # avoid integer division problems
            if y > self.lastminmax[yaxis][posaxis]:
                self.minmax[yaxis][posaxis] = y
                if (self.lastminmax[yaxis][posaxis] == 0 or
                        (y >= self.ytop)):
                    redo = 1
            elif -y > self.lastminmax[yaxis][negaxis]:
                self.minmax[yaxis][negaxis] = -y
                if (self.lastminmax[yaxis][negaxis] == 0 or
                        (-y >= self.ybottom)):
                    redo = 1
            if redo:
                self.setrangey()
                for axissign in range(2):
                    self.axisdisplay(yaxis,axissign)
                    self.lastminmax[yaxis][axissign] = self.minmax[yaxis][axissign]

def getgdisplay():
    return gdisplay()

def constructorargs(obj,arguments):
    if arguments.has_key('gdisplay'):
        obj.gdisplay = arguments['gdisplay']
    else:
        if lastgdisplay is None:
            obj.gdisplay = getgdisplay()
        else:
            obj.gdisplay = lastgdisplay
    if arguments.has_key('delta'):
        obj.delta = arguments['delta']
    else:
        obj.delta = 1.
    if arguments.has_key('color'):
        obj.color = arguments['color']
    else:
        obj.color = obj.gdisplay.foreground
    if arguments.has_key('pos'):
        return arguments['pos']
    else:
        return None

def primitiveargs(obj,arguments):
        if arguments.has_key('color'):
            obj.color = arguments['color']
        if arguments.has_key('pos'):
            return arguments['pos']
        else:
            raise SyntaxError

class gcurve:
    def __init__(self, **args):
        pos = constructorargs(self,args)
        self.gcurve = curve(display=self.gdisplay.display, color=self.color)
        if pos is not None:
            for xy in pos:
                self.plot(pos=xy)

    def plot(self, **args):
        pos = primitiveargs(self,args)
        self.gdisplay.resize(pos[0], pos[1])
        self.gcurve.append(pos=(pos[0],pos[1],dz), color=self.color)
                    
class gdots:
    def __init__(self, **args):
        pos = constructorargs(self,args)
        self.dots = [] # eventually should permit changing visible for all dots
        if pos is not None:
            for xy in pos:
                self.plot(pos=xy)

    def plot(self, **args):
        pos = primitiveargs(self,args)
        self.gdisplay.resize(pos[0], pos[1])
        self.dots.append(label(display=self.gdisplay.display,
              pos=(pos[0],pos[1],dz), color=self.color,
              text='O', height=5, opacity=0, box=0, line=0))

class gvbars:
    def __init__(self, **args):
        self.vbars = []
        pos = constructorargs(self,args)
        if pos is not None:
            for xy in pos:
                self.plot(pos=xy)

    def plot(self, **args):
        pos = primitiveargs(self,args)
        self.gdisplay.resize(pos[0], pos[1])
        self.vbars.append(box(display=self.gdisplay.display,
                pos=(pos[0],pos[1]/2.,-dz), size=(self.delta,pos[1],dz), color=self.color))

class ghbars:
    def __init__(self, **args):
        self.hbars = []
        pos = constructorargs(self,args)
        if pos is not None:
            for xy in pos:
                self.plot(pos=xy)

    def plot(self, **args):
        pos = primitiveargs(self,args)
        self.gdisplay.resize(pos[0], pos[1])
        self.hbars.append(box(display=self.gdisplay.display,
                pos=(pos[0]/2.,pos[1],-dz), size=(pos[0],self.delta,dz), color=self.color))

# For backwards compatibility:
class gvbar(gvbars):
    def __init__(self, **args):
        print "New version: you should change 'gvbar' to 'gvbars'"
        apply(gvbars.__init__(self,),args)

class ghbar(ghbars):
    def __init__(self, **args):
        print "New version: you should change 'ghbar' to 'ghbars'"
        apply(ghbars.__init__(self,),args)

class gdot(gdots):
    def __init__(self, **args):
        print "New version: you should change 'gdot' to 'gdots'"
        apply(gdots.__init__(self,),args)

class ghistogram:
    def __init__(self, bins=None, accumulate=0, average=0,
                 delta=None, gdisplay=None, color=None):
        if gdisplay is None:
            if lastgdisplay is None:
                gdisplay = getgdisplay()
            else:
                gdisplay = lastgdisplay
        self.gdisplay = gdisplay
        self.bins = bins
        self.nhist = 0 # number of calls to plot routine
        self.accumulate = accumulate # add successive data sets
        self.average = average # display accumulated histogram divided by self.nhist
        if color is None:
            self.color = self.gdisplay.foreground
        else:
            self.color = color
        self.histaccum = zeros(len(bins))
        if delta is None:
            self.delta = (bins[1]-bins[0])
        else:
            self.delta = delta
        self.vbars = []

    def plot(self, data=None, accumulate=None, average=None, color=None):
        if color is None:
            color = self.color
        if accumulate is not None:
            self.accumulate = accumulate
        if average is not None:
            self.average = average
        n = searchsorted(sort(data), self.bins)
        n = concatenate([n, [len(data)]])
        histo = n[1:]-n[:-1]
        if self.accumulate:
            self.histaccum = self.histaccum+histo
        else:
            self.histaccum = histo
        self.nhist = self.nhist+1.
        ymax = max(self.histaccum)
        if ymax == 0.: ymax == 1.
        self.gdisplay.resize(self.bins[-1],ymax)
        for nbin in range(len(self.bins)):
            x = (nbin+0.5)*self.delta
            y = self.histaccum[nbin]
            if self.nhist == 1.:
                self.vbars.append(box(display=self.gdisplay.display, pos=(x,y/2.,-dz),
                    size=(0.8*self.delta,y,dz), color=color))
            else:
                if self.accumulate and self.average: 
                    y = y/self.nhist
                self.vbars[nbin].y = y/2.
                self.vbars[nbin].height = y

if __name__ == '__main__':
    from time import clock
    # If xmax, xmin, ymax, or ymin specified, the related axis is not autoscaled
    # Can turn off autoscaling with
    #    oscillation.autoscale[0]=0 for x or oscillation.autoscale[1]=0 for y
    
    oscillation = gdisplay(title='Test Plotting', xtitle='Time', ytitle='Response')
##        xmax=70, xmin=-30, ymax=20, ymin=-6)

    funct1 = gcurve(color=color.cyan)
    funct2 = gvbars(color=color.red, delta=0.8)
    funct3 = gdots(color=color.yellow)

    tt = clock()
    pos = []
    for t in arange(-30, 76, 1):
        funct1.plot(pos=(t, 5.0+5.0*cos(-0.2*t)*exp(0.015*t)) )
        funct2.plot(pos=(t, 2.0+5.0*cos(-0.1*t)*exp(0.015*t)) )
        funct3.plot(pos=(t, 5.0*cos(-0.03*t)*exp(0.015*t)) )

    tt = clock()-tt
    print '%0.1f sec' % tt

