/***************************************************************************
  hopmodel.cpp
  -------------------
  Hop model
  -------------------
  Copyright 2006, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <QMessageBox>
#include <QPair>

#include "data.h"
#include "hopmodel.h"

const int extra = 1; // always display one extra blank row

//////////////////////////////////////////////////////////////////////////////
// HopModel()
// ------------
// Constructor

HopModel::HopModel(QObject *parent, HopList *list)
    : QAbstractTableModel(parent), list_(list)
{}

HopModel::~HopModel() {}

//////////////////////////////////////////////////////////////////////////////
// data()
// ------
// Return data at index

QVariant HopModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid()) return QVariant();
    if (index.row() >= list_->count()) return QVariant();

    // row is the entry in the QList
    const Hop &hop = list_->at(index.row());

    // column is the hop "field"
    if (role == Qt::DisplayRole) {
        switch (index.column()) {
          case Hop::NAME:
              return hop.name();
          case Hop::QUANTITY:
              return hop.weight().toString(2);
          case Hop::ALPHA:
              return QString::number(hop.alpha(), 'f', 1) + '%';
          case Hop::TIME:
              return QString::number(hop.time()) + " min";
          case Hop::FORM:
              return hop.form();
          default:
              return QVariant();
        }
    } else if (role == Qt::EditRole) {
        switch (index.column()) {
          case Hop::NAME:
              return hop.name();
          case Hop::QUANTITY: {
              // return converted quantity
              Weight weight = hop.weight();
              weight.convert(Data::instance()->defaultHopUnit());
              return weight.amount();
          }
          case Hop::ALPHA:
              return hop.alpha();
          case Hop::TIME:
              return hop.time();
          case Hop::FORM:
              return hop.form();
          default:
              return QVariant();
        }
    } else if (role == Qt::TextAlignmentRole) {
        switch (index.column()) {
          case Hop::NAME:
          case Hop::FORM:
              return Qt::AlignLeft;
          case Hop::QUANTITY:
          case Hop::ALPHA:
          case Hop::TIME:
          default:
              return Qt::AlignRight;
        }
    } else {
        return QVariant();
    }
}

//////////////////////////////////////////////////////////////////////////////
// setData()
// ---------
// Set data at index

bool HopModel::setData(const QModelIndex &index,
                       const QVariant &value, int role)
{
    static bool deleting = false;

    Hop hop;
    QString name;
    Data *data = Data::instance();
    int row = index.row();
    int column = index.column();

    if (!index.isValid()) return false;
    if (role != Qt::EditRole) return false;
    if ((row >= list_->count())
        && (column != Hop::NAME)) return false;

    // grab existing hop
    if (row < list_->count()) hop = list_->value(row);

    switch (column) {
      case Hop::NAME:
          // editing name as several special cases
          name = value.toString();

          // deleting name deletes hop
          if (name.isEmpty()) {
              if (row >= list_->count()) return false; // already blank

              // TODO: for some reason this gets entered recursively...
              if (deleting) return false;
              deleting = true;

              // remove hop
              beginRemoveRows(index.parent(), row, row);

              list_->removeAt(row);
              emit modified();
              endRemoveRows();

              deleting = false;
              return true;
          }

          // setting name on blank row adds hop
          if (row >= list_->count()) {
              if (data->hasHop(name)) {
                  hop = data->hop(name);
              } else {
                  hop.setName(name);
              }

              beginInsertRows(QModelIndex(), row, row);
              list_->append(hop);
              emit modified();
              endInsertRows();

              return true;
          }

          // changed name
          hop.setName(name);
          break;

      case Hop::QUANTITY:
          hop.setWeight(Weight(value.toDouble(),
                               Data::instance()->defaultHopUnit()));
          break;

      case Hop::ALPHA:
          hop.setAlpha(value.toDouble());
          break;

      case Hop::TIME:
          hop.setTime(value.toUInt());
          break;

      case Hop::FORM:
          hop.setForm(value.toString());
          break;

      default:
          return false;
    }

    list_->replace(row, hop);
    emit modified();

    // whole row may have changed
    emit dataChanged(index.sibling(row, Hop::NAME),
                     index.sibling(row, Hop::TIME));

    return true;
}

//////////////////////////////////////////////////////////////////////////////
// headerData()
// ------------
// Return header information

QVariant HopModel::headerData(int section, Qt::Orientation orientation,
                                int role) const
{
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
        switch (section) {
          case Hop::NAME:
              return "Hop";
          case Hop::QUANTITY:
              return "Quantity";
          case Hop::ALPHA:
              return "Alpha";
          case Hop::TIME:
              return "Time";
          case Hop::FORM:
              return "Form";
          default:
              return QVariant();
        }
    }

    return QVariant();
}

//////////////////////////////////////////////////////////////////////////////
// flags()
// ------
// Return flags at index

Qt::ItemFlags HopModel::flags(const QModelIndex &index) const
{
    if (!index.isValid()) return Qt::ItemIsEnabled;

    return QAbstractTableModel::flags(index) | Qt::ItemIsEditable;
}

//////////////////////////////////////////////////////////////////////////////
// rowCount()
// ----------
// Return number of rows of data

int HopModel::rowCount(const QModelIndex &) const
{
    return list_->count() + extra;
}

//////////////////////////////////////////////////////////////////////////////
// columnCount()
// -------------
// Return number of columns of data

int HopModel::columnCount(const QModelIndex &) const
{
    return Hop::COUNT;
}

//////////////////////////////////////////////////////////////////////////////
// sort()
// ------
// Sort by column

void HopModel::sort(int column, Qt::SortOrder order)
{
    QList<QPair<QString,Hop> > sortlist;

    foreach(Hop hop, *list_) {
        QString field;
        switch (column) {
          case Hop::NAME:
              field = hop.name();
              break;
          case Hop::QUANTITY:
              field = QString::number(hop.weight().amount()).rightJustified(6,'0');
              break;
          case Hop::ALPHA:
              field = QString::number(hop.alpha(),'f',1).rightJustified(6,'0');
              break;
          case Hop::FORM:
              field = hop.form();
              break;
          case Hop::TIME:
          default:
              field = QString::number(hop.time()).rightJustified(6,'0');
              break;
        }
        sortlist.append(QPair<QString,Hop>(field, hop));
    }

    // sort list
    qSort(sortlist.begin(), sortlist.end());

    // create new list
    list_->clear();
    QPair<QString,Hop> pair;
    foreach(pair, sortlist) {
        if (order == Qt::AscendingOrder)
            list_->append(pair.second);
        else
            list_->prepend(pair.second);
    }

    emit layoutChanged();
}
