/* main program for quicklist */

/* Current owner:  Bob */ 

/*  QuickList  
 *  www.quicklist.org for more information.
 *  Copyright (C) 1999
 *  Author of this code file:  Robert Lissner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 */

#include "includes.h"

/* This would be a global variable */
FILE* f;
whole_file *front = 0;
whole_file *back [MAX_FILES]; /* Pointers to the non focused files */
char dialog_mode; 
gint16 file_ct = 0;
gint curr_year;
char* cb [MAX_CLIPBOARD]; /* address of Glist for clipboard */
char cb_sel_type = '\0';
gint cb_rows = 0;
gint16 cb_cols = 0;
GtkWidget *dialog1_win = NULL; 
whole_file  *dialog1_file = NULL; 
GtkWidget *dialog2_win = NULL; 
whole_file *dialog2_file = NULL; 
char close_mode; 
gint file_type; 
char blockme [] = "blockme";
char last_path_name [128] = " ";

/* prototypes */
gint qls2d_fix_year (gint year);


static char*  months [] = {
  "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};


int main (int argc, char** argv) {
  gint i;
  struct tm *tm;
  time_t t;

  /* I18N */
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
  
  t = time (NULL);
  tm = localtime (&t);
  curr_year = tm->tm_year + 1900;
  
  gtk_init (&argc, &argv);
  for (i=0; i < MAX_FILES; i++)
    back [i] = 0;  /* clear out the table.  Memset could do if I knew how */
  if (argc >= 2 && argv [1]) {
    file_type = 1;
    actually_open (argv [1]);
  }
  maybe_create_initial_db (); 
  gtk_main ();
  return 0;
} /* end of main */  


/* Cleanly and neatly get rid of a window */
void clean_destroy (GtkWidget **win) {
  if (*win) {
    gtk_widget_destroy (*win);
    *win = NULL;
  }
}


/* This routine destroys level2 dialog box if it's around */
void destroy_dialog2 (void) {
  clean_destroy (&dialog2_win);
  dialog2_file = NULL;
}  

/* This routine destroys all dialog boxes if they're around */
void destroy_dialog () {
  destroy_dialog2 ();
  clean_destroy (&dialog1_win);
  dialog1_file = NULL;
}

/* This callback results from:
   1.  Any action on a level1 error dialog box
   2.  Cancel or destroy on any level 1 dialog box */
gint cancel_level1 (GtkWidget *widget, GdkEvent *event, gpointer data ) {
  destroy_dialog ();
  maybe_create_initial_db ();
  return (TRUE); /* Don't cancel app */
}

 /* This callback results from:
   1.  Any action on a level2 error dialog box
   2.  Cancel or destroy on any level 2 dialog box */
gint cancel_level2 ( GtkWidget *widget, GdkEvent  *event, gpointer data ) {
  destroy_dialog2 ();
  return (TRUE);
}
/* ________________________________
  |                                |
  |      add1row                   |
  |________________________________|*/
void add1row (void) {
  gtk_sheet_add_row ( GTK_SHEET (front->sheet), 1);
  gtk_sheet_row_button_add_label (GTK_SHEET (front->sheet), 
				  ++front->last_row, " "); 
  front_is_changed ();
}

void big_draw_start () {
  if (front->display_mode == 'L') {
    gtk_signal_handler_block_by_data (GTK_OBJECT (front->sheet), blockme);
    gtk_sheet_freeze (GTK_SHEET (front->sheet));
  }
  else 
    gtk_sheet_freeze (GTK_SHEET (front->report_sheet));
}

void big_draw_end () {
  if (front->display_mode == 'L') {
    gtk_sheet_thaw (GTK_SHEET (front->sheet));
    gtk_signal_handler_unblock_by_data (GTK_OBJECT (front->sheet), blockme); 
  }
  else 
    gtk_sheet_thaw (GTK_SHEET (front->report_sheet));
}

/* ________________________________
  |                                |
  |      check_entry               |
  |________________________________|*/
gint check_entry (char* entry) {
  gint length;
  char error_message [] = 
    "Text must contain valid characters and not contain \\ * or %";
  char button [] = "Go back";
  if (!entry)
    return (0);
  length = strlen (entry);
  if (!length)
    return (0);
  while (length && entry [length -1] == ' ')
    entry [--length] = '\0';
  if (length && !strpbrk (entry, "\\*%"))
    return (0);
  level2_error (error_message, button);
  return (-1);
}



/* |--------------------------------|
   |     check_if_changed           |
   |--------------------------------|*/
gint check_if_changed () {
  gint fieldx;
  double temp_double;
  char linebuf [40];
  char* newtext;

  if (!front)
    return (FALSE);
  destroy_dialog ();
  if (front->activate_row < 0 || front->activate_col < 0) /* initial vals */
    return (FALSE);

  newtext = gtk_sheet_cell_get_text (GTK_SHEET (front->sheet),
				     front->activate_row,
				     front->activate_col);

  if (!front->activate_text && !newtext)
    return (FALSE); /* both zero */
  if (front->activate_text && newtext) 
    if (!(strcmp (newtext, front->activate_text)))
      return (FALSE);

  front_is_changed (); /* entries clearly don't match */ 
  if (!newtext) 
    return (FALSE);

  /* check that input is valid in non-text fields */
  fieldx = front->col_to_field [front->activate_col];
  if (front->fields [fieldx].type != FIELD_TYPE_TEXT) {
    temp_double = qls2d (newtext, front->fields [fieldx].type,
			 front->fields [fieldx].formatting);
   
    if (temp_double >= HUGE_VAL) {
      level2_error ("      The cell contents are not valid.       ",
		    "  Please try again  ");
      return (TRUE);
    }
    d2qls (linebuf, temp_double, front->fields [fieldx].type,
	   front->fields [fieldx].formatting,
	   front->fields [fieldx].decimal_places);
    gtk_sheet_set_cell_text (GTK_SHEET (front->sheet), 
			     front->activate_row, 
			     front->activate_col, linebuf);
  }

  return (FALSE);
} /* end of check_if_changed () */

/* ________________________________
  |                                |
  |      connect_signals           |
  |________________________________|
*/
void connect_signals () {
  gtk_signal_connect (GTK_OBJECT (front->list_win), "focus-in-event",  
		      GTK_SIGNAL_FUNC (focus_in_event_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (front->list_win), "delete_event",
		      GTK_SIGNAL_FUNC (delete_on_front), NULL);
  gtk_signal_connect(GTK_OBJECT( front->sheet),
		     "activate",
		     GTK_SIGNAL_FUNC(activate_callback), blockme);

  gtk_signal_connect(GTK_OBJECT( front->sheet),
		     "new_column_width",
		     GTK_SIGNAL_FUNC(new_column_width), blockme); 
  gtk_signal_connect(GTK_OBJECT( front->sheet),
		     "select_range",
		     GTK_SIGNAL_FUNC(select_range_callback), blockme);
 } /* end of connect_signals */

/* ________________________________
  |                                |
  |        d2qls                   |
  |________________________________|*/
gint d2qls (char* texto, double input, gint type, gint formatting,
	    gint dec_places) {


  /* these are for numeric conversions */
  gboolean is_negative;
  char use_for_currency;
  gboolean is_percent;
  char use_for_comma;
  char use_for_decimal;
  char texti [40];
  gint inx, outx;
  double temp_val;
  gint digits_so_far;

  /* these are for date conversion */
  char date_format [20];
  char* date_formatting [] = {
  "M /D0/Y2",
  "M /D0/Y4",
  "M .D0.Y2",
  "M .D0.Y4",
  "MA D  Y4",
  "Y4-M0-D0",
  "D  MA Y4",
  "D /M0/Y2",
  "D /M0/Y4" };
  gint year, month, day;
  gint written;
  char* outptr;

  /* these are for time conversions */
  gint hour;
  gint min;
  
  temp_val = input;
  if (temp_val == HUGE_VAL) {
    strcpy (texto, "######");
    return (6);
  }
  switch (type) {
  case FIELD_TYPE_NUMERIC:
    is_negative = FALSE;
    is_percent = FALSE;
    use_for_currency = '\0';
    use_for_comma = '\0';
    use_for_decimal = '.';
    digits_so_far = 0;
    
    if (temp_val < 0)
      is_negative = TRUE;
    switch (formatting) {
    case 0:
      use_for_comma = ',';
      break;
    case 1:
      use_for_comma = '.';
      use_for_decimal = ',';
      break;
    case 2:
      break;
    case 3:
      use_for_decimal = ',';
      break;
    case 4:
      use_for_currency = '$';
      use_for_comma = ',';
      break;
    case 5:
      use_for_comma = ',';
      is_percent = TRUE;
      break;
    default: /* this would be case 6 */
      use_for_comma = '.';
      use_for_decimal = ',';
      is_percent = TRUE;
    }
    
    if (is_percent)
      temp_val = temp_val * 100;
    switch (dec_places) {
    case 0: 
      sprintf (texti, "%.0f", temp_val);
      break;
    case 1: 
      sprintf (texti, "%.1f", temp_val);
      break;
    case 2: 
      sprintf (texti, "%.2f", temp_val);
      break;
    case 3: 
      sprintf (texti, "%.3f", temp_val);
      break;
    case 4: 
      sprintf (texti, "%.4f", temp_val);
      break;
    case 5: 
      sprintf (texti, "%.5f", temp_val);
      break;
    case 6: 
      sprintf (texti, "%.6f", temp_val);
      break;
    case 7: 
      sprintf (texti, "%.7f", temp_val);
      break;    
    case 8: 
      sprintf (texti, "%.8f", temp_val);
      break;  
    default: /* this would be case 9 */
      sprintf (texti, "%.9f", temp_val);
      break;
    }  
    
    /* now count the digits before the decimal place or the end */
    if (use_for_comma) 
      for (inx = 0; inx < 40; inx++) {
	if (texti [inx] >= '0' && texti [inx] <= '9')
	  digits_so_far++;
	else if (!texti [inx] || texti [inx] == '.')
	  break;
      }
    
    /* have counted places in front of decimal.  Now set up counter of how
       many digits to skip before inserting a comma */
    digits_so_far %= 3;
    if (!digits_so_far)
      digits_so_far = 3;
    
    /* start formatting output */
    outx = 0;
    if (use_for_currency) {
      if (is_negative)
	texto [outx++] = '(';
      texto [outx++] = use_for_currency;
    }
    else if (is_negative)
      texto [outx++] = '-';
    for (inx = 0; inx < 40; inx++) {
      if (texti [inx] >= '0' && texti [inx] <= '9') {
	if (use_for_comma && !digits_so_far) {
	  texto [outx++] = use_for_comma;
	  digits_so_far = 3;
	}
	texto [outx++] = texti [inx];
	digits_so_far--;
      }
      else if (texti [inx] == '.') {
	use_for_comma = '\0'; /* turn off commas past decimal point */
	texto [outx++] = use_for_decimal;
      }
      else if (!texti [inx])
	break;
    }
    
    /* finished moving everything.  Put percent or minus on end */
    if (use_for_currency && is_negative)
      texto [outx++] = ')';
    if (is_percent)
      texto [outx++] = '%';
    texto [outx] = '\0';
    return (outx);
    break; /* end of FIELD_TYPE_NUMERIC */

  case FIELD_TYPE_DATE:
    strcpy (date_format, date_formatting [formatting]);
    year = (gint) temp_val / 10000;
    month = (gint) temp_val /100 % 100;
    day = (gint) temp_val % 100;
    inx = 0;
    outptr = texto;
    
    while (date_format [inx]) {
      if (date_format [inx] == 'M') {
	if (date_format [inx+1] == 'A') {
	  strcpy (outptr, months [month]);
	  outptr += 3;
	}
	else {
	  if (month < 10) 
	  *outptr++ = date_format [inx+1];
	  sprintf (outptr, "%u%n", month, &written);
	  outptr += written;
	}  
	inx += 2;
      }
      else if (date_format [inx] == 'D') {
	if (day  < 10) 
	  *outptr++ = date_format [inx+1];
	sprintf (outptr, "%u%n", day, &written);
	outptr += written;
	inx += 2;
      }
      else if (date_format [inx] == 'Y') {
	if (date_format [inx+1] == '2')
	  year = year % 100;
	if (year  < 10) 
	  *outptr++ = '0';
	sprintf (outptr, "%u%n", year, &written);
	outptr += written;
	inx += 2;
      }
      else
	*outptr++ = date_format [inx++];
    }
    *outptr = '\0';
    return (outptr - texto);
    break;

  default: /* that would be FIELD_TYPE_TIME */
    hour = temp_val / 60;
    min = (gint) temp_val % 60;
    if (!formatting) { /* 0 is 7:15 PM */
      if (!hour)
	hour = 12;
      if (hour > 12)
	hour -= 12;
      if (temp_val > 719) /* half a day */ 
	return (sprintf (texto, "%u:%u%u PM", hour, min/10, min%10));
      else
	return (sprintf (texto, "%u:%u%u AM", hour, min/10, min%10));
    }
  
    if (formatting == 1)
      return (sprintf (texto, "%u%u:%u%u", hour/10, hour%10, min/10, min%10));
    
    /* all that remains is 18.00 */
    return (sprintf (texto, "%u.%u%u", hour, min/10, min%10));
    break;
  }
  return (0);
} /* end of d2qls */
      
      
/* ________________________________
  |                                |
  |      focus-in-event            |
  |________________________________|*/
gint focus_in_event_cb (GtkWidget  *win, GdkEvent *name, gpointer *value) { 
gint i;

 if (front)
   if (win == front->list_win ||
       win == front->report_win) {
     return (TRUE);
   }
 if (win == dialog1_win) { /* watchout for initial dialog */
   front = dialog1_file;
   return (TRUE);
 }
 if (win == dialog2_win) {
   front = dialog2_file;
   return (TRUE);
 }

 /* It must be another file's main window.  Go find it */
  for (i = 0; i < MAX_FILES; i++)
    if (back [i])
      if ( win == back [i]->list_win ||
	   win == back [i]->report_win) {
	front = back [i];
	dim_all_menus();
	return (TRUE);
    }
  return (TRUE); 
}

/* ________________________________
  |                                |
  |     front_is_changed           |
  |________________________________|
 */
void front_is_changed () {
   if (front->changed)
    return;
  front->changed = TRUE;
  dim_list_file_menu (); 
}
 
/* |--------------------------------|
   |    get_rid_of_front            |
   |--------------------------------|*/
void get_rid_of_front () {
  gint16 i;

  /* if dialog windows belong to this file, get rid of them */
  if (dialog1_file == front || dialog2_file == front)
    destroy_dialog ();
  
  for (i=0; i < MAX_FILES; i++)  /* remove from table */
    if (back [i] == front) {
      back [i] = 0;
      break;
    }
  if (front->list_win) { /* may not be if creating new file */
    big_draw_start (); /* kills off a bunch of unwanted signals */
    clean_destroy (&front->list_win);
  }
  clean_destroy (&front->report_win);
  if (front->file_path) {
    g_mem_check(front->file_path);
    g_free (front->file_path);
  }
  if (front->activate_text)
    g_free (front->activate_text);
  g_free  (front);
  front = 0;
  file_ct--;

  /* now see if some other file is available */
  if (file_ct)
    for (i=0; i < MAX_FILES; i++)
    if (back [i]) {
      front = back [i];
      break;
    }
} /* End of get rid of front */


/* ________________________________
  |                                |
  |    get_window_size_loc         |
  |________________________________|
  Save the x and y and height and width of window */
void get_window_size_loc (GtkWidget *win) {
  gint x, y;
  
  front->width = win->allocation.width;
  front->height = win->allocation.height;   
  gdk_window_get_deskrelative_origin (win->window, &x, &y);
  front->x = x;
  front->y = y;
}
  
/* ________________________________
  |                                |
  |      level1-error              |
  |________________________________|
 */
void level1_error (char* message, char* button){
  GtkWidget  *the_button;
  GtkWidget *label;
  
  make_basic_dialog1 ();
  gtk_signal_connect (GTK_OBJECT  (dialog1_win),
		      "delete_event",
		      (GtkSignalFunc) cancel_level1, NULL);
  gtk_window_set_title (GTK_WINDOW (dialog1_win), 
			_("For your information"));
  label = gtk_label_new (message);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog1_win)->vbox),
		      label, TRUE, TRUE, 0);
  gtk_widget_show (label);  
  the_button = gtk_button_new_with_label (button);
  gtk_widget_show (the_button);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog1_win)->action_area),
		      the_button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT(the_button),
		      "clicked", 
		      GTK_SIGNAL_FUNC (cancel_level1), NULL);
  GTK_WIDGET_SET_FLAGS (the_button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (the_button);
  gtk_widget_show(dialog1_win);  
  gtk_grab_add (dialog1_win);

} /* End of level1_error */

/* ________________________________
  |                                |
  |      level2-error              |
  |________________________________|
 */
/* Error dialog box when there's already a dialog on the screen, and
   it has to stay */
void level2_error (char* message, char* button) {
  GtkWidget  *the_button;
  GtkWidget *label;
  
  destroy_dialog2 (); /* get rid of any existing dialog box */
  dialog2_win = gtk_dialog_new ();
  dialog2_file = front; 
  gtk_signal_connect (GTK_OBJECT (dialog2_win), "focus-in-event",  
		      GTK_SIGNAL_FUNC (focus_in_event_cb), NULL); 
  gtk_window_position (GTK_WINDOW (dialog2_win), GTK_WIN_POS_CENTER); 
  gtk_window_set_title (GTK_WINDOW (dialog2_win), 
			_("For your information"));
  gtk_window_set_policy (GTK_WINDOW (dialog2_win), FALSE, FALSE, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog2_win), 5);
  label = gtk_label_new (message);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog2_win)->vbox),
		      label, TRUE, TRUE, 0);
  gtk_widget_show (label);  
  the_button = gtk_button_new_with_label (button); 

  gtk_widget_show (the_button);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog2_win)->action_area),
		      the_button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT(the_button),
		      "clicked", 
		      GTK_SIGNAL_FUNC (cancel_level2), NULL);
  gtk_signal_connect (GTK_OBJECT (dialog2_win), 
		      "delete_event",
		      GTK_SIGNAL_FUNC (cancel_level2), NULL);
  GTK_WIDGET_SET_FLAGS (the_button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (the_button);
  gtk_widget_show(dialog2_win);
  gtk_grab_add (dialog2_win);
} /* level2_error */


/* ________________________________
  |                                |
  |     make_basic_dialog1         |
  |________________________________|
 */
void make_basic_dialog1 () {
  destroy_dialog ();
  dialog1_win = gtk_dialog_new ();
  gtk_window_set_modal (GTK_WINDOW (dialog1_win), TRUE);
  dialog1_file = front; /* might be zero */
  gtk_window_position (GTK_WINDOW (dialog1_win), GTK_WIN_POS_CENTER); 
  gtk_window_set_policy (GTK_WINDOW (dialog1_win), FALSE, FALSE, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog1_win), 5);
  gtk_signal_connect (GTK_OBJECT (dialog1_win), "focus-in-event",  
		      GTK_SIGNAL_FUNC (focus_in_event_cb), NULL); 
}

/* ________________________________
  |                                |
  |       qls2d                    |
  |________________________________|
  converts a quicklist string (1,234.56) to double.  Have to check for
  \0 and \n at end of line because of reading input files */
double qls2d (char* input, gint type, gint formatting) {
  gboolean got_pct;
  char linebuf [24];
  gint inx, x;
  gint outx;
  char sep3;
  char decimal;
  gboolean negatory;
  double temp_double;
  gboolean got_digit;
  gboolean got_space_after_digit;

  /* these are for time conversion */
  char before_char [15];
  gint beforex;
  char after_char [15];
  gint afterx;
  gboolean got_sep;
  gboolean got_pm;
  gboolean got_am;

  /* these are for date conversion */
  gint temp_val;
  gint value0, value1, value2;
  gboolean alpha1;
  gint level;
  const gint16 days_in_month [13] = {0, 31, 29, 31, 30, 31, 30, 31, 31, 
    30, 31, 30, 31};

  if (!input || !input [0])
    return (HUGE_VAL);
  for (inx = 0; input [inx] == ' '; inx++);
  if (!input [inx])
    return (HUGE_VAL);

  if (type == FIELD_TYPE_NUMERIC) {
    outx = 0;
    sep3 = ',';
    decimal = '.';
    got_digit = FALSE;
    got_space_after_digit = FALSE;
    got_pct = negatory = FALSE;
    if (formatting == 1 || formatting == 3 || formatting == 6) {
      sep3 = '.';
      decimal = ',';
    }
    for (; inx < 24; inx++) {
      if (input [inx] >= '0' && input [inx] <= '9') {
	got_digit = TRUE;
	if (got_space_after_digit) /* prevent "14 10" for example */
	  return (HUGE_VAL);
	linebuf [outx++] = input [inx];
      }
      else if (input [inx] < ' ') /* end of string */
	break;
      else if (input [inx] == decimal) 
	linebuf [outx++] = '.';
      else if (input [inx] == '-' 
	       || input [inx] == '('
	       || input [inx] == ')') 
	negatory = TRUE;
      else if (input [inx] == sep3)
	continue;
      else if (input [inx] == '$' 
	       || input [inx] == ' ') {
	if (got_digit)
	  got_space_after_digit = TRUE;
      }
      else if (input [inx] == '+' && !got_digit); /* ignore + if early */
      else if (input [inx] == '%')
	got_pct = TRUE;
      else return (HUGE_VAL); /* unacceptable character */
    }
    
    if (!outx || outx > 20)
      return (HUGE_VAL);
    linebuf [outx] = '\0';
    temp_double = strtod (linebuf, NULL);
    if (got_pct)
      temp_double /= 100;
    if (negatory)
      temp_double = -temp_double;
    return (temp_double);
  }

  else if (type == FIELD_TYPE_DATE) {
    value0 = value1 = value2 = -1;
    alpha1  = FALSE;
    level = 0;
    while (inx < 24) {
      if (input [inx] < ' ')
	break;
      if (isalpha (input [inx])) {
	if (level >= 2) /* month not allowed as third field */
	  return (HUGE_VAL);
	linebuf [0] = toupper (input [inx++]);
	outx = 1;
	while (isalpha (input [inx])) { 
	  if (outx > 10)
	    return (HUGE_VAL);
	  linebuf [outx++] = tolower (input [inx++]);
	}
	if (outx < 3)
	  return (HUGE_VAL);
	temp_val = 0;
	for (x = 1; x <= 12; x++)
	  if (months [x] [0] == linebuf [0] &&
	      months [x] [1] == linebuf [1] &&
	      months [x] [2] == linebuf [2]) {
		temp_val = x;
		break;
	      }
	if (x >= 13)
	  return (HUGE_VAL);
	if (level == 0)
	  value0 = temp_val;
	else  {
	  value1 = temp_val;
	  alpha1 = TRUE;
	}
	level++;
      }
      else if (isdigit (input [inx])) {
	if (level >= 3) 
	  return (HUGE_VAL);
	linebuf [0] = input [inx++];
	outx = 1;
	while (isdigit (input [inx])) { 
	  if (outx > 4)
	    return (HUGE_VAL);
	  linebuf [outx++] = input [inx++];
	}
	linebuf [outx] = '\0';
	temp_val = atoi (linebuf);
	if (temp_val < 0)
	  return (HUGE_VAL);
	if (level == 0) 
	  value0 = temp_val;
	else if (level == 1)
	  value1 = temp_val;
	else
	  value2 = temp_val;
	level++;
      }
      else if (input [inx] == ' ' ||
	       input [inx] == '/' ||
	       input [inx] == '-' ||
	       input [inx] == '.')
	inx++;
      else
	return (HUGE_VAL);
    }

    /* end of loop.  Now figure out what we have */
    if (value1 < 0) /* automatically checks value0 */
      return (HUGE_VAL);
    if (formatting < 5 && value0 < 13 && value1 <= days_in_month [value0]) {
      value2 = qls2d_fix_year (value2);
      if (!value2)
	return (HUGE_VAL);
      return (value0 * 100 + value1 + value2 * 10000);
    }
    if (formatting == 5 && value1 < 13 && value2 <= days_in_month [value1]) {
      value0 = qls2d_fix_year (value0);
      if (!value0)
	return (HUGE_VAL);
      return (value1 * 100 + value2 + value0 * 10000);
    }
    if (formatting > 5 && value1 < 13 && value0 <= days_in_month [value1]) {
      value2 = qls2d_fix_year (value2);
      if (!value2)
	return (HUGE_VAL);
      return (value1 * 100 + value0 + value2 * 10000);
    }
    
    /* didn't match any.  try m/d/y */
       
    if (value0 < 13 && value1 <= days_in_month [value0]) {
      value2 = qls2d_fix_year (value2);
      if (value2)
	return (value0 * 100 + value1 + value2 * 10000);
    }

    /* all that's left is 14 Dec type notation.  */
    if (alpha1 && value0 <= days_in_month [value1]) {
        value2 = qls2d_fix_year (value2);
      if (value2)
	return (value0 + value1 * 100 + value2 * 10000);
    }
   }

  else { /* this would be FIELD_TYPE_TIME */
    beforex = afterx = 0;
    got_sep = got_pm = got_am = FALSE; 
    for (; inx < 15; inx++) {
      if (input [inx] >= '0' && input [inx] <= '9') {
	if (got_sep)
	  after_char [afterx++] = input [inx];
	else
	  before_char [beforex++] = input [inx];
      }
      else if (input [inx] == ':' 
	       || input [inx] == ';' /* for efficient typists */
	       || input [inx] == '.')
	got_sep = TRUE;
      else if (input [inx] == ' ') {
	if (beforex && !afterx)
	  got_sep = TRUE;
      }
      else if ((input [inx] == 'P' || input [inx] == 'p') 
	       && !got_am && !got_pm)
	got_pm = TRUE;
      else if ((input [inx] == 'A' || input [inx] == 'a') 
	       && !got_am && !got_pm) 
	got_am = TRUE;
      else if (input [inx] == 'm' || input [inx] == 'M');
      else if (input [inx] < ' ') /* end of string */
	break;
      else return (HUGE_VAL);  /* invalid characters */
    }
    /* wrap up the time conversion */
    if (!beforex)
      return (HUGE_VAL); /* ie, no data at all */
    before_char [beforex] = '\0';
    temp_double = atof (before_char) * 60;
    if (temp_double == 720 /* 12:00 */
	&& got_am)
      temp_double = 0;
    if (afterx) {
        after_char [afterx] = '\0';
	temp_double += atof (after_char);
    }
    /* for 4:00 pm type formatting, change an entry of 4 to 4:00 pm */
    if (got_pm && temp_double < 720)
      temp_double += 720; /* 720 minutes is half a day */
    if (!got_am && temp_double < 419 && !formatting)
      temp_double += 720; /* if 1:00 to 6:59 make it pm by default */
    if (temp_double < 1440)
      return (temp_double);
    }
  return (HUGE_VAL);
} /* end of qls2d */

gint qls2d_fix_year (gint year) {
  if (year < 0)
    return (curr_year);
  if (year < 20)
    return (year + 2000);
  if (year < 100)
    return (year + 1900);
  if (year < 2500)
    return (year);
  return (0);
}



/* ________________________________
  |                                |
  |     reset_col_to_field         |
  |________________________________|
  Resets the table point at fields from columns */
void reset_col_to_field () {
  gint fieldx;
  gint colx;
  for (fieldx = 0; fieldx <= front->last_field; fieldx++) {
    colx = front->fields [fieldx].sheet_column;
    front->col_to_field [colx] = fieldx;
  }
}


/* ________________________________
  |                                |
  |     set_window_size_loc        |
  |________________________________|
  Force the size of the window we just opened */
void set_window_size_loc (GtkWidget *win) {
  gtk_widget_set_usize (win, front->width, front->height);
  gtk_widget_set_uposition (win, front->x, front->y);
}

void sorry () {
  level1_error ("            That function doesn't work yet             ", 
		"  Go back  ");
}

void unselect () {
  if (front && front->display_mode == 'L' && front->sel_type) {
    front->sel_type = '\0';
    gtk_sheet_unselect_range (GTK_SHEET (front->sheet), &front->sel_range);
  }
}








