/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_kill.c,v 1.30.2.3 2005/01/15 23:53:32 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "modules.h"
#include "xmode.h"
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

Module MOD_HEADER(m_kill) = {
	"m_kill",
	"/KILL command",
	6, "$Revision: 1.30.2.3 $"
};

int MOD_LOAD(m_kill)()
{
	if (register_command(&MOD_HEADER(m_kill), &CMD_KILL, m_kill) == NULL) {
		return MOD_FAILURE;
	}
	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_kill)()
{
	return MOD_SUCCESS;
}

/*
 * m_kill
 *	parv[0] = sender prefix
 *	parv[1] = target
 *	parv[2] = path
 */
int m_kill(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	static char buf[BUFSIZE];
	aClient *acptr;
	char *user, *path, *reason, *p, *nick;
	char mypath[KILLLEN + 1], mymsg[KILLLEN + 1];
	int chasing = 0, kcount = 0;

	if (!IsServer(cptr) && (MyClient(sptr) && !HasMode(sptr, UMODE_OPER))) {
		send_me_numericNA(sptr, ERR_NOPRIVILEGES);
		return 0;
	}
	if (parc < 2 || BadPtr(parv[1])) {
		send_me_numeric(sptr, ERR_NEEDMOREPARAMS, "KILL");
		return 0;
	}

	user = parv[1];
	path = parv[2];

	if (!BadPtr(path)) {
		if (strlen(path) > KILLLEN) {
			path[KILLLEN] = '\0';
		}
	}

	if (MyClient(sptr)) {
		user = canonize(user);
	}

	for (p = NULL, nick = strtoken(&p, user, ","); nick != NULL; nick = strtoken(&p, NULL, ",")) {
		chasing = 0;
		if (MyClient(sptr) || (*nick != '!')) {
			if ((acptr = find_client(nick, NULL)) == NULL) {
				if ((acptr = get_history(nick, (long)KILLCHASETIMELIMIT)) == NULL) {
					target_left(sptr, nick, MSG_KILL, NULL);
					continue;
				}
				send_me_notice(sptr, ":KILL target changed from %s to %s", nick,
					acptr->name);
				chasing = 1;
			}
		}
		else {
			if ((acptr = find_target(nick)) == NULL) {
				send_me_notice(sptr, ":KILL lost in transit - user signoff before "
					"delivery/desynch (ID: %s)", nick);
				continue;
			}
		}

		if (MyClient(cptr) && ((MyConnect(acptr) && !OPHasFlag(cptr, OFLAG_LKILL)) ||
		  (!MyConnect(acptr) && !OPHasFlag(cptr, OFLAG_GKILL)))) {
			send_me_numericNA(sptr, ERR_NOPRIVILEGES);
			continue;
		}
		if (IsServer(acptr) || IsMe(acptr) || (IsULine(acptr) && !IsServer(cptr))) {
			send_me_numericNA(sptr, ERR_CANTKILLSERVER);
			ircdlog(LOG_KILL, "client %s attempted to kill server or super client %s",
				get_client_name(cptr, FALSE), acptr->name);
			continue;
		}

		kcount++;
		if (!IsServer(sptr) && (kcount > GeneralConfig.max_kills)) {
			send_me_notice(sptr, ":Too many targets, kill list truncated (maximum is %d)",
				GeneralConfig.max_kills);
			break;
		}

		if (MyClient(sptr)) {
			char myname[HOSTLEN + 1], *s;

			if (!BadPtr(path)) {
				ircsnprintf(mymsg, KILLLEN + 1, "(%s)", path);
				reason = mymsg;
			}
			else {
				reason = "(No reason specified)";
			}

			strncpy(myname, me.name, HOSTLEN + 1);
			if ((s = strchr(myname, '.')) != NULL) {
				*s = '\0';
			}

			ircsnprintf(mypath, KILLLEN + 1, "%s!%s!%s", myname, MaskedHost(sptr), sptr->name);
		}
		else {
			if (BadPtr(path) || (reason = strchr(path, ' ')) == NULL) {
				path = sptr->name;
				reason = "(No reason specified)";
			}
			else {
				*reason++ = '\0';
			}
			strncpyzt(mypath, path, KILLLEN + 1);
		}

		sendto_realops_lev(SKILL_LEV, "Received KILL message for %s!%s@%s. "
			"From: %s Path: %s %s", acptr->name, acptr->username,
			acptr->user ? MaskedHost(acptr) : "<unknown>", parv[0], mypath, reason);

		if (HasMode(sptr, UMODE_OPER)) {
			ircdlog(LOG_KILL, "KILL for %s!%s@%s. From: %s Path: %s %s", acptr->name,
				*acptr->username != '\0' ? acptr->username : "<unknown>",
				*acptr->host != '\0' ? MaskedHost(acptr) : "<unknown>", parv[0], mypath,
				reason);
		}

		if (!MyConnect(acptr) || !MyConnect(sptr) || !HasMode(sptr, UMODE_OPER)) {
			sendto_serv_kill_msg_butone(cptr, sptr, acptr, ":%s %s", mypath, reason);
			if (chasing && IsServer(cptr)) {
				sendto_one_client_real(cptr, acptr, &me, &CMD_KILL, ":%s %s",
					mypath, reason);
			}
			SetKilled(acptr);
		}
		if (MyConnect(acptr)) {
			sendto_one_client_prefixed(acptr, sptr, &CMD_KILL, ":%s %s", mypath, reason);
		}
		if (MyConnect(acptr) && MyConnect(sptr) && HasMode(sptr, UMODE_OPER)) {
			ircsprintf(buf, "Local kill by %s %s", sptr->name, reason);
		}
		else {
			ircsprintf(buf, "Killed (%s %s)", sptr->name, reason);
		}
		if (exit_client(cptr, acptr, sptr, buf) == FLUSH_BUFFER) {
			return FLUSH_BUFFER;
		}
	}
	return 0;
}
