/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_kline.c,v 1.51.2.2 2005/01/15 23:53:30 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "modules.h"
#include "xmode.h"
#include "user_ban.h"
#include <utmp.h>
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

Module MOD_HEADER(m_kline) = {
	"m_kline",
	"/KLINE command",
	6, "$Revision: 1.51.2.2 $"
};

int MOD_LOAD(m_kline)()
{
	if (register_command(&MOD_HEADER(m_kline), &CMD_KLINE, m_kline) == NULL) {
		return MOD_FAILURE;
	}
	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_kline)()
{
	return MOD_SUCCESS;
}

static char *cluster_host(char *host)
{
	static char result[HOSTLEN + 1];
	char temphost[HOSTLEN + 1], *p, *zap_point = NULL, *tld, *hostmask;
	int is_ipaddr = 1, dot_count = 0;

	if (BadPtr(host)) {
		return NULL;
	}

	if (in_str(host, '@')) {
		strncpyzt(host, result, HOSTLEN + 1);
		return result;
	}

	strncpyzt(temphost, host, HOSTLEN + 1);

	for (p = temphost; *p != '\0'; p++) {
		if (*p == '.') {
			if (++dot_count == 3) {
				zap_point = p;
			}
		}
		else if (!IsDigit(*p)) {
			is_ipaddr = 0;
			break;
		}
	}

	if (is_ipaddr && (dot_count == 3)) {
		zap_point++;
		*zap_point++ = '*';
		*zap_point = '\0'; /* Truncate xxx.xxx.xxx.xxx to xxx.xxx.xxx.* */

		strncpyzt(result, temphost, HOSTLEN + 1);
		return result;
	}

	if ((tld = strrchr(temphost, '.')) != NULL) {
		dot_count = 2;

		if (*(tld + 3) != '\0') { /* 3-char tld */
			dot_count--;
		}
		if (tld != temphost) {
			hostmask = (tld - 1);
		}
		else {
			hostmask = tld;
		}

		while (hostmask != temphost) {
			if (*hostmask == '.') {
				dot_count--;
			}
			if (!dot_count) {
				*result = '*';
				strncpy(result + 1, hostmask, HOSTLEN - 1);
				return result;
			}
			hostmask--;
		}

		*result = '*';
		strncpy(result + 1, temphost, HOSTLEN - 1);
	}
	else {
		strncpyzt(result, temphost, HOSTLEN + 1);
	}

	return result;
}

static int isnumber(char *n)
{
	int result = 0;

	ASSERT(n != NULL);

	while (*n != '\0') {
		if (IsDigit(*n)) {
			result *= 10;
			result += ((*n) & 0xF);
			n++;
		}
		else {
			return -1;
		}
	}

	return result;
}

static void write_kline_conf(userBan *uban, aClient *sptr)
{
	char buf[BUFSIZE * 2];
	int fd = -1, len;

	if ((fd = open(Internal.conf_file, O_RDWR|O_APPEND)) == -1) {
		send_me_notice(sptr, ":Couldn't open config file: %s", strerror(errno));
		return;
	}

	if (!BadPtr(uban->reason)) {
		len = ircsprintf(buf, "// Kill added at %s by %s"LF"kill { mask = \"%s@%s\"; reason = \"%s\"; };"LF,
			smalldate(0), sptr->name, uban->user, uban->host, uban->reason);
	}
	else {
		len = ircsprintf(buf, "// Kill added at %s by %s"LF"kill { mask = \"%s@%s\"; };"LF,
			smalldate(0), sptr->name, uban->user, uban->host);
	}

	if ((len = write(fd, buf, len)) <= 0) {
		send_me_notice(sptr, ":Couldn't write to config file: %s",
			(len == -1) ? strerror(errno) : "write returned zero");
	}

	close(fd);
}

/*
 * m_kline
 *	parv[0] = sender prefix
 *
 *	parc > 3
 *		parv[1] = expire
 *		parv[2] = user@host|nick
 *		parv[3] = reason
 *	parc = 3
 *		parv[1] = user@host|nick
 *		parv[2] = reason
 */
int m_kline(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	userBan *uban, *uban2;
	aClient *acptr;
	char *arg;
	char *user, tempuser[USERLEN + 2]; /* wildcard prefix */
	char *host, temphost[HOSTLEN + 1];
	char *reason = NULL;
	int time_specified = 0;
	time_t expire_mins, expire_secs = 0;
	dlink_node *node, *next = NULL;

	if (!OPHasFlag(sptr, OFLAG_KLINE)) {
		send_me_numericNA(sptr, ERR_NOPRIVILEGES);
		return 0;
	}
	if (parc < 2 || BadPtr(parv[1])) {
		send_me_numeric(sptr, ERR_NEEDMOREPARAMS, "KLINE");
		return 0;
	}

	arg = parv[1];

	if ((expire_mins = isnumber(arg)) >= 0) {
		if (parc < 3) {
			send_me_numeric(sptr, ERR_NEEDMOREPARAMS, "KLINE");
			return 0;
		}

		if (expire_mins > (24 * 60 * 7)) {
			expire_mins = (24 * 60 * 7); /* Max: 1 week */
		}

		expire_secs = (time_t)expire_mins * 60;
		time_specified = 1;

		arg = parv[2];
		parc--;
	}
	else {
		expire_mins = 0;
	}

	if ((host = strchr(arg, '@')) != NULL || ((*arg == '*') || in_str(arg, '.'))) {
		if (host != NULL) {
			*host++ = '\0';
			user = arg;
		}
		else {
			host = arg;
			user = "*";
		}

		strncpyzt(tempuser, user, USERLEN + 2);
		strncpyzt(temphost, host, HOSTLEN + 1);

		user = tempuser;
		host = temphost;

		if (!irccmp(user, "*")) {
			int i = 0;
			char *p;

			for (p = host; *p != '\0'; p++) {
				if ((*p != '*') && (*p != '?') && (*p != '.') && (*p != '/')) {
					i++;
				}
			}

			if (i < 4) {
				send_me_notice(sptr, ":Cannot k-line %s@%s, the mask is too vague.", user, host);
				return 0;
			}
		}
	}
	else {
		if ((acptr = find_chasing(sptr, arg, NULL, MSG_KLINE)) == NULL) {
			return 0;
		}
 		if (!IsPerson(acptr)) {
			send_me_noticeNA(sptr, ":You can only k-line userhosts or clients.");
			return 0;
		}
		if (!MyClient(acptr)) {
			send_me_noticeNA(sptr, ":You cannot k-line remote users.");
			return 0;
		}

		ircsnprintf(tempuser, USERLEN + 1, "*%s", (*acptr->username == '~') ? (acptr->username + 1) : acptr->username);
		user = tempuser;
		host = cluster_host(acptr->host);
	}

	if (time_specified) {
		arg = parv[3];
	}
	else {
		arg = parv[2];

		if (GeneralConfig.default_kline_time) {
			expire_secs = GeneralConfig.default_kline_time;
		}
	}

	if (parc > 2 && !BadPtr(arg)) {
		reason = arg;

		if (strlen(reason) > TOPICLEN) {
			reason[TOPICLEN] = '\0';
		}
	}

	if ((uban = make_userban(user, host, reason, expire_secs)) == NULL) {
		send_me_notice(sptr, ":%s@%s is an invalid k-line mask.", user, host);
		return 0;
	}

	if ((uban2 = find_userban_exact(uban, 0)) != NULL) {
		send_me_notice(sptr, ":%s@%s is already %s: %s", user, uban->host,
			(uban2->flags & BAN_LOCAL) ? LOCAL_BANNED : NETWORK_BANNED, BanReason(uban2));
		userban_free(uban);
		return 0;
	}

	uban->flags |= BAN_LOCAL;

	if (user_match_ban(sptr, uban)) {
		send_me_notice(sptr, ":Aborted k-line on %s@%s, it would affect yourself.", user, uban->host);
		userban_free(uban);
		return 0;
	}

	add_userban(uban);
	DLINK_FOREACH_SAFE_DATA(lclient_list.head, node, next, acptr, aClient) {
		if (!IsKlineExempt(acptr) && user_match_ban(acptr, uban)) {
			if (uban->flags & (UBAN_CIDRBIG|UBAN_CIDR|UBAN_IPV4)) {
				exit_client_zap(acptr, acptr, uban);
			}
			else {
				exit_client_kill(acptr, acptr, uban);
			}
		}
	}

	if (expire_secs) {
		ircdlog(LOG_KILL, "%s added %d min. k-line on %s@%s (%s)", get_client_name(sptr, FALSE),
			(expire_secs / 60), user, uban->host, BanReason(uban));
		sendto_realops("%s added %d min. k-line on %s@%s (%s)", sptr->name,
			(expire_secs / 60), user, uban->host, BanReason(uban));
	}
	else {
		ircdlog(LOG_KILL, "%s added k-line on %s@%s (%s) to config file",
			get_client_name(sptr, FALSE), user, uban->host, BanReason(uban));
		sendto_realops("%s added k-line on %s@%s (%s) to %s", sptr->name, user, uban->host,
			BanReason(uban), Internal.conf_file);

		write_kline_conf(uban, sptr);
	}

	return 0;
}
