/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: s_bsd.c,v 1.186.2.4 2005/05/06 22:57:39 amcwilliam Exp $
 */

#include "memory.h"
#include "setup.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "res.h"
#include "numeric.h"
#include "patchlevel.h"
#include "zlink.h"
#include "ssl.h"
#include "h.h"
#include "fd.h"
#include "hook.h"
#include "sbuf.h"
#include <netinet/tcp.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <utmp.h>
#include <sys/resource.h>

#ifdef INCREASE_SOCK_BUFS
char *readbuf;
int rcvbufmax = 0;
int sndbufmax = 0;
#else
char readbuf[8192];
#endif

const char *ERR_set_non_blocking = "set_non_blocking() failed for %s: %s";
const char *ERR_set_sock_opts = "set_sock_opts() failed for %s: %s";
const char *ERR_unset_sock_ipopts = "unset_sock_ipopts() failed for %s: %s";
const char *ERR_increase_sock_bufs = "increase_sock_bufs() failed for %s: %s";

int get_sockerr(int fd)
{
	int err_no = errno;
#ifdef SO_ERROR
	int err, len = sizeof(err);

	if (fd >= 0) {
		if (!getsockopt(fd, SOL_SOCKET, SO_ERROR, (char *)&err, &len)) {
			if (err) {
				err_no = err;
			}
		}
	}
	errno = err_no;
#endif

	return err_no;
};

void report_error(int fd, const char *text, char *host)
{
	char err_buf[512];
	int err_no = get_sockerr(fd);

	ircsprintf(err_buf, text, (host != NULL) ? host : "-", strerror(err_no));

	Debug((DEBUG_ERROR, "%s", err_buf));
	sendto_realops_lev(DEBUG_LEV, err_buf);
	ircdlog(LOG_ERROR, err_buf);
}

static int match_allow(aClient *cptr)
{
	dlink_node *node;
	ConfigItem_allow *allow;
	char userhost[USERLEN + HOSTLEN + 2], *host = NULL;
	char userip[USERLEN + HOSTIPLEN + 1], *ip = NULL;

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	ircsprintf(userhost, "%s@%s", cptr->username, cptr->host);
	host = cptr->host;

	ircsprintf(userip, "%s@%s", cptr->username, cptr->hostip);
	ip = cptr->hostip;

	DLINK_FOREACH_PREV_DATA(conf_allow_list.tail, node, allow, ConfigItem_allow) {
		if (allow->item.temp || allow->class->item.temp) {
			continue;
		}
		if (allow->port && (cptr->localClient->listener != NULL)
		  && (cptr->localClient->listener->port != allow->port)) {
			continue;
		}

		if (!BadPtr(allow->hostname)
		  && (!match(allow->hostname, (allow->flags & ALLOW_HOSTNAME_AT) ? userhost : host))) {
			Debug((DEBUG_DEBUG, "Allowing %s: hostname %s", cptr->name, allow->hostname));
			return attach_allow(cptr, allow);
		}
		if (!BadPtr(allow->ipaddr)
		  && (!match(allow->ipaddr, (allow->flags & ALLOW_IPADDR_AT) ? userip : ip))) {
			Debug((DEBUG_DEBUG, "Allowing %s: ipaddr %s", cptr->name, allow->ipaddr));
			return attach_allow(cptr, allow);
		}
	}

	return CLIENTAUTH_NOMATCH;
}

int check_client(aClient *cptr)
{
	int auth_state;
	HookData hdata = HOOKDATA_INIT;

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	Debug((DEBUG_DEBUG, "ch_cl: check access for %s[%s@%s]", cptr->name, cptr->username,
		cptr->localClient->sockhost));

	if ((auth_state = match_allow(cptr)) != CLIENTAUTH_MATCHED) {
		Debug((DEBUG_DEBUG, "ch_cl: access denied: %s[%s@%s]", cptr->name, cptr->username,
			cptr->localClient->sockhost));
		return auth_state;
	}
	
	/* Check the h_user_access hook before we attach any confs */
	hdata.sptr = cptr;
	if (hook_run_until(h_user_access, &hdata, FLUSH_BUFFER) == FLUSH_BUFFER) {
		Debug((DEBUG_DEBUG, "ch_cl: access denied by hook: %s[%s@%s]", cptr->name, cptr->username,
			cptr->localClient->sockhost));
		return CLIENTAUTH_NOMATCH;
	}

	Debug((DEBUG_DEBUG, "ch_cl: access ok: %s[%s@%s]", cptr->name, cptr->username,
		cptr->localClient->sockhost));
	attach_class(cptr);
	
	return CLIENTAUTH_MATCHED;
}

void close_connection(aClient *cptr)
{
	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	if (IsServer(cptr)) {
		ircstp->is_sv++;
		ircstp->is_sbs += cptr->localClient->sendB;
		ircstp->is_sbr += cptr->localClient->receiveB;
		ircstp->is_sks += cptr->localClient->sendK;
		ircstp->is_skr += cptr->localClient->receiveK;
		ircstp->is_sti += timeofday - cptr->firsttime;

		if (ircstp->is_sbs > 2047) {
			ircstp->is_sks += (ircstp->is_sbs >> 10);
			ircstp->is_sbs &= 0x3ff;
		}
		if (ircstp->is_sbr > 2047) {
			ircstp->is_skr += (ircstp->is_sbr >> 10);
			ircstp->is_sbr &= 0x3ff;
		}
	} else if (IsClient(cptr)) {
		ircstp->is_cl++;
		ircstp->is_cbs += cptr->localClient->sendB;
		ircstp->is_cbr += cptr->localClient->receiveB;
		ircstp->is_cks += cptr->localClient->sendK;
		ircstp->is_ckr += cptr->localClient->receiveK;
		ircstp->is_cti += timeofday - cptr->firsttime;

		if (ircstp->is_cbs > 2047) {
			ircstp->is_cks += (ircstp->is_cbs >> 10);
			ircstp->is_cbs &= 0x3ff;
		}
		if (ircstp->is_cbr > 2047) {
			ircstp->is_ckr += (ircstp->is_cbr >> 10);
			ircstp->is_cbr &= 0x3ff;
		}
	}
	else {
		ircstp->is_ni++;
	}

	if (IsServer(cptr)) {
		ConfigItem_link *linkp = cptr->serv->conf;

		ASSERT(linkp != NULL);

		if (!linkp->item.temp && (linkp->flags & LINK_AUTOCONNECT)) {
			linkp->next_connect = timeofday;
			linkp->next_connect += ((timeofday - cptr->since) > HANGONGOODLINK) ?
				HANGONRETRYDELAY : TRY_CONNECTIONS_FREQ;
		}
	}

	ASSERT(cptr->localClient->fd > FD_UNUSED);

	if (!DeadSocket(cptr) && !IsBlocked(cptr)) {
		/* zip_out() will be 0 if !USE_ZLIB */
		if (SBufLength(&cptr->localClient->sendQ) || zip_out(cptr)) {
			engine_send_queued(cptr);
		}
	}

#ifdef USE_OPENSSL
	if (IsSecure(cptr)) {
		SSL_set_shutdown(cptr->localClient->ssl, SSL_RECEIVED_SHUTDOWN);
		SSL_smart_shutdown(cptr->localClient->ssl);
		SSL_free(cptr->localClient->ssl);
		cptr->localClient->ssl = NULL;
	}
#endif

	fd_close(cptr->localClient->fd);
	cptr->localClient->fd = FD_UNUSED;

	SBufClear(&cptr->localClient->sendQ);
	SBufClear(&cptr->localClient->recvQ);

	detach_confs(cptr);
	detach_listener(cptr);

	cptr->from = NULL;
}

#ifdef INCREASE_SOCK_BUFS

static int init_sock_buf(int fd, int sock_opt)
{
	int max = 0, s = sizeof(max);

	getsockopt(fd, SOL_SOCKET, sock_opt, (char *)&max, &s);
	while ((max < 16385) && setsockopt(fd, SOL_SOCKET, sock_opt, (char *)&max, s) >= 0) {
		max += 1024;
	}
	getsockopt(fd, SOL_SOCKET, sock_opt, (char *)&max, &s);

	if (sock_opt == SO_RCVBUF) {
		readbuf = (char *)MyMalloc(max * sizeof(char));
	}

	return max;
}

int increase_sock_bufs(int fd, int server)
{
	int s = (server) ? rcvbufmax : CLIENT_BUFFER_SIZE;
	if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (char *)&s, sizeof(s))) {
		return 0;
	}

	s = (server) ? sndbufmax : SEND_BUFFER_SIZE;
	if (setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (char *)&s, sizeof(s))) {
		return 0;
	}
	return 1;
}

#endif

int set_sock_opts(int fd)
{
	int s;

#ifdef SO_REUSEADDR
	s = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&s, sizeof(s))) {
		return 0;
	}
#endif

#if defined(SO_USELOOPBACK) && !defined(OS_CYGWIN)
	s = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_USELOOPBACK, (char *)&s, sizeof(s))) {
		return 0;
	}
#endif

#ifdef SO_RCVBUF
#ifdef INCREASE_SOCK_BUFS
	if (!rcvbufmax) {
		rcvbufmax = init_sock_buf(fd, SO_RCVBUF);
	}
	s = 4096;
#else
	s = 8192;
#endif
	if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (char *)&s, sizeof(s))) {
		return 0;
	}
#endif

#ifdef SO_SNDBUF
#ifdef INCREASE_SOCK_BUFS
	if (!sndbufmax) {
		sndbufmax = init_sock_buf(fd, SO_SNDBUF);
	}
	s = 4096;
#else
	s = 8192;
#endif
	if (setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (char *)&s, sizeof(s))) {
		return 0;
	}
#endif

	return 1;
}

int unset_sock_ipopts(int fd)
{
#if defined(IPPROTO_IP) && defined(IP_OPTIONS)
	if (setsockopt(fd, IPPROTO_IP, IP_OPTIONS, NULL, 0)) {
		return 0;
	}
#endif
	return 1;
}

int set_non_blocking(int fd)
{
	int opt;

#ifdef USE_RTSIGIO
	setup_sigio_fd(fd);
#endif

	if ((opt = fcntl(fd, F_GETFL, 0)) < 0) {
		return 0;
	}

	opt |= O_NONBLOCK;
	if (fcntl(fd, F_SETFL, opt) < 0) {
		return 0;
	}

	return 1;
}

void add_connection(Listener *l, int fd)
{
	aClient *acptr = NULL;
	struct sockaddr_in addr;
	int addrlen = sizeof(struct sockaddr_in);
	char ipaddr[HOSTIPLEN + 1];
#ifdef USE_OPENSSL
	SSL *ssl = NULL;
#endif
	HookData hdata = HOOKDATA_INIT;

	ASSERT(l != NULL);

	if (getpeername(fd, (struct sockaddr *)&addr, &addrlen) == -1) {
		report_error(fd, "Failed to accept connection on %s: %s", get_listener_name(l));
		ircstp->is_ref++;
		fd_close(fd);
		return;
	}

	inetntop(&addr.sin_addr, ipaddr, HOSTIPLEN + 1);

	if (!set_non_blocking(fd)) {
		report_error(fd, ERR_set_non_blocking, ipaddr);
	}
	if (!set_sock_opts(fd)) {
		report_error(fd, ERR_set_sock_opts, ipaddr);
	}
	if (!unset_sock_ipopts(fd)) {
		report_error(fd, ERR_unset_sock_ipopts, ipaddr);
	}

	Count.unknown++;

#ifdef USE_OPENSSL
	if (l->flags & LISTEN_SECURE) {
		if ((ssl = ssl_do_handshake(fd, ipaddr)) == NULL) {
			ircstp->is_ref++;
			fd_close(fd);
			return;
		}
	}
#endif

	hdata.i = fd;
	if (hook_run_until(h_post_accept, &hdata, FLUSH_BUFFER) == FLUSH_BUFFER) {
		ircstp->is_ref++;
		fd_close(fd);

#ifdef USE_OPENSSL
		if (ssl != NULL) {
			SSL_set_shutdown(ssl, SSL_RECEIVED_SHUTDOWN);
			SSL_smart_shutdown(ssl);
			SSL_free(ssl);
		}
#endif
		
		return;
	}

	acptr = make_client(NULL, &me);
	acptr->localClient->fd = fd;

	strncpy(acptr->localClient->sockhost, ipaddr, HOSTLEN + 1);
	strcpy(acptr->host, ipaddr);
	strcpy(acptr->hostip, ipaddr);
	acptr->ip.s_addr = addr.sin_addr.s_addr;

#ifdef USE_OPENSSL
	if ((acptr->localClient->ssl = ssl) != NULL) {
		SetSecure(acptr);
	}
#endif

	attach_listener(acptr, l);
	l->count++;

	connauth_init_client(acptr);
}

static void serv_connect_dns_callback(void *vlink, adns_answer *answer)
{
	ConfigItem_link *linkp = (ConfigItem_link *)vlink;

	ASSERT(linkp != NULL);

	if (answer != NULL && (answer->status == adns_s_ok)) {
		linkp->ip.s_addr = answer->rrs.addr->addr.inet.sin_addr.s_addr;
		MyFree(answer);
	}
	else {
		send_gnotice("Connect to %s[%s].%d failed: DNS lookup failed.",
			linkp->servername, linkp->host, linkp->port);
	}
	if (linkp->dns_query != NULL) {
		MyFree(linkp->dns_query);
		linkp->dns_query = NULL;
	}
	if (linkp->ip.s_addr) {
		serv_connect(linkp, NULL);
	}
}

static inline void serv_connect_exit(aClient *cptr)
{
	if (cptr->localClient->fd > FD_UNUSED) {
		fd_close(cptr->localClient->fd);
		cptr->localClient->fd = FD_UNUSED;
	}
	free_client(cptr);
}

static void serv_connect_callback(int fd, void *data, int engine_status)
{
	aClient *cptr = (aClient *)data;
	ConfigItem_link *linkp = NULL;
	unsigned int my_capabs;

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);
	ASSERT(cptr->serv != NULL);

	ASSERT(cptr->serv->conf != NULL);
	linkp = cptr->serv->conf;

	if (engine_status != ENGINE_OK) {
		engine_connect_error(cptr, engine_status, errno);
		return;
	}

	Count.unknown++;
	SetHandshake(cptr);

	my_capabs = Internal.default_capabs;
#ifdef USE_OPENSSL
	if (linkp->flags & LINK_SECURE) {
		my_capabs |= CAP_DKEY;
	}
#endif

	if (linkp->auth != NULL) {
		sendto_one_client_nopostfix(cptr, NULL, &CMD_PASS, "%s :TS", linkp->auth->string);
	}
	sendto_one_client_nopostfix(cptr, NULL, &CMD_CAPAB, "%s", get_my_cap(my_capabs));
	sendto_one_client_nopostfix(cptr, NULL, &CMD_MYID, "%s", me.id.string);
	sendto_one_client_nopostfix(cptr, NULL, &CMD_SERVER, "%s 1 :%s", me.name, me.info);

	if (DeadSocket(cptr)) {
		engine_connect_error(cptr, ENGINE_ERR_HANDSHAKE, cptr->localClient->sockerr);
	}
	else {
		engine_set_call(cptr->localClient->fd, FDEV_READ, engine_read_packet, cptr, 0);
	}
}

int serv_connect(ConfigItem_link *linkp, aClient *by)
{
	static struct sockaddr_in addr;
	int addrlen = sizeof(struct sockaddr_in);
	char ipaddr[HOSTIPLEN + 1];
	aClient *cptr;

	ASSERT(linkp != NULL);

	if (linkp->dns_query != NULL) {
		Debug((DEBUG_DEBUG, "serv_connect() ignored due to previous DNS lookup"));
		return 0;
	}

	if (has_wilds(linkp->host)) {
		ircdlog(LOG_SERVER, "Connect to %s[%s] failed: hostname with wild cards",
			linkp->servername, linkp->host);
		return 0;
	}
	if (!linkp->ip.s_addr) {
		if (!inetpton(linkp->host, &linkp->ip.s_addr)) {
			Debug((DEBUG_DEBUG, "serv_connect() submitting DNS lookup for %s", linkp->host));
			linkp->dns_query = (DNSQuery *)MyMalloc(sizeof(DNSQuery));
			linkp->dns_query->ptr = linkp;
			linkp->dns_query->callback = serv_connect_dns_callback;
			dns_gethost(linkp->host, linkp->dns_query);
			return 1;
		}
	}

	inetntop(&linkp->ip, ipaddr, HOSTIPLEN + 1);
	Debug((DEBUG_NOTICE, "Connect to %s[%s].%d @%s", linkp->servername, linkp->host, linkp->port, ipaddr));

	if ((cptr = find_server(linkp->servername, NULL)) != NULL) {
		char *inpath = get_client_name(cptr, HIDE_IP);

		sendto_realops("Server %s already present from %s.", linkp->servername, inpath);
		if ((by != NULL) && IsPerson(by) && !MyClient(by)) {
			send_me_notice(by, ":Server %s already present from %s", linkp->servername, inpath);
		}

		return 0;
	}

	cptr = make_client(NULL, &me);
	cptr->ip.s_addr = linkp->ip.s_addr;

	strncpyzt(cptr->name, linkp->servername, sizeof(cptr->name));
	strncpyzt(cptr->host, linkp->host, HOSTLEN + 1);
	inetntop(&linkp->ip, cptr->localClient->sockhost, HOSTIPLEN + 1);
	strncpyzt(cptr->hostip, cptr->localClient->sockhost, HOSTIPLEN + 1);

	if ((cptr->localClient->fd = engine_open(SOCK_STREAM, 0)) == -1) {
		report_error(cptr->localClient->fd, "Error opening socket to server %s: %s",
			get_client_name(cptr, SHOW_IP));
		serv_connect_exit(cptr);
		return 0;
	}
	if (cptr->localClient->fd > (HARD_FDLIMIT - 10)) {
		sendto_realops("No more connections allowed (%s)", cptr->name);
		serv_connect_exit(cptr);
		return 0;
	}

	if (!set_non_blocking(cptr->localClient->fd)) {
		report_error(cptr->localClient->fd, ERR_set_non_blocking, cptr->localClient->sockhost);
	}
	if (!set_sock_opts(cptr->localClient->fd)) {
		report_error(cptr->localClient->fd, ERR_set_sock_opts, cptr->localClient->sockhost);
	}
	if (!unset_sock_ipopts(cptr->localClient->fd)) {
		report_error(cptr->localClient->fd, ERR_unset_sock_ipopts, cptr->localClient->sockhost);
	}

	make_server(cptr);
	SetConnecting(cptr);
	cptr->serv->up = me.name;

	attach_link(cptr, linkp);
	attach_class(cptr);

	if (by != NULL && IsPerson(by)) {
		strcpy(cptr->serv->by_nick, by->name);
		strcpy(cptr->serv->by_user, by->username);
		strcpy(cptr->serv->by_host, by->host);
	}
	else {
		strcpy(cptr->serv->by_nick, "AutoConnect");
		*cptr->serv->by_user = '\0';
		*cptr->serv->by_host = '\0';
	}

	add_client_to_list(cptr);

	memset(&addr, '\0', addrlen);
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = 0;

	if (!BadPtr(linkp->bind_ip)) {
		Debug((DEBUG_DEBUG, "serv_connect() binding with bind_ip %s", linkp->bind_ip));
		if (!inetpton(linkp->bind_ip, &addr.sin_addr.s_addr)) {
			return 0;
		}
	}
	else if (!BadPtr(ServerInfo->default_bind_ip)) {
		Debug((DEBUG_DEBUG, "serv_connect() binding with default_bind_ip %s", ServerInfo->default_bind_ip));
		if (!inetpton(ServerInfo->default_bind_ip, &addr.sin_addr.s_addr)) {
			return 0;
		}
	}
	else {
		Debug((DEBUG_DEBUG, "serv_connect() no bind data, using INADDR_ANY!"));
	}

	Debug((DEBUG_DEBUG, "serv_connect() attempting TCP connection on %d to %s:%d",
		cptr->localClient->fd, cptr->localClient->sockhost, linkp->port));
	if (addr.sin_addr.s_addr != INADDR_ANY) {
		engine_connect_tcp(cptr->localClient->fd, cptr->localClient->sockhost, linkp->port,
			(struct sockaddr *)&addr, addrlen, serv_connect_callback, cptr, CONNECTTIMEOUT);
	}
	else {
		engine_connect_tcp(cptr->localClient->fd, cptr->localClient->sockhost, linkp->port,
			NULL, 0, serv_connect_callback, cptr, CONNECTTIMEOUT);
	}

	return 1;
}
