/*
   Name: $RCSfile: dbm.c,v $
   Author: Alan Moran
   $Date: 2005/11/20 13:20:51 $
   $Revision: 1.4 $
   $Id: dbm.c,v 1.4 2005/11/20 13:20:51 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
  @file

  @brief MySQL interface for storage of meta data in database.

  Meta data relating to a specific web asset (e.g., MIME type, filename, status
  of transformations etc.) are recorded during processing and can be made available
  to other tools via a database table.  Such tools might include administrative
  interfaces or content management layers.  This module provides a MySQL interface
  for the storage of such data.

  The relevant connectivity data (e.g., host, username etc.) is stored in the 
  &l;tdatabase&gt; nodeset of the configuration file.

*/

#include "globals.h"

#ifdef HAVE_MYSQL
#include <my_global.h>
#include <my_sys.h>
#include <mysql.h>

static MYSQL *conn;
#endif

/**
  Acquires a connection to a MySQL database.
 */
static void
rpl_db_init()
{
#ifdef HAVE_MYSQL
	assert(conn == NULL);
	if((conn = mysql_init(NULL)) == NULL) 
		rpl_log_fatal(rpl_message_get("DB_INIT_FAILED", RPL_EOM));
#endif
} 

/**
  Connects to the database (fatal error on failure).
 */
void
rpl_db_connect()
{
	rpl_db_init();
#ifdef HAVE_MYSQL
	if(mysql_real_connect(conn, rpl_cfg_get_db_host() , 
								rpl_cfg_get_db_user(), 
								rpl_cfg_get_db_pwd(), 
								rpl_cfg_get_db_name(), 
								rpl_cfg_get_db_port(), 
								NULL, 0) == NULL)
	{
		mysql_close(conn);
		rpl_log_fatal(rpl_message_get("DB_CONNECT_FAILED", mysql_error(conn), RPL_EOM));
	}
#endif
}

/**
   Performs MySQL character escaping on the string str.

   @param str the string to be escaped.

   @return string in escaped format.
 */
static rpl_str_t
rpl_db_escape_string(rpl_c_str_t str)
{
	rpl_str_t escaped_str = NULL;
	
#ifdef HAVE_MYSQL
	int length; 

	assert((str != NULL) && (conn != NULL));
	
   	length = strlen(str);
	escaped_str = (rpl_str_t)rpl_me_malloc(2 * length + 1);
	/* derivation of length from strlen limits this function to escaping null terminated strings */
	(void)mysql_real_escape_string(conn, escaped_str, str, length);
#endif
	return escaped_str;
}	

/**
   Disconnects from the database.
 */
void
rpl_db_disconnect()
{
#ifdef HAVE_MYSQL
	assert(conn != NULL);

	mysql_close(conn);
#endif
}

/**
   Parses and executes the query.

   @param query the SQL string for the query to be executed.
 */
void
rpl_db_query(rpl_c_str_t query)
{
#ifdef HAVE_MYSQL
	rpl_str_t msg;

	assert((conn != NULL) && (query != NULL));

	if(mysql_query(conn, query) != 0)
	{
		msg = rpl_message_get("DB_INSERT_FAILED", mysql_error(conn), RPL_EOM);
		rpl_log_error(msg);
		rpl_me_free(msg);
	}
#endif
}

