/*
   Name: $RCSfile: fsutil.c,v $
   Author: Alan Moran
   $Date: 2005/11/26 21:33:56 $
   $Revision: 1.19 $
   $Id: fsutil.c,v 1.19 2005/11/26 21:33:56 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief Filesystem related utility functions.

   The file system utilities module caters for file system operations
   encountered when handling files or working with the file system datastore.
   It is intended to simplify such operations by providing a simple interface
   for invoking (and error handling) common file system functions.

*/

#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <unistd.h>

#include "globals.h"

static int  fs_dir_stat(rpl_c_str_t pathname);
static void fs_create_dir(rpl_c_str_t path);

/**
   Returns the contents of file filename as a string.

   @param filename is the relative (to the cwd) name of the file.

   @return the contents of file filename as a string.
 */
rpl_str_t
rpl_fs_f2str(rpl_c_str_t filename) {
    rpl_str_t f_cnts, f_buf;
    struct stat f_stat;
    size_t f_size;
    FILE *fp;

    assert(filename != NULL);

    /* open the file (and acquire a descriptor for it) */
    /* binary "b" is req'd for ANSI C portability but has no effect on POSIX platforms */
    errno = 0;
    if((fp=fopen(filename, "rb")) == NULL)
        rpl_log_fatal(rpl_message_get("FS_OPEN_FAILED", filename, " (", strerror(errno), ")", RPL_EOM));

    /* determine file size */
    errno = 0;
    if(stat(filename, &f_stat) == -1)
        rpl_log_fatal(rpl_message_get("FS_LSTAT_FAILED", strerror(errno), RPL_EOM));

    f_size = f_stat.st_size;
    /* assign sufficient memory to store file contents */
    f_cnts = (rpl_str_t)rpl_me_malloc(f_size + 1);
    /* buffer needs to be sufficiently large in case entire file is a single string */
    f_buf = (rpl_str_t)rpl_me_malloc(f_size + 1);

    /* read file contents into string (fgets retains \n on each line and appends \0 at the end) */
    (void)fgets(f_cnts,f_size,fp);
    while(fgets(f_buf,f_size,fp) != NULL)
        strcat(f_cnts,f_buf);

    /* tidy up */
    fclose(fp);
    /* free(f_buf); */

    return f_cnts;
}

/**
   Stores the string sout into the file identified by filename.

   @param sout
   @param filename
 */
void
rpl_fs_str2f(rpl_c_str_t sout, rpl_c_str_t filename) {
    FILE *fpout;

    assert((sout != NULL) && (filename != NULL));

    errno = 0;
    if ((fpout = fopen(filename,"w")) == NULL)
        rpl_log_fatal(rpl_message_get("FS_FILE_WRITE_FAILED", filename, " : ", strerror(errno), RPL_EOM));

    fprintf(fpout,"%s",sout);

    fclose(fpout);
}

/**
   Stat the directory indicated by pathname. Intended to query the filesystem when deciding
   whether or not to create a directory.

   @param pathname

   @return -1 on error, 0 on success.
 */
static int
fs_dir_stat(rpl_c_str_t pathname) {
    struct stat stbuf;

    assert(pathname != NULL);

    /* if the pathname does not exist return to the caller */
    if(lstat(pathname, &stbuf)<0)
        return -1;

    /* if directory already exists then notify the caller */
    if(S_ISDIR(stbuf.st_mode))
        return 0;

    /* if an existing entity already exists then fail entirely */
    if(S_ISREG(stbuf.st_mode) || S_ISLNK(stbuf.st_mode))
        rpl_log_fatal(rpl_message_get("FS_MKDIR_REG_FILE_FAILED", pathname, RPL_EOM));
    if(S_ISCHR(stbuf.st_mode) || S_ISBLK(stbuf.st_mode) || S_ISFIFO(stbuf.st_mode) || S_ISSOCK(stbuf.st_mode))
        rpl_log_fatal(rpl_message_get("FS_MKDIR_DV_FAILED", pathname, RPL_EOM));
    /* return at least an indication of error (all avoids -Wall complaints) */
    return -1;
}

/**
   Creates the directory indicated by pathname. Assumes that fs_dir_stat has been called
   first to ascertain whether creation is required and legal.

   @param pathname
 */
static void
fs_create_dir(rpl_c_str_t pathname) {
    /* directory permissions */
    static mode_t perms = S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH;

    assert(pathname != NULL);

    errno = 0;
    if(mkdir(pathname, perms) < 0)
        rpl_log_fatal(rpl_message_get("FS_MKDIR_FAILED", pathname, " (",strerror(errno),")", RPL_EOM));
}

/**
   Recursively creates the directory tree indicated by pathname (akin to the mkdir command
   with the -p option).

   @param pathname
 */
void
rpl_fs_mkdir(rpl_c_str_t pathname) {
    rpl_str_t path, sp, ep, cwd;
    size_t length;

    /* for security reasons do not permit the creation of absolute paths */
    assert((pathname != NULL)); /* && (pathname[0] != '/')); */

    /* note the directory from which this function was called (you'll have to return to it later) */
    cwd = rpl_fs_get_cwd();

    /* make a copy of the path since we intend to alter it during processing */
    length = strlen(pathname) + 1;
    path = (rpl_str_t)rpl_me_malloc(length);
    snprintf(path, length, "%s", pathname);

    /* set sp to support both absolute and relative directory trees */
    sp = path;
    if(path[0] == '/')
        sp++;

    if((ep = strchr(sp, '/')) != NULL) {
        /* multiple consecutive "/" are permitted in paths but we still have to step over them
        while(*(ep+1) == '/')
            ep++;
        */
        *ep = '\0';

        /* if the pathname does not exist then create it */
        if(fs_dir_stat(path) < 0)
            fs_create_dir(path);
        (void)chdir(path);
        rpl_fs_mkdir(ep + 1);
        (void)chdir("..");
    } else {
        if(fs_dir_stat(path) < 0)
            fs_create_dir(path);
    }

    /* tidy up before you leave */
    rpl_me_free(path);
    chdir(cwd);
    rpl_me_free(cwd);
}


void
rpl_fs_cp(rpl_c_str_t from_filename, rpl_c_str_t to_filename) 
{

	int from_fid, to_fid;
	struct stat statbuf;
	char *from, *to;

	/* open from and to files */
	if((from_fid = open(from_filename, O_RDONLY)) < 0)
	{
        rpl_log_error(rpl_message_get("FS_OPEN_FAILED", from_filename, " (", strerror(errno), ")", RPL_EOM));
		return;
	}
	if((to_fid = open(to_filename, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) < 0 ) 
	{
        rpl_log_error(rpl_message_get("FS_OPEN_FAILED", from_filename, " (", strerror(errno), ")", RPL_EOM));
		return;
	}

	/* fix the size of the output file */
    if(fstat(from_fid, &statbuf) < 0) {
        rpl_log_error(rpl_message_get("FS_FSTAT_FAILED", from_filename, " (", strerror(errno), ")", RPL_EOM));
        return;
    }
    if(lseek(to_fid, statbuf.st_size - 1, SEEK_SET) == -1) {
        rpl_log_error(rpl_message_get("FS_LSEEK_FAILED", to_filename, " (", strerror(errno), ")", RPL_EOM));
        return;
    }
    if(write(to_fid, "", 1) != 1) {
        rpl_log_error(rpl_message_get("FS_WRITE_FAILED", to_filename, " (", strerror(errno), ")", RPL_EOM));
        return;
    }

	/* memory copy the contents */
    if((from = mmap(0, statbuf.st_size, PROT_READ, MAP_FILE | MAP_SHARED, from_fid, 0)) == (caddr_t) - 1) {
        rpl_log_error(rpl_message_get("FS_READ_FAILED", from_filename, " (", strerror(errno), ")", RPL_EOM));
        return;
    }
    if((to = mmap(0, statbuf.st_size, PROT_READ | PROT_WRITE, MAP_FILE | MAP_SHARED, to_fid, 0)) == (caddr_t) - 1) {
        rpl_log_error(rpl_message_get("FS_WRITE_FAILED", to_filename, " (", strerror(errno), ")", RPL_EOM));
        return;
    }
    memcpy(to, from, statbuf.st_size);

}

/**
   Removes file indicated by filename.

   @param filename
 */
void
rpl_fs_remove(rpl_c_str_t filename) 
{
    rpl_str_t    msg;

    assert(filename != NULL);

    errno=0;
    /* remove performs unlink for files and rmdir for directories */
    /* some return codes are negative but only -1 indicates failure to remove file */
    if(remove(filename) == -1) {
        msg = rpl_message_get("FS_UNLINK_FAILED", filename, " (", strerror(errno), ")", RPL_EOM);
        rpl_log_warn(msg);
		rpl_me_free(msg);

    }
}

/**
   Performs deletes.  Conforms to rpl_mod_process_fp signature.

   @param filename
 */
static rpl_wk_status
rpl_fs_delete(rpl_c_str_t filename, struct stat statbuf)
{
	rpl_wk_status status = RPL_WK_OK;

    assert(filename != NULL);

    /* locate and delete each appropriate file in the datastore */
    rpl_fs_remove(filename);

	return status;
}


/**
   Destroys content of datastore.
 */
void
rpl_fs_delete_ds()
{
	rpl_str_t base_dir, tout_dir, terr_dir, parse_dir;

	base_dir = rpl_cfg_get_ds_basedir();
	/* be specific about which directories are being deleted ! */
	tout_dir = rpl_str_concat(base_dir, "/", RPL_DS_TOUT_DIR, RPL_STR_EOC);
	terr_dir = rpl_str_concat(base_dir, "/", RPL_DS_TERR_DIR, RPL_STR_EOC);
	parse_dir = rpl_str_concat(base_dir, "/", RPL_DS_PARSE_DIR, RPL_STR_EOC);
	
	rpl_fs_recurse(tout_dir, rpl_fs_delete);
	rpl_fs_recurse(terr_dir, rpl_fs_delete);
	rpl_fs_recurse(parse_dir, rpl_fs_delete);

	rpl_me_free(tout_dir);
	rpl_me_free(terr_dir);
	rpl_me_free(parse_dir);
}

/**
   Returns current working directory.

   @return current working directory.
 */
rpl_str_t
rpl_fs_get_cwd() {
    int         buf_size = RPL_FS_CWD_BUF_SIZE;
    rpl_str_t    cwd;

    /* buffer size required to hold the current working directory must be determined dynamically */
    while(1) {
        errno = 0;
        if((cwd = (rpl_str_t) rpl_me_malloc(buf_size)) == NULL)
            rpl_log_fatal(rpl_message_get("OUT_OF_MEMORY", strerror(errno), RPL_EOM));
        errno = 0;
        if(getcwd(cwd, buf_size-1) != NULL)
            return cwd;

        if(errno != ERANGE)
            rpl_log_fatal(rpl_message_get("FS_UNABLE_TO_DETERMINE_CWD", strerror(errno), RPL_EOM));
        rpl_me_free(cwd);
        buf_size += RPL_FS_CWD_BUF_SIZE;
    }
}

/**
   Checks whether filename exists or not.

   @return -1 if files does not exist otherwise 0.
 */
int
rpl_fs_file_exists(rpl_str_t filename) {
    int rc = 0;
    struct stat f_stat;

	assert(filename != NULL);

    errno = 0;
    rc = stat(filename, &f_stat);
    if(rc == -1) {
       rpl_log_error(rpl_message_get("FS_LSTAT_FAILED", strerror(errno), RPL_EOM));
	} else {
		rc = S_ISDIR(f_stat.st_mode) ? 1 : 0;
	}

    return rc;
}

/**
   Returns path resolved as an absolute path without trailing "/".

   @param path Path.

   @return path resolved as an absolute path without trailing "/".
 */
rpl_str_t
rpl_fs_normalize_path(rpl_str_t path) {
    rpl_str_t n_path = NULL, cwd;

	assert(path != NULL);
	
    /* silently ignore NULL paths */
    if(path) {
        /* resolve relative paths */
        if (path[0] != '/') {
            cwd = rpl_fs_get_cwd();
            n_path = (rpl_str_t) rpl_me_malloc(strlen(cwd) + strlen(path) + 2);
            sprintf(n_path, "%s%s%s", cwd, "/", path);
            rpl_me_free (cwd);
        } else {
            n_path = (rpl_str_t) rpl_me_malloc(strlen(path) + 1);
            sprintf(n_path, "%s", path);
        }

        /* strip trailing "/" */
        if(n_path[strlen(n_path) - 1] == '/')
            n_path[strlen(n_path) -1] = '\0';
    }

    return n_path;
}

/**
   Recurses a directory tree invoking the callback on each file (not directory)
   found therein.  Similar to ftw (Linux) or fts (BSD).

   @param filename absolute filename of the asset to be processed.
   @param fn callback to be invoked on each file found.
 */
void
rpl_fs_recurse(rpl_c_str_t filename, rpl_mod_process_fp process)
{
    struct dirent *dirp;
    DIR *dp;
    struct stat statbuf;
	rpl_str_t sub_dir, msg;

	assert(filename != NULL);

    /* check that the file exists */
    errno = 0;
    if(stat(filename,&statbuf)==-1) {
        msg = rpl_message_get("FS_PROCESS_FAILED", filename, " (", strerror(errno), ")", RPL_EOM);
        rpl_log_error(msg);
		rpl_me_free(msg);
        return;
    }

    /* recursively process the directory tree */
    if(S_ISREG(statbuf.st_mode)) {
        process(filename, statbuf);
    } else if(S_ISDIR(statbuf.st_mode)) {
        dp=opendir(filename);
        while ((dirp = readdir (dp)) != NULL) {
            if (strcmp (dirp->d_name, ".") == 0 || strcmp (dirp->d_name, "..") == 0)
                continue;
            sub_dir = (rpl_str_t) rpl_me_malloc(strlen(filename) + strlen(dirp->d_name) + 2);
            sprintf(sub_dir, "%s/%s", filename, dirp->d_name);
            rpl_fs_recurse(sub_dir, process);
            rpl_me_free(sub_dir);
        }
        if (closedir (dp) < 0) {
            msg = rpl_message_get("FS_DIR_CLOSE_FAILED", filename, RPL_EOM);
            fprintf(stderr, rpl_str_concat("Fatal: ", msg, RPL_STR_EOC));
            rpl_log_fatal(msg);
        }
		process(filename, statbuf);
	}
}

/**
   Extracts the relative (to a base directory) path and filename from an
   absolute filename.  This convenience function addresses a need to resolve
   filenames that frequently arises in processing modules.  In order to 
   extract the relative directory a check that the basedir matches the start
   of the absolute filename is made.

   @param filename absolute filename from which paths are to be extracted.
   @param basedir the directory relative to which the relative directory is formed.
   @param rel_dir return pointer to relative directory
   @param file return pointer to filename

   @return 1 if the path and filename cannot be extracted (basedir failes to match)
   0 otherwise.

 */
int
rpl_fs_resolve_paths(rpl_c_str_t filename, rpl_c_str_t basedir, rpl_str_t *rel_dir, rpl_str_t *file)
{
	rpl_str_t a_rel_dir, a_file;
	char *rfp;

	assert((filename != NULL) && (basedir != NULL));

	if(strcmp(filename, basedir) == 0)
	{
		*rel_dir = "";
		*file = "";
	} else {
		rfp = strdup(filename);
		if(strstr(rfp, basedir) == NULL)
		{
			/* str match check to validate the filename argument */
			rpl_log_error(rpl_message_get("REG_INVALID_PATH", filename, RPL_EOM));
			*rel_dir = RPL_STR_NUL;
			*file = RPL_STR_NUL;
			return 1;
		}
		/* make sure to step over the trailing "/" in the base pathname */
		rfp += strlen(basedir) + 1;
		rpl_str_rsplit(rfp, '/', &a_rel_dir, &a_file);
		*rel_dir = a_rel_dir;
		*file = a_file;
	}
	
	return 0;
}

