/*
   Name: $RCSfile: memory.c,v $
   Author: Alan Moran
   $Date: 2005/07/18 21:44:40 $
   $Revision: 1.7 $
   $Id: memory.c,v 1.7 2005/07/18 21:44:40 nloyola Exp $

   Description:
   Memory management functions.

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.
 *
 */

/**
   @file

   @brief Memory management functions.

   The memory module provides the most rudimentary of memory management
   functions by wrapping the memory allocation/deallocation calls.  These
   wrappers provide for a more consistent and safer use of the underlying
   system calls.  In addition they leave open the opportunity for the
   introduction of more sophisticated memory management (e.g., auditing or
   custom management) should this need arise at a future date.  For this reason
   it is essential that clients use the rpl_me_malloc and rpl_me_free function calls in
   preference to their underlying system counterparts.

*/

#include <stddef.h>

#include "globals.h"

/**
   Allocates memory and logs failed allocations as fatal.

   @param n allocated size in bytes.

   @return ptr to allocated memory.
 */
void *
rpl_me_malloc(size_t n)
{
    void *mp;
    errno = 0;
    if((mp = (void *)calloc(n, sizeof(rpl_str_t))) == NULL)
        rpl_log_fatal(rpl_message_get("OUT_OF_MEMORY", strerror(errno), RPL_EOM));
    return mp;
}

/**
   Frees memory and ensure that no dangling pointer remains.

   @param data ptr to be freed.
 */
void
rpl_me_free(void *data)
{
    free(data);
    data = NULL;
}
