/*
   Name: $RCSfile: messages.c,v $
   Author: Alan Moran
   $Date: 2005/11/13 20:57:21 $
   $Revision: 1.17 $
   $Id: messages.c,v 1.17 2005/11/13 20:57:21 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief Functions for the retrieval of message resources.

   The messages module is concerned with the display of application messages
   (typically but not exclusively within application logging).  For the most
   part these messages are identified by a key (e.g., DB_INIT_FAILED) which
   resolves to a message (e.g., "Unable to initialise database (probably out of
   memory)") that can be concatenated with custom strings supplied by the
   caller.  The caller must include the end of message identifier, EOM, to
   terminate this list (even if it only consists of a message key).  The use of
   message keys centralises messages in a single location and ensures
   consistent usage of messages that may appear across several modules.

   In situations where a custom message is called for no message key needs to
   be supplied.  However, the message will be prepended with a token identifier
   (!).  This makes it possible to grep the logs in order to identify custom
   messages or misspelt message keys.

*/

#include <stdarg.h>

#include "globals.h"

static const Message messages[] = {
    /* configuration */
    { "RPL_CFG_FILE_NOT_FOUND", "Unable to locate the config file " },
    /* logfiles */
    { "LOGFILE_NOT_FOUND", "Unable to locate the logfile" },
    { "LOGFILE_CLOSE_FAILURE", "Unable to close the logfile" },
    /* usage */
    { "UNIDENTIFIED_USER", "Unidentified user.  Remote IP addr: " },
    { "WEB_ASSERT_INSERT", "Web asset inserted: " },
    /* http */
    { "PARSE_FORM_FAILED", "Unable to parse the form." },
    { "UNSUPPORTED_METHOD", "Unknown or unsupported method: " },
    /* database */
    { "DB_INIT_FAILED", "Unable to initialise database (probably out of memory)" },
    { "DB_CONNECT_FAILED", "Unable to connect to database: " },
    { "DB_INSERT_FAILED", "Unable to insert webassert data into database: " },
    { "DB_DIGEST_RESOLVE_FAILED", "Unable to resolve filename against digest: " },
    { "DB_DIGEST_RESOLVE_WARNING", "Attempt to resolve filename against digest returned empty result set: " },
	/* config */
    { "CONFIG_STAT_FAILED", "Config Error! " },
    /* filesystem */
    { "FS_LSTAT_FAILED", "Unable to lstat file: " },
    { "FS_FSTAT_FAILED", "Unable to fstat file: " },
    { "FS_LSEEK_FAILED", "Unable to lseek file: " },
    { "FS_WRITE_FAILED", "Unable to write to file: " },
    { "FS_READ_FAILED", "Unable to read file: " },
    { "FS_PROCESS_FAILED", "Unable to process file: " },
    { "FS_OPEN_FAILED", "Unable to open file: " },
    { "FS_MKDIR_REG_FILE_FAILED", "Unable to create directory. Regular file of same name already exists: " },
    { "FS_MKDIR_DV_FAILED",
        "Unable to create directory.  Device (character, block, FIFO or socket) of same name already exists: " },
    { "FS_MKDIR_FAILED", "Unable to create directory: " },
    { "FS_FILE_WRITE_FAILED", "Unable to write file to filesystem: " },
    { "FS_TEMP_FILE_FAILED", "Unable to create temporary file to store file upload: " },
    { "FS_DIR_CLOSE_FAILED", "Unable to close directory: " },
    { "FS_UNABLE_TO_DETERMINE_CWD", "Unable to determine the current working directory: " },
    { "FS_UNLINK_FAILED", "Unable to delete file: " },
    /* transformations */
    { "UNSUPPORTED_MIME_TYPE", "Unable to process file (unsupported MIME type): " },
    { "TIDY_CONFIG_FAILED", "Unable to configure tidy : " },
    { "TIDY_TRANSFORM_FAILED", "Unable to perform tidy transformation of file: " },
    { "XSLT_UNABLE_TO_CREATE_SIT", "Unable to create XSLT situation (context)." },
    { "XSLT_UNABLE_TO_CREATE_PROC", "Unable to create processor for situation (context)." },
    { "XSLT_STYLESHEET_PARSE_FAILED", "Unable to parse XSLT stylesheet: " },
    { "XSLT_INPUT_PARSE_FAILED", "Unable to parse XSLT input: " },
    { "XSLT_TRANSFORM_FAILED", "Unable to perform XSLT transformation of file: " },
    { "XSLT_CATALOG_DEFAULT", "Unable to locate the XML Catalog (tried: config file and environment).  Using"
      "defaults." },
    { "XSLT_CATALOG_FAILED", "Unable to load XML Catalog: " },
    { "HTML_TIDY_REPORT_PREFIX", "Tidy Error Report : " },
    { "XML_PARSER_ERROR", "Unable to perform normalization parsing owing to an XML parse error." },
    { "XML_ENTITY_PARSER_ERROR", "Unable to resolve external entity." },
    { "XML_ENTITY_PARSER_CONFIG_ERROR", "Unable to configure entity parsing." },
    { "XML_READ_ERROR", "Error occured whilst reading configuration file." },
    { "XML_WELLFORMEDNESS_ERROR", "Unable to parse configuration document: " },
    /* digests */
    { "DIGEST_INDEX_FAILED", "Unable to index file contents: " },
    { "DIGEST_DIR_CLOSE_FAILED", "Unable to close directory being indexed: " },
    { "DIGEST_FILE_NOT_FOUND", "Unable to parse digest input file: " },
    { "DIGEST_FILE_NOT_WELLFORMED", "Digest file input is not well-formed: " },
    { "DIGEST_NON_POS_BUFFER_SZ", "Attempt to set the digest summary buffer size to non-positive value detected. " },
	/* SMTP client */
	{ "MAIL_MTA_NOT_FOUND", "Unable to access local delivery MTA. " },
	/* workflow */ 
	{ "WK_PROCESSING", "Processing: " },
	{ "WK_INIT_ERROR", "Unable to initialse module.  Terminated." },
	/* registry */
	{ "REG_INVALID_PATH", "The path does match in the anticipated manner: " },
	{ "REG_ASSET_NOT_FOUND", "The asset could not be found in the registry: " },
	/* parser */
	{ "PARSER_ILLEGAL_ATTR_CHAR", "Illegal attribute character detected (scipting language construct?)" },
	{ "PARSER_UNKNOWN_ENTITY", "Unknown entity detected: " },
	/* link checker */
	{ "LINK_CHECKER_MISSING_ENTRIES", "Unable to create link checking report owing to missing configuration file entires." },
    /* critical */
    { "OUT_OF_MEMORY", "Unable to allocate memory: " },
    { "PROCESS_TERMINATED", "Process terminated (Fatal error)" },
};

static rpl_msg_global_t msg_global = {
    .init = 1,
    .tptr = NULL,
    .size = (sizeof(messages) / sizeof(messages[0])),
    .ns_prefix = "!"
};

/**
    Initialize message global. Must be called before message subsystem can be used.
    The only thing that can (will) break this if someone messes with our returned pointer.
*/
void rpl_message_init()
{
    msg_global.init = 1;
    msg_global.tptr = NULL;
}

/**
    Release message pointer and uninitialize message global. Should be called after
    using message subsystem.
*/
void rpl_message_done()
{
    if (msg_global.init) {
        msg_global.init = 0;
        if (NULL != msg_global.tptr) {
            rpl_me_free(msg_global.tptr);
        }
    }
}

/**
    Return the message corresponding to label (if the label does not return a standard
    messages then label prepended with "!" consists the returned message).
    Calls to get_message must be terminated by EOM.  This function has refactored to include
    code derived original from the strutil:concat function in order to reduce the number
    of implicit memory leaks.

    msg.tptr is what we use for subsequent calls. Assume messages are not that important that
    we should
    1) keep them all around
    2) Risk memory leaks for them.
    If they are so important, or thread safety becomes an issue, a mutex and a strncpy should
    keep them.
    Note : message_init() and message_done() is a must for the proper working of this.
    DO NOT TRY TO FREE THE RETURNED PTR!!!

    @param label message identifier.

    @note This function has a variable argument list. Calls must be terminated
    by EOM.

    @return message corresponding to label.
 */
rpl_str_t
rpl_message_get(rpl_c_str_t label, ...)
{
    int i;
    rpl_str_t sp; //, msg = RPL_STR_NUL;
    size_t length;
    va_list argp;

    assert(label != NULL);

    /* No init - no messages... */
    if (0 == msg_global.init) {
        return NULL;
    }

    if (msg_global.tptr != NULL) {
        rpl_me_free(msg_global.tptr);
    }

    /* ascertain whether this is a standard or a custom message. */
    for(i = 0; i < msg_global.size; i++)
    {
        if(strcmp(label, messages[i].label) == 0)
            break;
    }

    /* determine the memory allocation required to hold the message */
    length = (i<msg_global.size) ? strlen(messages[i].message) + 1 : strlen(label) + 2;
    va_start(argp, label);
    while((sp = va_arg(argp, rpl_str_t)) != NULL)
        length += strlen(sp);
    va_end(argp);
    msg_global.tptr = (rpl_str_t)rpl_me_malloc(length + 1);

    /* create the message by concatenating the necessary strings */
    if(i<msg_global.size) {
        /* standard message */
        strcpy(msg_global.tptr, messages[i].message);
    } else {
        /* custom message */
        strcpy(msg_global.tptr, msg_global.ns_prefix);
        strcat(msg_global.tptr, label);
    }
    va_start(argp, label);
    while((sp=va_arg(argp, rpl_str_t)) != NULL)
       strcat(msg_global.tptr, sp);
    va_end(argp);

    return msg_global.tptr;
}
