/***************************************************************************
    file				 : kbase.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
						   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdarg.h>
#include	<stdlib.h>
#include	<errno.h>
#include	<time.h>

#include	<qlist.h>
#include	<qdict.h>
#include	<qevent.h>
#include	<qwidget.h>
#include	<qwidgetlist.h>
#include	<qguardedptr.h>
#include	<qapplication.h>
#include 	<qpalette.h>
#include 	<qtimer.h>
#include 	<qpopupmenu.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_database.h"
#include	"kb_attr.h"
#include	"kb_attrdict.h"
#include	"kb_dblink.h"
#include	"kb_dbinfo.h"
#include	"kb_node.h"
#include	"kb_display.h"
#include	"kb_prompt.h"
#include	"kb_gui.h"
#include	"kb_location.h"
#include	"kb_viewer.h"
#include	"kb_appptr.h"
#include	"kb_callback.h"
#include	"kb_director.h"
#include	"kb_rtbuild.h"

#ifndef 	_WIN32
#include	"kbase.moc"
#else
#include	"kbase.h"
#endif

#include	"kb_conductor.h"

#include	"kb_dbaseviewer.h"
#include	"kb_textlog.h"
#include	"kb_querylog.h"
#include	"kb_eventlog.h"
#include	"kb_script.h"
#include	"kb_debug.h"
#include	"kb_options.h"
#include	"kb_optionsdlg.h"
#include	"kb_metrics.h"
#include	"kb_notifier.h"
#include	"kb_formcopier.h"
#include	"kb_objbase.h"

#include	"kb_libloader.h"

#include	"tk_filedialog.h"

#include	"../parts/editor/kb_editor.h"

#if		! __KB_RUNTIME
#include	"kb_wizconnect.h"
#endif

KBaseApp		*kbaseApp 		;
ulong			kbLogFlags		;

bool			__useMDI		;
bool			__useSDI		;
bool			__noNoteDB              ;
bool			__noLastDB              ;
QString			__initDEB		;

bool			useMDI			;

QGuardedPtr<KBQueryLog>	queryLog		;
QGuardedPtr<KBEventLog>	eventLog		;
QGuardedPtr<KBTextLog >	scriptLog		;


TKAction		*actOpenDatabase	;
TKAction		*actNewDatabase		;
TKAction		*actShowOptions		;
TKAction		*actPlayScore		;
TKAction		*actPackage		;
TKAction		*actQuit		;
TKToggleAction		*actShowQueryLog	;
TKToggleAction		*actShowEventLog	;
TKToggleAction		*actShowScriptLog	;
TKToggleAction		*actShowDebugger	;
TKActionMenu		*actDatabases		;
TKRecentFilesAction	*actRecentFiles		;
TKToggleAction		*actSnapEnable		;
TKSelectAction		*actShowWindow		;
TKAction		*actHelpPlugins		;
TKAction		*actHelpContents	;
TKAction		*actHelpAbout		;
TKAction		*actHelpIndex		;
TKAction		*actHelpSearch		;


QDict<TKAction>		globalActions		;

KBDirector		director		;


static	QDict<QString>	  dummyDict		;
static	uint		  nApp			;

typedef	QGuardedPtr<KBDebug>	KBDebugPtr	;
static	QDict<KBDebugPtr>	debuggers	;


static	char	filePatterns[] = "*.rkl|Rekall files\n"
				 "*.edb|EasyDB files\n"
				 "*.kdb|Legacy kbase files"	;
static	char	defExtn	    [] = ".rkl"				;


extern	void	setCallBack	() ;
extern	void	setActiveWindow	(QWidget *) ;


/*  ------------------------------------------------------------------  */

/*  fixSaveExtn	: Fix extension on database file name			*/
/*  name	: QString &	: File name				*/
/*  extn	: cchar *	: Default extension			*/
/*  (returns)	: void		:					*/

static	void	fixSaveExtn
	(	QString	&name,
		cchar	*extn
	)
{
	int	od	= name.findRev ('.') ;
	int	os	= name.findRev ('/') ;

	if ((od < 0) || (od < os))  name += extn ;
}

/*  registerLanguage							*/
/*		: Register scripting language				*/
/*  language	: const QString & : The language			*/
/*  (returns)	: void		  :					*/

void	registerLanguage
	(	const QString	&language
	)
{
	if (debuggers.find (language) == 0)
	{
		debuggers.insert (language, new KBDebugPtr) ;
#if	! __KB_RUNTIME
		actShowDebugger->setEnabled (true) ;
#endif
	}
}


/*  ------------------------------------------------------------------  */

/*  KBaseApp								*/
/*  KBaseApp	: Constructor for main KBase widget			*/
/*  initDB	: QString	: Initial database			*/
/*  create	: bool		: Create new database			*/
/*  add		: bool		: Add to recent list			*/
/*  (returns)	: KBaseApp	:					*/

KBaseApp::KBaseApp
	(	QString		initDB,
		bool		create,
		bool		add
	)
	:
	TKMainWindow (0, 0, WDestructiveClose)
{
	setIcon	   (getSmallIcon("rekall")) ;
	setCaption ("Rekall") ;

	nApp	+= 1	;

	m_showWidget	= this	;
	m_available	= true	;
	m_runable	= true	;
	m_autoObj	= 0	;
	m_dbViewer	= 0	;

	m_pManager	= new TKPartManager (this) ;
	connect
	(	m_pManager, SIGNAL(activePartChanged(TKPart *)),
		this,	    SLOT  (createGUI        (TKPart *))
	) 	;

	TKConfig *config = TKConfig::getConfig () ;

	init	   (config) ;
	setXMLFile ("rekallui.empty") ;
	createGUI  (0)	;
	fixHelpMenu(director) ;

#if	! __KB_EMBEDDED
	if (useMDI)
	{	
		setCentralWidget (m_mdi = new QWorkspace (this)) ;

		connect
		(	m_mdi,	SIGNAL(windowActivated(QWidget *)),
			this,	SLOT  (windowActivated(QWidget *))
		)	;
			
#if	QT_VERSION >= 300
		m_mdi->setScrollBarsEnabled (true) ;
		m_mdi->setEraseColor	    (colorGroup().color(QColorGroup::Background)) ;
		m_mdi->setBackgroundMode    (Qt::FixedColor) ;
#endif
		m_mdi->show () ;
	}
	else
#endif
		m_mdi = 0 ;


	config->setGroup("General Options") ;
	QSize 	size	   = config->readSizeEntry(useMDI ? "MDIGeometry" : "SDIGeometry") ;

	if (!size.isEmpty()) resize (size)  ;

	show	() ;
	KBMetrics::noteMenuBarHeight   (menuBarHeight()) ;
	KBMetrics::noteToolBarHeight   (toolBarHeight()) ;
	KBMetrics::noteStatusBarHeight (toolBarHeight()) ;


	fprintf
	(	stderr,
		"KBaseApp::KBaseApp: initDB=[%s] noLast=%d openLast=%d\n",
		(cchar *)initDB,
		__noLastDB,
		KBOptions::getOpenLast()
	)	;

#if	! __KB_RUNTIME
	/* In the runtime version, never honour the last-opened setting	*/
	/* Since we cannot get at the options dialog, should this be	*/
	/* set then the user could not invoke tkcRekall and get the	*/
	/* file-open dialog.						*/
	if (initDB.isEmpty() && !__noLastDB && KBOptions::getOpenLast())
	{
		initDB	= config->readEntry ("lastOpened") ;
		create	= false	;
	}
#endif

#if	__KB_RUNTIME
	/* In the runtime version, we must open a database, so if none	*/
	/* is specified then prompt now and exit if none is given.	*/
	if (initDB.isEmpty())
		initDB	= KBFileDialog::getOpenFileName
			  (	".",
				filePatterns,
				parentWidget (),
				"Open database ..."
			  ) ;

	if (initDB.isEmpty()) 
	{
		fprintf
		(	stderr,
			"KBaseApp::KBaseApp: runtime no initDB\n"
		)	;
		m_runable = false ;
		return  ;
	}
#endif

	fprintf
	(	stderr,
		"KBaseApp::KBaseApp: initDB=[%s]\n",
		(cchar *)initDB
	)	;

	/* If there is an initial database (and for the runtime there	*/
	/* will be), or we are creating, then open it now. If the	*/
	/* database has a startup form then this will be show instead	*/
	/* of the database window.					*/
	if (!initDB.isEmpty())
		if ((QFile::exists (initDB) || create))
		{
			QWidget	*dbw = openDBaseViewer (initDB, create, add) ;
			if (dbw != 0)
			{
				fprintf
				(	stderr,
					"KBaseApp::KBaseApp: openDBaseViewer true: [%p] useMDI=%d create=%d\n",
					(void *)dbw,
					useMDI,
					create
				)	;

				if (!useMDI && !create)
					m_showWidget = dbw ;
			}
		}

#if	__KB_RUNTIME
	/* When the runtime starts, the database must exist and must	*/
	/* have a startup form. If not then barf and exit.		*/
	if (m_showWidget == this)
	{
		TKMessageBox::sorry
		(	0,
			TR("Database does not exist or does not have a start-up form"),
			TR("Runtime start-up failed")
		)	;
		m_runable = false ;
		return ;
	}
#endif

	/* The database window is "available" if it is being show. This	*/
	/* affects the embedded version, since top-level widgets seem	*/
	/* to return "::isVisible()" always as true.			*/
	m_available = m_showWidget == this ;


#if	! __KB_EMBEDDED
	/* Hide the database window if we have started a form.		*/
	/* EXCEPT: Don't do this when embedded since (a) top-level hide	*/
	/* seems to be broken and (b) if we do then the tool bat gets	*/
	/* screwed up!							*/
	if (!m_available)
	{
		fprintf	(stderr, "KBaseApp::KBaseApp: hiding\n") ;
		hide	() ;
	}
#endif

#if	! __KB_RUNTIME
	if (!__initDEB.isEmpty()) showDebugger () ;
#endif
}

/*  KBaseApp								*/
/*  ~KBaseApp	: Desctructor for the application			*/
/*  (returns)	: void		:					*/

KBaseApp::~KBaseApp()
{
	fprintf
	(	stderr,
		"KBaseApp::~KBaseApp: nApp=%d\n",
		nApp
	)	;

	/* We need to shut down in an orderly way on account of various	*/
	/* dependancies (eg., forms depending on the database viewer).	*/
	/* Since closing a database viewer will close all associated	*/
	/* form, etc., viewers, we need only close these.		*/
	QListIterator<__TKPart>	piter (*m_pManager->parts()) ;
	__TKPart	        *part ;

	while ((part = piter.current()) != 0)
	{	KBDBaseViewer *dbv = ((KBasePart *)part)->isDBaseViewer() ;
		if (dbv != 0) delete dbv ;
		piter += 1 ;
        }

	/* All that remains, possibly, are the query and script log	*/
	/* windows.							*/
	if (queryLog ) queryLog ->widget()->close() ;
	if (scriptLog) scriptLog->widget()->close() ;

	if ((nApp -= 1) == 0)
	{
		m_runable = false ;
		qApp->exit (0)	  ;
	}
}

/*  KBaseApp								*/
/*  init	: Initialise common stuff				*/
/*  config	: TKConfig *	: Configuration information		*/
/*  (returns)	: void		:					*/

void	KBaseApp::init
	(	TKConfig	*config
	)
{
	static	bool	first	= true	;

	if (first)
	{
		kbaseApp	= this	;
		first		= false	;

		setCallBack		() ;
		KBOptions::loadOptions	() ;

#if	! __KB_EMBEDDED
		if	(__useMDI) useMDI = true  ;
		else if (__useSDI) useMDI = false ;
		else		   useMDI = KBOptions::getUseMDI () ;

//		fprintf
//		(	stderr,
//			"KBaseApp::init: __useMDI=%d __useSDI=%d useMDI=%d\n",
//			__useMDI,
//			__useSDI,
//			useMDI
//		)	;
#endif


		/* Create a single instance of the GUI elements which	*/
		/* appear in the default "empty" GUI, or are common to	*/
		/* multiple GUIs, and which we need to access		*/
		/* directly.						*/
		actOpenDatabase	= new TKAction
				  (	TR("&Open Database"),
				  	"fileopen",
				  	0,
				  	&director,
				  	SLOT(openDatabase()),
				  	0,
				  	"KB_openDatabase"
				  )	;
		actNewDatabase	= new TKAction
				  (	TR("&New Database"),
				  	"filenew",
				  	0,
				  	&director,
				  	SLOT(newDatabase ()),
				  	0,
				  	"KB_newDatabase"
				  )	;
		actQuit		= new TKAction
				  (	TR("&Quit"),
					"exit",
					0,
				  	&director,
					SLOT(quit()),
					0,
					"KB_quit"
				  )	;
		actShowOptions	= new TKAction
				  (	TR("&Options"),
				  	"options",
					0,
					&director,
					SLOT(showOptions ()),
					0,
					"KB_options"
				  )	;
		actRecentFiles	= new TKRecentFilesAction
				  (	TR("Open &Recent"),
					0,
					0,
					&director,
					SLOT(openRecent(const TKURL &)),
					0,
					"KB_openRecent"
				  )	;
#if	! __KB_RUNTIME
		actDatabases	= new TKActionMenu
				  (	TR("Show D&atabase"),
					(QObject    *)0,
					"KB_databases"
				  )	;
//		actPlayScore	= new TKAction
//				  (	TR("&Play Score"),
//				  	0,
//					0,
//					&director,
//					SLOT(playScore ()),
//					0,
//					"KB_score"
//				  )	;
		actPackage	= new TKAction
				  (	TR("Package database"),
				  	0,
					0,
					&director,
					SLOT(package   ()),
					0,
					"KB_package"
				  )	;
		actShowQueryLog = new TKToggleAction
				  (	TR("Show &Query Log"),
					"querylog",
					0,
					&director,
					SLOT(showQueryLog ()),
					0,
					"KB_queryLog"
				  )	;
		actShowEventLog = new TKToggleAction
				  (	TR("Show &Event Log"),
					0,
					0,
					&director,
					SLOT(showEventLog ()),
					0,
					"KB_eventLog"
				  )	;
		actShowScriptLog= new TKToggleAction
				  (	TR("Show &Script Log"),
					"scriptlog",
					0,
					&director,
					SLOT(showScriptLog()),
					0,
					"KB_scriptLog"
				  )	;
		actShowDebugger = new TKToggleAction
				  (	TR("Show &Debugger"),
					"debugger",
					0,
					&director,
					SLOT(showDebugger ()),
					0,
					"KB_showDebug"
				  )	;
		actSnapEnable	= new TKToggleAction
				  (	TR("Enable Snap To Grid"),
				        "snapenable",
				        0,
			  		0,
			  		0,
			  		0,
			  		"KB_snapEnable"
				  )	;
#endif
		actShowWindow	= new TKSelectAction
				  (	TR("Show &Window"),
					0,
					0,
					&director,
					SLOT(showWindow(const QString &)),
					0,
					"KB_showWindow"
				  )	;

		connect
		(	actShowWindow->popupMenu(),
			SIGNAL(aboutToShow   ()),
			&director,
			SLOT  (loadWindowList())
		)	;

#if	__KB_TKC 
		

		actHelpContents	= new TKAction
				  (	TR("&Contents"),
					"helpcontents",
					0,
					&director,
					SLOT(showHelpContents()),
					0,
					"KB_contents"
				  )	;

		actHelpAbout	= new TKAction
				  (	TR("&About Rekall"),
					"helpabout",
					0,
					&director,
					SLOT(showHelpAbout()),
					0,
					"KB_about"
				  )	;
#ifdef	_WIN32

		actHelpIndex	= new TKAction
				  (	TR("&Index"),
					"helpindex",
					0,
					&director, 
					SLOT(showHelpIndex()),
					0,
					"KB_index"
				  )	;

		actHelpSearch	= new TKAction
				  (	TR("&Search"),
				  	"helpsearch",
				  	0,
				  	&director, 
					SLOT(showHelpSearch()),
					0,
					"KB_search"
				  )	;

#endif
		actHelpPlugins	= new TKAction
				  (	TR("&Database Plugins"),
					"helpplugins",
					0,
					&director,
					SLOT(showHelpPlugins()),
					0,
					"KB_plugins"
				  )	;
#endif

		actRecentFiles ->loadEntries (config) ;

#if	! __KB_RUNTIME
		actShowDebugger->setEnabled  (false ) ;
		actSnapEnable  ->setChecked  (config->readBoolEntry("snapenable", true)) ;
#endif

		extern	void	loadRekallPlugins () ;
		loadRekallPlugins () ;
	}

	/* Add the global actions. The "addGlobalAction" method checks	*/
	/* for null pointers (ie., actions that were not created), so	*/
	/* just go through the lot, irrespective of build version.	*/
	addGlobalAction (actOpenDatabase ) ;
	addGlobalAction (actNewDatabase  ) ;
	addGlobalAction (actQuit	 ) ;
	addGlobalAction (actShowOptions  ) ;
	addGlobalAction (actRecentFiles  ) ;
	addGlobalAction (actDatabases    ) ;
	addGlobalAction (actShowQueryLog ) ;
	addGlobalAction (actShowEventLog ) ;
	addGlobalAction (actShowScriptLog) ;
	addGlobalAction (actShowDebugger ) ;
	addGlobalAction (actSnapEnable   ) ;
	addGlobalAction (actShowWindow   ) ;
	addGlobalAction (actHelpContents ) ;
	addGlobalAction (actHelpAbout    ) ;
	addGlobalAction (actHelpContents ) ;
	addGlobalAction (actHelpIndex    ) ;
	addGlobalAction (actHelpSearch   ) ;
	addGlobalAction (actHelpPlugins  ) ;
//	addGlobalAction (actPlayScore    ) ;
}


void	KBaseApp::addGlobalAction
	(	TKAction	*action
	)
{
	if (action != 0)
	{
		actionCollection()->insert (action->getAction()	  ) ;
		globalActions	   .insert (action->name(), action) ;
	}
}



/*  KBaseApp	:							*/
/*  openDBaseViewer							*/
/*		: Open database viewer					*/
/*  db		: const QString & : Database name			*/
/*  create	: bool		  : Create new database			*/
/*  add		: bool		  : Add to recent list			*/
/*  (returns)	: QWidget *	  : Auto-started form widget		*/

QWidget	*KBaseApp::openDBaseViewer
	(	const QString	&db,
		bool		create,
		bool		add
	)
{
	if (!QFileInfo(db).exists() && !create)
	{
		TKMessageBox::sorry
		(	0,
			TR("There is no database with the specified name"),
			TR("No such database")
		)	;
		return	0 ;
	}

	/* If we are not in MDI mode and the app. window already has	*/
	/* a database viewer, then create a new instance of the app. to	*/
	/* handle this database.					*/
	if (!useMDI && (m_dbViewer != 0))
	{
		new KBaseApp (db, create, add) ;
		return	0 ;
	}

	if (!__noNoteDB)
	{
		TKConfig *config = TKConfig::getConfig () ;
		config->setGroup   ("General Options")    ;
		config->writeEntry ("lastOpened", db)	  ;
		config->sync	   () ;
	}

	KBDBaseViewer *dbV = new KBDBaseViewer
			     (	useMDI ? (QWidget *)m_mdi : (QWidget *)this,
				this,
				db,
				create
			     )	;

	addViewer (dbV) ;

	/* If not in MDI mode then note the new database viewer as the	*/
	/* viewer for this instance of the app., and set it as the	*/
	/* central widget.						*/
	if (!useMDI)
	{	m_dbViewer = dbV ;
		setCentralWidget (dbV->getPartWidget()) ;
		createGUI	 (dbV) ;
		setCaption (dbV->getPartWidget()->caption()) ;
	}

	/* If the add flag is set then the database is added to the	*/
	/* recent files (databases) list ...				*/
	if (add)
	{	TKConfig *config = TKConfig::getConfig () ;
		actRecentFiles->addURL (TKURL(db))  ;
		actRecentFiles->saveEntries (config) ;
		config->sync () ;
	}

	actNewDatabase ->setEnabled (!KBOptions::getSingleDBOnly()) ;
	actOpenDatabase->setEnabled (!KBOptions::getSingleDBOnly()) ;
	actRecentFiles ->setEnabled (!KBOptions::getSingleDBOnly()) ;

	/* Then see if there is an auto-start form for the database, in	*/
	/* which case we attempt to start it up. This is done manually	*/
	/* here so that we can record the object for later.		*/
	KBLocation autoLoc ;
	if (dbV->autoStart (autoLoc))
	{
		QDict<QString>	empty	;
		KBError		error   ;
		KB::ShowRC	rc	;

		/* Load the appropriate object and then build the	*/
		/* document. For things like forms and reports this	*/
		/* will mean parsing the XML.				*/
		QGuardedPtr<KBObjBase> ob = loadObject (autoLoc) ;
		if (!ob) return 0 ;


		if (!ob->build (autoLoc, false, error))
		{
			delete	m_autoObj ;
			error.DISPLAY()	  ;
			return	0	  ;
		}

		/* Ready to go. Show the object. This will return the	*/
		/* viewer part, with is duly noted in MDI mode (or	*/
		/* null on error).					*/
		rc = ob->show (KB::ShowAsData, empty, (QWidget *)m_mdi, error, 0) ;
		fprintf
		(	stderr,
			"KBaseApp::openDBaseViewer: autostart returned %d\n",
			(int)rc
		)	;

		switch (rc)
		{
			case KB::ShowRCModal	:
				/* FIX ME!				*/
				return	0	;

			case KB::ShowRCError	:
				/* Failed. Report the error and tidy up	*/
				/* before leaving.			*/
				error.DISPLAY() ;
				if (ob) delete ob ;
				return	0	;

			case KB::ShowRCOK 	:
				/* Form now running. Add he part to the	*/
				/* to the viewer, and hide the main	*/
				/* database window.			*/
				m_autoObj = ob	;
				addViewer (ob->getPart ()) ;

				if (useMDI) dbV->hide () ;

				return	ob->getPart()->getMainWindow() ;

			default	:
				/* Any other return code is unexpected.	*/
				/* Give the programmer some feedback!	*/
				KBError::EFault
				(	TR("Unexpected exit code from autoStart form"),
					QString(TR("Exit code: %1")).arg((int)rc),
					__ERRLOCN
				)	;
				break	;
		}

		return	0 ;
	}

	return	0 ;
}

/*  KBaseApp								*/
/*  showingObj	: See if specified object exists			*/
/*  location	: KBLocation &	: Object location			*/
/*  (returns)	: KBObjBase  *	: Object				*/

KBObjBase
	*KBaseApp::showingObj
	(	KBLocation	&location
	)
{
	LITER
	(	KBObjBase,
		m_objects,
		obj,
		if (obj->showing (location)) return obj ;
	)

	return	0 ;
}

/*  KBaseApp								*/
/*  objectNode	: Gte top level node for object				*/
/*  location	: KBLocation &	: Object location			*/
/*  (returns)	: KBObjBase  *	: Object				*/

KBNode	*KBaseApp::objectNode
	(	KBLocation	&location
	)
{
	KBObjBase *obj = showingObj (location) ;
	return	obj == 0 ? 0 : obj->getTopNode () ;
}

/*  KBaseApp								*/
/*  queryClose	: Query whether to close or not				*/
/*  (returns)	: bool		: True if OK to close			*/

bool	KBaseApp::queryClose ()
{
#if	! __KB_RUNTIME
	/* See if there are any unsaved changes, and generally query	*/
	/* whether to close. This is ignored in the runtime, where any	*/
	/* such querying must be done from a script.			*/
	QListIterator<__TKPart> piter (*m_pManager->parts()) ;
	__TKPart		*part ;

	while ((part = piter.current()) != 0)
	{
		if (((KBasePart *)part)->getChanged(true))
			return	TKMessageBox::questionYesNo
				(	this,
					TR("There are unsaved changes: quit anyway?"),
					TR("Database close")
				)
				== TKMessageBox::Yes ;

		piter += 1 ;
	}

	if (isAvailable())
		if ((m_dbViewer != 0) || useMDI)
			if (TKMessageBox::questionYesNo
				(	this,
					nApp > 1 ? TR("Really close database") : TR("Really quit?"),
					TR("Database close")   
				)
				!= TKMessageBox::Yes
		   	   )	return false ;
#endif

	DELOBJ	(m_dbViewer) ;
	return	true ;
}

/*  KBaseApp								*/
/*  queryExit	: Actually, do final state saving			*/
/*  (returns)	: bool		: Always true				*/

bool	KBaseApp::queryExit ()
{

	TKConfig *config = TKConfig::getConfig () ;

	config->setGroup   ("General Options") ;
	config->writeEntry (useMDI ? "MDIGeometry" : "SDIGeometry", size()) ;
#if	! __KB_RUNTIME
	config->writeEntry ("snapenable", actSnapEnable->isChecked()) ;
#endif
	actRecentFiles->saveEntries (config) ;

	config->sync () ;
	return	true 	;
}


/*  KBaseApp								*/
/*  openDatabase: Open an existing database				*/
/*  (returns)	: void		:					*/

void	KBaseApp::openDatabase ()
{
	extern	int	getNumTypeObjects ()	;
	extern	int	getNumDataArrays  ()	;
	fprintf	(stderr, "Rekall: %u type objects\n", getNumTypeObjects()) ;
	fprintf	(stderr, "Rekall: %u data arrays\n",  getNumDataArrays ()) ;

	QString	name = KBFileDialog::getOpenFileName
		       (	".",
				filePatterns,
				parentWidget (),
				"Open database ..."
		       ) ;

	if (name.isEmpty()) return  ;

	fixSaveExtn     (name, defExtn) ;
	openDBaseViewer (name, false, true) ;
}

/*  KBaseApp								*/
/*  closeDatabase: Close a currently open database			*/
/*  dbv		 : KBDBaseViewer * : Associated database viewer		*/
/*  (returns)	 : void		   :					*/

void	KBaseApp::closeDatabase
	(	KBDBaseViewer	*dbv
	)
{
	if (TKMessageBox::questionYesNo
		(	this,
			TR("Really close database"),
			TR("Database close")   
		)
		!= TKMessageBox::Yes
	   )	return ;

	delete	 dbv   ;
	m_dbViewer = 0 ;

	setCentralWidget (0) ;

	actNewDatabase ->setEnabled (true) ;
	actOpenDatabase->setEnabled (true) ;
	actRecentFiles ->setEnabled (true) ;

	fprintf
	(	stderr,
		"KBaseApp::closeDatabase: useMDI=%d nApp=%d\n",
		useMDI,
		nApp
	)	;

	if (!useMDI)
	{
		if (nApp > 1)
		{
			close	()   ;
			return	;
		}
		else
		{
			TKConfig *config = TKConfig::getConfig () ;
			config->setGroup   ("General Options")    ;
			config->writeEntry ("lastOpened", QString("")) ;
			config->sync	   () ;
		}
	}
}

/*  KBaseApp								*/
/*  openRecent	: Open a recent database				*/
/*  url		: KURL/QUrl &	: Database URL				*/
/*  (returns)	: void		:					*/

void	KBaseApp::openRecent
	(	const TKURL	&url
	)
{
	actRecentFiles->setCurrentItem (-1) ;

	if (!QFileInfo(url.path()).exists())
	{
		TKConfig *config = TKConfig::getConfig() ;
		actRecentFiles->removeURL   (url   ) ;
		actRecentFiles->saveEntries (config) ;
		config->sync () ;

		TKMessageBox::sorry
		(	0,
			TR
			(	"Database file does not exist\n"
				"and has been removed from the recent files list"
			),
			TR("No such database")
		)	;

		return	;
	}

	openDBaseViewer (url.path(), false, false) ;
}

/*  KBaseApp								*/
/*  playScore 	: Replay a score					*/
/*  (returns)	: void		:					*/

void	KBaseApp::playScore ()
{
	QString	name = KBFileDialog::getOpenFileName
		       (	".",
				filePatterns,
				parentWidget (),
				"Play score ..."
		       ) ;

	if (name.isEmpty()) return  ;

	KBConductor *conductor	= KBConductor::self() ;
	KBError	    error	;

	if (!conductor->init (name, error, true))
	{
		error.DISPLAY() ;
		return		;
	}

	conductor->start()	;
}

/*  KBaseApp								*/
/*  package	: Package database					*/
/*  (returns)	: void		:					*/

void	KBaseApp::package ()
{
#if	! __KB_RUNTIME && __KB_EMBEDDED
	if (m_dbViewer != 0)
	{
		KBDBInfo  *dbInfo = m_dbViewer->getDBInfo () ;
		QString	   name	  = dbInfo    ->getDBName () ;

		int i1 = name.findRev ('/') ;
		if (i1 >= 0) name = name.mid  (i1 + 1) ;
		int i2 = name.findRev ('.') ;
		if (i2 >= 0) name = name.left (i2) ;

		KBRTBuild(dbInfo->getDBPath(), name).exec () ;
	}
#endif
}

/*  KBaseApp								*/
/*  showQueryLog: Hide or show the query log window			*/
/*  (returns)	: void		:					*/

void	KBaseApp::showQueryLog ()
{
#if	! __KB_RUNTIME
	if (queryLog)
		queryLog  ->widget()->close() ;
	else	queryLog  = new KBQueryLog
			    (   (QWidget *)m_mdi,	
				TR("Query log"),
				actShowQueryLog
			    )	;
#endif
}

/*  KBaseApp								*/
/*  showEventLog: Hide or show the event log window			*/
/*  (returns)	: void		:					*/

void	KBaseApp::showEventLog ()
{
#if	! __KB_RUNTIME
	if (eventLog)
		eventLog  ->widget()->close() ;
	else	eventLog  = new KBEventLog
			    (   (QWidget *)m_mdi,	
				TR("event log"),
				actShowEventLog
			    )	;
#endif
}

/*  KBaseApp								*/
/*  showScriptLog: Hide or show the script log window			*/
/*  (returns)	 : void		:					*/

void	KBaseApp::showScriptLog ()
{
#if	! __KB_RUNTIME
	if (scriptLog)
		scriptLog ->widget()->close() ;
	else	scriptLog = new KBTextLog
			    (	(QWidget *)m_mdi,
				TR("Script log"),
				actShowScriptLog
			    )	;
#endif
}

/*  KBaseApp								*/
/*  showDebugger : Hide or show the debugger window			*/
/*  (returns)	 : void		:					*/

void	KBaseApp::showDebugger ()
{
#if	! __KB_RUNTIME
	QString	   lang	 = "py" ;

	KBDebugPtr *dPtr = debuggers.find (lang) ;
	if (dPtr == 0) return	;

	if (*dPtr)
	{	(*dPtr)->widget()->close() ;
		return	;
	}

	KBError	   error    ;
	KBScriptIF *scrIface = LinkKBScript (lang, error) ;
	if (scrIface == 0)
	{
		error.DISPLAY () ;
		actShowDebugger->setChecked (false) ;
		return	;
	}

	if (!(*dPtr = scrIface->showDebug (actShowDebugger)))
	{	
		KBError::EError
		(	TR("Unable to load debugger"),
			QString("Language: %1").arg(lang),
			__ERRLOCN
		)	;
		actShowDebugger->setChecked (false) ;
		return	;
	}
#endif
}

/*  KBaseApp								*/
/*  newDatabase	: Create a mew database					*/
/*  (returns)	: void		:					*/

void	KBaseApp::newDatabase ()
{
#if	! __KB_RUNTIME
	/* Start by offering the new connection wizard. If the user	*/
	/* cancels then return. Otherwise, unless the user finishes	*/
	/* the wizard on the "initial" page, create the new database.	*/
	/* If this step is OK then the retrurn value from ::create is	*/
	/* the part to the rekall database file, and we can open it.	*/
	KBWizardConnect wizConn (0, QString::null) ;
	if (!wizConn.exec()) return ;

	if (wizConn.currentPageName() != "initial")
	{
		QString	dbPath	= wizConn.create () ;
		if (!dbPath.isEmpty())
			openDBaseViewer (dbPath, false, true) ;
		return	;
	}

	/* User exits wizard on the "initial" page, which means that	*/
	/* they want to do it manually ....				*/
	QString	name = KBFileDialog::getSaveFileName
		       (	"",
				filePatterns,
				parentWidget (),
				TR("Create database ...")
		       ) ;

	if (name.isEmpty()) return ;

	fixSaveExtn     (name, defExtn) ;
	openDBaseViewer (name, true, true) ;
#endif
}

/*  KBaseApp	:							*/
/*  showOptions	: Show global options dialog				*/
/*  (returns)	: void		:					*/

void	KBaseApp::showOptions ()
{
#if	! __KB_RUNTIME
	KBOptionsDlg optDlg  (this) ;
	if (!optDlg.exec ()) return ;
#endif
}

/*  KBaseApp								*/
/*  addObject	: Add object						*/
/*  object	: KBObjBase  *	: Object				*/
/*  location	: KBLocation &	: Location being displayed		*/
/*  (returns)	: void		:					*/

void	KBaseApp::addObject
	(	KBObjBase	*object,
		KBLocation	&location
	)
{
	if (m_dbViewer == 0)
	{
		QListIterator<__TKPart> piter (*m_pManager->parts()) ;
		__TKPart	        *part ;

		while ((part = piter.current()) != 0)
		{	KBDBaseViewer *dbv = ((KBasePart *)part)->isDBaseViewer() ;
			if ((dbv != 0) && dbv->addObject (location, object)) break ;
			piter += 1 ;
        	}
	}
	else	m_dbViewer->addObject (location, object) ;

	m_objects  .append (object) ;

	connect
	(	object, SIGNAL(destroyed   ()),
		this,   SLOT  (removeObject())
	)	;
}

/*  KBaseApp								*/
/*  removeObjec	: Remove object 					*/
/*  (returns)	: void		:					*/

void	KBaseApp::removeObject ()
{
	const KBObjBase *object  = (const KBObjBase *)sender () ;

	fprintf
	(	stderr,
		"KBaseApp::removeObject [%p][%p]\n",
		(void *)object,
		(void *)m_autoObj
	)	;


	/* The object is removed from the objects list, which is all we	*/
	/* need to do unless this is the object that was auto-started.	*/
	m_objects.remove (object) ;
	if (object != m_autoObj) return ;

	/* If this is the auto-start object being closed then close all	*/
	/* the other objects. This will result in the entire database	*/
	/* being closed down.						*/
	m_autoObj = 0 ;
	while (m_objects.count() > 0)
		delete m_objects.at(0) ;
}

/*  KBaseApp								*/
/*  addViewer	: Add child viewer					*/
/*  viewer	: KBasePart *	: Child viewer				*/
/*  (returns)	: void		:					*/

void	KBaseApp::addViewer
	(	KBasePart	*viewer
	)
{
	if (useMDI)
	{
		m_pManager->addPart (viewer, true) ;
		createGUI	    (viewer) ;
	}
}

/*  KBaseApp								*/
/*  loadObject	: Load an object					*/
/*  location	: KBLocation &  : Location to be displayed		*/
/*  (returns)	: KBObjBase *	: Object base or null on error		*/

KBObjBase
	*KBaseApp::loadObject
	(	KBLocation	&location
	)
{
	QString	      objname = QString("%1_obj").arg(location.docType) ;
	KBPartFactory *f      = location.getFactory() ;

	if (f == 0)
	{	fprintf	(stderr, "Failed to locate %s factory\n", (cchar *)location.docType) ;
		return	0 ;
	}

	KBObjBase *ob = (KBObjBase *)f->create((QWidget *)m_mdi, objname) ;
	if (ob == 0)
	{	fprintf	(stderr, "Failed to create %s\n", (cchar *)objname) ;
		return	0 ;
	}

	addObject (ob, location) ;
	return	  ob	;
}

/*  KBaseApp	:							*/
/*  openObject	: Open an extant object					*/
/*  location	: KBLocation &	 : Document location			*/
/*  showAs	: KB::ShowAs	 : Show in specified mode		*/
/*  pDict	: QDict<QString>&: Parameter dictionary			*/
/*  pError	: KBerror &	 : Error return				*/
/*  key		: const KBValue &: Link key				*/
/*  rDict	: QDict<QString>*: Results dictionary			*/
/*  (returns)	: KB::ShowRC	 : Success				*/

KB::ShowRC
	KBaseApp::openObject
	(	KBLocation	&location,
		KB::ShowAs	showAs,
		QDict<QString>	&pDict,
		KBError		&pError,
		const KBValue	&key,
		QDict<QString>	*rDict
	)
{
	QGuardedPtr<KBObjBase> 	ob	;
	KB::ShowRC		rc	;

	/* See if we are currently showing the sepcified object. If so	*/
	/* then just get it into the right mode. Special case: If this	*/
	/* is a request to directly print a report then we will always	*/
	/* create a new object.						*/
	if (showAs != KB::ShowAsReport)
	{	ob = showingObj(location) ;
		if (ob)
		{	::setActiveWindow (ob->getPart()->getMainWindow()) ;
			return	ob->show (showAs, pDict, 0, pError, key, rDict) ;
		}
	}

	/* Load the appropriate object and then build the document.	*/
	/* For things like forms and reports this will mean parsing the	*/
	/* XML.								*/
	ob = loadObject (location) ;
	if (!ob)
		return	KB::ShowRCError ;


	if (!ob->build (location, false, pError))
	{	delete	ob	;	
		return	KB::ShowRCError ;
	}

	/* Ready to go. Show the object in the required more with the	*/
	/* specified paramater dictionary. This will return the viewer	*/
	/* part, with is duly noted in MDI mode (or null on error).	*/
	rc = ob->show (showAs, pDict, (QWidget *)m_mdi, pError, key, rDict) ;

	switch (rc)
	{
		case KB::ShowRCReport	:
		case KB::ShowRCModal	:
			if (ob) delete ob	;
			return	KB::ShowRCOK	;

		case KB::ShowRCCancel	:
			if (ob) delete ob	;
			return	rc		;

		case KB::ShowRCError	 :
			if (ob) delete	ob 	;
			return	rc		;

		default	:
			break	;
	}

	addViewer (ob->getPart()) ;
	::setActiveWindow (ob->getPart()->getMainWindow()) ;

//	fprintf	  (stderr, "createGUI on %d %d\n",
//			   ob->getPart(),
//			   ob->getPart()->widget()) ;
//	createGUI (ob->getPart ()) ;

	return	KB::ShowRCOK ;
}

/*  KBaseApp	:							*/
/*  newObject	: Open an new object					*/
/*  location	: KBLocation &	: Document location			*/
/*  pError	: KBerror &	: Error return				*/
/*  (returns)	: bool		:					*/

bool	KBaseApp::newObject
	(	KBLocation	&location,
		KBError		&pError
	)
{
	QGuardedPtr<KBObjBase> 	ob	;
	KB::ShowRC		rc	;

	ob = loadObject (location) ;
	if (!ob)
		return false ;

	if (!ob->build (location, true, pError))
	{	delete	ob	;	
		return	false	;
	}

	/* Ready to go. Show the object in the required more with the	*/
	/* specified paramater dictionary. This will return the viewer	*/
	/* part, with is duly noted in MDI mode (or null on error).	*/
	rc = ob->show
	     (		KB::ShowAsDesign,
			dummyDict,
			(QWidget *)m_mdi,
			pError
	     )	;

	switch (rc)
	{
		case KB::ShowRCReport	:
		case KB::ShowRCModal	:
			if (ob) delete	ob ;
			return	true	;

		case KB::ShowRCCancel	:
			if (ob) delete	ob ;
			return	true	;

		case KB::ShowRCError	 :
			if (ob) delete	ob  ;
			return	false	;

		default	:
			break	;
	}

	addViewer (ob->getPart()) ;
	return	true	 ;
}

/*  KBaseApp								*/
/*  guiEnable	: Enable or disabled named GUI element			*/
/*  name	: cchar *	: Element name				*/
/*  enable	: bool		: Enable/disable			*/
/*  (returns)	: void		:					*/

void	KBaseApp::guiEnable
	(	cchar	*name,
		bool	enable
	)
{
	KBasePart *child = (KBasePart *)m_pManager->activePart() ;
	if (child != 0) child->guiEnable (name, enable) ;
}

/*  KBaseApp								*/
/*  guiEnable	: Enable or disabled a group of GUI elements		*/
/*  ggroup	: KB::GGroup	: Group					*/
/*  enable	: bool		: Enable/disable			*/
/*  (returns)	: void		:					*/

void	KBaseApp::guiEnable
	(	KB::GGroup	ggroup,
		bool		enable
	)
{
	KBasePart *child = (KBasePart *)m_pManager->activePart() ;
	if (child != 0) child->guiEnable (ggroup, enable) ;
}

void	KBaseApp::resizeEvent
	(	QResizeEvent	*e
	)
{
	TKMainWindow::resizeEvent (e) ;
}

/*  KBaseApp								*/
/*  getDebugIface: Get script interface for debugging			*/
/*  language	 : const QString & : Language				*/
/*  (returns)	 : KBScriptIF	   : Script interface or null		*/

KBScriptIF
	*KBaseApp::getDebugIface
	(	const QString	&
	)
{
	/* This (static) method is used to when we want to pass a	*/
	/* script to a debugger to be displayed. We work through the	*/
	/* script interface since we do not then need to expose the	*/
	/* debugger interface to the outside world.			*/
	KBDebugPtr *dPtr = debuggers.find ("py") ;

	return	dPtr == 0 ? 0 :
		!*dPtr	  ? 0 : (*dPtr)->getScriptIface() ;


}

/*  KBaseApp								*/
/*  showRawSQL	: Show raw SQL window					*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  server	: const QString & : Server name				*/
/*  (returns)	: KBRawSLQ *	  : Raw SQL display			*/

KBRawSQL*KBaseApp::showRawSQL
	(	KBDBInfo	*dbInfo,
		const QString	&server
	)
{
#if	! __KB_RUNTIME

	bool	 ok	 ;

#if	! __KB_EMBEDDED
	KBRawSQL *rawSQL = new KBRawSQL (m_mdi, dbInfo, server, ok) ;
#else
	KBRawSQL *rawSQL = new KBRawSQL (0,	dbInfo, server, ok) ;
#endif
	if (!ok)
	{	delete	rawSQL	;
		return	0	;
	}

	return	rawSQL	;
#else
	return	0	;
#endif
}

/*  KBaseApp								*/
/*  isRunable	: Return runable flag					*/
/*  (returns)	: bool		: Rekall is runnable			*/

bool	KBaseApp::isRunable ()
{
	/* This flag is cleared at startup if rekall should not run at	*/
	/* all, for instance the user cancels the initial open database	*/
	/* dialog when in the runtime version.				*/
	return	m_runable	;
}

/*  KBaseApp								*/
/*  isAvailable	: Return available flag					*/
/*  (returns)	: bool		: Database window can be shown		*/

bool	KBaseApp::isAvailable ()
{
#if	__KB_RUNTIME

	/* Runtime version, the database window is *never* made visible	*/
	/* to the user, whatever happens.				*/
	return	false		;

#elif	__KB_EMBEDDED

	/* Embedded non-runtime version. Since Qtopia does not seem to	*/
	/* handle hidden top-level window, we have to decide by hand if	*/
	/* the database window can be show. The flag is set if either	*/
	/* we open to the database window, or the user selects it.	*/
	return	m_available	;
#else

	/* Desktop non-runtime version, nice and easy, just use the	*/
	/* window's visibility.						*/
	return	isVisible ()	;
#endif
}

/*  KBaseApp								*/
/*  setAvailable: Set the available flag				*/
/*  (returns)	: void		:					*/

void	KBaseApp::setAvailable
	(	bool	available
	)
{
	/* This is called if the database window becomes displayed,	*/
	/* such as if the user selects it from the windows menu.	*/
	m_available = available ;
}
