/***************************************************************************
    file	         : kb_filelist.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>

#include	<qdatetime.h>
#include	<qtextstream.h>
#include	<qtabwidget.h>

#if		__KB_EMBEDDED
#include	<qpe/qpeapplication.h>
#endif


#include	"kb_dbinfo.h"
#include	"kb_dbdociter.h"
#include	"kb_serverinfo.h"
#include	"kb_notifier.h"
#include	"kb_svrchooser.h"
#include	"kb_prompt.h"

#ifndef		_WIN32
#include	"kb_filelist.moc"
#else
#include 	"kb_filelist.h"
#endif

#include	"kb_callback.h"
#include	"kb_appptr.h"

#include	"tk_icons.h"
#include	"tk_messagebox.h"
#include	"tk_filedialog.h"


static	QString	lHead	("0_")	;
static	QString	lSort	("1_")	;
static	QString	lTail	("2_")	;



/*  KBFileList								*/
/*  KBFileList	: Constructor for file list				*/
/*  parent	: QWidget *	  : Parent widget			*/
/*  dbInfo	: KBDBInfo *	  : Database information object		*/
/*  newtext	: cchar *	  : Text for new object entry		*/
/*  icon	: cchar *	  : Icon for entries			*/
/*  col0	: cchar *	  : Column zero title			*/
/*  col2	: cchar *	  : Column two title or null		*/
/*  col3	: cchar *	  : Column three title or null		*/
/*  (returns)	: KBFileList	  :					*/

KBFileList::KBFileList
	(	QWidget		*parent,
		KBDBInfo	*dbInfo,
		cchar		*newtext,
		cchar		*icon,
		cchar		*tabType,
		cchar		*col0,
		cchar		*col2,
		cchar		*col3
	)
	:
	QListView	(parent),
	KBPlayer	("filelist", tabType, this),
	m_parent	(parent),
	m_dbInfo	(dbInfo),
	m_newtext	(newtext),
	m_icon		(icon),
	m_tabType	(tabType),
	m_wizard	(false)
{
#if	__KB_EMBEDDED
	QPEApplication::setStylusOperation
	(	this->viewport(),
		QPEApplication::RightOnHold
	)	;
#endif

	if (col0 != 0)
	{
		addColumn (col0,           200) ;
		addColumn (TR("Modified"), 200) ;
		if (col2 != 0) addColumn (col2) ;
		if (col3 != 0) addColumn (col3) ;
	}

	setRootIsDecorated (true)  ;
	setSorting	   (0)	   ;


	connect	 (this,	SIGNAL(doubleClicked     (QListViewItem *)),
		  	SLOT  (showDefault       (QListViewItem *))) ;
	connect	 (this,	SIGNAL(returnPressed     (QListViewItem *)),
		  	SLOT  (showDefault       (QListViewItem *))) ;
	connect	 (this,	SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int)),
		  	SLOT  (showMenu          (QListViewItem *, const QPoint &, int))) ;

	connect  (KBNotifier::self(),
		  SIGNAL (sServerChanged(const KBLocation &)),
		  SLOT   (serverChanged (const KBLocation &))) ;

	connect	 (KBNotifier::self(),
		  SIGNAL (sObjectChanged(const KBLocation &)),
		  SLOT	 (objChange	(const KBLocation &))) ;


	KBListItem *item ;

	item	= new KBServerItem (this, lHead, KBLocation::m_pFile) ;
	item->setPixmap (0, getSmallIcon(m_tabType == "table" ? "database" : "folder_open")) ;

	QListIterator<KBServerInfo> *svIter = dbInfo->getServerIter() ;
	KBServerInfo		    *svInfo ;

	while ((svInfo = svIter->current()) != 0)
	{
		if (!svInfo->disabled())
		{
			item = new KBServerItem (this, lSort, svInfo->serverName()) ;
			item->setPixmap (0, getSmallIcon("database")) ;
		}

		(*svIter) += 1 ;
	}

	delete	svIter	;
}

/*  KBFileList								*/
/*  ~KBFileList	: Destructor for form list widget			*/
/*  (returns)	:		:					*/

KBFileList::~KBFileList ()
{
}

/*  KBFileList								*/
/*  itemToLocation							*/
/*		: Build location object for specified item		*/
/*  item	: KBListItem *	: The item				*/
/*  location	: KBLocation &	: Return location			*/
/*  (returns)	: bool		: Success				*/

bool	KBFileList::itemToLocation
	(	KBListItem	*item,
		KBLocation	&location
	)
{
	if (item->type() != KBListItem::Object) return false ;


	QString	docLocn  = item->parent()->text (0) ;
	QString	docName  = item->	   text (0) ;

	location = KBLocation
		   (	m_dbInfo,
			m_tabType,
			docLocn,
			docName,
			getDocExtension()
		   )	;
	return	true	;
}


/*  KBFileList								*/
/*  reloadServer: Load/reload lists					*/
/*  server	: KBServerItem * : Server to reload			*/
/*  (returns)	: void		 :					*/

void	KBFileList::reloadServer
	(	KBServerItem	*server
	)
{
	KBError	    error   ;
	KBDBDocIter docIter ;

	/* Clear the existing entries. There doesn't seem to be a Qt	*/
	/* call to do this, so do it by hand. Then create the new-entry	*/
	/* item or items.						*/
	KBListItem  *t	;
	while ((t = (KBListItem *)server->firstChild()) != 0)
		delete t ;

#if	! __KB_RUNTIME
	t = new KBListItem (server, lHead, m_newtext) ;
	t->setPixmap (0, getSmallIcon("filenew")) ;
	t->setType   (KBListItem::Create) ;

	if (m_wizard)
	{
		t = new KBListItem (server, lHead, QString(TR("%1 with wizard")).arg(m_newtext)) ;
		t->setPixmap (0, getSmallIcon("filenew")) ;
		t->setType   (KBListItem::Wizard) ;
	}
#endif

	/* Nothing to do if the server is disabled ....			*/
	KBServerInfo *s = m_dbInfo->findServer (server->text(0)) ;
	if ((s != 0) && s->disabled()) return ;


	/* Initialise the document iterator. Here we do not consider it	*/
	/* an error if there is no objects table, we simply won't	*/
	/* display anything.						*/
	if (!docIter.init
		(	m_dbInfo,
			server->text (0),
			m_tabType,
			KBLocation::extnForType (m_dbInfo, m_tabType, getDocExtension()),
			error,
			false
		))
	{
		error.DISPLAY() ;
		return	;
	}

	QString	name	;
	QString	stamp	;

	/* Now iterate through and add entries for each document. We	*/
	/* convert the timestamp to something more readable.		*/
	while (docIter.getNextDoc (name, stamp))
	{
		int	tm_year	;
		int	tm_mon	;
		int	tm_mday	;
		int	tm_hour	;
		int	tm_min	;
		int	tm_sec	;

		if (!stamp.isNull())
			sscanf	  ((cchar *)stamp, "%4d%2d%2d%2d%2d%2d",
						   &tm_year, &tm_mon,
						   &tm_mday, &tm_hour,
						   &tm_min,  &tm_sec
				  )	;

		t = new KBObjectItem
		    (	server,
			lSort,
			name,
			QDateTime
			(	QDate(tm_year, tm_mon, tm_mday),
				QTime(tm_hour, tm_min, tm_sec )
			)	.toString()
		    )	;

		t->setPixmap (0, getSmallIcon(m_icon)) ;
	}
}

/*  KBFileList								*/
/*  reloadServer: Load/reload lists					*/
/*  (returns)	: void		  :					*/

void	KBFileList::reloadServer ()
{
	if ((m_curItem != 0) && (m_curItem->type() == KBListItem::Server))
		reloadServer ((KBServerItem *)m_curItem) ;
}

/*  KBFileList								*/
/*  showObjecAs: Handle request to open an object in specified mode	*/
/*  item	: KBListItem *	: Selected item				*/
/*  showAs	: KB::ShowAs	: Show in design mode			*/
/*  (returns)	: void		:					*/

void	KBFileList::showObjectAs
	(	KBListItem	*item,
		KB::ShowAs	showAs
	)
{
	KBLocation	location ;
	KBError		error 	 ;
	QDict<QString>	empty	 ;
	KBCallback	*cb	 = KBAppPtr::getCallback() ;

	if (!itemToLocation (item, location))
		return ;

	if (cb->openObject (location, showAs, empty, error) == KB::ShowRCError)
		error.DISPLAY () ;
}

/*  KBFileList								*/
/*  createByWizard							*/
/*		: Handle request to create object using a wizard	*/
/*  item	: KBListItem *	: Selected item				*/
/*  (returns)	: void		:					*/

void	KBFileList::createByWizard
	(	KBListItem	*
	)
{
	/* This is just a place holder, and will be implemented by	*/
	/* object types that have creation wizards.			*/
}

/*  KBFileList								*/
/*  showDefault	: Default double-click/return operation			*/
/*  item	: QListViewItem *: Selected item			*/
/*  (returns)	: void		 :					*/

void	KBFileList::showDefault
	(	QListViewItem *item
	)
{
	if ((m_curItem = (KBListItem *)item) == 0)
		return ;

	if (m_curItem->type() == KBListItem::Create)
	{
		KBError		error	 ;
		KBCallback	*cb	= KBAppPtr::getCallback() ;

		KBLocation	location
				(	m_dbInfo,
					m_tabType,
					m_curItem->parent()->text(0),
					"",
					getDocExtension()
				)	;

		if (! cb->newObject (location, error)) error.DISPLAY() ;
		return	;
	}

	if (m_curItem->type() == KBListItem::Wizard)
	{
		createByWizard (m_curItem) ;
		return	;
	}

	if (m_curItem->type() == KBListItem::Object)
	{
		showObjectAs   (m_curItem, KB::ShowAsData) ;
		return	;
	}
}

/*  KBFileList								*/
/*  showAsData	: Show popup item in data view				*/
/*  (returns)	: void		 :					*/

void	KBFileList::showAsData ()
{
	showObjectAs (m_curItem, KB::ShowAsData  ) ;
}

/*  KBFileList								*/
/*  showAsDesign: Show popup item in design view				*/
/*  (returns)	: void		 :					*/

void	KBFileList::showAsDesign ()
{
#if	! __KB_RUNTIME
	showObjectAs (m_curItem, KB::ShowAsDesign) ;
#endif
}

/*  KBFileList								*/
/*  canOperate	: Check if we can operate on a specified object type	*/
/*  location	: KBLocation &	: Location for operation		*/
/*  op		: cchar *	: Putative operation			*/
/*  (returns)	: bool		: Operation allowed			*/

bool	KBFileList::canOperate
	(	KBLocation	&location,
		cchar		*op
	)
{
	if (KBAppPtr::getCallback()->showingObj (location) != 0)
	{
		TKMessageBox::sorry
		(	0,
			QString (TR("%1 %2 is currently open")).arg(m_tabType).arg(location.docName),
			QString	(TR("Unable to %1")).arg(op)
		)	;
		return	false	;
	}

	return	true	;
}

/*  KBFileList								*/
/*  canOperate	: Check if we can operate on a specified object type	*/
/*  item	: KBListItem *	: Item selected for operation		*/
/*  op		: cchar *	: Putative operation			*/
/*  (returns)	: bool		: Operation allowed			*/

bool	KBFileList::canOperate
	(	KBListItem	*item,
		cchar		*op
	)
{
	KBLocation location ;

	if (!itemToLocation (item, location))
		return false ;

	return	canOperate (location, op) ;
}

/*  KBFileList								*/
/*  rename	: Rename an object					*/
/*  item	: KBListItem *	: Item selected for operation		*/
/*  (returns)	: void		:					*/

void	KBFileList::rename
	(	KBListItem	*item
	)
{
	KBLocation location ;
	KBError	   error    ;

	if (!itemToLocation (item, location)) return ;
	if (!canOperate (location, "rename")) return ;

	/* Prompt the user for a new name for the object, with the	*/
	/* usual get-out.						*/
	QString	newName	;
	if (!doPrompt
	    (	QString (TR("Rename %1 ...")).arg(m_tabType),
	    	QString (TR("Please enter the new %1 name")).arg(m_tabType),
		newName
	    ))	return  ;


	if (!location.rename (newName, error)) error.DISPLAY() ;

	/* Reload the form list. Do this whether or not there was an	*/
	/* error; this ensures that the displayed list is correct.	*/
	reloadServer ((KBServerItem *)item->parent()) ;
}

/*  KBFileList								*/
/*  delobj	: Delete an object					*/
/*  item	: KBListItem *	: Item selected for operation		*/
/*  (returns)	: void		:					*/

void	KBFileList::delobj
	(	KBListItem	*item
	)
{
	KBLocation location ;
	KBError	   error    ;

	if (!itemToLocation (item, location)) return ;
	if (!canOperate (location, "delete")) return ;

	if (TKMessageBox::questionYesNo
	   	(	0,
			QString (TR("Definitely delete %1?")).arg(location.docName),
			TR("Delete document")
		)
		!= TKMessageBox::Yes) return ;


	if (!location.remove (error)) error.DISPLAY() ;

	/* Reload the entire list. Do this whether or not there was an	*/
	/* error; this ensures that the displayed list is correct.	*/
	reloadServer ((KBServerItem *)item->parent()) ;
}

/*  KBFileList								*/
/*  serverChange: Notification of server changes			*/
/*  location	: const KBLocation & : Location changed			*/
/*  (returns)	: void		     :					*/

void	KBFileList::serverChanged
	(	const KBLocation &location
	)
{
	KBServerItem *item = (KBServerItem *)firstChild() ;

	/* Scan looking for the corresponding entry. When found, update	*/
	/* the name (may or may not have changed), and drop all the	*/
	/* children documents (the server may now point somewhere else)	*/
	while (item != 0)
	{
		if (location.docLocn == item->text(0))
		{
			item->setText (0, location.docName) ;
			reloadServer (item) ;
			return	;
		}

		item	= (KBServerItem *)item->nextSibling() ;
	}

	/* Should only get here for new servers ....			*/
	KBServerInfo *svInfo = m_dbInfo->findServer (location.docName) ;
	if ((svInfo != 0) && !svInfo->disabled())
	{
		item = new KBServerItem (this, lSort, svInfo->serverName()) ;
		item->setPixmap (0, getSmallIcon("database")) ;
	}
}

/*  KBFileList								*/
/*  objChange	: Notification of object changes			*/
/*  location	: const KBLocation & : Location changed			*/
/*  (returns)	: void		     :					*/

void	KBFileList::objChange
	(	const KBLocation &location
	)
{
	KBServerItem *item = (KBServerItem *)firstChild() ;

	while (item != 0)
	{
		if (location.docLocn == item->text(0))
		{
			reloadServer (item) ;
			break	;
		}

		item	= (KBServerItem *)item->nextSibling() ;
	}
}

/*  KBFileList								*/
/*  saveToFile	: Save document to a file				*/
/*  (returns)	: void		L					*/

void	KBFileList::saveToFile ()
{
	KBLocation location ;
	KBError	   error    ;

	if (!itemToLocation (m_curItem, location)) return ;
	if (!canOperate     (location, " delete")) return ;

	KBFileDialog	fDlg
			(	".",
				"*.*|All file types",
				qApp->activeWindow(),
				"saveobject",
				true
			)	;

#if	! __KB_EMBEDDED
	fDlg.setSelection (location.docName)		;
#endif
	fDlg.setMode	  (KBFileDialog::AnyFile)	;
	fDlg.setCaption   (TR("Save to file ...."))	;

	if (!fDlg.exec ()) return ;

	QFile	file	;
	QString	name	= fDlg.selectedFile () ;
	file.setName	(name) ;

	if (QFileInfo(file).exists())
		if (TKMessageBox::questionYesNo
		   	(	0,
				QString (TR("%1 already exists: overwrite?")).arg(name),
				TR("Save to file ....")
			)
			!= TKMessageBox::Yes) return ;

	QString	doc	= location.contents (error) ;
	if (doc.isNull())
	{	error.DISPLAY() ;
		return	;
	}

	if (!file.open (IO_WriteOnly|IO_Truncate))
	{
		KBError::EError
		(	QString	(TR("Cannot open \"%1\"")).arg(name),
			strerror(errno),
			__ERRLOCN
		)	;
		return	;
	}


	QTextStream (&file) << doc ;
}

/*  KBFileList								*/
/*  copyToServer: Copy objects to specified server			*/
/*  (returns)	: void		:					*/

void	KBFileList::copyToServer ()
{
	/* If there is no child then the branch has not been opened	*/
	/* so do it now. The check that follows is then for the sake of	*/
	/* paranoia.							*/
	if (m_curItem->firstChild() == 0)
		reloadServer () ;

	if (m_curItem->firstChild() == 0)
		return ;

	QString		docLocn = m_curItem->text(0) ;
	QListViewItem   *item   = m_curItem->firstChild()->nextSibling() ;
	KBSvrChooserDlg chooser (m_dbInfo, docLocn, true) ;

	while (item != 0)
	{
		chooser.addEntry (item->text(0)) ;
		item = item->nextSibling() ;
	}

	if (!chooser.exec ()) return ;

	QString		target	;
	QStringList	forms	;
	bool		over	;

	chooser.getInfo (target, forms, over) ;

	for (uint idx = 0 ; idx < forms.count() ; idx += 1)
	{
		QString		name	= forms[idx] ;
		KBLocation	srce	(m_dbInfo, m_tabType, docLocn, name, getDocExtension()) ;
		KBLocation	dest	(m_dbInfo, m_tabType, target,  name, getDocExtension()) ;

		if (!over && dest.exists())
			if (TKMessageBox::questionYesNo
			   	(	0,
					QString (TR("%1 exists on server %2: overwrite>"))
						.arg(name  )
						.arg(target),
					TR("Overwrite document ...")
				)
				!= TKMessageBox::Yes) continue ;

		KBError		error	;
		QString		text	= srce.contents (error) ;

		if (text.isNull())
		{
			error.DISPLAY() ;
			continue	;
		}

		if (!dest.save (QString::null, QString::null, text, error))
		{
			error.DISPLAY() ;
			continue	;
		}
	}
}

#if	! __KB_RUNTIME

/*  KBFileList								*/
/*  playerPerform							*/
/*		: Perform a stanza from a score				*/
/*  action	: const QString		: Action			*/
/*  args	: const QStringList &	: Arguemnts			*/
/*  pError	: KBError &		: Error return			*/
/*  (returns)	; bool			: Success			*/

bool	KBFileList::playerPerform
	(	const QString		&action,
		const QStringList	&args,
		KBError			&pError
	)
{
	/* Show								*/
	/* Make sure that this list is showing within a QTabWidget. We	*/
	/* have to check by hand that we are in a tabbed widget.	*/
	if (action == "show")
	{
		if (m_parent->inherits("QTabWidget"))
		{
			QTabWidget *tw = (QTabWidget *)m_parent ;
			tw->setCurrentPage (tw->indexOf(this))  ;
			return	true	;
		}

		pError	= KBError
			  (	KBError::Error,
				"File list does not have tab parent",
				QString	("%1: %1")
					.arg(m_tabType)
					.arg(m_parent->className()),
				__ERRLOCN
			  )	;
		return	false	;
	}

	/* Remaining actions need a server which will be specified in	*/
	/* argument zero, so look for that first.			*/
	QString		server	= args[0]	;
	QListViewItem	*item	= firstChild () ;
	while (item != 0)
	{	if (item->text(0) == server) break ;
		item	= item->nextSibling() ;
	}

	if (item == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				"File list does not contain server",
				QString("Server: %1").arg(server),
				__ERRLOCN
			  )	;
		return	false	;
	}


	/* Expand							*/
	/* Expand or collapse the branch. Argument one specifies which	*/
	/* is the case.							*/
	if (action == "expand")
	{
		item->setOpen (args[1].toInt()) ;
		return	true  ;
	}

	/* Create							*/
	/* Create a new object. Unlike user operation, we supply the	*/
	/* name at creation, to avoid the SaveAs dialog later.		*/
	if (action == "create")
	{
		KBError		error	 ;
		KBLocation	location (m_dbInfo, m_tabType, server, args[1]) ;
		KBCallback	*cb	= KBAppPtr::getCallback() ;

		return	cb->newObject (location, pError) ;
	}

	/* Remaining actions need a form which will be specified in	*/
	/* argument one, so look for that.				*/
	QString	name	= args[1]	;
	item	= item->firstChild ()	;
	while (item != 0)
	{	if (item->text(0) == name) break ;
		item	= item->nextSibling() ;
	}

	if (item == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				"File list does not contain object",
				QString("%1, %2").arg(server).arg(name),
				__ERRLOCN
			  )	;
		return	false	;
	}

	/* Open								*/
	/* Open a document. Argument two specifies whether this should	*/
	/* be in design mode or not.					*/
	if (action == "open")
	{
		bool	design	= args[2].toInt() ;
		showObjectAs ((KBListItem *)item, design ? KB::ShowAsDesign : KB::ShowAsData) ;
		return	 true	;
	}

	/* Anything else is unrecognised and is an error.		*/
	pError	= KBError
		  (	KBError::Error,
			"Player: Unknown filelist action",
			action,
			__ERRLOCN
		  )	;
	return	false	;
}
#endif

/*  KBFileList								*/
/*  getDocExtension							*/
/*		: Get document name extension				*/
/*  (returns)	: QString	: Extension				*/

QString	KBFileList::getDocExtension ()
{
	/* This method is overridden for object types where there is	*/
	/* an extension which is independant of Rekall itself, eg., for	*/
	/* scripts where the entension is the script file extension	*/
	/* (such as "py" for python). In all other cases it is empty.	*/
	return	""	;
}

/*  KBFileList								*/
/*  getTabType	: Return the type string as used in the objects table	*/
/*  (returns)	: QString	: Type					*/

QString	KBFileList::getTabType ()
{
	return	m_tabType	;
}

/*  KBFileList								*/
/*  getObjectNames							*/
/*		: Get list of all objects on specified server		*/
/*  server	: KBServerInfo * : Server				*/
/*  (returns)	: QStringList	 : List of names			*/

QStringList
	KBFileList::getObjectNames
	(	KBServerInfo	*server
	)
{
	KBError		error   ;
	KBDBDocIter	docIter ;

	if (!docIter.init
		(	m_dbInfo,
			server->serverName(),
			m_tabType,
			KBLocation::extnForType (m_dbInfo, m_tabType, getDocExtension()),
			error,
			false
		))
	{
		error.DISPLAY() ;
		return	QStringList () ;
	}


	QString		name	;
	QString		stamp	;
	QStringList	nameList;

	while (docIter.getNextDoc (name, stamp))
	{
		nameList.append (name) ;
	}

	return	nameList ;
}


/*  showAsCode	: Convert show-as value from strings to code		*/
/*  text	: const QString & : String value			*/
/*  deflt	: KB::ShowAs	  : Default value			*/
/*  (returns)	: KB::ShowAs	  : Valuu as code			*/

LIBAPP_API
	KB::ShowAs showAsCode
	(	const QString	&text,
		KB::ShowAs	deflt
	)
{
	if (text == "ShowAsData"   ) return KB::ShowAsData	;
	if (text == "ShowAsPrint"  ) return KB::ShowAsPrint	;
	if (text == "ShowAsPreview") return KB::ShowAsPreview	;
	if (text == "ShowAsReport" ) return KB::ShowAsReport	;
	if (text == "ShowAsDesign" ) return KB::ShowAsDesign	;

	TKMessageBox::sorry
	(	0,
		QString(TR("Unknown show-as code: %1")).arg(text),
		TR("ShowAs error")
	)	;

	return	deflt	;
}

void	KBFileList::showServerMenu ()
{
}

void	KBFileList::showCreateMenu ()
{
}

void	KBFileList::showObjectMenu ()
{
}

void	KBFileList::showMenu
	(	QListViewItem	*item,
		const QPoint	&,
		int
	)
{
	if ((m_curItem = (KBListItem *)item) == 0)
		return	;

	switch (m_curItem->type())
	{
		case KBListItem::Server	:
			showServerMenu	() ;
			return	;

		case KBListItem::Create	:
			showCreateMenu	() ;
			return	;

		case KBListItem::Object	:
			showObjectMenu	() ;
			return	;

		default	:
			break	;
	}
}

