/***************************************************************************
    file	         : kb_dbdociter.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qimage.h>


#include	"kb_dbdociter.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_dbinfo.h"
#include	"kb_location.h"

#include	"kb_locator.h"


/*  imageFilters: Get filters for images for QDir::setNameFilter	*/
/*  (returns)	: QString	: Filter				*/

static	QString	imageFilters ()
{
	QString		res	;
	QStrList	formats	= QImageIO::inputFormats() ;
	cchar		*sep	= "" ;

	for (uint idx = 0 ; idx < formats.count() ; idx += 1)
	{
		QString	extn	= QString(formats.at(idx)).lower() ;

		res	+= QString("%1*.%2").arg(sep).arg(extn) ;
		sep	 = ";"	;

		if (extn == "jpeg") res	+= ";*.jpg" ;
	}

	fprintf	(stderr, "imageFilters->[%s]\n", (cchar *)res) ;
	return	res	;
}


/*  KBDBDocIter								*/
/*  KBDBDocIter	: Constructor for document iterator			*/
/*  withExtn	: bool		: Return names with extensions		*/
/*  (returns)	: KBDBDocIter	: Iterator				*/

KBDBDocIter::KBDBDocIter
	(	bool	withExtn
	)
	:
	m_withExtn (withExtn)
{
	m_dbIter = 0	;
	m_select = 0	;
}

/*  KBDBDocIter								*/
/*  ~KBDBDocIter: Destructor for document iterator			*/
/*  (returns)	:		:					*/


KBDBDocIter::~KBDBDocIter ()
{
	DELOBJ	(m_dbIter) ;
	DELOBJ	(m_select) ;
}

/*  KBDBDocIter								*/
/*  init	: Initialise iterator					*/
/*  dbInfo	: KBDBinfo *	  : Database information		*/
/*  svrName	: const QString & : Server name (or _files)		*/
/*  type	: const QString & : Document type			*/
/*  extn	: const QString & : Document entension			*/
/*  pError	: KBError &	  : Error return			*/
/*  needObjTab	: bool		  : Missing object table is an error	*/
/*  (returns)	: bool		  : Success				*/

bool	KBDBDocIter::init
	(	KBDBInfo	*dbInfo,
		const QString	&svrName,
		const QString	&type,
		const QString	&extn,
		KBError		&pError,
		bool		needObjTab
	)
{
	DELOBJ	(m_dbIter) ;
	DELOBJ	(m_select) ;

	fprintf
	(	stderr,
		"KBDBDocIter::init(%s,%s,%s,%s)\n",
		(cchar *)dbInfo->getDBExtn(),
		(cchar *)svrName,
		(cchar *)type,
		(cchar *)extn
	)	;

	/* Easy case is if we are iterating files, in which case the QT	*/
	/* classes can do all the work.	First case is the directory in	*/
	/* which the database file exists.				*/
	if (svrName == KBLocation::m_pFile)
	{
		m_dbDir.setPath		(dbInfo->getDBPath()) ;

		if (type == "graphic")
			m_dbDir.setNameFilter (imageFilters ()) ;
		else	m_dbDir.setNameFilter ("*." + extn) ;

		m_dbDir.setFilter	(QDir::Files | QDir::Readable) ;
		m_dbDir.setSorting	(QDir::Name ) ;


		if (m_dbDir.isReadable() && (m_dbDir.entryInfoList() != 0))
			m_dbIter	= new QFileInfoListIterator (*m_dbDir.entryInfoList()) ;
		return	true	;
	}

	/* The second similar case is for the stock components, which	*/
	/* are distributed with rekall.					*/
	if (svrName == KBLocation::m_pStock)
	{
		QString	dir	= locateDir
				  (	"appdata",
					QString("stock/%1/dummy").arg(type)
				  )	;

		fprintf
		(	stderr,
			"KBDBDocIter::init: [%s][%s]->[%s]\n",
			(cchar *)type,
			(cchar *)extn,
			(cchar *)dir
		)	;

		m_dbDir.setPath		(QString("%1/stock/%2").arg(dir).arg(type)) ;
		m_dbDir.setNameFilter	("*." + extn) ;
		m_dbDir.setFilter	(QDir::Files | QDir::Readable) ;
		m_dbDir.setSorting	(QDir::Name ) ;

		if (m_dbDir.isReadable() && (m_dbDir.entryInfoList() != 0))
			m_dbIter	= new QFileInfoListIterator (*m_dbDir.entryInfoList()) ;
		return	true	;
	}

	/* OK, database. First connect to the server and check that the	*/
	/* objects table exists therein. If not, then it may be treated	*/
	/* as an error, or as OK in which case no documents will result	*/
	if (!m_dbLink.connect (dbInfo, svrName, true))
	{	pError	= m_dbLink.lastError() ;
		return	false	;
	}

	bool	exists	  ;
	QString	objectTab = m_dbLink.rekallPrefix (OBJECTTABNAME) ;

	if (!m_dbLink.tableExists (objectTab, exists))
	{	pError	= m_dbLink.lastError() ;
		return	false	;
	}
	if (!exists)
	{
		if (needObjTab)
		{	pError	= KBError
				  (	KBError::Error,
					"Server does not have an objects table",
					QString::null,
					__ERRLOCN
				  )	;
			return	false	;
		}
		return true	;
	}

	/* Construct the query. The special case is scripts, in which	*/
	/* case we need to add the extension to the query (to handle	*/
	/* multiple scripting languages).				*/
	QString	query	;
	KBValue	args[2]	;
	uint	nargs	;

	query	= QString ("select %1, %2, %3 from %4 where %5 = %6")
			  .arg(m_dbLink.mapExpression("Name"	 ))
			  .arg(m_dbLink.mapExpression("SaveDate" ))
			  .arg(m_dbLink.mapExpression("Extension"))
			  .arg(m_dbLink.mapExpression(objectTab  ))
			  .arg(m_dbLink.mapExpression("Type"	 ))
			  .arg(m_dbLink.placeHolder  (0))
			  ;

	args[0]	= QString(type)	;
	nargs	= 1	;

	if (type == "script")
	{
		query	+= QString(" and %1 = %2")
				  .arg(m_dbLink.mapExpression("Extension"))
				  .arg(m_dbLink.placeHolder  (1)) ;
		args[1]	 = QString(extn) ;
		nargs	 = 2 ;
	}

	query	+= QString(" order by %1").arg(m_dbLink.mapExpression("Name")) ;


	/* Instantiate and execute the query, checking for errors. If	*/
	/* OK then start the row iterator at zero ...			*/
	if ((m_select = m_dbLink.qrySelect (false, query)) == 0)
	{	pError	= m_dbLink.lastError() ;
		return	false	;
	}

	if (!m_select->execute (nargs, args))	
	{	pError	= m_select->lastError() ;
		return	false	;
	}

	m_nextRow = 0	;
	return	true	;
}


/*  KBDBDocIter								*/
/*  getNextDoc	: Get next document					*/
/*  name	: QString &	: Return name				*/
/*  stamp	: QString &	: Return time stamp			*/
/*  (returns)	: bool		: Document forthcoming			*/

bool	KBDBDocIter::getNextDoc
	(	QString 	&name,
		QString		&stamp
	)
{
	/* First case is iterating in the file system. Get the name	*/
	/* (less the extension) and the time-and-date (which is		*/
	/* converted to YYYYMMDDHHMMSS format).				*/
	if (m_dbIter != 0)
	{
		QFileInfo *fi = m_dbIter->current() ;

		if (fi != 0)
		{
			QDateTime dt = fi->lastModified () ;

			name	= m_withExtn ?
					fi->fileName () :
					fi->baseName () ;

			stamp.sprintf
			(	"%04d%02d%02d%02d%02d%02d",
				dt.date().year  (),
				dt.date().month (),
				dt.date().day   (),
				dt.time().hour  (),
				dt.time().minute(),
				dt.time().second()
			)	;

			(*m_dbIter) += 1	;
			return	true	;
		}

		return	false	;
	}

	/* Second case is a database objects table scan. In this case	*/
	/* the timestamp is in the right format already.		*/
	if (m_select != 0)
	{
		if (!m_select->rowExists (m_nextRow)) return false ;

		name	  = m_select->getField (m_nextRow, 0).getRawText() ;
		stamp	  = m_select->getField (m_nextRow, 1).getRawText() ;

		if (m_withExtn)
			name += "." + m_select->getField (m_nextRow, 2).getRawText() ;

		m_nextRow += 1	;
		return	true	;
	}

	/* We get where for a null iterator, which can occur for a	*/
	/* database scan where there is no objectb table.		*/
	return	false	;
}

