/***************************************************************************
    file	         : kb_serverinfo.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#ifndef 	_WIN32
#include	<dlfcn.h>
#define		DLERROR()	dlerror()
#else
#define		DLERROR()	QString::null
#endif

#include	<qstringlist.h>
#include	<qregexp.h>

#include	"kb_serverinfo.h"
#include	"kb_dbinfo.h"
#include	"kb_dblink.h"

#include	"kb_libloader.h"


#if		defined(_WIN32) || __KB_EMBEDDED
#define		DLERROR()	QString::null
#else
#define		DLERROR()	dlerror()
#endif


typedef	void	*(*IFPROC)()	;

static	const QString nullToEmpty
	(	const QString	str
	)
{
	return	str.isNull() ? QString("") : str ;
}


/*  ------------------------------------------------------------------  */

static	QDict<QString>	identStrings	;

static	void	addIdentString
	(	const QString	&rdbms,
		cchar		*ident
	)
{
	identStrings.setAutoDelete (true) ;
	identStrings.replace	   (rdbms, new QString(ident)) ;
}

/*  getDriverFactory							*/
/*		: Get factory for specified RDBMS			*/
/*  dbType	: const QString & : RDBMS type				*/
/*  (returns)	: KBFactory *	  : Factory or null on error		*/

KBFactory
	*getDriverFactory
	(	const QString	&dbType
	)  
{
	KBFactory *factory = 0 ;

	if (dbType.isEmpty())
	{
		KBError::EFault
		(	TR("getDriverFactory called with no type"),
			QString::null,
			__ERRLOCN
		)	;
		return	0	;
	}

	KBLibLoader	*loader	= KBLibLoader::self  () ;
	KBLibrary	*lib	= loader->getLibrary ("libkbase_driver_" + dbType) ;

	if (lib == 0)
	{
		KBError::EFault
		(	TR(QString("%1: cannot load driver").arg(dbType)),
			loader->lastErrorMessage(),
			__ERRLOCN
		)	;
		return	0 ;
	}

	factory	= (KBFactory *)lib->factory() ;

	if (factory == 0)
	{
		KBError::EError
		(	QString (TR("Cannot load driver library for \"%1\"")).arg(dbType),
			QString::null,
			__ERRLOCN
		)	;
		return	0 ;
	}


	addIdentString	(dbType, factory->ident()) ;
	return	factory	;
}

/*  getDriverAdvanced							*/
/*		: Get advanced settings object for RDBMS		*/
/*  dbType	: const QString & : RDBMS type				*/
/*  (returns)	: KBDBAdvanced *  : Settings object or null on error	*/

KBDBAdvanced
	*getDriverAdvanced
	(	const QString	&dbType
	)  
{
	KBFactory *factory = getDriverFactory (dbType) ;
	if (factory == 0) return 0 ;

	return	(KBDBAdvanced *)factory->create(0, "advanced") ;
}

KBServer
	*getDriverServer
	(	const QString	&dbType
	)  
{

	KBFactory *factory = getDriverFactory (dbType) ;
	if (factory == 0) return 0 ;

	return	(KBServer *)factory->create(0, "driver") ;
}

/*  ------------------------------------------------------------------  */

/*  KBServerInfo							*/
/*  KBServerInfo: Constructor for connection information class		*/
/*  dbInfo	: KBDBInfo *	: Parent database information		*/
/*  (returns)	: KBServerInfo	:					*/

KBServerInfo::KBServerInfo
	(	KBDBInfo	*dbInfo
	)
	:
	m_dbInfo	(dbInfo)
{
	m_isDisabled	= false 	;
	m_autoStart	= false 	;
	m_objState	= SysUnknown	;
	m_dictState	= SysUnknown	;
	m_server	= 0  		;
	m_checkTabs	= false		;
	m_tableInfoSet	= 0		;
	m_advanced	= 0		;
	m_noRekallTables= false		;
	m_showAllTables	= false		;
	m_cacheTables	= false		;
	m_printQueries	= false		;
	m_nullUserPwd	= false		;
	m_pkReadOnly	= false		;
	m_fakeKeys	= false		;
	m_readOnly	= false		;
}

/*  KBServerInfo							*/
/*  KBServerInfo: Constructor for connection information class		*/
/*  dbInfo	: KBDBInfo *	  : Parent database information		*/
/*  svInfo	: KBServerInfo  * : Server information structure	*/
/*  (returns)	: KBServerInfo	  :					*/

KBServerInfo::KBServerInfo
	(	KBDBInfo		*dbInfo,
		const KBServerInfo	*svInfo
	)
	:
	m_dbInfo	(dbInfo)
{
	m_serverName	= svInfo->m_serverName	;
	m_dbType	= svInfo->m_dbType	;
	m_hostName	= svInfo->m_hostName	;
	m_dbName	= svInfo->m_dbName	;
	m_userName	= svInfo->m_userName	;
	m_password	= svInfo->m_password	;
	m_portNumber	= svInfo->m_portNumber	;
	m_socketName	= svInfo->m_socketName	;
	m_flags		= svInfo->m_flags	;
	m_useUserName	= svInfo->m_userName	;

	m_usePassword	= svInfo->m_password	;
	m_isDisabled	= svInfo->m_isDisabled	;
	m_autoStart	= svInfo->m_autoStart	;
	m_autoForm	= svInfo->m_autoForm	;
	m_objState	= SysUnknown		;
	m_dictState	= SysUnknown		;
	m_checkTabs	= false			;
	m_server	= 0  			;
	m_tableInfoSet	= 0			;

	m_noRekallTables= svInfo->m_noRekallTables	;
	m_showAllTables	= svInfo->m_showAllTables	;
	m_cacheTables	= svInfo->m_cacheTables		;
	m_printQueries	= svInfo->m_printQueries	;
	m_nullUserPwd	= svInfo->m_nullUserPwd		;
	m_pkReadOnly	= svInfo->m_pkReadOnly		;
	m_fakeKeys	= svInfo->m_fakeKeys		;
	m_readOnly	= svInfo->m_readOnly		;
	m_initSQL	= svInfo->m_initSQL		;

	m_dataEncoding	= svInfo->m_dataEncoding	;
	m_objEncoding	= svInfo->m_objEncoding		;
#ifndef	_WIN32
	m_sshTarget	= svInfo->m_sshTarget		;
#endif
	if (svInfo->m_advanced != 0)
		m_advanced = svInfo->m_advanced->copy () ;
	else	m_advanced = 0	;
}

/*  KBServerInfo							*/
/*  KBServerInfo: Constructor for server information class		*/
/*  dbInfo	: KBDBInfo *	  : Parent database information		*/
/*  serverSpec	: char *	  : Specification string		*/
/*  (returns)	: KBServerInfo	  :					*/

KBServerInfo::KBServerInfo
	(	KBDBInfo	*dbInfo,
		cchar		*serverSpec
	)
	:
	m_dbInfo	(dbInfo)
{
	QStringList bits = QStringList::split (QRegExp("[|\n]"), serverSpec, true) ;


	m_serverName = nullToEmpty(bits[ 0]) ;
	m_dbType     = nullToEmpty(bits[ 1]) ;
	m_hostName   = nullToEmpty(bits[ 2]) ;
	m_dbName     = nullToEmpty(bits[ 3]) ;
	m_userName   = nullToEmpty(bits[ 4]) ;
	m_password   = nullToEmpty(bits[ 5]) ;
	m_portNumber = nullToEmpty(bits[ 6]) ;
	m_socketName = nullToEmpty(bits[ 7]) ;
	m_flags      = nullToEmpty(bits[ 8]) ;
	m_isDisabled = nullToEmpty(bits[ 9]).lower() == "yes" ;
	m_autoStart  = nullToEmpty(bits[10]).lower() == "yes" ;

	m_objState   = SysUnknown	;
	m_dictState  = SysUnknown	;
	m_checkTabs  = false		;

	m_server        = 0		;
	m_tableInfoSet	= 0		;

	m_useUserName   = m_userName	;
	m_usePassword   = m_password	;
	m_noRekallTables= false		;
	m_showAllTables	= false		;
	m_cacheTables	= false		;
	m_printQueries	= false		;
	m_nullUserPwd	= false		;
	m_pkReadOnly	= false		;
	m_fakeKeys	= false		;
	m_readOnly	= false		;

	if (!m_dbType.isEmpty())
		m_advanced	= getDriverAdvanced (m_dbType) ;
	else	m_advanced	= 0	;
}

/*  KBServerInfo                                                	*/
/*  KBServerInfo: Constructor for m_pServer information class		*/
/*  dbInfo	: KBDBInfo *	  : Parent database information		*/
/*  serverSpec	: QDomElement     : Specification element       	*/
/*  (returns)	: KBServerInfo	  :                             	*/

KBServerInfo::KBServerInfo
	(	KBDBInfo	  *dbInfo,
		const QDomElement &serverSpec
	)
	:
	m_dbInfo (dbInfo)
{
	m_serverName	= nullToEmpty (serverSpec.attribute("ServerName"  )) ;
	m_dbType    	= nullToEmpty (serverSpec.attribute("DBType"      )) ;
	m_hostName  	= nullToEmpty (serverSpec.attribute("HostName"    )) ;
	m_dbName    	= nullToEmpty (serverSpec.attribute("DBName"      )) ;
	m_userName  	= nullToEmpty (serverSpec.attribute("UserName"    )) ;
	m_password  	= nullToEmpty (serverSpec.attribute("Password"    )) ;
	m_portNumber	= nullToEmpty (serverSpec.attribute("PortNumber"  )) ;
	m_socketName	= nullToEmpty (serverSpec.attribute("SocketName"  )) ;
	m_flags      	= nullToEmpty (serverSpec.attribute("Flags"       )) ;
	m_dataEncoding	= nullToEmpty (serverSpec.attribute("DataEncoding")) ;
	m_objEncoding	= nullToEmpty (serverSpec.attribute("ObjEncoding" )) ;
#ifndef	_WIN32
	m_sshTarget	= nullToEmpty (serverSpec.attribute("SSHTarget"   )) ;
#endif
	m_isDisabled    = serverSpec.attribute("IsDisabled").lower() == "yes" ;
	m_autoStart     = serverSpec.attribute("AutoStart" ).lower() == "yes" ;
	m_autoForm      = serverSpec.attribute("AutoForm"  ) ;

	m_objState      = SysUnknown 	;
	m_dictState  	= SysUnknown	;
	m_checkTabs  	= false	  	;

	m_useUserName   = m_userName	;
	m_usePassword   = m_password	;

	m_noRekallTables= serverSpec.attribute ("NoRekallTables").toUInt() ;
	m_showAllTables	= serverSpec.attribute ("ShowAllTables" ).toUInt() ;
	m_cacheTables	= serverSpec.attribute ("CacheTables"   ).toUInt() ;
	m_printQueries	= serverSpec.attribute ("PrintQueries"  ).toUInt() ;
	m_nullUserPwd	= serverSpec.attribute ("NullUserPwd"   ).toUInt() ;
	m_pkReadOnly	= serverSpec.attribute ("PKReadOnly"    ).toUInt() ;
	m_fakeKeys	= serverSpec.attribute ("FakeKeys"      ).toUInt() ;
	m_readOnly	= serverSpec.attribute ("ReadOnly"      ).toUInt() ;
	m_initSQL	= serverSpec.attribute ("InitSQL"       )	   ;

	m_server        = 0	  ;
	m_tableInfoSet	= 0	  ;

	if (m_dbType.isEmpty())
		m_advanced = 0 ;
	else	m_advanced = getDriverAdvanced (m_dbType) ;

	if (m_advanced != 0)
		for (QDomNode child = serverSpec.firstChild() ;
				      !child	.isNull() ;
			      child = child	.nextSibling())
		{
			if (child.isElement() && child.toElement().tagName() == "driver")
			{
				m_advanced->load (child.toElement()) ;
				break	;
			}
		}
}

/*  KBServerInfo							*/
/*  KBServerInfo: Constructor for connection information class		*/
/*  dbInfo	: KBDBInfo *	: Parent database information		*/
/*  serverName	: cchar *	: Server name				*/
/*  dbType	: cchar *	: Database type				*/
/*  hostName	: cchar *	: Host name				*/
/*  dbName	: cchar *	: Database name on host			*/
/*  userName	: cchar *	: User name				*/
/*  password	: cchar *	: Password				*/
/*  (returns)	: KBServerInfo	:					*/

KBServerInfo::KBServerInfo
	(	KBDBInfo	*dbInfo,
		cchar		*serverName,
		cchar		*dbType,
		cchar		*hostName,
		cchar		*dbName,
		cchar		*userName,
		cchar		*password
	)
	:
	m_dbInfo	(dbInfo),
	m_serverName	(serverName),
	m_dbType	(dbType),
	m_hostName	(hostName),
	m_dbName	(dbName),
	m_userName	(userName),
	m_password	(password)
{
	m_isDisabled	= false 	;
	m_autoStart	= false 	;
	m_objState	= SysUnknown	;
	m_dictState	= SysUnknown	;
	m_server	= 0  		;
	m_checkTabs	= false		;
	m_tableInfoSet	= 0		;
	m_advanced	= 0		;
	m_noRekallTables= false		;
	m_showAllTables	= false		;
	m_cacheTables	= false		;
	m_printQueries	= false		;
	m_nullUserPwd	= false		;
	m_pkReadOnly	= false		;
	m_fakeKeys	= false		;
	m_readOnly	= false		;

	m_useUserName	= userName	;
	m_usePassword	= password	;
}

/*  KBServerInfo							*/
/*  ~KBServerInfo: Destructor for connection information class		*/
/*  (returns)	 :		:					*/

KBServerInfo::~KBServerInfo ()
{
//	if (m_links.count() > 0)
//		KBError::EError
//		(	TR("Releasing server with open connections"),
//			QString(TR("Server: %1, with %1 connections"))
//				.arg(m_serverName)
//				.arg(m_links.count()),
//			__ERRLOCN
//		)	;

	DELOBJ	(m_advanced) ;

	if (m_server != 0)
	{
//		fprintf	(stderr, "KBServerInfo::~KBServerInfo(%s) closing link\n", (cchar *)m_serverName) ;
		DELOBJ	(m_server) ;
	}
}

/*  KBServerInfo                                                        */
/*  saveTableInfo: Save table information				*/
/*  (returns)	 : void		:					*/

void	KBServerInfo::saveTableInfo ()
{
	if (m_tableInfoSet != 0)
	{
		m_tableInfoSet->save (true) ;
		DELOBJ	(m_tableInfoSet)    ;
	}
}

/*  KBServerInfo                                                        */
/*  buildSpecElement							*/
/*		: Build specification element				*/
/*  info	: QDomElement &	: DOM element for specification		*/
/*  (returns)   : void		:					*/

void	KBServerInfo::buildSpecElement
	(	QDomElement	&info
	)
{
	info.setAttribute("ServerName", 	m_serverName	) ;
	info.setAttribute("DBType",     	m_dbType    	) ;
	info.setAttribute("HostName",   	m_hostName  	) ;
	info.setAttribute("DBName",     	m_dbName    	) ;
	info.setAttribute("UserName",   	m_userName  	) ;
	info.setAttribute("Password",   	m_password  	) ;
	info.setAttribute("PortNumber", 	m_portNumber	) ;
	info.setAttribute("SocketName", 	m_socketName	) ;
	info.setAttribute("Flags",      	m_flags     	) ;
	info.setAttribute("IsDisabled", 	m_isDisabled ? "Yes" : "No") ;
	info.setAttribute("AutoStart",  	m_autoStart  ? "Yes" : "No") ;
	info.setAttribute("AutoForm",  		m_autoForm	) ;

	info.setAttribute("NoRekallTables",	m_noRekallTables) ;
	info.setAttribute("ShowAllTables",	m_showAllTables ) ;
	info.setAttribute("CacheTables",	m_cacheTables   ) ;
	info.setAttribute("PrintQueries",	m_printQueries  ) ;
	info.setAttribute("NullUserPwd",	m_nullUserPwd   ) ;
	info.setAttribute("PKReadOnly",		m_pkReadOnly    ) ;
	info.setAttribute("FakeKeys",		m_fakeKeys      ) ;
	info.setAttribute("ReadOnly",		m_readOnly      ) ;
	info.setAttribute("InitSQL",		m_initSQL       ) ;
	info.setAttribute("DataEncoding",	m_dataEncoding  ) ;
	info.setAttribute("ObjEncoding",	m_objEncoding   ) ;
#ifndef	_WIN32
	info.setAttribute("SSHTarget",		m_sshTarget     ) ;
#endif
	if (m_advanced != 0)
	{
		QDomElement drvElem = info.ownerDocument().createElement("driver") ;
		info.appendChild (drvElem) ;
		m_advanced->save (drvElem) ;
	}
}


/*  KBServerInfo							*/
/*  attachLink	: Attach a link to the server				*/
/*  dbLink	: KBDBLink *	: Requesting link			*/
/*  (returns)	: void		:					*/

void	KBServerInfo::attachLink
	(	KBDBLink	*dbLink
	)
{
	/* All we do here is to attach the link, so that we keep track	*/
	/* of who is using us.						*/
	m_links.append (dbLink) ;
}

/*  KBServerInfo							*/
/*  detachLink	: Detach a logical link					*/
/*  dbLink	: KBDBLink *	: Link being closed			*/
/*  (returns)	: void		:					*/

void	KBServerInfo::detachLink
	(	KBDBLink	*dbLink
	)
{
	/* Remove the link from the list. We can allow the server	*/
	/* connection to be closed when the link count is zero.		*/
	m_links.remove (dbLink) ;
}

#if	0

/*  KBServerInfo							*/
/*  makeDesignDict							*/
/*		: Make a new design dictionary				*/
/*  (returns)	: void		:					*/

void	KBServerInfo::makeDesignDict ()
{
	if (!promptMakeDesignDict())
	{
		m_dictState = SysMissing ;
		return	  ;
	}

	extern	bool	GetKBDesignDictSpec (QList<KBFieldSpec> &) ;

	KBTableSpec	tabSpec (m_server->rekallPrefix(DESIGNTABNAME)) ;
	GetKBDesignDictSpec (tabSpec.m_fldList) ;

	if (!m_server->createTable (tabSpec, true, false))
	{
		m_server->lastError().DISPLAY() ;
		m_dictState = SysMissing ;
	}
	else	m_dictState = SysPresent ;
}

#endif

/*  KBServerInfo							*/
/*  makeObjTable: Make a new objects table				*/
/*  (returns)	: void		:					*/

void	KBServerInfo::makeObjTable ()
{
	if (!promptMakeObjTable ())
	{
		m_objState = SysMissing ;
		return	  ;
	}

	extern	bool	GetKBObjectTableSpec (QList<KBFieldSpec> &) ;

	KBTableSpec	tabSpec (m_server->rekallPrefix(OBJECTTABNAME)) ;
	GetKBObjectTableSpec (tabSpec.m_fldList) ;

	if (!m_server->createTable (tabSpec, true, false))
	{
		m_server->lastError().DISPLAY() ;
		m_objState = SysMissing ;
	}
	else	m_objState = SysPresent ;
}

/*  KBServerInfo							*/
/*  checkForTables							*/
/* 		: Check for object and design tables			*/
/*  (returns)	: KBServer *	: Connection or null			*/

void	KBServerInfo::checkForTables ()
{
	if (m_checkTabs)
		return	;

	m_checkTabs = true ;

#if	0
	if (m_dictState == SysUnknown)
	{
		bool	exists	;

		if (!m_server->tableExists (m_server->rekallPrefix(DESIGNTABNAME), exists))
		{
			m_checkTabs = false ;
			m_server->lastError().DISPLAY() ;
			return	;
		}

		fprintf
		(	stderr,
			"KBServerInfo::checkForTables: dict %d, nor=%d\n",
			exists,
			m_noRekallTables
		)	;

		if	(exists) 
			m_dictState = SysPresent ;
		else if	(m_noRekallTables) 
			m_dictState = SysMissing ;
		else
			makeDesignDict();
	}
#endif

	if (m_serverName != KBLocation::m_pFile)
	{
		if (m_objState == SysUnknown)
		{
			bool	exists	;

			if (!m_server->tableExists (m_server->rekallPrefix(OBJECTTABNAME), exists))
			{
				m_checkTabs = false ;
				m_server->lastError().DISPLAY() ;
				return	;
			}

			fprintf
			(	stderr,
				"KBServerInfo::checkForTables: objs %d, nor=%d\n",
				exists,
				m_noRekallTables
			)	;

			if 	(exists)
				m_objState = SysPresent ;
			else if (m_noRekallTables)
				m_objState = SysMissing ;
			else
				makeObjTable () ;
		}
	}
	else
		m_objState  = SysMissing ;

	m_checkTabs = false ;
}

/*  KBServerInfo							*/
/*  getServer	: Get an actual connection to the server		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: KBServer *	: Connection or null			*/

KBServer*KBServerInfo::getServer
	(	KBError		&pError
	)
{
	if (m_isDisabled)
	{
		pError	= KBError
			  (	KBError::Error,
				QString (TR("Server %1 is disabled")).arg(m_serverName),
				QString::null,
				__ERRLOCN
			  ) ;
		return	0 ;
	}

	if (m_dbType.isEmpty() && (m_serverName != KBLocation::m_pFile))
	{
		m_isDisabled = true ;
		pError	= KBError
			  (	KBError::Error,
				QString (TR("Server %1 has no database type")).arg(m_serverName),
				QString::null,
				__ERRLOCN
			  ) ;
		return	0 ;
	}

	/* If we do not have a connection to the server then now is the	*/
	/* time to open one.						*/
	if (m_server == 0)
	{
		KBFactory *factory = getDriverFactory (m_dbType) ;

		if (factory == 0)
		{
			m_isDisabled = true ;
			pError	= KBError
				  (	KBError::Fault,
					QString (TR("Error loading driver library for \"%1\"")).arg(m_dbType),
					QString::null,
					__ERRLOCN
				  )	;
			return	0	;
		}

		if ((m_server = (KBServer *)factory->create(0, "driver")) == 0)
		{
			m_isDisabled = true ;
			pError	= KBError
				  (	KBError::Fault,
					QString (TR("Cannot create driver for \"%1\"")).arg(m_dbType),
					QString::null,
					__ERRLOCN
				  )	;
			return	0	;
		}

		/* If the server requires a username and password and	*/
		/* they are not set then prompt for them now.		*/
		if ((m_server->optionFlags() & AF_USERPASSWORD) != 0)
			if (m_useUserName.isEmpty() || m_password.isEmpty())
				if (!m_nullUserPwd)
					if (!promptUserPassword(pError))
					{
						m_isDisabled = true ;
						DELOBJ	(m_server) ;
						return	0 ;
					}

		if (!m_server->connect (this))
		{
			m_isDisabled = true ;
			pError	= m_server->lastError () ;
			DELOBJ	(m_server) ;
			return	0 ;
		}

		if (!m_server->execInitSQL (m_initSQL))
		{
			m_isDisabled = true ;
			pError	= m_server->lastError () ;
			DELOBJ	(m_server) ;
			return	0 ;
		}

	}

	checkForTables () ;
	return	m_server  ;
}

/*  KBServerInfo							*/
/*  linkIsOpen	: Report whether there is an open link			*/
/*  (returns)	: bool		: Link open				*/

bool	KBServerInfo::linkIsOpen ()
{
	if (m_links.count() > 0) return true ;

	fprintf	(stderr, "KBServerInfo::linkIsOpen(%s) closing link\n", (cchar *)m_server) ;
	DELOBJ	(m_server) ;
	return	false	 ;
}

/*  KBServerInfo							*/
/*  tableInfoSet: Get table information set for this server		*/
/*  (returns)	: KBTableInfoSet * : Table information set		*/

KBTableInfoSet	*KBServerInfo::tableInfoSet ()
{
	if (m_tableInfoSet == 0)
		m_tableInfoSet = new KBTableInfoSet (m_dbInfo, m_serverName) ;

	return	m_tableInfoSet ;
}

/*  KBServerInfo							*/
/*  getDBPath	: Get the path to the underlying database directory	*/
/*  (returns)	: const QString & : Path				*/

const QString
	&KBServerInfo::getDBPath ()
{
	/* Check in case there is no database information object, which	*/
	/* can happen if the drivers are being used outside of rekall.	*/
	/* If so, just return the current directory.			*/
	static	QString	dot (".") ;
	return	m_dbInfo == 0 ? dot : m_dbInfo->getDBPath() ;
}


bool	KBServerInfo::promptUserPassword
	(	KBError		&
	)
{
	return	false	;
}

bool	KBServerInfo::promptMakeObjTable ()
{
	return	false	;
}

bool	KBServerInfo::promptMakeDesignDict ()
{
	return	false	;
}


QString	KBServerInfo::getIdentStrings ()
{
	QString			res  ;
	QDictIterator<QString>	iter (identStrings) ;

	while (iter.current() != 0)
	{
		res	+= QString("<tr><td>Driver</td><td><b>%1</b></td><td><nobr>%2</nobr></td></tr>")
				  .arg( iter.currentKey())
				  .arg(*iter.current   ()) ;

		iter	+= 1 ;
	}

	return	res	;
}


