/***************************************************************************
    file	         : kb_tableinfo.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>
#include	<qregexp.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"
#include	"kb_dbdociter.h"
#include	"kb_database.h"
#include	"kb_dbinfo.h"
#include	"kb_dblink.h"
#include	"kb_databuffer.h"
#include	"kb_tableinfo.h"


static	KBTableUniqueList	emptyUnique	;

static	cchar	*designNames[] =
{
	"description",
	"evalid",
	"igncase",
	"default",
	"format",
	"link",
	"width"
}	;

#define	DOMITER_BEGIN(root,tag,child)	\
	{										\
	for (QDomNode n = root.firstChild() ; !n.isNull() ; n = n.nextSibling())	\
	{	QDomElement child = n.toElement() ;					\
		if (child.tagName() != tag) continue ;

#define	DOMITER_END	\
	}		\
	}

/*  ------------------------------------------------------------------  */

KBTableUniqueList::KBTableUniqueList ()
{
}

KBTableUniqueList::KBTableUniqueList
	(	const KBTableUniqueList	&unique
	)
	:
	QValueList<KBTableUnique> (unique)
{
}

/*  ------------------------------------------------------------------  */


/*  KBTableColumn							*/
/*  KBTableColumn: Constructor for empty table column object		*/
/*  name	 : const QString & : New column name			*/
/*  (returns)	 : KBTableColumn   :					*/

KBTableColumn::KBTableColumn
	(	const QString	&name
	)
	:
	m_colname (name)
{
}

/*  KBTableColumn							*/
/*  KBTableColumn: Constructor for duplicate table column object	*/
/*  column	 : KBTableColumn * : Extant column object		*/
/*  (returns)	 : KBTableColumn   :					*/

KBTableColumn::KBTableColumn
	(	const KBTableColumn	*column
	)
{
	for (uint idx = 0 ; idx < TI_COUNT ; idx += 1)
		m_design[idx] = column->m_design[idx] ;

	m_colname = column->m_colname ;
}

/*  KBTableColumn							*/
/*  setColumnName: Set the column name					*/
/*  name	 : const QString & : New column name			*/
/*  (returns)	 : void		   :					*/

void	KBTableColumn::setColumnName
	(	const QString	&name
	)
{
	m_colname = name ;
}

/*  KBTableColumn							*/
/*  columnName	: Get the column name					*/
/*  (reutrns)	: const QString & : Column name				*/

const QString
	&KBTableColumn::columnName ()
{
	return	m_colname ;
}


/*  KBTableColumn							*/
/*  designValue	: Get design value					*/
/*  idx		: uint		  : Required value			*/
/*  (returns)	: const QString & :					*/

const	QString	&KBTableColumn::designValue
	(	uint 		idx
	)
{
	return	m_design[idx]	;
}

/*  KBTableColumn							*/
/*  setDesignValue							*/
/*		: Set design value					*/
/*  idx		: uint		: Value to update			*/
/*  value	: QString	: New value				*/
/*  (returns)	: bool		: Value changed				*/

bool	KBTableColumn::setDesignValue
	(	uint		idx,
		QString		value
	)
{
	/* HACK: If all values are empty then the column need not be	*/
	/* explitely saved in the XML document. To this end, convert	*/
	/* the numeric cases with zero values to nulls.			*/
	switch (idx)
	{
		case TI_IGNCASE :
		case TI_WIDTH	:
			if (value.toUInt() == 0) value = QString::null ;
			break	;

		default	:
			break	;
	}

	if ((value.length() == 0) && (m_design[idx].length() == 0))
		return	false	;
	if (value == m_design[idx])
		return	false	;

	m_design[idx] = value	;
	return	true		;
}

/*  KBTableColumn							*/
/*  anyValueSet	: See if any value is non-empty				*/
/*  (returns)	: bool		: True if so				*/

bool	KBTableColumn::anyValueSet ()
{
	for (uint idx = 0 ; idx < TI_COUNT ; idx += 1)
		if (!m_design[idx].isEmpty())
			return	true	;

	return	false	;
}



/*  ------------------------------------------------------------------  */

/*  KBTableSort								*/
/*  KBTableSort	: Constuctor for table sort definition			*/
/*  sortElem	: QDomElement &	: Definition element			*/
/*  (returns)	: KBTableSort	:					*/

KBTableSort::KBTableSort
	(	const QDomElement	&sortElem
	)
{

	m_name	= sortElem.attribute ("name") ;

	DOMITER_BEGIN(sortElem, "column", colElem)
	{
		addColumn (colElem.attribute("name"), colElem.attribute("asc").toUInt()) ;
	}
	DOMITER_END

	fprintf	(stderr, "KBTableSort::KBTableSort [%s]\n", (cchar *)m_name) ;
}

/*  KBTableSort								*/
/*  addColumn	: Add column to definition				*/
/*  column	: const QString & : Column name				*/
/*  asc		: bool		  : Ascending sort			*/
/*  (returns)	: void		  :					*/

void	KBTableSort::addColumn
	(	const QString	&column,
		bool		asc
	)
{
	m_columns.append (column) ;
	m_orders .append (asc   ) ;
}

/*  KBTableSort								*/
/*  save	: Save definition					*/
/*  parent	: QDomElement &	: Parent element into which to save	*/
/*  (returns)	: void		  :					*/

void	KBTableSort::save
	(	QDomElement	&parent
	)
{
	QDomElement sortElem = parent.ownerDocument().createElement("sort") ;
	parent  .appendChild  (sortElem) ;
	sortElem.setAttribute ("name", m_name) ;

	for (uint idx = 0 ; idx < m_columns.count() ; idx += 1)
	{
		QDomElement colElem = parent.ownerDocument().createElement("column") ;
		sortElem.appendChild  (colElem) ;
		colElem .setAttribute ("name", m_columns[idx]) ;
		colElem .setAttribute ("asc",  m_orders [idx]) ;
	}
}

/*  KBTableSort								*/
/*  sql		: Get SQL for sort					*/
/*  buffer	: KBDataBuffer & : SQL text buffer			*/
/*  (returns)	: void		  :					*/

void	KBTableSort::sql
	(	KBDataBuffer	&buffer
	)
{
	for (uint idx = 0 ; idx < m_columns.count() ; idx += 1)
	{
		if (idx > 0) buffer.append (", ") ;
		buffer.append (m_columns[idx])    ;
		if (!m_orders[idx]) buffer.append (" desc") ;
	}
}


/*  ------------------------------------------------------------------  */

/*  KBTableSelect							*/
/*  KBTableSelect							*/
/*		: Constuctor for table filter definition		*/
/*  filterElem	: QDomElement &	: Definition element			*/
/*  (returns)	: KBTableSelect	:					*/

KBTableSelect::KBTableSelect
	(	const QDomElement	&filterElem
	)
{
	m_name	= filterElem.attribute ("name") ;

	DOMITER_BEGIN(filterElem, "column", colElem)
	{
		addColumn
		(	colElem.attribute("name" ),
			(Operator)colElem.attribute("oper" ).toUInt(),
			colElem.attribute("value")
		)	;
	}
	DOMITER_END

	fprintf	(stderr, "KBTableSelect::KBTableSelect [%s]\n", (cchar *)m_name) ;
}

/*  KBTableSelect								*/
/*  addColumn	: Add column to definition				*/
/*  column	: const QString & : Column name				*/
/*  oper	: Operator	  : Value test operator			*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBTableSelect::addColumn
	(	const QString	&column,
		Operator	oper,
		const QString	&value
	)
{
	m_columns  .append (column) ;
	m_operators.append (oper  ) ;
	m_values   .append (value ) ;
}

/*  KBTableSelect								*/
/*  save	: Save definition					*/
/*  parent	: QDomElement &	: Parent element into which to save	*/
/*  (returns)	: void		  :					*/

void	KBTableSelect::save
	(	QDomElement	&parent
	)
{
	QDomElement sortElem = parent.ownerDocument().createElement("filter") ;
	parent  .appendChild  (sortElem) ;
	sortElem.setAttribute ("name", m_name) ;

	for (uint idx = 0 ; idx < m_columns.count() ; idx += 1)
	{
		QDomElement colElem = parent.ownerDocument().createElement("column") ;
		sortElem.appendChild  (colElem) ;
		colElem .setAttribute ("name",  m_columns  [idx]) ;
		colElem .setAttribute ("oper",  m_operators[idx]) ;
		colElem .setAttribute ("value", m_values   [idx]) ;
	}
}

/*  KBTableSelect								*/
/*  sql		: Get SQL for filer					*/
/*  buffer	: KBDataBuffer  & : SQL text buffer			*/
/*  typeMap	: QDict<KBType> & : Column to type mapping		*/
/*  (returns)	: void		  :					*/

void	KBTableSelect::sql
	(	KBDataBuffer	&buffer,
		QDict<KBType>	&typeMap
	)
{
	cchar	*oper	;

	for (uint idx = 0 ; idx < m_columns.count() ; idx += 1)
	{
		switch (m_operators[idx])
		{
			case Eq		: oper = " =  "		 ; break ;
			case Neq	: oper = " <> "		 ; break ;
			case Gt		: oper = " >  "		 ; break ;
			case Ge		: oper = " >= "		 ; break ;
			case Lt		: oper = " <  "		 ; break ;
			case Le		: oper = " <= " 	 ; break ;
			case Like	: oper = " like "	 ; break ;
			case NotNull	: oper = " is not null " ; break ;
			case IsNull	: oper = " is null "	 ; break ;
			default		: oper = " <unknown> "	 ; break ;
		}

		if (idx > 0) buffer.append(", ") ;
		buffer.append (m_columns[idx])   ;
		buffer.append (oper) ;
	
		if ((m_operators[idx] != NotNull) && (m_operators[idx] != IsNull))
		{
			KBType	    *fType = typeMap.find(m_columns[idx]) ;
			KBValue	    value  (m_values[idx], fType == 0 ? &_kbUnknown : fType) ;
			value.getQueryText (buffer) ;
		}
	}
}

/*  ------------------------------------------------------------------  */

/*  KBTableView								*/
/*  KBTableView	: Constuctor for table view definition			*/
/*  viewElem	: QDomElement &	: Definition element			*/
/*  (returns)	: KBTableView	:					*/

KBTableView::KBTableView
	(	const QDomElement	&viewElem
	)
{
	m_name	= viewElem.attribute ("name") ;

	DOMITER_BEGIN(viewElem, "column", colElem)
	{
		addColumn (colElem.attribute("name")) ;
	}
	DOMITER_END

	fprintf	(stderr, "KBTableView::KBTableView [%s]\n", (cchar *)m_name) ;
}

/*  KBTableView								*/
/*  addColumn	: Add column to definition				*/
/*  column	: const QString & : Column name				*/
/*  (returns)	: void		  :					*/

void	KBTableView::addColumn
	(	const QString	&column
	)
{
	m_columns.append (column) ;
}

/*  KBTableView								*/
/*  save	: Save definition					*/
/*  parent	: QDomElement &	: Parent element into which to save	*/
/*  (returns)	: void		  :					*/

void	KBTableView::save
	(	QDomElement	&parent
	)
{
	QDomElement sortElem = parent.ownerDocument().createElement("view") ;
	parent  .appendChild  (sortElem) ;
	sortElem.setAttribute ("name", m_name) ;

	for (uint idx = 0 ; idx < m_columns.count() ; idx += 1)
	{
		QDomElement colElem = parent.ownerDocument().createElement("column") ;
		sortElem.appendChild  (colElem) ;
		colElem .setAttribute ("name",  m_columns  [idx]) ;
	}
}

/*  ------------------------------------------------------------------  */

/*  KBTableInfo								*/
/*  KBTableInfo	: Constructor for table information object		*/
/*  name	: const QString &: Table name				*/
/*  (returns)	: KBTableInfo	 :					*/

KBTableInfo::KBTableInfo
	(	const QString	&name
	)
	:
	m_name	(name)
{
	m_changed	= false		 ;
	m_widthChanged	= false		 ;

	m_columns  .setAutoDelete (true) ;
	m_sortSet  .setAutoDelete (true) ;
	m_selectSet.setAutoDelete (true) ;
	m_viewSet  .setAutoDelete (true) ;
}

/*  KBTableInfo								*/
/*  getColumn	: Get inforamtion object for named column		*/
/*  name	: const QString & : Column name				*/
/*  (returns)	: KBTableColumn * : Column information object		*/

KBTableColumn
	*KBTableInfo::getColumn
	(	const QString	&name
	)
{
	KBTableColumn	*column	= m_columns.find (name) ;
	if (column == 0)
		m_columns.insert (name, column = new KBTableColumn (name)) ;
	return	column	;
}


/*  KBTableInfo								*/
/*  designValue	: Get design value					*/
/*  column	: const QString & : Column name				*/
/*  idx		: uint		  : Required value			*/
/*  (returns)	: const QString & :					*/

const	QString	&KBTableInfo::designValue
	(	const QString	&column,
		uint 		idx
	)
{
	return	getColumn(column)->designValue(idx) ;
}

/*  KBTableInfo								*/
/*  setDesignValue							*/
/*		: Set design value					*/
/*  column	: const QString & : Column name				*/
/*  idx		: uint		  : Value to update			*/
/*  value	: const QString & : New value				*/
/*  (returns)	: bool		  : Value changed			*/

void	KBTableInfo::setDesignValue
	(	const QString	&column,
		uint		idx,
		const QString	&value
	)
{
	if (getColumn(column)->setDesignValue(idx, value))
	{
		if (idx == TI_WIDTH)
			m_widthChanged	= true ;
		else	m_changed	= true ;
	}
}


/*  KBTableInfo								*/
/*  loadFromInfo: Load information from DOM document			*/
/*  dbInfo	: KBDBInfo *	  : Database inforamtion		*/
/*  server	: const QString & : Server name				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: int		  : Status				*/

int	KBTableInfo::loadFromInfo
	(	KBDBInfo	*dbInfo,
		const QString	&server,
		KBError		&pError
	)
{
	/* See if the document exists. If not then this is not actually	*/
	/* an error, but returning zero indicates that no information	*/
	/* was found.							*/
	KBLocation infoLoc(dbInfo, "info", server, m_name) ;
	if (!infoLoc.exists())
	{
		return	0 ;
	}

	QString	   info   = infoLoc.contents(pError) ;
	if (info    .isNull())
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Table information error"),
			  	QString(TR("Meta-object for %1/%2 is empty"))
			  		.arg(server)
			  		.arg(m_name),
			  	__ERRLOCN
			  )	;
		return	-1	;
	}


	QDomDocument 	doc    ;
	if (!doc.setContent(info))
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Table information error"),
			  	QString(TR("Meta-object for %1/%2 is corrupt"))
			  		.arg(server)
			  		.arg(m_name),
			  	__ERRLOCN
			  )	;

		return	-1	;
	}

	QDomElement rootElem	= doc.documentElement() ;


	/* Locate the "unique" node (if any) and from that locate the	*/
	/* unique key columns and their default values (again, if any).	*/
	QDomElement uniElem	= rootElem.namedItem("unique").toElement() ;

	DOMITER_BEGIN(uniElem, "key", keyElem)
	{
		QDomElement keyElem = n.toElement();

		if(keyElem.tagName() != "key")
			continue;
	
		m_unique.append(KBTableUnique(keyElem.attribute ("column"),	keyElem.attribute ("defval")));
	}
	DOMITER_END

	/* Next locate the columns element and pick up each column.	*/
	/* From these extract the design fields.			*/
	QDomElement colsElem	= rootElem.namedItem("columns").toElement() ;

	DOMITER_BEGIN(colsElem, "column", colElem)
	{
		QDomElement colElem = n.toElement();

		if(colElem.tagName() != "column")
			continue;

		KBTableColumn *column = getColumn (colElem.attribute("name")) ;

		for (uint idx = 0 ; idx < TI_COUNT ; idx += 1)
			column->setDesignValue
			(
				idx,
				colElem.namedItem(designNames[idx]).toElement().attribute("value")
			)	;
	}
	DOMITER_END

	/* The next three blocks of code pick up sort, filter and view	*/
	/* definitions.							*/
	QDomElement sortSetElem	= rootElem.namedItem("sortset").toElement() ;

	DOMITER_BEGIN(sortSetElem, "sort", sortElem)
	{
		KBTableSort *ts = new KBTableSort (sortElem) ;
		m_sortSet.append (ts) ;
	}
	DOMITER_END

	QDomElement selectSetElem = rootElem.namedItem("selectset").toElement() ;

	DOMITER_BEGIN(selectSetElem, "filter", filterElem)
	{
		KBTableSelect *tf = new KBTableSelect (filterElem) ;
		m_selectSet.append (tf) ;
	}
	DOMITER_END

	QDomElement viewSetElem	= rootElem.namedItem("viewset").toElement() ;

	DOMITER_BEGIN(viewSetElem, "view", viewElem)
	{
		KBTableView *tv = new KBTableView (viewElem) ;
		m_viewSet.append (tv) ;
	}
	DOMITER_END

	m_changed	= false	;
	m_widthChanged	= false	;
	return	1 ;
}

#if	0

/*  The "loadFromDesign" routine remained after the switch from a	*/
/*  design table to information meta objects for backward compatability	*/
/*  This is removed as of Rekall 2.2.x					*/

/*  KBTableInfo								*/
/*  loadFromDesign							*/
/*		: Load information from design table			*/
/*  dbInfo	: KBDBInfo *	  : Database inforamtion		*/
/*  server	: const QString & : Server name				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: int		  : Status				*/

int	KBTableInfo::loadFromDesign
	(	KBDBInfo	*dbInfo,
		const QString	&server,
		KBError		&pError
	)
{
	/* Connect to the server and get a list of tables. We will then	*/
	/* iterate through each, picking up the table design 		*/
	/* information.							*/
	KBDBLink dbLink ;
	if (!dbLink.connect (dbInfo, server))
	{	pError	= dbLink.lastError() ;
		return	-1 ;
	}

	/* See if the design table exists. This will not be the case	*/
	/* for databases created from 2.0.0 onwards, or where the user	*/
	/* previously elected not to create this table.			*/
	bool	exists	;
	if (!dbLink.tableExists (dbLink.rekallPrefix("RekallDesign"), exists))
	{	pError	= dbLink.lastError() ;
		return	-1 ;
	}
	if (!exists) return 0 ;
	
	QString qryString   =
		"select TabName, ColName, Descrip, Evalid, Igncase, Defval, Format, Link, Width "
		"from " 	    + dbLink.rekallPrefix("RekallDesign") +
		" where TabName = " + dbLink.placeHolder (0) ;

	KBSQLSelect *select = dbLink.qrySelect (false, qryString) ;
	KBValue	    tn (m_name) ;
	if (!select->execute (1, &tn))
	{
		select->lastError().DISPLAY() ;
		delete	select	;
		return	-1	;
	}

	/* Special case, query executes OK but returns no data, in this	*/
	/* case return zero to indicate this.				*/
	if (!select->rowExists (0))
	{
		delete	select	;
		return	0	;
	}

	for (uint row = 0 ; select->rowExists(row) ; row += 1)
	{
		QString	column	= select->getField(row, 1).getRawText() ;

		setDesignValue (column, TI_DESCR,   select->getField (row, 2).getRawText()) ;
		setDesignValue (column, TI_EVALID,  select->getField (row, 3).getRawText()) ;
		setDesignValue (column, TI_IGNCASE, select->getField (row, 4).getRawText()) ;
		setDesignValue (column, TI_DEFAULT, select->getField (row, 5).getRawText()) ;
		setDesignValue (column, TI_FORMAT,  select->getField (row, 6).getRawText()) ;
		setDesignValue (column, TI_LINK,    select->getField (row, 7).getRawText()) ;
		setDesignValue (column, TI_WIDTH,   select->getField (row, 8).getRawText()) ;			
	}

	delete	select	;
	return	1	;
}
#endif

/*  KBTableInfo								*/
/*  load	: Load information for table				*/
/*  dbInfo	: KBDBInfo *	  : Database inforamtion		*/
/*  server	: const QString & : Server name				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBTableInfo::load
	(	KBDBInfo	*dbInfo,
		const QString	&server,
		KBError		&pError
	)
{
	fprintf
	(	stderr,
		"KBTableInfo::load: %s/%s\n",
		(cchar *)server,
		(cchar *)m_name
	)	;

	/* Load table information from the meta-object. If then event	*/
	/* that there is no information (but no error) then we will try	*/
	/* the meta-object, which handles design migration.		*/
	switch (loadFromInfo   (dbInfo, server, pError))
	{
		case	-1 :
			return	false	;

		case	0  :
			break	;

		default	:
			return	true	;
	}

#if	0
	switch (loadFromDesign (dbInfo, server, pError))
	{
		case	-1 :
			return	false	;

		default	:
			break	;
	}
#endif

	fprintf
	(	stderr,
		"KBTableInfo::load: %s/%s: calling save for migration\n",
		(cchar *)server,
		(cchar *)m_name
	)	;

	return	save (dbInfo, server, pError, true) ;
}

/*  KBTableInfo								*/
/*  update	: Update column information				*/
/*  columns	: QList<KBTableColumn> &				*/
/*				: New columns				*/
/*  (returns)	: void		:					*/

void	KBTableInfo::update
	(	QList<KBTableColumn> &columns
	)
{
	m_columns.clear() ;

	LITER
	(	KBTableColumn,
		columns,
		col,

		m_columns.insert (col->columnName(), col)
	)

	m_changed = true ;
}


/*  KBTableInfo								*/
/*  save	: Save information as DOM document			*/
/*  dbInfo	: KBDBInfo *	  : Database inforamtion		*/
/*  server	: const QString & : Server name				*/
/*  pError	: KBError &	  : Error return			*/
/*  anyChange	: bool		  : Force save on any change		*/
/*  (returns)	: bool		  : Success				*/

bool	KBTableInfo::save
	(	KBDBInfo	*dbInfo,
		const QString	&server,
		KBError		&pError,
		bool		anyChange
	)
{
	extern	QString	kbXMLEncoding () ;

	/* Only need to do anything if the design data has changed, or	*/
	/* the width has changed _and_ the "anyChange" flag is set.	*/
	/* This is used to avoid saving the information when only a	*/
	/* width changes except when closing down.			*/
	if (!m_changed && !(anyChange && m_widthChanged))
		return	true	;

	QDomDocument	tXML	("tableInfo") ;
	QDomElement	root	;

	tXML.appendChild
	(	tXML.createProcessingInstruction
		(	"xml",
			QString("version=\"1.0\" encoding=\"%1\"").arg(kbXMLEncoding())
	)	)	;

	tXML.appendChild     (root = tXML.createElement ("tableInfo")) ;

	/* Add a top-level element which gives the table name. This is	*/
	/* not otherwise used by rekall, but might be useful elsewhere.	*/
	QDomElement	tblElem	= tXML.createElement ("info") ;
	root   .appendChild  (tblElem) ;
	tblElem.setAttribute ("name", m_name) ;


	/* Create the "unique" element and attach to it all the unique	*/
	/* key/default pairs. If there are none and the list is	empty	*/
	/* then rekaLL will fall back on its automatic mechanisms for	*/
	/* determining unique columns.					*/
	QDomElement	uniElem	= tXML.createElement ("unique") ;
	root.appendChild (uniElem) ;

	for (uint idx = 0 ; idx < m_unique.count() ; idx += 1)
	{
		QDomElement keyElem = tXML.createElement ("key") ;
		uniElem.appendChild  (keyElem) ;

		keyElem.setAttribute ("column", m_unique[idx].column()) ;
		keyElem.setAttribute ("defval", m_unique[idx].defval()) ;
	}


	/* Next create a "columns" element. Below this are added	*/
	/* entries for each column containing the associated design	*/
	/* information. Note that we skip columns for which no value is	*/
	/* set, so that the output document size is kept down.		*/
	QDomElement colsElem = tXML.createElement ("columns") ;
	root.appendChild (colsElem) ;

	QDictIterator<KBTableColumn>	colIter (m_columns) ;
	KBTableColumn			*column	;

	while ((column = colIter.current()) != 0)
	{
		if (!column->anyValueSet())
		{	colIter	 += 1 ;
			continue ;
		}

		QDomElement colElem = tXML.createElement ("column") ;
		colsElem.appendChild  (colElem) ;
		colElem .setAttribute ("name", colIter.currentKey()) ;

		for (uint idx = 0 ; idx < TI_COUNT ; idx += 1)
		{
			QDomElement designElem = tXML.createElement (designNames[idx]) ;
			colElem   .appendChild  (designElem) ;
			designElem.setAttribute ("value", column->designValue(idx)) ;
		}

		colIter	+= 1 ;
	}

	/* Next save the sorting filtering and view sets ...		*/
	QDomElement sortSetElem   = tXML.createElement("sortset"  ) ;
	QDomElement selectSetElem = tXML.createElement("selectset") ;
	QDomElement viewSetElem   = tXML.createElement("viewset"  ) ;

	root.appendChild (sortSetElem  ) ;
	root.appendChild (selectSetElem) ;
	root.appendChild (viewSetElem  ) ;

	LITER
	(	KBTableSort,
		m_sortSet,
		sort,
		sort  ->save(sortSetElem  )
	)
	LITER
	(	KBTableSelect,
		m_selectSet,
		filter,
		filter->save(selectSetElem)
	)
	LITER
	(	KBTableView,
		m_viewSet,
		view,
		view  ->save(viewSetElem  )
	)




	/* All done, save the document ....				*/
	KBLocation infoLoc(dbInfo, "info", server, m_name) ;

	return	infoLoc.save
		(	QString::null,
			QString::null,
			tXML.toString(),
			pError
		)	;
}

/*  KBTableInfo								*/
/*  unique	: Get unique columns list				*/
/*  (returns)	: KBTablUniqueList &	: Columns list			*/

const KBTableUniqueList	&KBTableInfo::unique ()
{
	return	m_unique ;
}

/*  KBTableInfo								*/
/*  setUnique	: Set unique columns list				*/
/*  copy	: KBTablUniqueList &	: Columns list			*/
/*  (returns)	: void			:				*/

void	KBTableInfo::setUnique
	(	const KBTableUniqueList	&copy
	)
{
	m_unique.clear () ;
	for (uint idx = 0 ; idx < copy.count() ; idx += 1)
		m_unique.append (copy[idx]) ;

	m_changed = true ;
}

void	KBTableInfo::sortList
	(	QStringList	&list
	)
{
	LITER
	(	KBTableSort,
		m_sortSet,
		sort,
		list.append(sort->name())
	)
}

void	KBTableInfo::selectList
	(	QStringList	&list
	)
{
	LITER
	(	KBTableSelect,
		m_selectSet,
		filter,
		list.append(filter->name())
	)
}

void	KBTableInfo::viewList
	(	QStringList	&list
	)
{
	LITER
	(	KBTableView,
		m_viewSet,
		view,
		list.append(view->name())
	)
}

KBTableSort
	*KBTableInfo::addSort
	(	const QString	&name
	)
{
	KBTableSort *ts = new KBTableSort (name) ;
	m_sortSet.append (ts) ;
	m_changed = true ;
	return	ts ;
}

KBTableSelect
	*KBTableInfo::addSelect
	(	const QString	&name
	)
{
	KBTableSelect *tf = new KBTableSelect (name) ;
	m_selectSet.append (tf) ;
	m_changed = true ;
	return	tf ;
}

KBTableView
	*KBTableInfo::addView
	(	const QString	&name
	)
{
	KBTableView *tv = new KBTableView (name) ;
	m_viewSet.append (tv) ;
	m_changed = true ;
	return	tv ;
}

KBTableSort
	*KBTableInfo::getSort
	(	const QString	&name
	)
{
	for (uint idx = 0 ; idx < m_sortSet  .count() ; idx += 1)
		if (m_sortSet  .at(idx)->name() == name)
			return	m_sortSet  .at(idx) ;

	return	0 ;
}

KBTableSelect
	*KBTableInfo::getSelect
	(	const QString	&name
	)
{
	for (uint idx = 0 ; idx < m_selectSet.count() ; idx += 1)
		if (m_selectSet.at(idx)->name() == name)
			return	m_selectSet.at(idx) ;

	return	0 ;
}

KBTableView
	*KBTableInfo::getView	
	(	const QString	&name
	)
{
	for (uint idx = 0 ; idx < m_viewSet  .count() ; idx += 1)
		if (m_viewSet  .at(idx)->name() == name)
			return	m_viewSet  .at(idx) ;

	return	0 ;
}

void	KBTableInfo::dropSort
	(	const QString	&name
	)
{
	m_sortSet  .remove (getSort   (name)) ;
	m_changed = true ;
}

void	KBTableInfo::dropSelect
	(	const QString	&name
	)
{
	m_selectSet.remove (getSelect (name)) ;
	m_changed = true ;
}

void	KBTableInfo::dropView	
	(	const QString	&name
	)
{
	m_viewSet  .remove (getView   (name)) ;
	m_changed = true ;
}

/*  ------------------------------------------------------------------  */

/*  KBTableInfoSet							*/
/*  KBTableInfoSet							*/
/* 		: Constructor for all tables information object		*/
/*  dbInfo	: KBDBInfo *	  : Database information object		*/
/*  server	: const QString & : Server name				*/
/*  (returns)	: KBTableInfo	  :					*/

KBTableInfoSet::KBTableInfoSet
	(	KBDBInfo	*dbInfo,
		const QString	&server
	)
	:
	m_dbInfo (dbInfo),
	m_server (server)
{
	static	bool	first	= true ;
	if (first)
	{
		KBLocation::registerType ("info", "inf", "Table Information", 0) ;
		first	= false	;
	}

	m_tableMap.setAutoDelete (true) ;
}


/*  KBTableInfoSet							*/
/*  getTableInfo: Get information for specified table			*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KBTableInfo *	  : Table information			*/

KBTableInfo
	*KBTableInfoSet::getTableInfo
	(	const QString	&tabName
	)
{
	KBTableInfo	*tabInfo ;
	KBError		error	 ;

//	fprintf
//	(	stderr,
//		"KBTableInfoSet::getTableInfo: [%s]->[%p]\n",
//		(cchar *)tabName,
//		(void  *)m_tableMap.find (tabName)
//	)	;

	if ((tabInfo = m_tableMap.find (tabName)) == 0)
	{
		m_tableMap.insert  (tabName, tabInfo = new KBTableInfo (tabName)) ;
		if (!tabInfo->load (m_dbInfo, m_server, error))
			error.DISPLAY() ;
	}

	return	tabInfo	;
}

/*  KBTableInfo								*/
/*  save	: Save all table information				*/
/*  anyChange	: bool		: True to save changes including width	*/
/*  (returns)	: void		:					*/

void	KBTableInfoSet::save
	(	bool	anyChange
	)
{
	QDictIterator<KBTableInfo>  tabIter (m_tableMap) ;
	KBTableInfo		   *tabInfo ;

	while ((tabInfo = tabIter.current()) != 0)
	{
		if (!tabInfo->save (m_dbInfo, m_server, m_error, anyChange))
			m_error.DISPLAY() ;

		tabIter	+= 1 ;
	}
}

/*  KBTableInfo								*/
/*  unique	: Get unique columns list				*/
/*  tabName	: const QString &	: Table name			*/
/*  (returns)	: KBTablUniqueList &	: Columns list			*/

const	KBTableUniqueList	&KBTableInfoSet::unique
	(	const QString	&tabName
	)
{
	return	getTableInfo(tabName)->unique() ;
}

/*  KBTableInfo								*/
/*  setUnique	: Set unique columns list				*/
/*  tabName	: const QString &	: Table name			*/
/*  copy	: KBTablUniqueList &	: Columns list			*/
/*  (returns)	: void			:				*/

void	KBTableInfoSet::setUnique
	(	const QString		&tabName,
		const KBTableUniqueList	&copy
	)
{
	getTableInfo(tabName)->setUnique(copy) ;
}

/*  KBTableInfo								*/
/*  designValue	: Get design value					*/
/*  tabName	: const QString & : Table name				*/
/*  column	: const QString & : Column name				*/
/*  idx		: uint		  : Design code				*/
/*  (returns)	: const QString & : Value				*/

const QString
	&KBTableInfoSet::designValue
	(	const QString	&tabName,
		const QString	&column,
		uint		idx
	)
{
	return	getTableInfo(tabName)->designValue(column, idx) ;
}

/*  KBTableInfo								*/
/*  setDesignValue							*/
/*		: Set design value					*/
/*  tabName	: const QString & : Table name				*/
/*  column	: const QString & : Column name				*/
/*  idx		: uint		  : Design code				*/
/*  column	: const QString & : New value				*/
/*  (returns)	: void		  :					*/

void	KBTableInfoSet::setDesignValue
	(	const QString	&tabName,
		const QString	&column,
		uint		idx,
		const QString	&value
	)
{
	getTableInfo(tabName)->setDesignValue(column, idx, value) ;
}


/*  KBTableInfoSet							*/
/*  load	: Load all table information				*/
/*  (returns)	: void		:					*/

void	KBTableInfoSet::load ()
{
	KBDBDocIter	docIter	;

	if (!docIter.init
		(	m_dbInfo,
			m_server,
			"info",
			"inf",
			m_error,
			false
		))
	{
		m_error.DISPLAY() ;
		return	;
	}

	QString	name	;
	QString	stamp	;

	while (docIter.getNextDoc (name, stamp))
		getTableInfo (name) ;
}


/*  KBTableInfoSet							*/
/*  renameTable	: Rename table						*/
/*  oldName	: const QString & : Old table name			*/
/*  newName	: const QString & : New table name			*/
/*  (returns)	: void		  :					*/

void	KBTableInfoSet::renameTable
	(	const QString	&oldName,
		const QString	&newName
	)
{
	KBError	     error	;
	KBLocation   infoLoc (m_dbInfo, "info", m_server, oldName) ;

	if (!infoLoc.rename (newName, error))
		if (error.getErrno() != ENOENT)
			error.DISPLAY() ;

	KBTableInfo *tabInfo = m_tableMap.find (oldName) ;
	if (tabInfo != 0)
	{
		m_tableMap.take   (oldName) ;
		m_tableMap.insert (newName, tabInfo) ;
		tabInfo->setName  (newName) ;
	}
	
}

/*  KBTableInfoSet							*/
/*  removeTable	: Remove table						*/
/*  tabName	: const QString & : Old table name			*/
/*  (returns)	: void		  :					*/

void	KBTableInfoSet::dropTable
	(	const QString	&tabName
	)
{
	KBError	   error	;
	KBLocation infoLoc (m_dbInfo, "info", m_server, tabName) ;

	if (!infoLoc.remove (error))
		if (error.getErrno() != ENOENT)
			error.DISPLAY() ;

	m_tableMap.remove (tabName) ;
}
