/***************************************************************************
    file	         : kb_blockact.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"

#include	"kb_qrybase.h"
#include	"kb_rowmark.h"

#include	"kb_display.h"
#include	"kb_block.h"
#include	"kb_formblock.h"
#include	"kb_framer.h"
#include	"kb_form.h"
#include	"kb_event.h"
#include	"kb_blockevents.h"



/*  KBFormBlock								*/
/*  doSyncAll	: Handle all-rows synchronisation			*/
/*  mvp		: KBValue *	: Value pointer to parent block		*/
/*  (returns)	: bool		: Success				*/

bool	KBFormBlock::doSyncAll
	(	KBValue		*mvp
	)
{
	if (!m_query->syncAll
		(	qryLvl,
			mvp,
			cexpr.getValue(),
			this
	   )	)
	{
		setError (m_query->lastError ()) ;
		return	 false ;
	}

	getLayout()->setChanged (false) ;

	KBValue	args[2] ;
	bool	evRc 	;

	args[0]	= 0	;
	args[1]	= (int)KB::SyncAll ;
	if (!eventHook (events->postSync, 2, args, evRc))
		return false ;

	m_dChanged = false ;
	return	true ;
}

/*  KBFormBlock								*/
/*  doSyncRow	: Handle current row synchronisation			*/
/*  mvp		: KBValue *	: Value pointer to parent block		*/
/*  qrow	: uint		: Row to synchronise			*/
/*  (returns)	: bool		: Success				*/

bool	KBFormBlock::doSyncRow
	(	KBValue		*mvp,
		uint		qrow
	)
{
	KB::Action	oper	;
	KBValue		args[3] ;
	bool		evRc 	;

	if (!m_query->syncRow
		(	qryLvl,
			qrow,
			mvp,
			cexpr.getValue(),
			this,
			oper,
			args[2]
	    )	)
	{
		setError (m_query->lastError ()) ;
		return	 false ;
	}

	getLayout()->setChanged (false) ;


	if (oper != KB::Null)
	{
		args[0]	= 0		;
		args[1]	= (int)oper	;
		if (!eventHook (events->postSync, 3, args, evRc))
			return false ;

		m_dChanged = false	;
	}

	return	true	;
}

/*  KBFormBlock								*/
/*  doOperation	: Handle various block actions				*/
/*  action	: KB::Action	   : Action required			*/
/*  toQRow	: uint		   : Target row for GotoQRow action	*/
/*  tabList	: KBTabOrderList * : Tab order list to be used		*/
/*  (returns)	: bool		   : Success				*/

bool	KBFormBlock::doOperation
	(	KB::Action	action,
		uint		toQRow,
		KBTabOrderList	*tabList
	)
{
	if (action == KB::Reload)
		return	doOperation (KB::Query,   0, tabList) &&
			doOperation (KB::Execute, 0, tabList) ;

	/* If the block is a null (menu-only) block then there is	*/
	/* nothing to do within this block, but we do scan down through	*/
	/* any nested form blocks or framers.				*/
	if (blkType == BTNull)
	{
		switch (action)
		{
			case KB::Save	 :
			case KB::SyncAll :
			case KB::Store	 :
				CITER
				(	FormBlock,
					b,
					if (!b->doOperation (action, 0))
					{	setError (b->lastError()) ;
						return   false ;
					}
				)
				CITER
				(	Framer,
					f,
					if (!f->doAction (action))
					{	setError (f->lastError()) ;
						return   false ;
					}
				)
				break	;

			default	:
				break	;
		}

		return	true	;
	}


	/* Note whether focus is in this block. If not then some	*/
	/* operations will not change focus.				*/
	bool	focusInBlock	= getForm()->focusInBlock (this) ;


	/* If there is a current item and it has changed then mark the	*/
	/* form as changed.						*/
	markChanged () ;

	/* As a prelude, carry out some checks which are common to	*/
	/* several of the actions. The first checks are for actions	*/
	/* which cannot be carried out during a query, or only during a	*/
	/* query.							*/
	switch (action)
	{
		case KB::First	 :
		case KB::Previous:
		case KB::Next	 :
		case KB::Last	 :
		case KB::Add	 :
		case KB::Insert	 :
		case KB::PrevBack:
		case KB::NextForw:
		case KB::PrevPage:
		case KB::NextPage:
		case KB::Query	 :
		case KB::Save	 :
		case KB::SyncAll :
		case KB::Store	 :
		case KB::Delete	 :
		case KB::GotoQRow:
			if (m_inQuery)
			{	setError
				(	KBError::Warning,
					TR("Executing a query"),
					QString::null,
					__ERRLOCN
				)	;
				return	 false ;
			}
			break	;

		case KB::Execute :
		case KB::Cancel	 :
			if (!m_inQuery)
			{	setError
				(	KBError::Warning,
					TR("Not executing a query"),
					QString::null,
					__ERRLOCN
				)	;
				return	false	;
			}
			break	;

		case KB::Reset	:
			break	;

		default		:
			break	;
	}


	/* Let the event hook if any have a look in here. If this	*/
	/* returns false then the action is cancelled (although it is	*/
	/* not an error).						*/
	KBValue	arg	((int)action, &_kbFixed) ;
	bool	evRc 	;

	if (!eventHook (events->onAction, 1, &arg, evRc)) return false ;
	if (!evRc) return true ;


	/* Next check for permission constraints. These may result from	*/
	/* underlying serverdatabase grant restrictions, or from unique	*/
	/* key restrictions.						*/
	switch (action)
	{
		case KB::Next	 :
		case KB::NextForw:
		case KB::NextPage:
			/* These are no problem unless we are at the	*/
			/* last row, in which case we need to check	*/
			/* whether insert is allowed ...		*/
			if (curQRow + 1 < m_query->getNumRows (qryLvl))
				break	;

		case KB::Add	 :
		case KB::Insert	 :
			/* ... also Next when at the last record.	*/
			if ((m_query->getPermission (qryLvl) & QP_INSERT) == 0)
			{	setError
				(	KBError::Warning,
					TR("Cannot insert rows"),
					TR("No unique key available after insert"),
					__ERRLOCN
				)	;
				return	false	;
			}
			break	;

		case KB::Delete	 :
			if ((m_query->getPermission (qryLvl) & QP_DELETE) == 0)
			{	setError
				(	KBError::Warning,
					TR("Cannot delete rows"),
					TR("No unique key column"),
					__ERRLOCN
				)	;
				return	false	;
			}

			/* Run the pre-delete hook at this point, as it	*/
			/* may cancel the deletion.			*/
			{
				KBValue	aCurQRow ((int)curQRow, &_kbFixed) ;

				if (!eventHook (events->preDelete, 1, &aCurQRow, evRc))
					return	false	;
				if (!evRc)
					return	true	;
			}

			break	;

		case KB::Save	 :
		case KB::SyncAll :
		case KB::Store	 :
			if ((m_query->getPermission (qryLvl) & QP_UPDATE) == 0)
			{	setError
				(	KBError::Warning,
					TR("Cannot update rows"),
					TR("No unique key column"),
					__ERRLOCN
				)	;
				return	false	;
			}
			break	;


		default	:
			break	;
	}

	/* This flag gets set if the row is saved to the database ...	*/
	bool	rowDelta = false ;

	/* Next pick up all the actions which may result in an implicit	*/
	/* save to the database, and see if there are any changes.	*/
	switch (action)
	{
		case KB::First	 :
		case KB::Previous:
		case KB::Next	 :
		case KB::Last	 :
		case KB::Add	 :
		case KB::Insert	 :
		case KB::PrevBack:
		case KB::NextForw:
		case KB::PrevPage:
		case KB::NextPage:
		case KB::Query	 :
		case KB::GotoQRow:
			if (!checkChange (true,  rowDelta)) return false ;
			break	;

		case KB::Save	 :
		case KB::SyncAll :
		case KB::Store	 :
			if (!checkChange (false, rowDelta)) return false ;
			break	;

		default	:
			break	;
	}

	/* In the case of a save request, see if anything has changed	*/
	/* here .... FIX ME!						*/
	switch (action)
	{
		case KB::Save	 :
		case KB::SyncAll :
		case KB::Store	 :
			CITER
			(	FormBlock,
				b,
				if (!b->doOperation (action, 0))
				{	setError (b->lastError()) ;
					return   false ;
				}
			)
			CITER
			(	Framer,
				f,
				if (!f->doAction (action))
				{	setError (f->lastError()) ;
					return   false ;
				}
			)
			break	;

		case KB::PrevPage :
			/* Next page. This is handled by scrolling by	*/
			/* one less than the number of rows, or one,	*/
			/* whichever is greater.			*/
			{	uint	newDRow	;
				uint	dr	= nRows <= 1 ? 1 : nRows - 1 ;

				if (dr < curDRow)
					newDRow	= curDRow - dr ;
				else	newDRow	= 0 ;

				scrollToRow (newDRow) ;
			}

			return	true	;

		case KB::NextPage :
			/* Similarly next page. Note that "scrollToRow"	*/
			/* will limit the movement to the number of	*/
			/* query rows.					*/
			{	uint	dr	= nRows <= 1 ? 1 : nRows - 1 ;
				scrollToRow (curDRow + dr) ;
			}
			return	true	;

		default	:
			break	;
	}


	/* Now the real work. In the process, set values for the item	*/
	/* and row which will become current, and note if the current	*/
	/* row may have changed.					*/
	KBObject	*newObj	 = 0	   ;
	KBItem		*newItem = m_curItem ;
	uint		newQRow	 = curQRow ;
	KBValue		*mvp	 = getBlockVal () ;
	KB::RState	rstate	 = m_query->getRowState (qryLvl, curQRow) ; 
	uint		nRowsQry = m_query->getNumRows  (qryLvl) ;
	uint		nRowsDel ;

	switch (action)
	{
		case KB::First	:
			/* Go to first extant record. Note that we do	*/
			/* not include the notional empty record at the	*/
			/* end.						*/
			if (nRowsQry > 0)
			{	newQRow	= 0 ;
				break	;
			}

			setError
			(	KBError::Warning,
				TR("No data was returned by server"),
				QString::null,
				__ERRLOCN
			)	;
			return	 false ;

		case KB::Previous:
		case KB::PrevBack:
			/* Go back one row ...				*/
			if (curQRow > 0)
			{	newQRow	= curQRow - 1 ;
				break	;
			}

			setError
			(	KBError::Warning,
				TR("Already at the first row"),
				QString::null,
				__ERRLOCN
			)	;
			return	 false ;

		case KB::Next	 :
		case KB::NextForw:
			/* Go to the next row. This is implemented such	*/
			/* that we can move from the last real record	*/
			/* to the notional empty record at the end.	*/
			if (curQRow < nRowsQry)
			{	newQRow	= curQRow + 1 ;
				break	;
			}

			setError
			(	KBError::Warning,
				TR("Already at the last row"),
				QString::null,
				__ERRLOCN
			)	;
			return	 false ;

		case KB::Last	:
			/* This is the reverse of first, again we do	*/
			/* not include the notional empty record unless	*/
			/* there are none at all.			*/
			if (m_query->getNumRows (qryLvl) > 0)
			{	newQRow	= m_query->getNumRows (qryLvl) - 1 ;
				break	;
			}

			setError
			(	KBError::Warning,
				TR("No data was returned by server"),
				QString::null,
				__ERRLOCN
			)	;
			return	 false ;

		case KB::GotoQRow:
			/* We require that the row actually exists, or	*/
			/* is the next new row.				*/
			if (toQRow <= m_query->getNumRows (qryLvl))
			{
				newQRow	= toQRow ;
				break	;
			}

			setError
			(	KBError::Warning,
				TR("Attempt to go to non-existent row"),
				QString	(TR("Target %1 but only %1 rows"))
					.arg(toQRow)
					.arg(m_query->getNumRows (qryLvl)),
				__ERRLOCN
			)	;
			return	false	;

		case KB::Add	:
			/* Actually, this doesn't add, it just moves	*/
			/* to the notional empty record.		*/
			newQRow	= m_query->getNumRows (qryLvl) ;
			break	;
	
		case KB::Save	:
		case KB::SyncAll:
			/* If auto-sync is not on then save does a sync	*/
			/* on all rows, in which case do a complete	*/
			/* requery and redisplay.			*/
			if (!autosync.getBoolValue())
			{
				if (!doSyncAll (mvp))
					return false ;

				if (!requery ())
					return false ;

				if (!showData ())
					return false ;

				enterBlock (true, 0) ;
				return	true  ;
			}

			/* If autosync was not set then the row will	*/
			/* have been synchronised in checkChange.	*/
			getLayout()->setChanged(false) ;
			break	 ;

		case KB::Insert	:
			/* Insert is not very meaningful for a normal	*/
			/* database query, since rows will be ordered	*/
			/* by the query, though it may be useful for	*/
			/* the user to add a new record while being	*/
			/* able see an extant record. The main use is	*/
			/* for table design, since we do have control	*/
			/* over the column order.			*/
			if (!m_query->insertRow (qryLvl, curQRow))
			{
				setError (m_query->lastError ()) ;
				return	 false ;
			}

			getLayout()->setChanged() ;

			/* The newly inserted row becomes the new	*/
			/* current row.					*/
			rowDelta = true ;
			break	 ;

		case KB::Delete	:
			/* Delete the current row, or all marked rows	*/
			/* if there are any; check for this first and	*/
			/* if not then delete the current row ....	*/
			if (!m_query->deleteAllMarked (qryLvl, nRowsDel))
			{
				setError (m_query->lastError ()) ;
				return	false	;
			}

			if (nRowsDel == 0)
			{
				/* No rows marked, so go for the	*/
				/* current row. This is silently	*/
				/* ignored if we are at the notional	*/
				/* empty row, except that the fields	*/
				/* are cleared.				*/
				if (curQRow >= m_query->getNumRows (qryLvl))
				{	clearFields (curQRow) ;
					return	 true  ;
				}

				if (!m_query->deleteRow (qryLvl, curQRow))
				{
					setError (m_query->lastError ()) ;
					return	 false ;
				}

				/* If autosync mode is on then sync	*/
				/* with the database, except if the row	*/
				/* was marked as previously inserted,	*/
				/* in which case it was never in the	*/
				/* database.				*/
				if (autosync.getBoolValue() && (rstate != KB::RSInserted))
					if (!doSyncRow(mvp, curQRow))
						return	false	;
			}
			else if (autosync.getBoolValue ())
			{
				/* Marked rows now marked for deletion,	*/
				/* and auto-sync is enabled. We do the	*/
				/* actual delete synchronisations one	*/
				/* at a time so the any post-sync hooks	*/
				/* are run.				*/
				uint	qrow	= 0 ;

				while (qrow < m_query->getNumRows (qryLvl))
					if (m_query->getRowMarked (qryLvl, qrow))
					{
						if (!doSyncRow (mvp, qrow))
							return	false	;
					}
					else	qrow	+= 1 ;
			}

			getLayout()->setChanged() ;

			/* Leave the query row where it is unless we	*/
			/* have just deleted the last row, in which	*/
			/* case go to the last row.			*/
			if (newQRow > 0)
			{	uint nQryRows = m_query->getNumRows(qryLvl) ;
				if (newQRow >= nQryRows)
					if (nQryRows > 0)
						newQRow	= nQryRows - 1 ;
					else	newQRow	= 0 ;
			}
			rowDelta = true ;
			break	;

		case KB::Query	:
			/* Clear the fields in the current row for the	*/
			/* used to enter query values, and flag that we	*/
			/* are in a query.				*/
			clearFields (curQRow, true) ;
			m_inQuery = true ;
			return	true ;

		case KB::Execute:
			/* Get the query to execute a SELECT, noting	*/
			/* that it should use the current field values	*/
			/* as query terms.				*/
			if (!eventHook (events->preQuery, 0, 0, evRc))
				return false ;

			if (!m_query->select
				(	qryLvl,
					mvp,
					cexpr.getValue(),
					m_userFilter,
					m_userSorting,
					true,
					curQRow
			   )	)
			{
				setError (m_query->lastError ()) ;
				return	 false ;
			}

			if (!eventHook (events->postQuery, 0, 0, evRc))
				return	false ;

			m_inQuery  = false ;
			newQRow    = 0	   ;
			rowDelta   = true  ;
			break	;

		case KB::Cancel	:
			/* Clear the query flag and redisplay the	*/
			/* original values.				*/
			m_query->loadItems (qryLvl, curQRow) ;
			m_inQuery  = false ;
			break	;

		case KB::Reset	:
			/* Terminate any avtive transaction first.	*/
			if (!endUpdate (false)) lastError().DISPLAY () ;

			m_dChanged = false ;
			/* Reset controls. If we are in a query, or if	*/
			/* this is an inserted row, then they are all	*/
			/* cleared.					*/

			if (m_inQuery)
			{	clearFields (curQRow) ;
				return	true ;
			}
			if (curQRow >= m_query->getNumRows(qryLvl))
			{	clearFields (curQRow) ;
				return	true ;
			}
			if (m_query->getRowState (qryLvl, curQRow) == KB::RSInserted)
			{	clearFields (curQRow) ;
				return	true ;
			}

			/* Otherwise, reload them ....			*/
			m_query->resetData (qryLvl, curQRow) ;
			m_query->loadItems (qryLvl, curQRow) ;

			getLayout()->setChanged (false) ;
			return	true ;

		case KB::Store	:
			/* Store does not move elsewhere, so nothing to	*/
			/* do here.					*/
			break	;

		default	:
			setError
			(	KBError::Fault,
				QString(TR("Unknown KBFormBlock action %1")).arg((int)action),
				QString::null,
				__ERRLOCN
			)	;
			return	 false ;
	}

	/* If the caller has not specified a tab order list then use	*/
	/* the one from our own navigation object.			*/
	if (tabList == 0) tabList = &m_tabList ;

	/* Next switch is just concerned with determining which control	*/
	/* item will get focus.						*/
	switch (action)
	{
		case KB::First	 :
		case KB::Last	 :
		case KB::Add	 :
		case KB::Insert	 :
		case KB::Delete	 :
		case KB::Query	 :
		case KB::Execute :
		case KB::Cancel	 :
		case KB::NextForw:
		case KB::GotoQRow:
			newObj	= goFirst (false) ;
			newItem = newObj != 0 ? newObj->isItem() : 0 ;
			break	;

		case KB::PrevBack:
			newObj	= goLast  (false) ;
			newItem = newObj != 0 ? newObj->isItem() : 0 ;
			break	;

		case KB::Store	 :
			/* Store does not move elsewhere, so nothing to	*/
			/* do here.					*/
			break	;

		default		:
			if (newItem == 0)
			{	newObj	= goFirst (false) ;
				newItem = newObj != 0 ? newObj->isItem() : 0 ;
			}
			break	;
	}

//	fprintf
//	(	stderr,
//		"KBFormBlock::doOperation: newobj=[%p](%s) newitem=[%p]\n",
//		(void *)newObj,
//		newObj == 0 ? "" : (cchar *)newObj->getElement(),
//		(void *)newItem
//	)	;

	/* Note whether we are switching to a new or item before	*/
	/* setting the now-current item and query row.			*/
	bool	chgItem	= (newItem != m_curItem) && (newItem != 0) ;
	bool	chgRow	= (newQRow != curQRow) ;


	/* Note whether we are switching to a new row before setting	*/
	/* the now-current item and query row.				*/
	bool	newRow	= newQRow != curQRow ;

	/* If on a new row or the row hash changed then notify the	*/
	/* query (neccessary for KBase-level queries) and redisplay.	*/
	/* We also run the UnCurrent event for the row we are leaving	*/
	/* (unless it has been deleted), and the Current event for the	*/
	/* noew current row.						*/
	if (newRow || rowDelta)
	{
		KBValue	argOld	((int)curQRow) ;
		KBValue	argNew	((int)newQRow) ;

		if (action != KB::Delete)
			if (!eventHook (events->onUnCurrent, 1, &argOld, evRc))
				return	false  ;

		curQRow = newQRow ;
		m_query->setCurrentRow (qryLvl, curQRow) ;

		if (!showData())
			return	false ;

		if (!eventHook (events->onCurrent, 1, &argNew, evRc))
			return	false ;
	}

	/* If focus is not in the block then don't do any focus		*/
	/* operations for queries, this means that script code can do	*/
	/* stuff like Query/Execute from another block without changing	*/
	/* focus.							*/
	if (!focusInBlock)
		switch (action)
		{
			case KB::Query	 :
			case KB::Execute :
			case KB::Cancel	 :
				return	true	;

			default	:
				break	;
		}

	/* Changing item. Update the current item pointer and give	*/
	/* focus to the current item at the current row.		*/
	if (chgItem)
	{
		m_curItem	= newItem ;
		m_curItem->giveFocus  (curQRow) ;
		getForm()->focusAtRow (curQRow) ;
		return	true ;
	}

	/* Changing row and an item has focus; give focus to that item	*/
	/* at the current row.						*/
	if (chgRow && (m_curItem != 0))
	{
		m_curItem->giveFocus  (curQRow) ;
		getForm()->focusAtRow (curQRow) ;
		return	true ;
	}

	/* Either the item and the row are both unchanged or there is	*/
	/* no current item, so nothing to do.				*/
	return true ;
}

/*  KBFormBlock								*/
/*  doAction	: Handle various query actions				*/
/*  action	: KBAction	   : Action required			*/
/*  tabList	: KBTabOrderList * : Tab order list to be used		*/
/*  (returns)	: bool		   : Success				*/

bool	KBFormBlock::doAction
	(	KB::Action	action,
		KBTabOrderList	*tabList
	)
{
//	KBFormBlock *fBlk = this ;
//
//	if (blkType == BTSubBlock)
//		if ((action == KB::Save) || (action == KB::SyncAll) || (action == KB::Store))
//		{
//			while (fBlk->getBlkType() == BTSubBlock)
//				if ((fBlk = fBlk->getParent()->isFormBlock()) == 0)
//					return	true ;
//		}

	bool	rc = doOperation (action, 0, tabList) ;
	getForm()->setFocusAtRow (this) ;
	return	rc ;
}

/*  KBFormBlock								*/
/*  gotoQRow	: Move to specified query row				*/
/*  qRow	: uint		: Query row number			*/
/*  (returns)	: bool		: Success				*/

bool	KBFormBlock::gotoQRow
	(	uint		qRow
	)
{
	bool	rc = doOperation (KB::GotoQRow, qRow) ;
	getForm()->setFocusAtRow (this) ;
	return	rc ;
}



/*  KBFormBlock								*/
/*  insertRow	: Insert empty row at specified position		*/
/*  qrow	: uint		: Insertion point			*/
/*  (returns)	: bool		: Success				*/

bool	KBFormBlock::insertRow
	(	uint	qrow
	)
{
	if (!checkChange (true)) return false ;
	curQRow	= qrow	;
	return	doAction (KB::Insert) ;
}

/*  KBFormBlock								*/
/*  deleteRow	: Delete row at specified position			*/
/*  qrow	: uint		: Deletion point			*/
/*  (returns)	: bool		: Success				*/

bool	KBFormBlock::deleteRow
	(	uint	qrow
	)
{
	if (!checkChange (false)) return false ;
	curQRow	= qrow	;
	return	doAction (KB::Delete) ;
}


/*  KBFormBlock								*/
/*  moveFocusTo	: Move focus to specified item				*/
/*  item	: KBItem *	: Item in question			*/
/*  (returns)	: void		:					*/

void	KBFormBlock::moveFocusTo
	(	KBItem		*item
	)
{
	if (item != m_curItem)
	{
		/* If moving controls, verify that the current item	*/
		/* has valid contents, unless we are in a query, in	*/
		/* which case anything goes.				*/
		if (!m_inQuery && (m_curItem != 0))
		{
			if (!m_curItem->doLeave (curQRow))
				return	;

			if (!m_curItem->isValid (curQRow, true))
			{	m_curItem->lastError().DISPLAY() ;
				return	;
			}
		}

		/* OK, so move ...					*/
		m_curItem = item ;
		getForm()->focusInEvent (m_curItem, curQRow) ;
		m_curItem->giveFocus    (curQRow) ;
	}
}

/*  KBFormBlock								*/
/*  scrollToRow	: Scroll to specified row				*/
/*  newDRow	: uint		: Row number				*/
/*  (returns)	: void		:					*/

void	KBFormBlock::scrollToRow
	(	uint	newDRow
	)
{
	uint	extra	= (m_query->getPermission(qryLvl) & QP_INSERT) != 0 ? 1 : 0 ;

	/* Start by bounding the upper display row to the valid range.	*/
	/* If it is unchanged then we can return.			*/
	if (newDRow + nRows > m_query->getNumRows (qryLvl) + extra)
		if (m_query->getNumRows (qryLvl) + extra < nRows)
			newDRow	= 0 ;
		else	newDRow = m_query->getNumRows (qryLvl) + extra - nRows ;

	if (newDRow == curDRow)
		return	;

	/* Generate a pseudo-action for the scroll so that the onAction	*/
	/* event handler can stop the scroll from occurring.		*/
	KBValue	arg	((int)KB::Scroll, &_kbFixed) ;
	bool	evRc 	;

	if (!eventHook (events->onAction, 1, &arg, evRc)) return ;
	if (!evRc) return ;

	/* If the current query row will be outside the displayed range	*/
	/* and (a) there are any child blocks or (b) sloppy focus is	*/
	/* off (the default) then we will always move the query row to	*/
	/* stay visible.						*/
	if ((curQRow < newDRow) || (curQRow >= newDRow + nRows))
		if (anyChildBlock || !m_sloppy.getBoolValue())
		{
			if (!checkChange (true))
			{
				lastError().DISPLAY()	;
				goto	revert		;
			}

			uint	newQRow	= curQRow < newDRow ?
						newDRow :
						newDRow + nRows - 1 ;

			curDRow	= newDRow ;
			focusMovesRow (newQRow) ;

			if (m_curItem != 0)
			{	focusMovesItem       (m_curItem, QFocusEvent::Tab) ;
				m_curItem->giveFocus (curQRow) ;
			}

			getForm()->setFocusAtRow   (this) ;
			return	;
		}


	/* We need to save any changed values into the query set so	*/
	/* that they will be correctly reloaded into the scrolled	*/
	/* control, unless this is an empty new row.			*/
	if ((curQRow >= curDRow) && (curQRow < curDRow + nRows))
		if (!m_query->newRowEmpty (qryLvl, curQRow))
			if (!m_query->saveRow (qryLvl, curQRow))
			{
				m_query->lastError().DISPLAY() ;
				goto	revert	;
			}

	/* Ensure that there are no unmorphed controls. This is needed	*/
	/* since focus may not enter another conrol.			*/
	getLayout ()->setUnMorphedItem (0, 0) ;


	curDRow	= newDRow ;
	showData  (true, false) ;

	/* If the current row is now out of sight then pass focus to	*/
	/* the block display widget and clear any rowmark current row.	*/
	if ((curQRow < curDRow) || (curQRow >= curDRow + nRows))
	{
		blkDisp->getDisplayWidget()->setFocus() ;
		if (blkInfo.rowmark != 0)
			blkInfo.rowmark->setCurrent (curQRow, true) ;
	}
	else	/* Current row on display, pass focus to the current	*/
		/* item if there is one.				*/
		if (m_curItem != 0) m_curItem->giveFocus(curQRow) ;

	return	;

	revert	:
		blkInfo.scroll->setRowRange
		(	m_query->getNumRows(qryLvl),
			extra,
			curQRow,
			curDRow,
			nRows
		)	;
}

/*  KBFormBlock								*/
/*  changeSizes	: Handle display size change				*/
/*  (returns)	: void		:					*/

void	KBFormBlock::changeSizes ()
{
	int	_dx	= dx.getIntValue() ;
	int	_dy	= dy.getIntValue() ;
	uint	numrows	= rowsInBlock   () ;

	if (numrows == nRows) return ;

	/*  FIXME! Need to handle the case where the current item is	*/
	/*  updated and disappears from view ....			*/

	/* Note the range of rows which will be exposed (which will be	*/
	/* negative if shrinking, then extent (or decrease) the number	*/
	/* of controls.							*/
	uint	lfrom	= nRows	  ;
	uint	lto	= numrows ;

	nRows	= numrows ;

	uint	extra	= 0	;
	uint	qrows	= 1	;

	if (m_query != 0)
	{
		extra	= (m_query->getPermission(qryLvl) & QP_INSERT) != 0 ? 1 : 0 ;
		qrows	= m_query->getNumRows(qryLvl) ;
	}

	CITER
	(	Item,
		i,
		i->extendCtrls (nRows, _dx, _dy) ;
		i->hideBelow   (qrows + extra) ;
	)
	CITER
	(	Framer,
		f,
		f->extendCtrls (nRows, _dx, _dy) ;
		f->hideBelow   (qrows + extra) ;
	)

	/* Display data in newly exposed rows (if any) and then fix up	*/
	/* the scroller.						*/
	if (showingData())
		displayData (true, lfrom + curDRow, lto + curDRow) ;
}

void	KBFormBlock::scrollBy
	(	int		delta
	)
{
	uint	newDRow	= curDRow	;

	if (delta < 0)
		newDRow	= (uint)(-delta) > newDRow ? 0 : newDRow - (uint)(-delta) ;
	else	newDRow	= newDRow + (uint)delta ;

	scrollToRow (newDRow)	;
}
