/***************************************************************************
    file	         : kb_builder.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<qstring.h>

#include	"kb_classes.h"
#include	"kb_database.h"
#include	"kb_dblink.h"
#include	"kb_location.h"
#include	"kb_tableinfo.h"
#include	"kb_attrdict.h"

#include	"kb_builder.h"


/*  formatText	: Get format text from wizard				*/
/*  wizard	: KBWizard &	: Wizard				*/
/*  tag		: cchar *	: Tag for wizard control		*/
/*  prefix	: cchar *	: Prefix to set on non-empty formats	*/
/*  (returns)	: void		:					*/

static	void	formatText
	(	KBWizard	&wizard,
		cchar		*tag,
		cchar		*prefix,
		QString		&raw,
		QString		&full
	)
{
	/* Get the format string (assumes control is in a page called	*/
	/* "format") and sets the prefix if the format is non-empty.	*/
	raw	= wizard.ctrlValue ("format", tag) ;
	full	= raw.isEmpty() ? QString::null : prefix + raw ;
}


/*  KBBuildFieldInfo							*/
/*  setFromWizard: Set format strings from wizard			*/
/*  wizard	 : KBWizard &	: Wizard				*/
/*  (returns)	 : void		:					*/

void	KBBuildFieldInfo::setFromWizard
	(	KBWizard	&wizard
	)
{
	formatText (wizard, "date",     "Date:",	m_rawFormatDate,	m_formatDate	) ;
	formatText (wizard, "time",     "Time:",	m_rawFormatTime,	m_formatTime	) ;
	formatText (wizard, "datetime", "DateTime:",	m_rawFormatDateTime,	m_formatDateTime) ;
	formatText (wizard, "float",    "Float:",	m_rawFormatFloat,	m_formatFloat	) ;
}

/*  ------------------------------------------------------------------  */

/*  builderCharWidth							*/
/*		: Calculate nominal character width			*/
/*  (returns)	: int		: Width					*/

int	builderCharWidth ()
{
	static	int	charWidth ;
	if (charWidth <= 0)
                charWidth = QFontMetrics(QFont()).size (0, "IWX").width() / 3 ;
	return	charWidth ;
}

/*  builderFieldWidth							*/
/*		: Calculate width for field				*/
/*  fieldSpec	: KBFieldSpec *	: Field specification			*/
/*  (returns)	: int		: Width or zero if not calculated	*/

int	builderFieldWidth
	(	KBFieldSpec	*fieldSpec
	)
{
	int	w	= builderCharWidth() * fieldSpec->m_length ;

	if	(w <  60) w = 60  ;
	else if	(w > 300) w = 300 ;

	return	w	;
}

/*  builderLinkWidth							*/
/*		: Calculate width for linked field			*/
/*  table	: const QString & : Table name				*/
/*  show	: const QString & : Expression shown from table		*/
/*  (returns)	: int		  : Width or zero if not calculated	*/

int	builderLinkWidth
	(	KBLocation	&location,
		const QString	&table,
		const QString	&show
	)
{
	KBDBLink    	dbLink		  ;
	KBTableSpec 	tableSpec (table) ;
	KBFieldSpec	*linkSpec	  ;

//	fprintf
//	(	stderr,
//		"builderLinkWidth: [%s] [%s] [%s]\n",
//		(cchar *)location.docLocn,
//		(cchar *)table,
//		(cchar *)show
//	)	;

	if (!dbLink.connect(location.dbInfo, location.docLocn))
		return	0 ;
	if (!dbLink.listFields (tableSpec))
		return	0 ;

	linkSpec = tableSpec.findField (show) ;
	if (linkSpec == 0) return 0 ;

	return	builderFieldWidth (linkSpec) ;
}

/*  ------------------------------------------------------------------  */

bool	builderSplitLookup
	(	const	QString	&value,
		QString		&vTable,
		QString		&vField,
		QString		&vShow
	)
{
	int	p1	;
	int	p2	;

	if ( ((p1 = value.find (":"	   )) >= 0) &&
	     ((p2 = value.find (":", p1 + 1)) >= 0) )
	{
		vTable	= value.left(p1) ;
		vField	= value.mid (p1 + 1, p2 - p1 - 1) ;
		vShow	= value.mid (p2 + 1) ;
		return	true ;
	}

	return	false	;
}

/*  ------------------------------------------------------------------  */

/*  builderMakeField							*/
/*		: Build field for use in form or report wizard		*/
/*  location	: KBLocation  &	: Dummy location			*/
/*  tabInfo	: KBTableInfo *	: Associated table information		*/
/*  fieldSpec	: KBFieldSpec &	: Field information			*/
/*  x, y, w, h	: int, ...	: Position and size			*/
/*  tabOrder	: int		: Tab ordering value			*/
/*  indent	: int		: Text layout indentation		*/
/*  info	: KBBuildFieldInfo &					*/
/*				: General field information		*/
/*  (returns)	: QString	: XML fragment				*/

QString	builderMakeField
	(	KBLocation		&location,
		KBTableInfo		*tabInfo,
		KBFieldSpec		*fieldSpec,
		int			x,
		int			y,
		int			&w,
		int			&h,
		int			tabOrder,
		int			indent,
		KBBuildFieldInfo	&info
	)
{
	KBAttrDict	fieldAttr  ;
	QString		vTable	   ;
	QString		vField	   ;
	QString		vShow	   ;
	bool		useLink	   = false ;
	QString		text	   ;


	fieldAttr.addValue ("x",	x) ;
	fieldAttr.addValue ("y",	y) ;
	fieldAttr.addValue ("h",	h) ;
	fieldAttr.addValue ("expr",	fieldSpec->m_name) ;

	if (tabOrder >= 0)
		fieldAttr.addValue ("taborder",	tabOrder) ;


	/* See if this is a linked field, in which case we construct a	*/
	/* link control ....						*/
	if (tabInfo != 0)
		if (builderSplitLookup (tabInfo->designValue(fieldSpec->m_name, TI_LINK), vTable, vField, vShow))
		{
			fieldAttr.addValue ("child", vField) ;
			fieldAttr.addValue ("show",  vShow ) ;
			useLink = true ;
		}

	if (useLink)
	{
		if (w <= 0) w = builderLinkWidth  (location, vTable, vShow) ;
		if (w <= 0) w = builderFieldWidth (fieldSpec) ;
		fieldAttr.addValue ("w", w) ;

		fieldAttr.addValue ("master",  fieldSpec->m_name) ;
		text	+= fieldAttr.print ("KBLink", false, indent) ;

		KBAttrDict qryAttr ;
		qryAttr.addValue ("server", location.docLocn) ;
		qryAttr.addValue ("table",  vTable) ;
		text	+= qryAttr.print ("KBQryTable", true, indent + 2) ;
		text	+= "    </KBLink>" ;

		return	text	;
	}

	if (w <= 0) w = builderFieldWidth (fieldSpec) ;
	fieldAttr.addValue ("w", w) ;


	/* Set the alignment and/or format based on the internal type	*/
	/* and the user format settings.				*/
	switch (fieldSpec->m_typeIntl)
	{
	 	case KB::ITFixed	:
			fieldAttr.addValue ("align",  Qt::AlignRight	   ) ;
			break	;

	     	case KB::ITFloat	:
			fieldAttr.addValue ("align",  Qt::AlignRight	   ) ;
			fieldAttr.addValue ("format", info.m_formatFloat   ) ;
			break	;

	     	case KB::ITDate		:
			fieldAttr.addValue ("format", info.m_formatDate    ) ;
			break	;

	     	case KB::ITTime		:
			fieldAttr.addValue ("format", info.m_formatTime    ) ;
			break	;

	     	case KB::ITDateTime	:
			fieldAttr.addValue ("format", info.m_formatDateTime) ;
			break	;

		case KB::ITBool		:
			/* Booleans will generate check boxes to we do	*/
			/* not need any format or alignment.		*/
			break	;

		default	:
			break	;
	}

	/* If the field is flagged as not-null or serial then it can	*/
	/* be empty when a record is saved, so flag it as null-OK.	*/
	/* NOTE: If this is a report then this value is ignored.	*/
	if ( ((fieldSpec->m_flags & KBFieldSpec::NotNull) == 0) ||
	     ((fieldSpec->m_flags & KBFieldSpec::Serial ) != 0) )
		fieldAttr.addValue ("nullok", "Yes") ;

	fieldAttr.addValue ("name",    fieldSpec->m_name) ;
	text	+= fieldAttr.print
		   (	fieldSpec->m_typeIntl == KB::ITBool ?
				"KBCheck" :
				"KBField",
			true,
			indent
		   )	;

	return	text ;
}

