/***************************************************************************
    file	         : kb_dispwidget.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#if		__KB_EMBEDDED
#include	<qpe/qpeapplication.h>
#endif

#include	<qpainter.h>
#include	<qstyle.h>

#ifndef 	_WIN32
#include	"kb_dispwidget.moc"
#else
#include	"kb_dispwidget.h"
#include	<qapplication.h>
#endif

#include	"kb_item.h"
#include	"kb_display.h"


/*  KBDispWidget: Display widget constructor				*/
/*  parent	: QWidget *	: Parent widget				*/
/*  display	: KBDisplay *	: Owning display object			*/
/*  showBar	: uint		: Show row scroll bar			*/
/*  (returns)	: KBDispScroller:					*/

KBDispWidget::KBDispWidget
	(	QWidget		*parent,
		KBDisplay	*display,
		uint		showBar
	)
	:
	QFrame 		(parent),
	m_display	(display),
	m_showBar	(0)
{
#if	__KB_EMBEDDED
	QPEApplication::setStylusOperation
	(	this,
		QPEApplication::RightOnHold
	)	;
#endif

	m_inSetRange = false		 ;
	m_vBar	     = 0		 ;
	m_rLabel     = 0		 ;
	m_tagLabel   = 0	 	 ;
	m_showing    = KB::ShowAsUnknown ;

	setShowbar (showBar) ;
	setMargin  (50)	;
}

/*  KBDispWidget							*/
/*  ~KBDispWidget: Display widget destructor				*/
/*  (returns)	 :		:					*/

KBDispWidget::~KBDispWidget ()
{
	DELOBJ (m_tagLabel) ;
	m_display->displayIsGone () ;
}


/*  KBDispScrollArea							*/
/*  setShowbar	: Enable/display display of row scroll bar		*/
/*  showBar	: uint		: Show row scroll bar			*/
/*  (returns)	: void		:					*/

void	KBDispWidget::setShowbar
	(	uint	showBar
	)
{
	if (showBar == m_showBar) return ;

	m_showBar = showBar ;

	if (m_showBar == 0)
	{
		DELOBJ	(m_vBar  ) ;
		DELOBJ	(m_rLabel) ;
		return	;
	}

	if (m_vBar == 0)
	{
		m_vBar	 = new QScrollBar (QScrollBar::Vertical, this) ;
		m_rLabel = new KBRecordNav (this, m_vBar->sizeHint().width()) ;

		connect
		(	m_vBar,
			SIGNAL(valueChanged(int)),
			SLOT(vbarMoved())
		)	;
		connect
		(	m_rLabel,
			SIGNAL(action(KB::Action)),
			SLOT  (slotAction(KB::Action))
		)	;
	}

	if ((m_showBar & NAV_SCROLL) == 0)
		m_vBar  ->hide () ;
	else	m_vBar  ->show () ;

	if ((m_showBar & NAV_MINI  ) == 0)
		m_rLabel->hide () ;
	else	m_rLabel->show () ;
}

/*  KBDispWidget							*/
/*  morphDestoyed: Catch destuction of morphed control			*/
/*  (returns)	 : void		:					*/

void	KBDispWidget::morphDestroyed ()
{
	m_morphList.removeRef ((KBItem *)sender()) ;
}

/*  KBDispWidget							*/
/*  addMorph	: Add a morphed control					*/
/*  morph	: KBItem *	: Morphed item				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::addMorph
	(	KBItem		*morph
	)
{
	m_morphList.append (morph) ;

	connect	(morph, SIGNAL(destroyed()), SLOT(morphDestroyed())) ;
}

/*  KBDispWidget							*/
/*  remMorph	: Remove a morphed control					*/
/*  morph	: KBItem *	: Morphed item				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::remMorph
	(	KBItem		*morph
	)
{
	if (m_morphList.removeRef (morph))
		disconnect (morph, SIGNAL(destroyed()), this, SLOT(morphDestroyed())) ;
}

/*  KBDispWidget							*/
/*  updateMorph	: Update a morphed control				*/
/*  morph	: KBItem *	: Morphed item				*/
/*  mRect	: const QRect &	: Area occupied by control		*/
/*  (returns)	: void		:					*/

void	KBDispWidget::updateMorph
	(	KBItem		*,
		const QRect	&mRect
	)
{
	update (mRect) ;
}

/*  KBDispWidget							*/
/*  updateMorph	: Update a morphed control				*/
/*  morph	: KBItem *	: Morphed item				*/
/*  drop	: uint		: Control display row			*/
/*  (returns)	: void		:					*/

void	KBDispWidget::updateMorph
	(	KBItem		*morph,
		uint		drow
	)
{
	QPainter p (this) ;
	morph->repaintMorph  (&p, drow) ;
}


/*  KBDispWidget							*/
/*  mousePressEvent							*/
/*		: Handler for mouse press events			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::mousePressEvent
	(	QMouseEvent	*e
	)
{
	m_mouseDownAt = e->pos () ;

	if (e->button() == QMouseEvent::LeftButton)
	{
		LITER
		(	KBItem,
			m_morphList,
			morph,

			if (morph->mouseClickHit (e->pos()))
				return ;
		)
#if	! __KB_RUNTIME
		if (m_display->markStartEvent (this, e))
			return ;
#endif
	}

	QWidget::mousePressEvent (e) ;
}

/*  KBDispWidget							*/
/*  mouseMoveEvent							*/
/*		: Handle mouse movement during marking			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::mouseMoveEvent
	(	QMouseEvent *e
	)
{
#if	! __KB_RUNTIME
	if ((e->state() & Qt::LeftButton) != 0)
		m_display->markMoveEvent (this, e) ;

	/* Lastly, a belt-and-braces check, since we seem otherwise to	*/
	/* miss some mouse release events.				*/
	if ((e->state() & Qt::LeftButton) == 0)
		m_display->markEndEvent (this, e) ;
#endif
}

/*  KBDispWidget							*/
/*  mouseReleaseEvent							*/
/*		: Handle mouse release to end marking			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::mouseReleaseEvent
	(	QMouseEvent *e
	)
{
#if	! __KB_RUNTIME
	if (e->button() == Qt::RightButton)
	{
		QPoint	mouseUpAt = e->pos() ;
		int	dx	  = mouseUpAt.x() - m_mouseDownAt.x() ;
		int	dy	  = mouseUpAt.y() - m_mouseDownAt.y() ;

		if (((dx * dx) + (dy * dy)) < 25)
		{
			m_display->contextEvent (this, e) ;
			return	;
		}
	}

	m_display->markEndEvent (this, e) ;
#endif
}

/*  KBDispWidet								*/
/*  mouseDoubleClickEvent						*/
/*		: Handler for mouse double click events			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::mouseDoubleClickEvent
	(	QMouseEvent	*e
	)
{
	/* Debugging hook. Double-click with the control and shift keys	*/
	/* down prints the widget tree.					*/
	if ((e->state() & (ShiftButton|ControlButton)) == (ShiftButton|ControlButton))
	{
		extern	void	printWidgetTree (QWidget *, uint, int) ;
		printWidgetTree (this, 0, -1) ;
		return	;
	}

#if	! __KB_RUNTIME
	m_display->contextEvent (this, e) ;
#endif
}

/*  KBDispWidget								*/
/*  paintEvent	: Handle painting for widget				*/
/*  e		: QPaintEvent *	: Paint event				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::paintEvent
	(	QPaintEvent *e
	)
{
	QPainter paint     (this)   ;
	QRect    r	= rect ()   ;

	if (!m_title.isEmpty ())
		r.setTop  (paint.fontMetrics().height() / 2) ;
	setFrameRect (r)  ;


	LITER
	(	KBItem,
		m_morphList,
		morph,

		morph->repaintMorph (&paint, e->rect()) ;
	)

	if (!m_title.isEmpty ())
	{
	        QFontMetrics fm = paint.fontMetrics () ;

		int	h	= fm.height() ;
		int	w	= fm.width (m_title) + 2 * fm.width (QChar(' ')) ;
		int	x	;
		int	align	;

#if	QT_VERSION >= 300
		align	= QApplication::reverseLayout() ? AlignRight : AlignLeft  ;
#else
		align	= AlignLeft ;
#endif

		switch (align)
		{
	        	case AlignHCenter :
				x = frameRect().width() / 2 - w / 2 ;
				break	;

			case AlignRight	  :
				x = frameRect().width() - w - 8 ;
				break	;

			default	:
				x = 8	;
				break	;
		}

		QRect	r (x, 0, w, h)	;

#if	QT_VERSION >= 300
		style().drawItem
		(	&paint,
			r,
			AlignCenter|ShowPrefix,
			colorGroup(),
                       	isEnabled (),
			0,
			m_title
		)	;
#else
		style().drawItem
		(	&paint,
			r.x	  (),
			r.y	  (),
			r.width   (),
			r.height  (),
			AlignCenter|ShowPrefix,
			colorGroup(),
                       	isEnabled (),
			0,
			m_title
		)	;
#endif
		paint.setClipRegion (e->region().subtract(r)) ;
	}

	drawFrame    (&paint) ;
	drawContents (&paint) ;

#if	! __KB_RUNTIME
	if (m_showing == KB::ShowAsDesign)
		m_display->drawDisplay (&paint, e->rect()) ;
#endif
}

/*  KBDispWidget								*/
/*  resizeEvent	: Handle resize for widget				*/
/*  e		: QResizeEvent *: Resize event				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::resizeEvent
	(	QResizeEvent *
	)
{
	if (m_showBar)
	{
		int sbw   = m_vBar->sizeHint().width() ;

		m_vBar->setGeometry
		(	width () - sbw,
			0,
			sbw,
			height()
		)	;

		m_rLabel->move (0, height() - m_rLabel->size().height()) ;
	}

	update	() ;
}

/*  KBDispWidget								*/
/*  setFrame	: Set frame style and width				*/
/*  style	: int		: Style					*/
/*  width	: int		: Width					*/
/*  (returns)	: void		:					*/

void	KBDispWidget::setFrame
	(	int	style,
		int	width
	)
{
	setFrameStyle	(style) ;
	setLineWidth	(width) ;
}

/*  KBDisplWidget								*/
/*  setTitle	: Set "group box" title					*/
/*  title	: const QString & : Title text				*/
/*  (returns)	: void		  :					*/

void	KBDispWidget::setTitle
	(	const QString	&title
	)
{
	m_title	= title	  ;
	update	     ()	  ;
}

/*  KBDispWidget							*/
/*  setRowRange	: Set range of rows being displayed			*/
/*  totalRows	: uint		: Total number of rows			*/
/*  extra	: uint		: Extra insertion rows			*/
/*  curQRow	: uint		: Current query row			*/
/*  curDRow	: uint		: Index of topmost displayed row	*/
/*  dispRows	: uint		: Number of rows actually visible	*/
/*  (returns)	: void		:					*/

void	KBDispWidget::setRowRange
	(	uint	totalRows,
		uint	extra,
		uint	curQRow,
		uint	curDRow,
		uint	dispRows
	)
{
	if (m_showBar)
	{
		/* Note the scrollbar-moved feedback is disabled while	*/
		/* setting the range. This is because range setting is	*/
		/* invoked from the block object, which will already	*/
		/* have scrolled its contents.				*/
		m_inSetRange = true  ;
		m_vBar->setRange  (0, totalRows + extra - dispRows) ;
		m_vBar->setSteps  (1, dispRows <= 1 ? 1 : dispRows  - 1) ;
		m_vBar->setValue  (curDRow) ;
		m_inSetRange = false ;

		m_rLabel->setRecord(curQRow, totalRows) ;
	}
}

/*  KBDispWidget								*/
/*  vbarMoved	: Notification of scrollbar change			*/
/*  (returns)	: void		:					*/

void	KBDispWidget::vbarMoved ()
{
	/* See comment in kb_dispscroller.cpp about this ...		*/
	if (!m_inSetRange)
		m_scrollTimer.singleShot (200, this, SLOT(scrollTimeout())) ;
}

/*  KBDispScrollWidget							*/
/*  scrollTimeout: Timeout on scroll bar movement			*/
/*  (returns)	 : void		:					*/

void	KBDispWidget::scrollTimeout ()
{
	int	value = m_vBar->value() ;
	m_display->scrollToRow (value < 0 ? 0 : value) ;
}

/*  KBDispWidget							*/
/*  slotAction	: Action requested					*/
/*  action	: KB::Action	: Action in question			*/
/*  (returns)	 : void		:					*/

void	KBDispWidget::slotAction
	(	KB::Action	action
	)
{
	m_display->doAction (action) ;
}

/*  KBDispWidget							*/
/*  effectiveSize: Get effective display size				*/
/*  (returns)	 : QSize	: Effective size			*/

QSize	KBDispWidget::effectiveSize ()
{
	int	w	= width	() ;
	int	h	= height() ;

	if ((m_showBar & NAV_SCROLL) != 0) w -= m_vBar  ->width () ;
	if ((m_showBar & NAV_MINI  ) != 0) h -= m_rLabel->height() ;

	return	QSize(w,h) ;
}

/*  KBDispWidget							*/
/*  showAs	: Set or clear design mode				*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBDispWidget::showAs
	(	KB::ShowAs	mode
	)
{
	m_showing = mode  ;
	setEnabled (true) ;
}

/*  KBDispWidget							*/
/*  event	: Event handler						*/
/*  e		: QEvent *	: Event					*/
/*  (returns)	: bool		: Event consumed			*/

bool	KBDispWidget::event
	(	QEvent		*e
	)
{
	/* Pass keypresses to the display object and thence the block	*/
	/* navigator. This is used to bring the current control back	*/
	/* into view if it has been scrolled out of site.		*/


	/* NOTE: If we do this at the QWidget::keyPressEvent level then	*/
	/* we don't get tabs.						*/
	if (e->type() == QEvent::KeyPress)
		return	m_display->keyStroke ((QKeyEvent *)e) ;


	/* Trap wheel events. These are passed to the display object	*/
	/* to scroll, and are then accepted and consumed.		*/
	if (e->type() == QEvent::Wheel)
	{
		QWheelEvent *w = (QWheelEvent *)e ;
		m_display->scrollBy (- w->delta() / 120) ;
		w->accept() 	;
		return	true	;
	}

	return	QWidget::event (e) ;
}

/*  KBDispWidget							*/
/*  setTagLabel	: Set tag label for report design			*/
/*  text	: const QString & : text				*/
/*  (returns)	: void		  :					*/

void	KBDispWidget::setTagLabel
	(	const QString	&text
	)
{
	m_tagText = text	;
}


/*  KBDispWidget							*/
/*  getTagLabel	: Get report design tagging label			*/
/*  parent	: QWindget *	: Parent for label			*/
/*  (returns)	: QLabel   *	: The label				*/

QLabel	*KBDispWidget::getTagLabel
	(	QWidget		*parent
	)
{
	if ((m_tagLabel == 0) && !m_tagText.isNull())
	{
		m_tagLabel = new QLabel (parent) ;

		m_tagLabel->setFrameStyle (QFrame::Box|QFrame::Plain) ;
		m_tagLabel->setLineWidth  (1) ;

		m_tagLabel->setText	  ("WWWWWWWWWWWWWWWWWW") ;
		m_tagLabel->setFixedSize  (m_tagLabel->sizeHint()) ;
		m_tagLabel->setText	  (m_tagText) ;

		m_tagLabel->setBackgroundMode (QWidget::PaletteLight) ;
	}

	return	m_tagLabel ;
}

