/***************************************************************************
    file	         : kb_item.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qapp.h>
#include	<qregexp.h>

#ifndef 	_WIN32
#include	"kb_item.moc"
#else
#include	"kb_item.h"
#endif

#include	"kb_ctrl.h"
#include	"kb_formblock.h"
#include	"kb_qrybase.h"
#include	"kb_nodemonitor.h"
#include	"kb_writer.h"
#include	"kb_layout.h"
#include	"kb_display.h"
#include	"kb_sizer.h"



QPalette *getMarkedPalette ()
{
	static	QPalette *markedPal ;

	if (markedPal == 0)
	{
		const QColor fg = QColor (255, 255, 255) ;
		const QColor bg = QColor (  0,   0,   0) ;
	
		markedPal = new QPalette (QApplication::palette()) ;
		markedPal->setColor  (QColorGroup::Text,       fg) ;
		markedPal->setColor  (QColorGroup::Foreground, fg) ;
		markedPal->setColor  (QColorGroup::Base,       bg) ;
		markedPal->setColor  (QColorGroup::Background, bg) ;
	}

	return	markedPal ;
}

/*  KBItem								*/
/*  KBItem	: Constructor for form item class			*/
/*  node	: KBNode *	: Parent node				*/
/*  name	: cchar *	: Name of actual node			*/
/*  ename	: cchar *	: Name of expression attribute		*/
/*  aList	: const QDict<QString> &				*/
/*				: Attribute list			*/
/*  (returns)	: KBItem	:					*/

KBItem::KBItem
	(	KBNode			*node,
		cchar 			*name,
		cchar			*ename,
		const QDict<QString>	&aList
	)
	:
	KBObject (node, name,       		aList),
	expr	 (this, ename,      		aList),
	rdonly	 (this, "rdonly",   		aList, KF_FORM),
	noupdate (this, "noupdate",   		aList, KF_FORM),
	tabOrd	 (this,	"taborder", 		aList, KF_FORM),
	defVal	 (this, "default",  		aList),
	errtext	 (this, "errtext",		aList),
	onEnter	 (this, "onenter",  "onItem",	aList, KF_FORM),
	onLeave	 (this, "onleave",  "onItem",	aList, KF_FORM),
	onSet	 (this, "onset",    "onItem",	aList)
{
	nCtrls		= 0	;
	ctrls		= 0	;
	ctrlMon		= 0	;
	m_table		= 0	;
	m_flags		= 0	;
	m_allEnabled	= true	;
	m_allVisible	= true	;
	m_isUpdateVal	= -1	;

	type		= 0	;
}

/*  KBItem								*/
/*  KBItem	: Constructor for form item class			*/
/*  parent	: KBObject *	: Parent object				*/
/*  name	: cchar *	: Name of actual node			*/
/*  rect	: const QRect &	: Size and position			*/
/*  ename	: cchar *	: Name of expression attribute		*/
/*  value	: cchar *	: Value for exprssion attribute		*/
/*  tabOrd	: uint		: Tab order				*/
/*  (returns)	: KBItem	:					*/

KBItem::KBItem
	(	KBObject	*parent,
		cchar 		*name,
		const QRect &	rect,
		cchar 		*ename,
		cchar		*value,
		uint		tabOrd
	)
	:
	KBObject (parent, name, rect),
	expr	 (this,   ename,       		   value),
	rdonly	 (this,   "rdonly",    		   false,	KF_FORM),
	noupdate (this,   "noupdate",  		   false,	KF_FORM),
	tabOrd	 (this,	  "taborder",  		   tabOrd + 1,	KF_FORM),
	defVal	 (this,   "default",   		   ""),
	errtext	 (this,   "errtext",		   ""),
	onEnter	 (this,   "onenter",   "onItem",   "",		KF_FORM),
	onLeave	 (this,   "onleave",   "onItem",   "",		KF_FORM),
	onSet	 (this,   "onset",     "onItem",   "")
{
	nCtrls		= 0	;
	ctrls		= 0	;
	ctrlMon		= 0	;
	m_table		= 0	;
	m_flags		= 0	;
	m_allEnabled	= true	;
	m_allVisible	= true	;
	m_isUpdateVal	= -1	;

	type		= 0	;
}

/*  KBItem								*/
/*  KBItem	: Constructor for form item class			*/
/*  _parent	: KBNode *	: Parent node				*/
/*  _ename	: cchar *	: Name of expression attribute		*/
/*  _item	: KBItem *	: Extant item				*/
/*  (returns)	: KBItem	:					*/

KBItem::KBItem
	(	KBNode		*_parent,
		cchar		*_ename,
		KBItem		*_item
	)
	:
	KBObject (_parent, _item),
	expr	 (this, _ename,     		_item),
	rdonly	 (this, "rdonly",   		_item, KF_FORM),
	noupdate (this, "noupdate",   		_item, KF_FORM),
	tabOrd	 (this,	"taborder", 		_item, KF_FORM),
	defVal	 (this, "default",  		_item),
	errtext	 (this, "errtext",		_item),
	onEnter	 (this, "onenter",  "onItem",   _item, KF_FORM),
	onLeave	 (this, "onleave",  "onItem",   _item, KF_FORM),
	onSet	 (this, "onset",    "onItem",   _item)
{
	nCtrls		= 0	;
	ctrls		= 0	;
	ctrlMon		= 0	;
	m_table		= 0	;
	m_flags		= 0	;
	m_allEnabled	= true	;
	m_allVisible	= true	;
	m_isUpdateVal	= -1	;

	/* See comment in constructor above ....			*/
	type		= 0	;
}

/*  KBItem								*/
/*  ~KBItem	: Destructor for form item class			*/
/*  (returns)	:		:					*/

KBItem::~KBItem ()
{
	if (ctrls != 0)
	{
		for (uint idx = 0 ; idx < nCtrls ; idx += 1)
			DELOBJ (ctrls[idx]) ;

		delete	[] ctrls ;
	}

	if (type != 0) type->deref() ;
}

/*  KBItem								*/
/*  moveFocusOK	: Check whether focus can be set			*/
/*  drow	: uint		: Display row				*/
/*  (returns)	: bool		: True if focus can be set		*/

bool	KBItem::moveFocusOK
	(	uint	drow
	)
{
	/* Unless in design mode, or there is no parent block (and	*/
	/* there should be!), check with the block whether focus is	*/
	/* allowed into this item (really, out of a current item).	*/
	return	showingDesign()	     ? true :
		getFormBlock () == 0 ? true :
				     getFormBlock()->moveFocusOK (this, drow) ;
}

/*  KBItem								*/
/*  focusInEvent: Handle focus in					*/
/*  drow	: uint		: Display row				*/
/*  reason	: Reason	: Reason focus arrives			*/
/*  (returns)	: void		:					*/

void	KBItem::focusInEvent
	(	uint			drow,
		QFocusEvent::Reason	reason
	)
{
	if (getFormBlock() != 0)
		getFormBlock()->focusInEvent (this, drow, reason) ;
}

/*  KBItem								*/
/*  doEnter	: Handle focus entry					*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: void		:					*/

void	KBItem::doEnter
	(	uint	qrow
	)
{
	KBValue	arg	= (int)qrow ;
	bool	evRc	;
	eventHook (onEnter, 1, &arg, evRc) ;
}

/*  KBItem								*/
/*  doLeave	: Handle focus exit					*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: bool		: OK to leave				*/

bool	KBItem::doLeave
	(	uint	qrow
	)
{
	KBValue	arg	= (int)qrow ;
	bool	evRc	;
	bool	rc	= eventHook (onLeave, 1, &arg, evRc) ? evRc : false ;

//	fprintf
//	(	stderr,
//		"KBItem::doLeave [%s][%s] -> %d\n",
//		(cchar *)getElement(),
//		(cchar *)getName(),
//		rc
//	)	;

	return	rc	;
}

/*  KBItem								*/
/*  setupCtrls	: Setup controls					*/
/*  nRows	: uint		: New number of rows			*/
/*  dx		: int		: X offset between rows			*/
/*  dy		: int		: Y offset between rows			*/
/*  (returns)	: void		:					*/

void	KBItem::setupCtrls
	(	uint	nRows,
		int	dx,
		int	dy
	)
{
	/* Reducing the number of controls is easy; just delete those	*/
	/* that are no longer needed (and note the new number). If	*/
	/* there are none left then clear the object control pointer.	*/
	if (nRows <= nCtrls)
	{
		for (uint idx = nRows ; idx < nCtrls ; idx += 1)
			DELOBJ (ctrls[idx]) ;

		nCtrls	= nRows	;

		if (nCtrls == 0) setControl (0) ;
		return	;
	}

	/* Increasing the number of controls. Start by allocating a	*/
	/* new controls vector and copying pointers at the extant	*/
	/* controls.							*/
	KBControl **save = ctrls	  ;
	ctrls	= new KBControl * [nRows] ;

	for (uint idx1 = 0 ; idx1 < nCtrls ; idx1 += 1)
		ctrls[idx1] = save[idx1] ;

	if (save != 0) delete [] save ;

	/* Now allocate as many more controls as are needed. These	*/
	/* are stored in the controls arrany and appropriately		*/
	/* positioned.							*/
	QRect	  r	= geometry() ;
	int	  ax	= r.x() + dx * nCtrls ;
	int	  ay	= r.y() + dy * nCtrls ;
	QPalette  *pal 	= getPalette  (false) ;
	QFont	  *font = getFont     (false) ;

	for (uint idx2 = nCtrls ; idx2 < nRows ; idx2 += 1)
	{
		KBControl *ctrl   = makeCtrl (idx2) ;

#if	__KB_RUNTIME
		ctrl->showAs  	  (KB::ShowAsData) ;
#else
		ctrl->showAs  	  (isShowing   ()) ;
#endif
		ctrl->setGeometry (ax, ay, r.width(), r.height()) ;

		ctrl->setVisible  (m_allEnabled) ;
		ctrl->setEnabled  (m_allVisible) ;

		ctrl->setPalette  (pal ) ;
		ctrl->setFont     (font) ;
		ctrls[idx2] = ctrl ;

		ax += dx ;
		ay += dy ;

		if (ctrlMon != 0)
			ctrl->showMonitor (ctrlMon) ;

		if (showingData())
			ctrl->setMorphed (isMorphing()) ;
	}


	/* Note the number of controls now present, make sure all	*/
	/* controls have the correct palette and font, and set the	*/
	/* object control as this may be the first call.		*/
	nCtrls	= nRows	;
	setControl (nCtrls == 0 ? 0 : ctrls[0]) ;
}

/*  KBItem								*/
/*  extendCtrls	: Extend (or decrease) the number of controls		*/
/*  nRows	: uint		: New number of rows			*/
/*  dx		: int		: X offset between rows			*/
/*  dy		: int		: Y offset between rows			*/
/*  (returns)	: void		:					*/

void	KBItem::extendCtrls
	(	uint	nRows,
		int	dx,
		int	dy
	)
{

	/* Setup the required number of controls and then prepare any	*/
	/* that are new.						*/
	uint	_nCtrls	= nCtrls ;
	setupCtrls (nRows, dx, dy) ;
	if (nCtrls > _nCtrls) prepareCtrls (_nCtrls, nCtrls) ;
}

/*  KBItem								*/
/*  prepare	: Prepare for execution					*/
/*  (returns)	: void		:					*/

void	KBItem::prepare ()
{
	KBObject::prepare ()  ;
	prepareCtrls	  (0, nCtrls) ;
}

/*  KBItem								*/
/*  prepareCtrls: Prepare controls for execution			*/
/*  first	: uint		: First control to be prepared		*/
/*  limit	: uint		: Limit of controls to be prepared	*/
/*  (returns)	: void		:					*/

void	KBItem::prepareCtrls
	(	uint	,
		uint
	)
{
	/* Placeholder only. Classes like KBLink will reimplement this	*/
}

/*  KBItem								*/
/*  reposition	: Repostision individual controls			*/
/*  (returns)	: void		:					*/

void	KBItem::reposition ()
{
	if (getBlock() != 0)
	{
		QRect	r	= geometry() ;
		int	ax	= r.x	  () ;
		int	ay	= r.y	  () ;
		int	dx	= getBlock()->getAttrVal("dx").toInt() ;
		int	dy	= getBlock()->getAttrVal("dy").toInt() ;

		for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		{
			ctrls[idx]->setGeometry (ax, ay, r.width(), r.height()) ;
			ax	+= dx	;
			ay	+= dy	;
		}
	}
}

/*  KBItem								*/
/*  ctrlGone	: Note death of control					*/
/*  ctrl	: KBControl *	: The control				*/
/*  (returns)	: void		:					*/

void	KBItem::ctrlGone
	(	KBControl	*ctrl
	)
{
	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		if (ctrls[idx] == ctrl)
		{	ctrls[idx] = 0	;
			break		;
		}

	KBObject::ctrlGone (ctrl) ;
}

/*  KBItem								*/
/*  move	: Move control(s)					*/
/*  _x		: int		: New X-position			*/
/*  _y		: int		: New Y-position			*/
/*  (returns)	: void		:					*/

void	KBItem::move
	(	int	_x,
		int	_y
	)
{
	KBObject::move (_x, _y) ;
	reposition     ()	;
}

/*  KBItem								*/
/*  resize	: Move control(s)					*/
/*  _w		: int		: New width				*/
/*  _h		: int		: New height				*/
/*  (returns)	: bool		: Size changed				*/

bool	KBItem::resize
	(	int	_w,
		int	_h
	)
{
	if (KBObject::resize (_w, _h))
	{	reposition () ;
		return	true  ;
	}

	return	false	;
}

/*  KBItem								*/
/*  redraw	: Redraw controls after design change			*/
/*  (returns)	: void		:					*/

void	KBItem::redraw ()
{
	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		ctrls[idx]->redraw () ;
}

/*  KBItem								*/
/*  setPalette	: Set palette						*/
/*  (returns)	: void		:					*/

void	KBItem::setPalette ()
{
	KBObject::setPalette () ;

	QPalette  *pal 	= getPalette (false) ;
	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		ctrls[idx]->setPalette (pal) ;
}

/*  KBItem								*/
/*  setFont	: Set font						*/
/*  (returns)	: void		:					*/

void	KBItem::setFont ()
{
	KBObject::setFont () ;

	QFont	*font 	= getFont (false) ;
	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		ctrls[idx]->setFont (font) ;
}

/*  KBItem								*/
/*  getVTrans	: Get value translation mode				*/
/*  (returns)	: KBValue::VTrans : Translation mode			*/

KBValue::VTrans	KBItem::getVTrans ()
{
	/* Default is text, since this is true for almost all items,	*/
	/* the obvious exception being pixmaps.				*/
	return	KBValue::VText	;
}

/*  KBItem								*/
/*  makeCtrl	: Make specific control					*/
/*  drow	: uint		: Display row number			*/
/*  (returns)	: KBControl *	: New control				*/

KBControl *KBItem::makeCtrl
	 (	uint
	 )
{
	KBError::EFault
	(	QString	(TR("KBItem::makeCtrl called for \"%1::%2\""))
			.arg (getElement())
			.arg (name.getValue()),
		QString::null,
		__ERRLOCN
	)	;
	return	0 ;
}

/*  buildCtrls	: Build controls for this item				*/
/*  numrows	: uint		: Number of rows to be displayed	*/
/*  dx		: int		: X offset between rows			*/
/*  dy		: int		: Y offset between rows			*/
/*  (returns)	: void		:					*/

void	KBItem::buildCtrls
	(	uint		numrows,
		int		dx,
		int		dy
	)
{
	setupCtrls (numrows, dx, dy) ;
}

/*  KBItem								*/
/*  getExpr	: Get item expression and set query column index	*/
/*  (returns)	: QString	: Expression				*/

QString	KBItem::getExpr ()
{
	return	expr.getValue ()   ;
}


/*  KBItem								*/
/*  setFieldType: Set database field type for the item			*/
/*  _type	: KBType *	: Type					*/
/*  (returns)	: void		:					*/

void	KBItem::setFieldType
	(	KBType	*_type
	)
{
	if (type != 0) type->deref() ;
	type	= _type	;
	if (type != 0) type->ref  () ;
}

/*  KBItem								*/
/*  isUpdateVal	: Test if item is used when updating database		*/
/*  (returns)	: bool		: TRUE if so				*/

bool	KBItem::isUpdateVal ()
{
	if (m_isUpdateVal < 0)
	{
		/* NOTE : Qt2.2.1: Setting the case sensitive argument	*/
		/* false does not seem to work, hence [a-zA-Z]		*/

		/* FIXME: This should be handled in the drivers ...	*/
		static	QRegExp e1	("^\\s*[_a-zA-Z0-9]*[_a-zA-Z][_a-zA-Z0-9]*\\s*$", false) ;
		static	QRegExp e2	("^\\s*[_a-zA-Z0-9]*[_a-zA-Z][_a-zA-Z0-9]*\\s*\\.\\s*[_a-zA-Z0-9]*[_a-zA-Z][_a-zA-Z0-9]*\\s*$", false) ;
		const 	QString	&e	= expr.getValue () ;

		m_isUpdateVal = (e1.match (e) >= 0) || (e2.match (e) >= 0) ;

		fprintf
		(	stderr,
			"KBItem::showAs: [%s] update [%d]\n",
			(cchar *)e,
			m_isUpdateVal
		)	;
	}

	return	noupdate.getBoolValue() ? false : m_isUpdateVal ;
}

/*  KBItem								*/
/*  getValue	: Get value						*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBItem::getValue
	(	uint	qrow
	)
{
	static	KBValue	nullVal	;
	KBControl *ctrl = ctrlAtQRow (qrow) ;
	return	ctrl == 0 ? nullVal : ctrl->getValue()  ;
}

/*  KBItem								*/
/*  getRowValue	: Get value for row from query				*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBItem::getRowValue
	(	uint	qrow
	)
{
	return	getBlock()->getQuery()->getField
				(	qryIdx.qryLvl(),
					qrow,
					qryIdx.qryIdx()
				)	;
}

/*  KBItem								*/
/*  getOrderType: Get ordering type					*/
/*  (returns)	: KB::IType	: Type					*/

KB::IType KBItem::getOrderType()
{
	/* Return the ordering type for the item given a raw value.	*/
	/* The default is the value itself, but the method may be	*/
	/* overridden, for instance by a link control.			*/
	return	type == 0 ? KB::ITUnknown : type->getIType() ;
}

/*  KBItem								*/
/*  getOrderValue							*/
/*		: Get ordering value					*/
/*  value	: const KBValue & : Value				*/
/*  (returns)	: QString	  : Ordering value			*/

QString	KBItem::getOrderValue
	(	const KBValue	&value
	)
{
	/* Return the ordering value for the item given a raw value.	*/
	/* The default is the value itself, but the method may be	*/
	/* overridden, for instance by a link control.			*/
	return	value.getRawText() ;
}

/*  KBItem								*/
/*  setData	: Set control-specific data				*/
/*  qrow	: uint		: Query row				*/
/*  data	: void *	: Data					*/
/*  (returns)	: void		:					*/

void	KBItem::setData
	(	uint	qrow,
		void	*data
	)
{
	KBControl *ctrl = ctrlAtQRow (qrow) ;
	if (ctrl != 0) ctrl->setData (data) ;
}

/*  KBItem								*/
/*  setValue	: Set value display row					*/
/*  qrow	: uint		 : Query row				*/
/*  value	: const KBValue &: Value to set				*/
/*  (returns)	: bool		 : Success				*/

bool	KBItem::setValue
	(	uint		qrow,
		const KBValue	&value
	)
{
	KB::ScriptRC	eRC	;

	fprintf
	(	stderr,
		"KBItem::setValue: [%s][%s]: %d: [%s]\n",
		(cchar *)getElement(),
		(cchar *)getName(),
		qrow,
		(cchar *)value.getRawText()
	)	;

	curVal	= expr.evaluate (value, eRC, fSubs) ;

	switch (eRC)
	{
		case KB::ScriptInlineError :
			propertyDlg (expr.getName()) ;
			return	false	   ;

		case KB::ScriptGlobalError :
			return	false	   ;

		default	:
			break	;
	}

	KBControl *ctrl = ctrlAtQRow  (qrow)   ;

//	fprintf
//	(	stderr,
//		"KBItem::setValue: [%d/%d][%s][%s]->[%p]\n",
//		qrow,
//		getBlock()->getCurQRow(),
//		(cchar *)value .getRawText(),
//		(cchar *)curVal.getRawText(),
//		(void  *)ctrl
//	)	;

	if (ctrl != 0) ctrl->setValue (curVal) ;

	/* Normally the type will be set from the query as soon as the	*/
	/* select query has been executed, but in case we set a value	*/
	/* before the query (eg., in the block "PreQuery" event) then	*/
	/* use the value type if the current type is unknown.		*/
	if (type->getIType() == KB::ITUnknown)
	{
		type->deref() ;
		type	= curVal.getType() ;
		type->ref  () ;
	}

	KBValue	args[2]	    ;
	bool	evRc	    ;

	args[0] = (int)qrow ;
	args[1] = curVal    ;

	return	eventHook (onSet, 2, args, evRc) ;
}

/*  KBItem								*/
/*  clearValue	: Clear value for specified query and display row	*/
/*  qrow	: uint		: Starting row in query			*/
/*  query	: bool		: Clearing for query			*/
/*  (returns)	: void		:					*/

void	KBItem::clearValue
	(	uint	qrow,
		bool	query
	)
{
	KBControl *ctrl = ctrlAtQRow (qrow) ;
	if (ctrl != 0) ctrl->clearValue (query) ;
}

/*  KBItem								*/
/*  changed	: See if user has changed an item			*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: bool		: Item changed				*/

bool	KBItem::changed
	(	uint	qrow
	)
{
	if (isUpdateVal())
	{
		KBControl *ctrl = ctrlAtQRow (qrow) ;
		return	ctrl == 0 ? false : ctrl->changed () ;
	}

	return	false	;
}

/*  KBItem								*/
/*  isEmpty	: See if control is empty				*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: bool		: Empty					*/

bool	KBItem::isEmpty
	(	uint	qrow
	)
{
	KBControl *ctrl = ctrlAtQRow (qrow) ;
	return	ctrl == 0 ? true : ctrl->isEmpty () ;
}

/*  KBItem								*/
/*  isValid	: Test if value is valid to be saved			*/
/*  qrow	: uint		: Query row				*/
/*  allowNull	: bool		: Ignore not-null checks		*/
/*  (returns)	: bool		: Value can be saved			*/

bool	KBItem::isValid
	(	uint	qrow,
		bool	allowNull
	)
{
	KBControl *ctrl = ctrlAtQRow (qrow) ;
	if (ctrl == 0) return true	    ;

	/* At this point check for an empty value along with a default	*/
	/* value or expression, in which case evaluate the default	*/
	/* expression: note that we pass the default itself as the	*/
	/* 'default' for the evaluation, so that writing a simple	*/
	/* default value has the obvious effect. On return check for	*/
	/* errors.							*/
	KBValue	  value	= ctrl->getValue () ;
	if (value.isEmpty() && !defVal.getValue().isEmpty())
	{
		KB::ScriptRC	eRC	;
		bool		_fSubs	;
		KBValue		value2	= defVal.evaluate
					  (	KBValue(defVal.getValue()),
						eRC,
						_fSubs
					  )	;

		switch (eRC)
		{
			case KB::ScriptInlineError :
				propertyDlg (expr.getName()) ;
				return	false	   ;

			case KB::ScriptGlobalError :
				return	false	   ;

			default	:
				break	;
		}

		/* Got a default, so pass this down to the control,	*/
		/* whence it can be retrieved and checked ...		*/
		if (!setValue (qrow, KBValue(value2.getRawText(), type)))
			return	false	;
	}

	if (!ctrl->isValid (allowNull))
	{	setError (ctrl->lastError ()) ;
		return	 false ;
	}

	return	true ;
}

/*  KBItem								*/
/*  updateProps	: Update properties					*/
/*  (returns)	: void		:					*/

void	KBItem::updateProps ()
{
	/* This method handles common properties for items over and	*/
	/* above those at the object level.				*/
	KBObject::updateProps () ;

	reposition () ;
	redraw	   () ;
	if (getNavigator() != 0) getNavigator()->fixTabOrder () ;
}

/*  KBItem								*/
/*  showAs	: Set or clear design mode				*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBItem::showAs
	(	KB::ShowAs	mode
	)
{
	m_allEnabled	= true	;
	m_allVisible	= true	;

	if (mode != KB::ShowAsData) m_isUpdateVal = -1 ;

//	if (expr.getValue().isEmpty() || (type == 0))
//	{	if (type != 0) type->deref() ;
//		type = &_kbString ;
//		type->ref () ;
//	}

	/* Note: If the expression is empty then set the type to string	*/
	/* so that anything is acceptable. This is needed since without	*/
	/* an expression, the item will not be set from the database	*/
	/* and hence we would not get a type.				*/
	if (type != 0) type->deref() ;

	type = expr.getValue().isEmpty() ? &_kbString : &_kbUnknown ;
	type->ref  ()	   ;

//	fprintf
//	(	stderr,
//		"KBItem::showAs: [%s] [0 ... %d]\n",
//		(cchar *)getName(),
//		nCtrls
//	)	;

	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
	{
		ctrls[idx]->showAs	(mode) ;
		ctrls[idx]->setMorphed	(isMorphing() && (mode == KB::ShowAsData)) ;
	}

	setControl (ctrls[0]) ;

	if (isMorphing() && (mode == KB::ShowAsData))
		getDisplay()->addMorph (this) ;
	else	getDisplay()->remMorph (this) ;

	KBObject::showAs (mode) ;
}

/*  KBItem								*/
/*  hideBelow	: Hide controls from specified row downwards		*/
/*  qrow	: uint		: Specific row				*/
/*  (returns)	: void		:					*/

void	KBItem::hideBelow
	(	uint	qrow
	)
{
	bool	show	= true ;

#if	! __KB_RUNTIME
	if (!showingData ())
	{
		for (uint idx = 0 ; idx < nCtrls ; idx += 1)
			ctrls[idx]->setVisible (true) ;
		return	;
	}
#endif
	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
	{
		if (show && (qrow - getBlock()->getCurDRow() == idx))
			show	= false	;

		ctrls[idx]->setVisible (show && m_allVisible) ;
	}
}

/*  KBItem								*/
/*  clearBelow	: Clear controls from specified row downwards		*/
/*  qrow	: uint		: Specific row				*/
/*  (returns)	: void		:					*/

void	KBItem::clearBelow
	(	uint	qrow
	)
{
	bool	clear	= false ;

	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
	{
		if (!clear && (qrow - getBlock()->getCurDRow() == idx))
			clear	= true ;

		ctrls[idx]->clearValue (false) ;
	}
}

/*  KBItem								*/
/*  ctrlAtQRow	: Get control at specified query row if any		*/
/*  qrow	: uint		: Specific row				*/
/*  (returns)	: KBControl *&	: Control or none			*/

KBControl *&KBItem::ctrlAtQRow
	(	uint	qrow
	)
{
	static	KBControl  *noCtrl = 0 ;

	uint	curDRow  = getBlock()->getCurDRow    () ;

	if (ctrls != 0)
		if ((qrow >= curDRow) && (qrow < curDRow + nCtrls))
			return	ctrls[qrow - curDRow] ;

	setError
	(	KBError::Error,
		TR("Row in query not currently displayed"),
		QString	(TR("Field %1, query row %2: showing %3 ... %4"))
			.arg(name.getValue())
			.arg(qrow)
			.arg(curDRow)
			.arg(curDRow + nCtrls   - 1),
		__ERRLOCN
	)	;
	return	noCtrl	;
}

/*  KBItem								*/
/*  setMarked	: Set or clear marked state				*/
/*  qrow	: uint		: Specific row				*/
/*  enable	: bool		: Enable				*/
/*  (returns)	: void		:					*/

void	KBItem::setMarked
	(	uint	qrow,
		bool	marked
	)
{
	KBControl *ctrl = ctrlAtQRow (qrow) ;
	if (ctrl == 0) return ;

//	static	QPalette *markedPal ;
//	if (markedPal == 0)
//	{
//		const QColor fg = QColor (255, 255, 255) ;
//		const QColor bg = QColor (  0,   0,   0) ;
//	
//		markedPal = new QPalette (QApplication::palette()) ;
//		markedPal->setColor  (QColorGroup::Text,       fg) ;
//		markedPal->setColor  (QColorGroup::Foreground, fg) ;
//		markedPal->setColor  (QColorGroup::Base,       bg) ;
//		markedPal->setColor  (QColorGroup::Background, bg) ;
//	}

	QPalette *pal = marked ? getMarkedPalette() : getPalette(true) ;
	ctrl->setPalette (pal) ;
}

/*  KBItem								*/
/*  setVisible	: Set control visibility				*/
/*  qrow	: uint		: Specific row				*/
/*  visible	: bool		: Visibility				*/
/*  (returns)	: void		:					*/

void	KBItem::setVisible
	(	uint	qrow,
		bool	visible
	)
{
	KBControl *ctrl	= ctrlAtQRow (qrow) ;
	if (ctrl != 0) ctrl->setVisible (visible) ;
}

/*  KBItem								*/
/*  setEnabled	: Enable or disable a control				*/
/*  qrow	: uint		: Specific row				*/
/*  enable	: bool		: Enable				*/
/*  (returns)	: void		:					*/

void	KBItem::setEnabled
	(	uint	qrow,
		bool	enabled
	)
{
	KBControl *ctrl	= ctrlAtQRow (qrow) ;
	if (ctrl != 0) ctrl->setEnabled (enabled) ;
}

/*  KBItem								*/
/*  qrow	: uint		: Specific row				*/
/*  isEnabled	: Get control enabled/disabled				*/
/*  (returns)	: bool		: Enabled/disabled			*/

bool	KBItem::isEnabled
	(	uint	qrow
	)

{
	KBControl *ctrl	= ctrlAtQRow (qrow) ;
	return	ctrl == 0 ? false : ctrl->isEnabled () ;
}

/*  KBItem								*/
/*  isVisible	: Get control visibility				*/
/*  qrow	: uint		: Specific row				*/
/*  (returns)	: bool		: Visibility				*/

bool	KBItem::isVisible 
	(	uint	qrow
	)
{
	KBControl *ctrl	= ctrlAtQRow (qrow) ;
	return	ctrl == 0 ? false : ctrl->isVisible () ;
}

/*  KBItem								*/
/*  setAllVisible: Set control visibility of all controls		*/
/*  visible	 : bool		: Visibility				*/
/*  (returns)	 : void		:					*/

void	KBItem::setAllVisible
	(	bool	visible
	)
{
	m_allVisible = visible ;
	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		if (ctrls[idx] != 0)
			ctrls[idx]->setVisible (m_allVisible) ;
}

/*  KBItem								*/
/*  setAllEnabled: Enable or disable all controls			*/
/*  enabled	 : bool		: Enabled				*/
/*  (returns)	 : void		:					*/

void	KBItem::setAllEnabled
	(	bool	enabled
	)
{
	m_allEnabled = enabled ;
	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		if (ctrls[idx] != 0)
			ctrls[idx]->setEnabled (m_allEnabled) ;
}

/*  KBItem								*/
/*  reMorphItem	: Convert control back to morph on losing focus		*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: void		:					*/

void	KBItem::reMorphItem
	(	uint	drow
	)
{
	/* If this item is morphing, and the control at this row is not	*/
	/* a morph control, then convert it.				*/
	if (isMorphing() && (drow < nCtrls) && !ctrls[drow]->isMorphed())
		ctrls[drow]->setMorphed (true) ;
}

/*  KBItem								*/
/*  giveFocus	: Give focus to specified row				*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: void		:					*/

void	KBItem::giveFocus
	(	uint	qrow
	)
{
	KBControl *&ctrl = ctrlAtQRow (qrow) ;
	if (ctrl == 0) return ;

	getLayout()->setUnMorphedItem (0, 0) ;

	/* If this is a morphing item and the control is currently	*/
	/* morphed, then convert it to the real control.		*/
	if (isMorphing() && ctrl->isMorphed())
	{	ctrl->setMorphed (false) ;
		getLayout()->setUnMorphedItem (this, ctrl->getDRow())  ;
	}

	ctrl->giveFocus() ;
}

/*  KBItem								*/
/*  setTabOrder	: Return tab order value				*/
/*  order	: int		: Tab order				*/
/*  (returns)	: void		:					*/

void	KBItem::setTabOrder
	(	int	order
	)
{
	tabOrd.setValue (order) ;
}

/*  KBItem								*/
/*  getTabOrder	: Return tab order value				*/
/*  (returns)	: int		: Tab order				*/

int	KBItem::getTabOrder ()
{
	return	tabOrd.getIntValue () ;
}

/*  KBItem								*/
/*  ctrlGeometry: Get control geometry					*/
/*  qrow	: uint		: Query row number			*/
/*  rCtrl	: QRect &	: Result				*/
/*  (returns)	: bool		: Success				*/

bool	KBItem::ctrlGeometry
	(	uint	qrow,
		QRect	&rCtrl
	)
{
	rCtrl	= ctrls[qrow - getBlock()->getCurDRow()]->geometry() ;
	return	true ;
}


/*  KBItem								*/
/*  keyStroke	: Control has received keystroke			*/
/*  k		: QKeyEvent *	: Key event in question			*/
/*  (returns)	: bool		: Key event consumed			*/

bool	KBItem::keyStroke
	(	QKeyEvent *k
	)
{
	KBNavigator *navi = getNavigator() ;
	return	navi == 0 ? false : navi->keyStroke (this, k) ;
}

/*  KBItem								*/
/*  isReadOnly	: See if item is read only				*/
/*  (returns)	: bool		: True for read only			*/

bool	KBItem::isReadOnly ()
{
	/* Check the embedding form block. Fields are never read-only	*/
	/* if we are in a query; fields are always read-only if the	*/
	/* block is marked read-only.					*/
	KBFormBlock *fb	  = getBlock()->isFormBlock()  ;
	if (fb != 0)
	{	if (fb->isInQuery    ()) return false ;
		if (fb->isBlkReadOnly()) return true  ;
	}

	if (rdonly.getBoolValue ()) return true ;
	if ((m_flags & KBFieldSpec::ReadOnly) != 0) return true ;

	return	false ;
}

/*  KBItem								*/
/*  getReportValue							*/
/*		: Get value for report writing				*/
/*  first	: bool		: First output flag			*/
/*  prior	: bool		: Write prior value			*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBItem::getReportValue
	(	bool		,
		bool
	)
{
	return	curVal	;
}

/*  KBItem								*/
/*  write	: Write control values					*/
/*  writer	: KBWriter *	: Writer				*/
/*  offset	: QPoint	: Writer offset				*/
/*  first	: bool		: First on page flag			*/
/*  extra	: init &	: Return extra space			*/
/*  prior	: bool		: Write prior value			*/
/*  (returns)	: bool		: Success				*/

bool	KBItem::write
	(	KBWriter	*writer,
		QPoint		offset,
		bool		first,
		int		&extra,
		bool		prior
	)
{
	if (!writer->asReport())
	{
		/* This method will be called in design mode and when	*/
		/* printing forms. The work is all done at the control	*/
		/* level, but we iterate through each row of controls.	*/
		QRect	posn	= geometry(offset) ;
		int	dx	= getBlock()->getAttrVal("dx").toInt() ;
		int	dy	= getBlock()->getAttrVal("dy").toInt() ;

		for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		{
			if (showingDesign() || ctrls[idx]->isVisible ())
				ctrls[idx]->write (writer, posn, KBValue(), false, extra) ;
			posn.moveBy	  (dx, dy) ;
		}

		return	true	;
	}

	return	ctrls[0]->write
		(	writer,
			geometry (offset),
			getReportValue (first, prior),
			fSubs,
			extra
		)	;
}

/*  KBItem								*/
/*  setMonitor	: Set monitor values for this node			*/
/*  myitem	: KBNodeMonitor * : Monitor item for this node		*/
/*  (returns)	: void		  :					*/

void	KBItem::setMonitor
	(	KBNodeMonitor	*myitem
	)
{
	KBNode::setMonitor (myitem) ;

	if (myitem != 0)
	{
		ctrlMon = new KBNodeMonitor (0, myitem) ;
		ctrlMon->setText (0, "Controls") ;
		ctrlMon->setSelectable (false) ;
	}
	else	ctrlMon	= 0 ;

	for (uint idx = 0 ; idx < nCtrls ; idx += 1)
		if (ctrls[idx] != 0)
			ctrls[idx]->showMonitor (ctrlMon) ;
}

#if	! __KB_EMBEDDED

/*  KBItem								*/
/*  property	: Get QT widget property				*/
/*  qrow	: uint		 : Query row				*/
/*  name	: cchar *	: Property name				*/
/*  (returns)	: QVariant	: Property value			*/

QVariant
	KBItem::property
	(	uint		qrow,
		cchar		*name
	)
{
	KBControl *ctrl = ctrlAtQRow  (qrow)   ;
	return	ctrl == 0 ? QVariant() : ctrl->property (name) ;
}

/*  KBItem								*/
/*  setProperty	: Set QT widget property				*/
/*  qrow	: uint		 : Query row				*/
/*  name	: cchar *	   : Property name			*/
/*  value	: const QVariant & : Value				*/
/*  (returns)	: bool		   : Success				*/

bool	KBItem::setProperty
	(	uint		qrow,
		cchar		*name,
		const QVariant	&value
	)
{
	KBControl *ctrl = ctrlAtQRow  (qrow)   ;
	return	ctrl == 0 ? false : ctrl->setProperty (name, value) ;
}

#endif

/*  KBItem								*/
/*  repaintMorph: Repaint morphed control				*/
/*  p		: QPainter *	: Painter				*/
/*  drow	: uint		: Display row				*/
/*  (returns)	: void							*/

void	KBItem::repaintMorph
	(	QPainter	*p,
		uint		drow
	)
{
	if (drow < nCtrls) ctrls[drow]->repaintMorph (p) ;
}

/*  KBItem								*/
/*  repaintMorph: Repaint morphed controls				*/
/*  p		: QPainter *	: Painter				*/
/*  area	: const QRect &	: Area to be repainted			*/
/*  (returns)	: void							*/

void	KBItem::repaintMorph
	(	QPainter	*p,
		const QRect	&area
	)
{
//	fprintf	(stderr, "repaintMorph [%d,%d,%d,%d]\n",
//			 area.x	    (),
//			 area.y     (),
//			 area.width (),
//			 area.height()
//		)	;

	for (uint drow = 0 ; drow < nCtrls ; drow += 1)
	{
		KBControl	*ctrl	= ctrls[drow]	;
		QRect		cRect	= ctrl->geometry() ;

		if (cRect.intersects(area)) ctrl->repaintMorph (p) ;
			ctrl->repaintMorph (p) ;
	}
}

bool	KBItem::mouseClickHit
	(	const QPoint	&pos
	)
{
	KBBlock	*block	= getBlock () ;
	uint	curDRow	= block->getCurDRow () ;
	uint	numRows	= block->getNumRows () ;

	for (uint drow = 0 ; drow < nCtrls ; drow += 1)
	{
		KBControl *ctrl = ctrls[drow] ;

		if (drow + curDRow > numRows + 1)
			break	 ;

		if (!ctrl->isVisible ()) continue ;
		if (!ctrl->isEnabled ()) continue ;

		if (ctrl->geometry().contains(pos))
		{
			if (moveFocusOK (drow))
			{	giveFocus (curDRow + drow) ;
				if (showingData())
					focusInEvent (drow, QFocusEvent::Mouse) ;
			}

			return	  true	;
		}
	}

	return	false	;
}

/*  KBItem								*/
/*  isMorphing	: Test if item is morphed				*/
/*  (returns)	: bool		: Morphed				*/

bool	KBItem::isMorphing ()
{
	/* This is always false. Specific data controls like fields and	*/
	/* links may override this.					*/
	return	false	;
}

/*  KBItem								*/
/*  getIniValue	: Get initial value					*/
/*  qrow	: uint		: Row number				*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBItem::getIniValue
	(	uint	qrow
	)
{
	KBQryBase	*query	= getBlock()->getQuery() ;

	return	query->getField (getQueryLvl(), qrow, getQueryIdx(), true) ;
}

/*  KBItem								*/
/*  userChange	: User has changed value				*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: void		:					*/

void	KBItem::userChange
	(	uint	qrow
	)
{
	KBFormBlock *fBlk = getFormBlock() ;

	/* Check if this item is updatable (should not get called	*/
	/* unless it is, but ...), and that it is in a form (ditto ...)	*/
	/* and ignore otherwise; also, more interestingly, ignore if we	*/
	/* are in a query.						*/
	if (!isUpdateVal() || (fBlk == 0) || fBlk->isInQuery())
		return	;

	/* Also ignore changes if the associated query is the null	*/
	/* (menu-only) query.						*/
	if (fBlk->getQuery()->isQryNull() != 0)
		return	;

	/* Notify the layout object so that the toolbar gets updated,	*/
	/* and the form block so that the block-level change event can	*/
	/* be triggered.						*/
	getLayout()->setChanged() ;
	fBlk->dataChanged  (qrow) ;
}

bool	KBItem::isInQuery ()
{
	KBFormBlock *fBlk = getBlock()->isFormBlock() ;
	return	fBlk == 0 ? false : fBlk->isInQuery() ;
}

bool	KBItem::startUpdate
	(	uint		qrow
	)
{
	if (showingData() && !isInQuery())
	{
		KBFormBlock *fBlk = getFormBlock() ;
		if ((fBlk != 0) && !fBlk->startUpdate (qrow))
		{
			fBlk->lastError().DISPLAY() ;
			return	false ;
		}
	}

	return	true	;
}

/*  KBItem								*/
/*  doSearch	: Do record search					*/
/*  (returns)	: void		:					*/

void	KBItem::doSearch ()
{
	/* Default is to do nothing. Controls that can actually do a	*/
	/* record search override this.					*/
}

/*  KBItem								*/
/*  getSearchText: Get sarch text for record search			*/
/*  qrow	 : uint		: Query row				*/
/*  (returns)	 : QString	: Text					*/

QString	KBItem::getSearchText
	(	uint	qrow
	)
{
	return	getBlock()->getRowValue(this, qrow).getRawText() ;
}

