/***************************************************************************
    file	         : kb_macro.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qlayout.h>


#include	"kb_classes.h"
#include	"kb_node.h"
#include	"kb_appptr.h"
#include	"kb_callback.h"

#ifndef		_WIN32
#include	"kb_macro.moc"
#else
#include	"kb_macro.h"
#endif

#include	"kb_macrodef.h"
#include	"kb_macrodebdlg.h"




static	QDict<MKMACRO*>		macroDict 	;


/*  ------------------------------------------------------------------  */

/*  KBMacroInstr							*/
/*  KBMacroInstr: Constructor for macro instruction base class		*/
/*  exec	: KBMacroExec *	  : Parent macro executor		*/
/*  action	: const QString & : Macro action			*/
/*  (returns)	: KBMacroInstr	  :					*/

KBMacroInstr::KBMacroInstr
	(	KBMacroExec	*exec,
		const QString	&action
	)
	:
	m_exec	(exec),
	m_action(action)
{
}

/*  KBMacroInstr							*/
/*  init	: Initialise macro instruction				*/
/*  args	: const QStringList &	: Macro arguments		*/
/*  comment	: const QString &	: Macro comment			*/
/*  minArgs	: uint			: Minimum number of arguments	*/
/*  maxArgs	: uint			: Maximum number of arguments	*/
/*  pError	: KBError &		: Return error			*/
/*  (returns)	: bool			: Success			*/

bool	KBMacroInstr::init
	(	const QStringList	&args,
		const QString		&comment,
		uint			minArgs,
		uint			maxArgs,
		KBError			&pError
	)
{
	/* BASE LEVEL INITIALISATION ROUTINE				*/
	/* ---------------------------------				*/

	/* Check that the number of arguments is within the specified	*/
	/* bounds. This is about all we can do as a common validation.	*/
	/* Note that the name must be valid or we could not have found	*/
	/* a constructor function.					*/
	if (args.count() < minArgs)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Macro instruction has too few arguments"),
				QString(TR("Action: %1: Needs %2 but has %3"))
					.arg(m_action)
					.arg(minArgs )
					.arg(args.count()),
				__ERRLOCN
			  )	;
		return	false	;
	}
	if (args.count() > maxArgs)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Macro instruction has too many arguments"),
				QString(TR("Action: %1: Needs %2 but has %3"))
					.arg(m_action)
					.arg(maxArgs )
					.arg(args.count()),
				__ERRLOCN
			  )	;
		return	false	;
	}

	m_comment = comment	;
	m_args	  = args	;
	return	true ;
}

/*  KBMacroInstr							*/
/*  init	: Initialise macro instruction				*/
/*  elem	: QDomElement &	: Definition				*/
/*  (returns)	: bool		: Success				*/

bool	KBMacroInstr::init
	(	QDomElement	&elem,
		KBError		&pError
	)
{
	/* Extract the arguments from the DOM element, then do the real	*/
	/* initialisation with these and the comment.			*/
	QStringList	args	;

	DOMITER_BEGIN(elem,"arg",arg)
	{
		args.append (arg.text()) ;
	}
	DOMITER_END

	return	init
		(	args,
			elem.attribute("comment"),
			pError
		)	;
}

/*  KBMacroInstr							*/
/*  ~KBMacroInstr: Destructor for macro instruction base class		*/
/*  (returns)	 :		:					*/

KBMacroInstr::~KBMacroInstr ()
{
}

/*  KBMacroInstr							*/
/*  save	: Save macro definition to DOM document			*/
/*  elem	: QDomElement &	: Element into which to save		*/
/*  (returns)	: void		:					*/

void	KBMacroInstr::save
	(	QDomElement	&elem
	)
{
	QDomElement instr = elem.ownerDocument().createElement("instruction") ;

	instr.setAttribute ("action",  m_action ) ;
	instr.setAttribute ("comment", m_comment) ;
	elem .appendChild  (instr) ;

	for (uint idx = 0 ; idx < m_args.count() ; idx += 1)
	{
		QDomElement arg  = elem.ownerDocument().createElement ("arg") ;
		QDomText    text = elem.ownerDocument().createTextNode(m_args[idx]) ;

		arg  .appendChild (text) ;
		instr.appendChild (arg ) ;
	}
}

void	KBMacroInstr::save
	(	QString		&text,
		int		indent
	)
{
	text	+= QString("%1<instruction action=\"%2\" comment=\"%3\">\n")
			  .arg ("", indent)
			  .arg (m_action  )
			  .arg (KBAttr::escapeText(m_comment)) ;

	for (uint idx = 0 ; idx < m_args.count() ; idx += 1)
	{
		text	+= QString("%1<arg>%2</arg>\n")
				  .arg ("", indent + 2)
				  .arg (KBAttr::escapeText(m_args[idx])) ;
	}

	text	+= QString("%1</instruction>\n")
			  .arg ("", indent) ;
}



/*  ------------------------------------------------------------------  */

/*  KBMacroExec								*/
/*  KBMacroExec	: Constructor for macro execution object		*/
/*  dbInfo	: KBDBInfo *	  : Database information object		*/
/*  server	: const QString & : Server name				*/
/*  (returns)	: KBMacroExec	  :					*/

KBMacroExec::KBMacroExec
	(	KBDBInfo	*dbInfo,
		const QString	&server
	)
	:
	m_dbInfo 	(dbInfo),
	m_server 	(server)
{
	m_instructions.setAutoDelete (true) ;
	m_debug	   = KBOptions::getMacroDebug() == KBOptions::MacroDebugAlwaysOn ;
	m_debugDlg = 0	;
	m_invoker  = 0  ;
}

/*  KBMacroExec								*/
/*  KBMacroExec	: Constructor for macro execution object		*/
/*  extant	: KBMacroExec *	: Extant macro executor			*/
/*  (returns)	: KBMacroExec	:					*/

KBMacroExec::KBMacroExec
	(	KBMacroExec	*extant
	)
	:
	m_dbInfo	(extant->m_dbInfo),
	m_server	(extant->m_server)
{
	KBError	dummy	;

	m_instructions.setAutoDelete (true) ;
	m_debug	   = KBOptions::getMacroDebug() == KBOptions::MacroDebugAlwaysOn ;
	m_debugDlg = 0	;
	m_invoker  = 0  ;

	LITER
	(	KBMacroInstr,
		extant->m_instructions,
		instr,

		append
		(	instr->action (),
			instr->args   (),
			instr->comment(),
			dummy
		)	;
	)
}

/*  KBMacroExec								*/
/*  ~KBMacroExec: Destructor for macro execution object			*/
/*  (returns)	: void		:					*/

KBMacroExec::~KBMacroExec ()
{
	if (m_debugDlg != 0) delete m_debugDlg ;
}

/*  KBMacroExec								*/
/*  load	: Load macro definition from DOM document		*/
/*  elem	: QDomElement &	: Definition				*/
/*  (returns)	: bool		: Success				*/

bool	KBMacroExec::load
	(	QDomElement	&elem,
		KBError		&pError
	)
{
	DOMITER_BEGIN(elem,"instruction",instr)
	{
		/* Get the macro action and use it to locate the	*/
		/* constructor function for the action, Grumble and	*/
		/* give up if it cannot be found.			*/
		QString	action	  = instr.attribute("action") ;
		MKMACRO	**mkMacro = macroDict.find ( action ) ;

		if (mkMacro == 0)
		{
			pError	= KBError
				  (	KBError::Error,
					TR("Unrecognised macro action"),
					QString(TR("Action: %1")).arg(action),
					__ERRLOCN
				  )	;
			return	false	;
		}

		/* Create and initialise the new macro instruction,	*/
		/* and append to the list unless there is an error.	*/
		KBMacroInstr *i = (*mkMacro) (this) ;

		if (!i->init (instr, pError))
		{	delete	i	;
			return	false	;
		}
 
		m_instructions.append	(i) ;
	}
	DOMITER_END

	return	true ;
}

/*  KBMacroExec								*/
/*  append	: Append an instruction					*/
/*  action	: const QString &	: Macro action			*/
/*  args	: const QStringList &	: Macro arguments		*/
/*  comment	: const QString &	: Macro comment			*/
/*  pError	: KBError &		: Error return			*/
/*  (returns)	: bool			: Success			*/

bool	KBMacroExec::append
	(	const QString		&action,
		const QStringList	&args,
		const QString		&comment,
		KBError			&pError
	)	
{
	/* Get the macro action and use it to locate the constructor	*/
	/* function for the action, Grumble and give up if it cannot	*/
	/* be found.							*/
	MKMACRO	**mkMacro = macroDict.find (action) ;

	if (mkMacro == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Unrecognised macro action"),
				QString(TR("Action: %1")).arg(action),
				__ERRLOCN
			  )	;
		return	false	;
	}

	/* Create and initialise the new macro instruction, and append	*/
	/* to the list unless there is an error.			*/
	KBMacroInstr *instr = (*mkMacro) (this) ;

	if (!instr->init (args, comment, pError))
	{	delete	instr	;
		return	false	;
	}
 
	m_instructions.append	(instr) ;
	return	true	;
}

/*  KBMacroExec								*/
/*  save	: Save macro definition to DOM document			*/
/*  elem	: QDomElement &	: Definition				*/
/*  (returns)	: void		:					*/

void	KBMacroExec::save
	(	QDomElement	&elem
	)
{
	LITER
	(
		KBMacroInstr,
		m_instructions,
		inst,

		inst->save (elem) ;
	)
}

/*  KBMacroExec								*/
/*  save	: Save macro definition as XML text			*/
/*  text	: QString &	: String into which to save		*/
/*  indent	: int		: Node indentation			*/
/*  (returns)	: void		:					*/

void	KBMacroExec::save
	(	QString		&text,
		int		indent
	)
{
	LITER
	(
		KBMacroInstr,
		m_instructions,
		inst,

		inst->save (text, indent) ;
	)
}

/*  KBMacroExec								*/
/*  showDebug	: Show instruction for debugging			*/
/*  instr	: KBMacroInstr * : Current instruction			*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: bool		 : Continue execution			*/

bool	KBMacroExec::showDebug
	(	KBMacroInstr	*instr,
		KBError		&pError
	)
{
#if	! __KB_RUNTIME
	if (m_debugDlg == 0)
		m_debugDlg = new KBMacroDebugDlg(m_instructions, m_invoker) ;

	if (!m_debugDlg->exec (instr, m_nodes))
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("User aborted macro"),
			  	QString::null,
			  	__ERRLOCN
			  )	;
		return	false	;
	}
#endif
	return	true	;
}

/*  KBMacroExec								*/
/*  execute	: Execute the macro					*/
/*  invoker	: KBNode *	: Invoking object			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBMacroExec::execute
	(	KBNode		*invoker,
		KBError		&pError
	)
{
	bool	rc	= true ;

	/* Set the executing flag. This will be cleared if a macro	*/
	/* instruction wants to terminate execution successfully. Also	*/
	/* note who invoked us.						*/
	m_executing = true 	;
	m_invoker   = invoker	;

	LITER
	(
		KBMacroInstr,
		m_instructions,
		instr,

		if (m_debug)
			if (!showDebug (instr, pError))
			{	rc	= false	;
				break	;
			}

		/* Execute the instruction and return failure if the	*/
		/* instruction fails, otherwise check in case the	*/
		/* instruction is successfully terminating execution.	*/
		if (!instr->execute (pError))
		{	rc	= false	;
			break	;
		}

		if (!m_executing)
		{	rc	= true	;
			break	;
		}
	)

	m_invoker = 0 ;
	return	true  ;
}

/*  KBMacroExec								*/
/*  addNode	: Add object to object map				*/
/*  type	: cchar  *	: Object type				*/
/*  node	: KBNode *	: Object in question			*/
/*  (returns)	: void		:					*/

void	KBMacroExec::addNode
	(	cchar		*type,
		KBNode		*node
	)
{
	/* The object map provides a general mechism whereby a macro	*/
	/* such as "openForm" can save the last opened form, such that	*/
	/* it can later be retrieved. We connect to its "destroyed()"	*/
	/* so that we can remove dangling pointers.			*/
	m_nodes.insert (type, node) ;
	connect	(node, SIGNAL(destroyed()), SLOT(slotNodeGone())) ;
}

/*  KBMacroExec								*/
/*  addValue	: Add value						*/
/*  tag		: cchar  *	  : Value tag				*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBMacroExec::addValue
	(	cchar		*tag,
		const QString	&value
	)
{
	m_values.insert (tag, value) ;
}

/*  KBMacroExec								*/
/*  slotNodeGone: Handle disappearence of object			*/
/*  (returns)	: void		:					*/

void	KBMacroExec::slotNodeGone ()
{
	fprintf
	(	stderr,
		"KBMacroExec::slotNodeGone: called [%p]\n",
		(void *)sender()
	)	;

        for (NodeMap::Iterator it = m_nodes.begin() ; it != m_nodes.end() ; ++it)
		if (it.data() == (KBNode *)sender())
		{
			m_nodes.remove (it) ;
			return	;
		}

	fprintf
	(	stderr,
		"KBMacroExec::slotNodeGone: not found!\n"
	)	;
}

/*  KBMacroExec								*/
/*  execute	: Static execution method				*/
/*  location	: KBLocation &	: Macro location			*/
/*  pError	: KBError &	: Error return				*/
/*  invoker	: KBNode *	: Invoking object			*/
/*  (returns	: bool		: Success				*/

bool	KBMacroExec::execute
	(	KBLocation	&location,
		KBError		&pError,
		KBNode		*invoker
	)
{
	QString	    text = location.contents (pError) ;
	if (text.isNull()) return false ;

	QDomDocument	doc	;
	doc.setContent	(text)	;

	QDomElement	root	= doc.documentElement() ;
	if (root.isNull())
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Macro definition has no root element"),
			  	location.title(),
			  	__ERRLOCN
			  )	;
		return	false	;
	}

	KBMacroExec exec (location.dbInfo, location.docLocn) ;
	if (!exec.load (root, pError)) return false ;

	return	exec.execute
		(	invoker == 0 ? 0 : invoker->getRoot(),
			pError
		)	;
}

/*  KBMacroExec								*/
/*  setDebug	: Set debugging on or off				*/
/*  debug	: bool		: On or off				*/
/*  (returns)	: void		:					*/

void	KBMacroExec::setDebug
	(	bool	debug
	)
{
	switch (KBOptions::getMacroDebug())
	{
		case KBOptions::MacroDebugOnDemand  :
			/* For on-demand debugging, this call turns	*/
			/* debug on or off as requested.		*/
			m_debug	= debug ;
			break	;

		case KBOptions::MacroDebugAlwaysOn  :
			/* In the case of always-on debugging, debug is	*/
			/* enabled whatever and the argument is ignored	*/
			m_debug	= true	;
			break	;

		case KBOptions::MacroDebugAlwaysOff :
			/* If debugging is always off then the argument	*/
			/* is ignored.					*/
			m_debug	= false	;
			break	;

		default	:
			break	;
	}
}

/*  KBMacroExec								*/
/*  getNode	: Locate top node for specified object			*/
/*  name	: const QString & : Object name				*/
/*  type	: cchar *	  : Object type				*/
/*  (returns)	: KBNode	  : Node or null if not found		*/

KBNode	*KBMacroExec::getNode
	(	const QString 	&name,
		cchar 		*type
	)
{
	/* If the name is empty then get the node for the last opened	*/
	/* object of the specified type, if any.			*/
	if (name.isEmpty())
		return	m_nodes[type] ;

	/* The name "[Invoker]" means to use the object which invoked	*/
	/* this macro, if any.						*/
	if (name == "[Invoker]")
		return	m_invoker  ;

	/* OK, need to look for the object and return its top node,	*/
	/* using the specified type and name.				*/
	KBLocation location 
	(	getDBInfo (),
		type,
		getServer (),
		name
	)	;

	return	KBAppPtr::getCallback()->objectNode (location) ;
}

/*  KBMacroExec								*/
/*  getValue	: Get a stored value					*/
/*  tag		: cchar *	: Value tag				*/
/*  (returns)	: QString	: Value					*/

QString	KBMacroExec::getValue
	(	cchar 		*tag
	)
{
	return	m_values[tag] ;
}


/*  ------------------------------------------------------------------  */

/*  KBMacroReg								*/
/*  KBMacroReg	: Constructor for macro registration object		*/
/*  name	: cchar *	: Macro name				*/
/*  mkMacro	: MKMACRO *	: Macro construction object		*/
/*  (returns)	: KBMacroReg	:					*/

KBMacroReg::KBMacroReg
	(	cchar		*name,
		MKMACRO		*mkMacro
	)
{
	/* Nasty hack, a QDict cannot hold pointers-to-functions, since	*/
	/* the template generates a delete, which is not allowed. So,	*/
	/* add a level of indirection.					*/
	MKMACRO	**redir	= new MKMACRO *	;
	*redir	= mkMacro ;
	macroDict .insert (name, redir)  ;

	fprintf
	(	stderr,
		"KBMacroReg::KBMacroReg: registered [%s]\n",
		name
	)	;
}

/*  ------------------------------------------------------------------  */


/*  KBMacroDebugEnable							*/
/*  KBMacroDebugEnable							*/
/*		: Constructor for message box macro			*/
/*  exec	: KBMacroExec *		: Parent macro executor		*/
/*  (returns)	: KBMacroMessageBox 	:				*/

KBMacroDebugEnable::KBMacroDebugEnable
	(	KBMacroExec	*exec
	)
	:
	KBMacroInstr (exec, "DebugEnable")
{
}

/*  KBMacroDebugEnable							*/
/*  init	: Initialise macro instruction				*/
/*  args	: const QStringList &	: Macro arguments		*/
/*  comment	: const QString &	: Macro comment			*/
/*  pError	: KBError &		: Error return			*/
/*  (returns)	: bool			: Success			*/

bool	KBMacroDebugEnable::init
	(	const QStringList	&args,
		const QString		&comment,
		KBError			&pError
	)
{
	return	KBMacroInstr::init (args, comment, 1, 1, pError) ;
}

/*  KBMacroDebugEnable							*/
/*  execute	: Execute macro						*/
/*  pError	: KBError &		: Error return			*/
/*  (returns)	: bool			: Success			*/

bool	KBMacroDebugEnable::execute
	(	KBError		&
	)
{
	m_exec->setDebug (m_args[0] == "On") ;
	return	true	;
}


NEWMACRO(DebugEnable)
