/***************************************************************************
    file	         : kb_event.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qregexp.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_database.h"
#include	"kb_location.h"
#include	"kb_attr.h"
#include	"kb_script.h"
#include	"kb_object.h"
#include	"kb_event.h"
#include	"kb_callback.h"
#include	"kb_location.h"
#include	"kb_callback.h"
#include	"kb_appptr.h"
#include	"kb_docroot.h"
#include	"kb_eventdlg.h"

#define		EFLAGS	(KAF_GRPEVENT|KAF_EDITOR|KAF_CLEAR|KAF_CUSTOM)

/*  KBEvent								*/
/*  KBEvent	: Constructor for event class				*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  base	: cchar *	: Event base				*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBEvent	:					*/

KBEvent::KBEvent
	(	KBNode			*node,
		cchar			*name,
		cchar			*base,
		const QDict<QString>	&aList,
		uint			nFlags
	)
	:
	KBAttrStr (node, name, aList, nFlags | EFLAGS),
	m_parent  (node),
	m_base	  (base)
{
	init	  () ;

	QString	*bpt = aList.find (QString("%1_bpt").arg(name)) ;
	if (bpt != 0)
	{	QStringList l = QStringList::split (',', *bpt) ;
		for (uint idx = 0 ; idx < l.count() ; idx += 1)
			m_breakpoints.append (l[idx].toInt()) ;
	}
}

/*  KBEvent								*/
/*  KBEvent	: Constructor for event class				*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  base	: cchar *	: Event base				*/
/*  text	: cchar *	: Initial text value			*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBEvent	:					*/

KBEvent::KBEvent
	(	KBNode	*node,
		cchar	*name,
		cchar	*base,
		cchar	*text,
		uint	nFlags
	)
	:
	KBAttrStr (node, name, text, nFlags | EFLAGS),
	m_parent  (node),
	m_base	  (base)
{
	init	  () ;
}

/*  KBEvent								*/
/*  KBEvent	: Constructor for event class				*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  base	: cchar *	: Event base				*/
/*  srce	: KBNode *	: Extant source				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBEvent	:					*/

KBEvent::KBEvent
	(	KBNode	*node,
		cchar	*name,
		cchar	*base,
		KBNode	*srce,
		uint	nFlags
	)
	:
	KBAttrStr (node, name, srce, nFlags | EFLAGS),
	m_parent  (node),
	m_base	  (base)
{
	init	  () ;

	/* See if we have an extant event and if such an event has a	*/
	/* macro, in which case replicate the macro.			*/
	KBAttr	*attr	= srce->getAttr(name) ;
	if (attr  == 0) return ;

	KBEvent	*event	= attr->isEvent() ;
	if (event == 0) return ;

	KBMacroExec *macro = event->getMacro() ;
	if (macro == 0) return ;

	m_macro	= new KBMacroExec(macro) ;
}

/*  KBEvent								*/
/*  ~KBEvent	: Destructor for event class				*/
/*  (returns)	:		:					*/

KBEvent::~KBEvent ()
{
	clearOverride () ;

	if (m_code    != 0) delete m_code    ;
	if (m_emitter != 0) delete m_emitter ;
	if (m_macro   != 0) delete m_macro   ;
}

void	KBEvent::init ()
{
	m_disable	= false	;
	m_code		= 0	;
	m_emitter	= 0	;
	m_cppFunc	= 0	;
	m_macro		= 0	;
	m_override	= 0	;
	m_inherit	= 0	;
}

/*  KBEvent								*/
/*  clearOverride:							*/
/*  (returns)	 : void		  :					*/

void	KBEvent::clearOverride ()
{
	/* Clear the override/inherit chain. Note that such a chain	*/
	/* ends up back at this event so careful with the loop		*/
	/* termiantion.							*/
	/* NOTE: The m_override pointer is only set in the original	*/
	/*	 event so we don't need to worry about enexpected	*/
	/*	 recursion.						*/
	KBEvent	*over	;
	KBEvent	*next	;

	if ((over = m_override) != 0)
		while ((over != 0) && (over != this))
		{	next	= over->m_inherit ;
			delete	over	;
			over	= next	;
		}

	m_override = 0 ;
	m_inherit  = 0 ;
}

/*  KBEvent								*/
/*  showAs	: Set display mode					*/
/*  showAs	: KB::ShowAs	: New mode				*/
/*  (returns)	: bool		: Value has been changed		*/

bool	KBEvent::showAs
	(	KB::ShowAs	showAs
	)
{
	/* Always clear the override/inherit chain. This only matters	*/
	/* if the user goes data->design->data but just to it on all	*/
	/* changes.							*/
	if (showAs != m_showing) clearOverride () ;

	return	KBAttrStr::showAs (showAs) ;
}

/*  KBEvent								*/
/*  setOverride	: Set override event function				*/
/*  over	: const QString & : Override function			*/
/*  (returns)	: void		  :					*/

void	KBEvent::setOverride
	(	const QString	&over
	)
{
	/* Create a new event attribute object, set the override and	*/
	/* link into this event. The new evet is chained such that	*/
	/* if goes to the head of a list which will chain back to this	*/
	/* event (which is effectively the base method).		*/
	static QDict<QString> dummy ;
	KBEvent	*oe = new KBEvent
			  (	m_parent,
				name,
				m_base,
				dummy,
				nFlags|EFLAGS
			  )	;

	oe->setValue  (over)	;
	oe->m_inherit = m_override == 0 ? this : m_override ;
	m_override    = oe	;
}

/*  KBEvent								*/
/*  setValue	: Set event text value					*/
/*  value	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBEvent::setValue
	(	const QString	&value
	)
{
	/* This method is overridden so that we can clear any code	*/
	/* compiled for an earlier value, and clear the disabled flag	*/
	DELOBJ	(m_code)  ;
	m_disable = false ;

	KBAttrStr::setValue (value) ;
}


void	KBEvent::setValue
	(	KBAttrItem	*item
	)
{
#if	! __KB_RUNTIME
	setValue (item->value()) ;
#endif
}


void	KBEvent::setCPPFunc
	(	CPPFunc		cppFunc
	)
{
	m_disable = false 	;
	m_cppFunc = cppFunc	;
}

/*  KBEvent								*/
/*  doExecute	: Execute event						*/
/*  resval	: KBValue &	: Return value 				*/
/*  argc	: uint		: Argument count			*/
/*  argv	: KBValue *	: Argument vector			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: KB::ScriptRC	: Execution return code			*/

KB::ScriptRC
	KBEvent::doExecute
	(	KBValue		&resval,
		uint		argc,
		KBValue		*argv,
		KBError		&pError
	)
{
	QString		event	  = getValue () ;

//	fprintf
//	(	stderr,
//		"KBEvent::doExecute: [%s] [%s]\n",
//		(cchar *)name,
//		(cchar *)event
//	)	;
//	for (uint idx = 0 ; idx < argc ; idx += 1)
//		fprintf
//		(	stderr,
//			"..... %d: %s\n",
//			idx,
//			(cchar *)argv[idx].getRawText()
//		)	;


	/* SIGNAL CODE							*/
	/* -----------							*/
	if (m_emitter != 0)
		if (m_emitter->doSignal (argc, argv) != KB::ScriptOK)
			return	KB::ScriptOK ;


	/* MACRO							*/
	/* -----							*/
	if (m_macro != 0)
	{
		if (!m_macro->execute (m_parent->getRoot(), pError))
		{
			pError.DISPLAY() ;
			m_parent->getDocRoot()->doExecError() ;
			return	KB::ScriptInlineError ;
		}

		return	KB::ScriptOK ;
	}

	/* EVENT CODE							*/
	/* ----------							*/
	/* C++-level function. Used internally, should really never	*/
	/* fail!							*/
	if (m_cppFunc != 0)
	{
		KBScript::ExeRC rc = (*m_cppFunc) (this, resval, argc, argv, pError) ;

		if ((rc == KBScript::ExeError) || (rc == KBScript::ExeFail))
		{
			/* Ho-hum, got an error which executing the	*/
			/* func. Report the error, switch to design	*/
			/* mode, and maybe try to edit the script ....	*/
			pError.DISPLAY() ;
			m_parent->getDocRoot()->doExecError() ;
			return	KB::ScriptInlineError ;
		}

		return	KB::ScriptOK ;
	}

	/* Next case, there is no text associated with the event, in	*/
	/* which case do nothing. This is considered successful.	*/
	if (event.isEmpty())
	{	resval.setTrue ()    ;
		return	KB::ScriptOK ;
	}

	/* Set the result value false until proven otherwise. This is	*/
	/* the catch-all case.						*/
	resval.setFalse ()   ;


	/* A script interface is needed to be able to execute the event	*/
	/* whether the code is implicit or explicit ....		*/
	bool		ok	  ;
	KBDocRoot	*root	  = m_parent->getDocRoot  () ;
	KBScriptIF	*scrIface = root  ->loadScripting (ok, pError) ;
	KBScript::ExeRC rc 	  ;

	if (!ok)
	{
		pError.DISPLAY () ;
		return	KB::ScriptInlineError ;
	}
	if (scrIface == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("No scripting language specified"),
				QString(TR("Trying to execute event %1")).arg(getLegend()),
				__ERRLOCN
			  )	;
		pError.DISPLAY();

		return	KB::ScriptInlineError ;
	}

	/* If the event is flagged as disabled then there has been an	*/
	/* earlier error ...						*/
	if (m_disable)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Event has been disabled due to earlier error"),
				QString::null,
				__ERRLOCN
			  )	;
		pError.DISPLAY();

		m_parent->getDocRoot()->doExecError () ;
		return	KB::ScriptInlineError ;
	}

	/* If the event is marked with #[A-Za-z]* then this is an	*/
	/* implicit event, and we execute a routine named for the	*/
	/* event base (eg., "onClick" for a button)  and the reset of	*/
	/* the event text.						*/
	if ((event.at(0) == '#') && event.at(1).isLetter())
	{
		QString	name	= m_base + event.mid(1).stripWhiteSpace() ;

		rc	= scrIface->execute
			  (	m_parent->getDocRoot()->getScripts(),
				name,
				m_parent,
				argc,
				argv,
				resval
			  )	;

		if ((rc == KBScript::ExeError) || (rc == KBScript::ExeFail))
		{
			/* Ho-hum, got an error which executing the	*/
			/* code. Report the error, switch to design	*/
			/* mode, and maybe try to edit the script ....	*/
			QString	   errMsg   ;
			uint	   errLno   ;
			QString	   errText  ;
			KBLocation errLocn  = scrIface->exeError (errMsg, errLno, errText) ;

			pError = KBError
				 (	KBError::Error,
					errMsg,
					QString (TR("%1, line %2\n%3"))
						.arg(errLocn.docName)
						.arg(errLno)
						.arg(errText),
					__ERRLOCN
				  )	  ;
			pError.DISPLAY () ;

			m_parent->getDocRoot()->doExecError() ;

			/* Fire up the script editor unless a debugger	*/
			/* is handling it.				*/
			if (rc != KBScript::ExeFail)
			{	KBError se ;
				if (!KBAppPtr::getCallback()->editScript (errLocn, errText, "", errLno, se))
					se.DISPLAY () ;
			}

			return	KB::ScriptInlineError ;
		}

		return	KB::ScriptOK ;
	}

	/* Now on to explicit events, where we compile the code and	*/
	/* keep a copy. First see if the code has already been compiled	*/
	/* and if not then so so now.					*/
	if (m_code == 0)
	{
		QString		eText	;
		QString		ePatt	;

		if ((m_code = scrIface->compileFunc
			    (	event,
				m_parent->isObject  ()->getPath   (),
				"eventFunc",
				eText,
				ePatt,
				m_parent->getDocRoot()->getImports(),
				m_inherit,
				pError
			    )	) == 0)
		{
			/* Compile error in the inline code. Display	*/
			/* the error message, and return the form or	*/
			/* report to design mode.				*/
			pError.DISPLAY  () ;
			m_parent->getDocRoot()->doExecError () ;

			/* The event is disabled, and we return false	*/
			/* so that the caller will bring up the		*/
			/* required property dialog.			*/
			m_disable = true ;
			return	KB::ScriptInlineError ;
		}

		m_code->setBreakpoints (m_breakpoints) ;
	}

	/* Code was or has been compiled successfully, in which case	*/
	/* execute it. Check for an execution time error ....		*/
	rc = m_code->execute (m_parent, argc, argv, resval) ;

	if ((rc == KBScript::ExeError) || (rc == KBScript::ExeFail))
	{
		QString	   errMsg   ;
		uint	   errLno   ;
		QString	   errText  ;
		KBLocation errLocn  = scrIface->exeError (errMsg, errLno, errText) ;

		pError	= KBError
			  (	KBError::Error,
				errMsg,
				QString (TR("%1, line %2\n%3"))
					.arg(errLocn.docName)
					.arg(errLno)
					.arg(errText),
				__ERRLOCN
			  )	;
		pError.DISPLAY();

		m_parent->getDocRoot()->doExecError () ;

		if (errLocn.docName != KBLocation::m_pInline)
		{
			if (rc != KBScript::ExeFail)
			{	KBError se ;
				if (!KBAppPtr::getCallback()->editScript (errLocn, errText, "", errLno, se))
					se.DISPLAY () ;
			}

			return	KB::ScriptGlobalError ;
		}

		m_disable = true	;
		return	KB::ScriptInlineError ;
	}

	/* Return true; success here means that there were no execution	*/
	/* errors, not that the event script itself returned true.	*/
	return	KB::ScriptOK ;
}

/*  KBEvent								*/
/*  execute	: Execute event						*/
/*  resval	: KBValue &	: Return value 				*/
/*  argc	: uint		: Argument count			*/
/*  argv	: KBValue *	: Argument vector			*/
/*  (returns)	: KB::ScriptRC	: Execution return code			*/

KB::ScriptRC
	KBEvent::execute
	(	KBValue		&resval,
		uint		argc,
		KBValue		*argv
	)
{
//	fprintf
//	(	stderr,
//		"KBEvent::execute: [%s][%p][%p]\n",
//		(cchar *)name,
//		(void  *)m_override,
//		(void  *)m_inherit
//	)	;
//	for (uint idx = 0 ; idx < argc ; idx += 1)
//		fprintf
//		(	stderr,
//			".... %d: %s\n",
//			idx,
//			(cchar *)argv[idx].getRawText()
//		)	;

	KBCallback *cb  = KBAppPtr::getCallback() ;
	void	*logid	= cb->logEvent
			  (	"Event",
				ownerName,
				getOwner()->getAttrVal("name"),
				name,
				argc,
				argv
			  )	;

	/* If the override pointer is set then there is a chain of	*/
	/* overriding (and inheriting) events starting here and leading	*/
	/* back to this event, so start following the chain.		*/
	KBError		error	;
	KB::ScriptRC	rc	;

	if (m_override != 0)
		rc	= m_override->doExecute (resval, argc, argv, error) ;
	else	rc	= doExecute		(resval, argc, argv, error) ;

	if (rc == KB::ScriptOK)
		cb->logEventResult (resval,		         true, logid) ;
	else	cb->logEventResult (KBValue(error.getMessage()), true, logid) ;

//	fprintf
//	(	stderr,
//		"KBEvent::execute: returns [%d]\n",
//		(int)rc
//	)	;
	return	rc	;
}

/*  KBEvent								*/
/*  getDescription							*/
/*		: Get description for property dialogs			*/
/*  (returns)	: QString	: Legend				*/


QString	KBEvent::getDescription ()
{
	QString	descr = KBAttr::getDescription () ;
	if (descr.find ('%') >= 0) descr = descr.arg(m_base) ;
	return	descr ;
}

/*  KBEvent								*/
/*  getEmitter	: Get emitter object					*/
/*  (returns)	: KBEmitter *	: Emitter				*/

KBEmitter
	*KBEvent::getEmitter ()
{
	/* The emitter object is created as needed, so that we do not	*/
	/* waste space by needing KBEvent to embed or be derived from	*/
	/* QObject.							*/
	if (m_emitter == 0)
		if (owner->isObject() != 0)
			m_emitter = new KBEmitter (owner->isObject(), this) ;

	return	m_emitter	;
}

/*  KBEvent								*/
/*  clearEmitter: Clear emitter object if any				*/
/*  (returns)	: void		:					*/

void	KBEvent::clearEmitter ()
{
	/* Having a dynamically allocated emitter also gives an easy	*/
	/* way to clear down all signals ...				*/
	DELOBJ	(m_emitter) ;
}


/*  KBEvent								*/
/*  isEvent	: Checkes whether attribute is an event			*/
/*  (returns)	: KBEvent *	: Always the object			*/

KBEvent	*KBEvent::isEvent ()
{
	return	this ;
}

/*  KBSlot								*/
/*  setEvent	: Set code associated with event				*/
/*  code	: const QString & : Code				*/
/*  append	: bool		  : Append code else replace		*/
/*  (returns)	: void		  :					*/

void	KBEvent::setCode
	(	const QString	&code,
		bool		append
	)
{
	if (append)
	{
		setValue (getValue() + code) ;
		return	;
	}

	setValue (code) ;
}

/*  KBEvent								*/
/*  tidy	: Tidy up after loading					*/
/*  (returns)	: void		:					*/

void	KBEvent::tidy ()
{
	/* This is just so we can output the code with a leading 	*/
	/* newline in the XML document.					*/
	QString	v = getValue().stripWhiteSpace() ;
	if (!v.isEmpty()) v += "\n" ;
	setValue (v) ;
}

/*  KBEvent								*/
/*  setMacro	: Set a macro						*/
/*  macro	: KBMacro *	: Macro in question			*/
/*  (returns)	: void		:					*/

void	KBEvent::setMacro
	(	KBMacroExec	*macro
	)
{
	if (m_macro != 0) delete m_macro ;
	m_macro	= macro	;
}

/*  KBEvent								*/
/*  getMacro	: Get macro if any					*/
/*  (returns)	: KBMacro *	: Macro or null of none			*/

KBMacroExec *KBEvent::getMacro ()
{
	return	m_macro	;
}

/*  KBEvent								*/
/*  printAttr	: Print attribute					*/
/*  attrText	: QString &	: Result text for attribute output	*/
/*  nodeText	: QString &	: Result text for node output		*/
/*  indent	: int		: Indent depth for node output		*/
/*  (returns)	: void		:					*/

void	KBEvent::printAttr
	(	QString		&attrText,
		QString		&nodeText,
		int		indent
	)

{
	if ((getFlags() & (KAF_HIDDEN|KAF_USER)) != 0)
		return	;

	if (m_macro != 0)
	{
		nodeText += QString("%1<macro name=\"%2\">\n")
				   .arg("", indent)
				   .arg(name) ;

		m_macro->save (nodeText, indent + 2) ;

		nodeText += QString("%1</macro>\n")
				   .arg("", indent) ;
		return	;
	}

	KBAttrStr::printAttr (attrText, nodeText, indent) ;

	if (m_breakpoints.count() != 0)
	{
		QString	list	;
		for (uint idx = 0 ; idx < m_breakpoints.count(); idx += 1)
		{	if (idx > 0) list += ',' ;
			list += QString::number(m_breakpoints[idx]) ;
		}

		attrText += QString (" %1_bpt=\"%2\"").arg(name).arg(list) ;
	}

#if	0
	if (!value.isEmpty())
	{
		nodeText += QString("%1<event name=\"%2\">\n%3%4</event>\n")
				   .arg("", indent)
				   .arg(name)
				   .arg(escapeText(value, false))
				   .arg("", indent) ;
		return	;
	}
#endif
}

void	KBEvent::setBreakpoints
	(	QValueList<int>	&breakpoints
	)
{
	m_breakpoints = breakpoints ;

	for (uint idx = 0 ; idx < m_breakpoints.count() ; idx += 1)
		fprintf
		(	stderr,
			"KBEvent::setBreakpoints: breakpoint %d\n",
			m_breakpoints[idx]
		)	;
}

const QValueList<int>
	&KBEvent::breakpoints ()
{
	for (uint idx = 0 ; idx < m_breakpoints.count() ; idx += 1)
		fprintf
		(	stderr,
			"KBEvent::breakpoints: breakpoint %d\n",
			m_breakpoints[idx]
		)	;

	return	m_breakpoints ;
}

#if	! __KB_RUNTIME

/*  KBEvent	:							*/
/*  trimEvent	: Trim event text					*/
/*  event	: QString	: User text				*/
/*  (returns)	: QString	: Trimmed text				*/

QString KBEvent::trimEvent
	(	QString		event
	)
{
	int	offset	;

	/* If this is a shortcut then strip leading whitespace and	*/
	/* anything from the first non-leading whitespace onwards.	*/
	/* This leaves just the shortcut itself.			*/
	if ((offset = QRegExp("^\\s*#[A-Za-z]").match (event)) >= 0)
	{
		event	= event.stripWhiteSpace() ;

		if ((offset = QRegExp("\\s").match (event)) >= 0)
			event	= event.left (offset) ;

		return	event	;
	}

	/* Otherwise, it is script code. In this case, strip leading	*/
	/* and trailing	whitespace and add a newline *unless* the text	*/
	/* is empty.							*/
	QString	trimmed	= event.stripWhiteSpace() ;
	return	trimmed.isEmpty() ? QString::null : trimmed + "\n" ;
}

/*  KBEvent								*/
/*  getAttrItem	: Get item for property dialogs				*/
/*  (returns)	: KBAttrItem *	: Actually KBEventItem			*/

KBAttrItem
	*KBEvent::getAttrItem ()
{
	return	new KBAttrEventItem (this) ;
}

/*  KBEvent								*/
/*  getAttrDlg	: Get dialog component for frame			*/
/*  parent	: QWidget *		: Parent widget			*/
/*  item	: KBAttrItem *		: Associated item		*/
/*  attrDict	: QDict<KBAttrItem> &	: Attribute dictionary		*/
/*  (returns)	: KBAttrDlg *		: Actually KBAttrFrameDlg	*/

KBAttrDlg
	*KBEvent::getAttrDlg
	(	QWidget			*parent,
		KBAttrItem		*,
		QDict<KBAttrItem>	&attrDict
	)
{
	return	new KBEventDlg (parent, this, 0, attrDict) ;
}

#endif
