/*
    Copyright (C) 2001-2002 by theKompany.com <www.thekompany.com>
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    See COPYING.GPL file.

    In addition, as a special exception, theKompany.com gives permission
    to link the code of this program with the tkwidgets library (or with
    modified versions of tkwidgets that use the same license as tkwidgets),
    and distribute linked combinations including the two.  You must obey
    the GNU General Public License in all respects for all of the code used
    other than tkwidgets.  If you modify this file, you may extend this
    exception to your version of the file, but you are not obligated to do so.
    If you do not wish to do so, delete this exception statement from your
    version.

    This license grants you the ability to use tkwidgets with Rekall only
    and may not be used outside of Rekall.
    See also http://www.rekall.a-i-s.co.uk/products/license.txt for details.
*/


#include	<qregexp.h>

#ifndef	_WIN32
#include "tktextview.moc"
#else
#include "tktextview.h"
#endif

#include "tktextmanager.h"
#include "tktextdoc.h"
#include "tktexteditor.h"
#include "tktextline.h"
#include "tktextaction.h"

#include <qapplication.h>
#include <qclipboard.h>
#include <qlistbox.h>
#include <qpainter.h>
#include <qkeycode.h>
#include <qlayout.h>
#include <qbitmap.h>
#include <qcursor.h>
#include <qdragobject.h>

#include "cursorbits.h"

void LineRegion::expand(int l1, int l2)
{
  start = QMIN(start == -1 ? l1 : start, l1);
  if (l2 == -1)
    l2 = l1;
  end = QMAX(end == -1 ? l2 : end, l2);
}

void LineRegion::expand(const LineRegion &lr)
{
  if (!lr.isEmpty())
    expand(lr.start, lr.end);
}

void LineRegion::intersect(int l1, int l2)
{
  start = QMAX(start, l1);
  end = QMIN(end, l2);
  if (end < start)
    clear();
}
/*****************************************************/
QCursor TKTextView::mouseCursor(int id)
{
  QBitmap normal(128, 96, cursor_bits, true);
  QBitmap mask(128, 96, cursor_mask_bits, true);

  QBitmap cnormal(32, 32);
  QBitmap cmask(32, 32);

  int x = 0;
  int y = 0;
  int xp = 0;
  int yp = 0;

  switch (id) {
    case 0:
      x = 0;
      y = 0;
      xp = 7;
      yp = 0;
      break;
    case 1:
      x = 32;
      y = 0;
      xp = 7;
      yp = 13;
      break;
    case 2:
      x = 64;
      y = 0;
      xp = 13;
      yp = 7;
      break;
    case 3:
      x = 96;
      y = 0;
      xp = 0;
      yp = 7;
      break;
    case 4:
      x = 0;
      y = 32;
      xp = 0;
      yp = 0;
      break;
    case 5:
      x = 32;
      y = 32;
      xp = 13;
      yp = 0;
      break;
    case 6:
      x = 64;
      y = 32;
      xp = 13;
      yp = 13;
      break;
    case 7:
      x = 96;
      y = 32;
      xp = 0;
      yp = 13;
      break;
    case 8:
      x = 0;
      y = 64;
      xp = 7;
      yp = 7;
      break;
    default:
      break;
  }

  bitBlt(&cnormal, 0, 0, &normal, x, y, 32, 32);
  bitBlt(&cmask, 0, 0, &mask, x, y, 32, 32);

  return QCursor(cnormal, cmask, xp, yp);
}
/****************************************************************************/
TKMargin::TKMargin(QWidget *parent, TKTextView *view)
: QWidget(parent, 0, Qt::WResizeNoErase | Qt::WRepaintNoErase),
  view(view), document(view->textDocument()), manager(view->textManager()), buffer(0)
{
  setBackgroundMode(NoBackground);
  setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding));
}

TKMargin::~TKMargin()
{
  delete buffer;
}

void TKMargin::resizeEvent(QResizeEvent *)
{
  resizeBuffer();
}

void TKMargin::resizeBuffer()
{
  delete buffer;
  buffer = new QPixmap(width(), document->lineHeight());
}
/****************************************************************************/
TKIndicatorMargin::TKIndicatorMargin(QWidget *parent, TKTextView *view)
: TKMargin(parent, view)
{
}

TKIndicatorMargin::~TKIndicatorMargin()
{
}

QSize TKIndicatorMargin::sizeHint() const
{
  return QSize(22, 0);
}

void TKIndicatorMargin::paintEvent(QPaintEvent *ev)
{
  QRect rect = ev->rect();
  int x = rect.x();
  int w = rect.width();
  int h = document->lineHeight();
  int line = document->lineNumber(view->contentsY() + rect.y());
  int y = document->linePosition(line) - view->contentsY();
  int end = rect.y() + rect.height();

  QPainter paint(buffer);
  paint.setBrushOrigin(0, -view->contentsY());
  paint.setPen(colorGroup().dark());
  paint.drawLine(width() - 1, 0, width() - 1, h);
  paint.setPen(colorGroup().light());
  paint.drawLine(width() - 2, 0, width() - 2, h);
  while (y < end) {
    paint.fillRect(0, 0, width() - 2, h, colorGroup().brush(QColorGroup::Background));
    if (line <= document->lastLine()) {
      TKTextLine *textLine = document->lineOf(line);
			
			int itemId;
			int classId;
			textLine->firstMargin(classId, itemId);
#ifdef KOBOL			
			if (manager->runLine()==line) {
				QPixmap pm(run_line_xpm);
        paint.drawPixmap((width() - 2 - pm.width()) / 2, (buffer->height() - pm.height()) / 2, pm);
			}
      
			if (textLine->isBreakpoint()) 
			{
				if (textLine->isBreakpointEnabled())
				{	
					QPixmap pm(breakpoint_xpm);
        	paint.drawPixmap((width() - 2 - pm.width()) / 2, (buffer->height() - pm.height()) / 2, pm);
				}
				else {
					QPixmap pm(breakpoint_disabled_xpm);
        	paint.drawPixmap((width() - 2 - pm.width()) / 2, (buffer->height() - pm.height()) / 2, pm);
				}
      }
#endif
    }
    bitBlt(this, x, y, buffer, x, 0, w, h);
    y += h;
    line++;
  }
}
/****************************************************************************/
TKSelectionMargin::TKSelectionMargin(QWidget *parent, TKTextView *view)
: TKMargin(parent, view), doSelect(false)
{
}

TKSelectionMargin::~TKSelectionMargin()
{
}

QSize TKSelectionMargin::sizeHint() const
{
  return QSize(16, 0);
}

void TKSelectionMargin::mousePressEvent(QMouseEvent *ev)
{
  if (ev->button() == LeftButton) {
    int line = document->lineNumber(view->contentsY() + ev->y());
    if (line > document->lastLine())
	return;

    int status = document->lineOf(line)->status();
    if (status & TKTextLine::Mater) {
      document->fold(view, line);
      return;
    }

    view->setCursorPosition(line, 0, false);
    if (line == document->lastLine())
      view->setCursorPosition(line, document->textLength(line), true);
    else
      view->setCursorPosition(line + 1, 0, true);

    doSelect = true;
    start = line;
    last = line;
  }
}

void TKSelectionMargin::mouseMoveEvent(QMouseEvent *ev)
{
  if (doSelect) {
    int line = document->lineNumber(view->contentsY() + ev->y());
    if (line > document->lastLine())
      return;

    if (last == line)
      return;

    if (upOrDown && line >= start) {
      upOrDown = false;
      view->setCursorPosition(start, 0, false);
    } else
      if (!upOrDown && line < start) {
        upOrDown = true;
        if (start == document->lastLine())
          view->setCursorPosition(start, document->textLength(start), false);
        else
          view->setCursorPosition(start + 1, 0, false);
      }

    if (!upOrDown) {
      if (line == document->lastLine())
        view->setCursorPosition(line, document->textLength(line), true);
      else
        view->setCursorPosition(line + 1, 0, true);
    } else {
      view->setCursorPosition(line, 0, true);
    }
    last = line;
  }
}

void TKSelectionMargin::mouseReleaseEvent(QMouseEvent *)
{
  doSelect = false;
}

void TKSelectionMargin::paintEvent(QPaintEvent *ev)
{
  QRect rect = ev->rect();
  int x = rect.x();
  int w = rect.width();
  int h = document->lineHeight();
  int line = document->lineNumber(view->contentsY() + rect.y());
  int y = document->linePosition(line) - view->contentsY();
  int end = rect.y() + rect.height();

  QPainter paint(buffer);
  paint.setPen(QColor(132, 132, 132));
  while (y < end) {
    int m = line < document->lastLine() ? 
                               document->lineOf(line)->marked() & 0x02 :
                               0 ;
    static QColor cm[] = { QColor(), QColor(0,0xff,0), QColor(0xff,0,0), QColor(0xff,0,0) } ;
    buffer->fill(m == 0 ? manager->normalBackgroundColor : cm[m]);
    if (line <= document->lastLine()) {
      int status = document->lineOf(line)->status();
      if (status & TKTextLine::Mater) {
        paint.drawRect(3, 3, 9, 9);
        paint.drawLine(5, 7, 9, 7);
        if (status & TKTextLine::Close)
          paint.drawLine(7, 5, 7, 9);
        else
          paint.drawLine(7, 14, 7, h);
      } else
        if (status & TKTextLine::Wrapped) {
           paint.drawLine(7, 0, 7, h);
        }
      else if (status & TKTextLine::Slave) {
          if (status & TKTextLine::Last) {
           paint.drawLine(7, 0, 7, h / 2);
           paint.drawLine(7, h / 2, width() - 3, h / 2);
          } else {
           paint.drawLine(7, 0, 7, h);
          }
        }
    }
    bitBlt(this, x, y, buffer, x, 0, w, h);
    y += h;
    line++;
  }
}
/****************************************************************************/
TKLinuNumberMargin::TKLinuNumberMargin(QWidget *parent, TKTextView *view)
: TKMargin(parent, view)
{
}

TKLinuNumberMargin::~TKLinuNumberMargin()
{
}

QSize TKLinuNumberMargin::sizeHint() const
{
#ifdef KOBOL
  return QSize(QFontMetrics(document->m->font()).width("000000"), 0);
#else
  return QSize(QFontMetrics(document->m->font()).width("10000"), 0);
#endif
}

void TKLinuNumberMargin::paintEvent(QPaintEvent *ev)
{
  QRect rect = ev->rect();
  int x = rect.x();
  int w = rect.width();
  int h = document->lineHeight();
  int line = document->lineNumber(view->contentsY() + rect.y());
  int y = document->linePosition(line) - view->contentsY();
  int end = rect.y() + rect.height();

  QPointArray p1((h + 1) / 2);
  QPointArray p2((h + 1) / 2);
  for (int p = 0; p < (h + 1) / 2; p++) {
    p1.setPoint(p, width() - 1, p * 2);
    p2.setPoint(p, width() - 1, p * 2 + 1);
  }

  QPainter paint(buffer);
  paint.setPen(QColor(0, 128, 128));
  paint.setFont( document->m->font() );

  while (y < end) {
    int m = line < document->lastLine() ? 
                               document->lineOf(line)->marked() & 0x01 :
                               0 ;
    static QColor cm[] = { QColor(), QColor(0,0xff,0), QColor(0xff,0,0), QColor(0xff,0,0) } ;
    buffer->fill(m == 0 ? manager->normalBackgroundColor : cm[m]);
    paint.drawPoints((y + view->contentsY()) % 2 == 0 ? p1 : p2);
    if (line <= document->lastLine())
#ifdef KOBOL
    {
      // int lnum = manager->fLinuNumberMarginZero ? document->lineOf(line)->number() : document->lineOf(line)->number()+1;
      int lnum = manager->fLinuNumberMarginZero ? document->lineNum(line) : document->lineNum(line) +1;
      paint.drawText(3, 0, width() - 6, h, AlignVCenter | AlignRight, QString().setNum(lnum));
    }
#else
      // paint.drawText(3, 0, width() - 6, h, AlignVCenter | AlignRight, QString().setNum(document->lineOf(line)->number() + 1));
      paint.drawText(3, 0, width() - 6, h, AlignVCenter | AlignRight, QString().setNum(document->lineNum(line) + 1));
#endif
    bitBlt(this, x, y, buffer, x, 0, w, h);
    y += h;
    line++;
  }
}
/****************************************************************************/
TKEditorCursor::TKEditorCursor(TKTextView *view)
: QObject(view),
  view(view), document(view->textDocument()), manager(view->textManager()),
  timer(0), xpos(0), ypos(0),
  visible(false), active(false), stack(0)
{
}

void TKEditorCursor::paint()
{
  xpos = QMAX(xpos, 0);
  ypos = QMIN(QMAX(ypos, 0), document->lastLine());

  int w = document->textWidth(ypos, xpos);
  int x = w - view->contentsX() - 2;
  int y = ypos * document->lineHeight() - view->contentsY();
  int b = y + document->lineHeight() - 2;

  QPainter paint(view);
  paint.setPen(white);
  paint.setRasterOp(XorROP);
  paint.drawLine(x + 2 +2, y, x + 2 +2, b);
  paint.drawLine(x +2, y, x + 4 +2, y);
  paint.drawLine(x +2, b, x + 4 +2, b);
}

QPoint TKEditorCursor::position()
{
  xpos = QMAX(xpos, 0);
  ypos = QMIN(QMAX(ypos, 0), document->lastLine());

  int w = document->textWidth(ypos, xpos);
  int x = w - view->contentsX() - 2;
  int y = ypos * document->lineHeight() - view->contentsY();

  return QPoint (x+2, y) ;
}


void TKEditorCursor::move(CursorAction action, bool select)
{
  QRegExp word("[0-9a-zA-Z_]+");
  QRegExp nonword("[^0-9a-zA-Z_\\s]+");
  QRegExp space("\\s+");

  int line = ypos;
  int column = xpos;

  TKTextLine *tktextline = document->lineOf(line);
  bool rewrap = false;
  switch (action) {
    case MoveBackward:
      if (column == 0 && line != 0) {
        --line;
        column = document->textLength(line);
      } else
        --column;
      break;
    case MoveForward:
      if ((manager->fCursorWrap || document->getWrap()) && column >= document->textLength(line)) {
        if (line != document->lastLine()) {
          line++;
          column = 0;
         }
      } else
        column++;
      break;
    case MoveWordBackward:
      if (column == 0) {
        if (line != 0) {
          --line;
          column = document->textLength(line);
         }
      } else {
        --column;
        word.searchRev(*tktextline, column);
        nonword.searchRev(*tktextline, column);
        space.searchRev(*tktextline, column);
        int pos = 0;
        if (word.pos() != column)
          pos = QMAX(pos, word.pos() + word.matchedLength());
        if (nonword.pos() != column)
          pos = QMAX(pos, nonword.pos() + nonword.matchedLength());
        if (space.pos() != column)
          pos = QMAX(pos, space.pos() + space.matchedLength());
        column = pos;
      }
      break;
    case MoveWordForward:
      if (word.search(*tktextline, column) == column)
        column += word.matchedLength();
      else
        if (nonword.search(*tktextline, column) == column)
          column += nonword.matchedLength();
        else
          if (space.search(*tktextline, column) == column)
            column += space.matchedLength();
          else {
            if (line != document->lastLine()) {
              line++;
              column = 0;
            }
          }
      break;
    case MoveUp:
      if (line != 0)
        --line;
        if(column > document->textLength(line))
            column = document->textLength(line);
      break;
    case MoveDown:
      if (line == document->lastLine())
        column = document->textLength(line);
      else
        line++;
        if(column > document->textLength(line))
            column = document->textLength(line);
      break;
    case MoveLineStart:
      if(tktextline->status() & TKTextLine::Wrapped)
          while( (line>=0) 
              && (document->lineOf(line)->status() & TKTextLine::Wrapped))
          line--;    
      tktextline = document->lineOf(line);
      column = (tktextline->firstChar()==column || !manager->fSmartHome) ? 0 : tktextline->firstChar();
      if ( column == -1 )
        column = 0;
      break;
    case MoveLineEnd:

      if(((line < document->lastLine()) 
        && (document->lineOf(line+1)->status() & TKTextLine::Wrapped)))
      {
          while( (line < document->lastLine()) 
              && (document->lineOf(line+1)->status() & TKTextLine::Wrapped))
          line++;    
          tktextline = document->lineOf(line);
      }
      column = document->textLength(line);
      break;
    case MoveHome:
      column = 0;
      line = 0;
      break;
    case MoveEnd:
      line = document->lastLine();
      column = document->textLength(line);
      break;
    case MovePgUp:
      document->rewrap(view);
      line -= view->height() / document->lineHeight();
      if (line < 0)
        line = 0;
      if(column > document->textLength(line))
          column = document->textLength(line);
      // rewrap whole view
      rewrap = true;
      break;
    case MovePgDown:
      document->rewrap(view);
      line += view->height() / document->lineHeight();
      if (line > document->lastLine())
        line = document->lastLine();
      if(column > document->textLength(line))
          column = document->textLength(line);
      // rewrap whole view
      rewrap = true;
      break;
  }
  if((!rewrap) && (
      !(document->lineOf(line)->status() & TKTextLine::Wrapped)
      && !((line < document->lastLine()) && document->lineOf(line+1)->status() & TKTextLine::Wrapped)))
      document->wrap(view, line, &line, &column);
  view->setCursorPosition(line, column, select);
}

void TKEditorCursor::set(int line, int column, bool update=true)
{
  xpos = QMAX(xpos, 0);
  ypos = QMIN(QMAX(ypos, 0), document->lastLine());
  if( update )
    setEnabled(false);
  xpos = QMAX(column, 0);
  ypos = QMIN(QMAX(line, 0), document->lastLine());
  if( update )
  {
    view->ensureCursorVisible();
    setEnabled(true);
  }
}

TKEditorCursor::~TKEditorCursor()
{
}

void TKEditorCursor::timerEvent(QTimerEvent *ev)
{
  if (ev->timerId() == timer) {
    visible = !visible;
    paint();
  }
}

void TKEditorCursor::setEnabled(bool isOn)
{
  if (isOn) {
    stack = QMAX(0, stack - 1);
    isOn = stack == 0;
  } else
    stack++;

  if (!active)
    return;

  if (isOn) {
    visible = true;
    if (timer)
      killTimer(timer);
    paint();
    timer = startTimer(QApplication::cursorFlashTime() / 2);
  } else {
    if (timer) {
      killTimer(timer);
      timer = 0;
    }
    if (visible) {
      visible = false;
      if (ypos <= document->lastLine())
        paint();
    }
  }
}

void TKEditorCursor::activate()
{
  active = true;
  setEnabled(true);
}

void TKEditorCursor::deactivate()
{
  setEnabled(false);
  active = false;
}
/****************************************************************************/
TKTextView::TKTextView(TKTextEditor *editor)
: QWidget(editor, 0, Qt::WResizeNoErase | Qt::WRepaintNoErase),
  editor(editor), document(editor->textDocument()), manager(document->textManager()),
  popupBox(0),
  hbar(QScrollBar::Horizontal, editor),
  vbar(QScrollBar::Vertical, editor),
  cursorLine(0), selectionStart(0,0)
{
  setCursor(ibeamCursor);
  setMouseTracking(true);
  setBackgroundMode(NoBackground);
  setFocusPolicy(StrongFocus);
  setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));
  setAcceptDrops(true);

  QGridLayout *layout = new QGridLayout(editor, 0, 0, 2, 0);

  indicatorMargin = new TKIndicatorMargin(editor, this);
  selectionMargin = new TKSelectionMargin(editor, this);
  linuNumberMargin   = new TKLinuNumberMargin(editor, this);

  layout->addWidget(indicatorMargin, 0, 0);
  layout->addWidget(linuNumberMargin, 0, 1);
  layout->addWidget(selectionMargin, 0, 2);
  layout->addWidget(this, 0, 3);
  layout->addWidget(&vbar, 0, 4);
  layout->addMultiCellWidget(&hbar, 1, 1, 0, 3);
  layout->activate();

  connect(&hbar, SIGNAL(valueChanged(int)), SLOT(hslide(int)));
  connect(&vbar, SIGNAL(valueChanged(int)), SLOT(vslide(int)));

  xPos = 0;
  yPos = 0;
  selectionFixed = false;
  mousePressed = false;
  mouseScrollMode = false;
  mouseCenter = 0;

  selected = QRect(QPoint(0, 0), QPoint(0, 0));
  found = QRect(QPoint(0, 0), QPoint(0, 0));

  cursor = new TKEditorCursor(this);
  document->registerView(this);

  updateMarginsState();
}

TKTextView::~TKTextView()
{
  document->removeView(this);
}

void TKTextView::updateMarginsState()
{
  if (manager->isIndicatorMarginVisible())
    indicatorMargin->show();
  else
    indicatorMargin->hide();

#ifndef KOBOL
  if (manager->isLinuNumberMarginVisible())
#else
  if (editor->isLinuNumberMarginVisible())
#endif
	linuNumberMargin->show();
  else
    linuNumberMargin->hide();


  if (manager->isSelectionMarginVisible())
    selectionMargin->show();
  else
    selectionMargin->hide();
}

void TKTextView::scrollX(int dx)
{
  if( document->getWrap() )
      return;

  if (QABS(dx) < width())
    scroll(dx, 0);
  else
    repaint(false);
}

void TKTextView::scrollY(int dy)
{
  if (QABS(dy) < height()) {
    scroll(0, dy);
    indicatorMargin->scroll(0, dy);
    selectionMargin->scroll(0, dy);
    linuNumberMargin->scroll(0, dy);
  } else {
    repaint(false);
    indicatorMargin->repaint(false);
    selectionMargin->repaint(false);
    linuNumberMargin->repaint(false);
  }
}

void TKTextView::hslide(int pos)
{
  cursor->setEnabled(false);
  int dx = xPos - pos;
  xPos = pos;
  scrollX(dx);
  cursor->setEnabled();
}

void TKTextView::vslide(int pos)
{

  cursor->setEnabled(false);
  if (!mouseScrollMode)
    pos *= document->lineHeight();
  int dy = yPos - pos;
  yPos = pos;
  scrollY(dy);
  cursor->setEnabled();
}

bool TKTextView::event(QEvent *ev)
{
  if (ev->type() == QEvent::KeyPress && ((QKeyEvent *)ev)->key() == Qt::Key_Tab) {
    keyPressEvent((QKeyEvent *)ev);
    return true;
  }
  if (ev->type() == QEvent::User + 1) {
    finishCompletion();
    popupBox->close();
  }
  return QWidget::event(ev);
}

bool TKTextView::eventFilter(QObject *obj, QEvent *ev)
{
  if (obj == popupBox) {
    bool hide = false;
    switch (ev->type()) {
      case QEvent::FocusIn:
        cursor->activate();
        break;
      case QEvent::FocusOut:
        cursor->deactivate();
        break;
      case QEvent::KeyPress: {
        QKeyEvent *kev = (QKeyEvent *)ev;
          if (kev->text().length() && (!kev->ascii() || kev->ascii() >= 32) || (kev->text() == "\t" && !(kev->state() & Qt::ControlButton))) {
          keyPressEvent(kev);
          completionWord += kev->text();
          completionGrep = completionGrep.grep(completionWord, false);
          if (completionGrep.count() == 0) {
            hide = true;
            break;
          }
          completionGrep.sort();
          popupBox->clear();
          popupBox->insertStringList(completionGrep);
          popupBox->setCurrentItem(0);
        } else
          if (kev->key() == Key_Backspace) {
            keyPressEvent(kev);
            completionWord.truncate(completionWord.length() - 1);
            if (completionWord.isEmpty()) {
              hide = true;
              break;
            }
            completionGrep = completionWords.grep(completionWord, false);
            if (completionGrep.count() == 0) {
              hide = true;
              break;
            }
            completionGrep.sort();
            popupBox->clear();
            popupBox->insertStringList(completionGrep);
            popupBox->setCurrentItem(0);
          }
          else if (kev->key() == Key_Escape)
            hide = true;
        break; }
      case QEvent::MouseButtonPress:
        hide = !popupBox->geometry().contains(((QMouseEvent *)ev)->globalPos());
      default:
        break;
    }
    if (hide)
      QApplication::postEvent(this, new QCustomEvent(QEvent::User + 1));
  }
  return QWidget::eventFilter(obj, ev);
}

void TKTextView::ensureCursorVisible()
{
  if (cursorLine != cursor->line() && cursorLine <= document->lastLine()) {
    if (manager->fRemoveSpaces) {
      TKTextLine *line = document->lineOf(cursorLine);
      if (line->removeSpaces())
        document->tagLine(cursorLine);
    }
  }
  cursorLine = cursor->line();

  cursor->setEnabled(false);
  int f = document->fontHeight;
  int t = document->tabWidth;
  int y = document->linePosition(cursor->line());
  int x = document->textWidth(cursor->line(), cursor->column());

  // shouldn't scroll when wrapping
  if(! document->getWrap() )
  {
    if (QMAX(0, x - t) < xPos) {
        int dx = xPos - QMAX(0, x - t);
        xPos = QMAX(0, x - t);
        scrollX(dx);
    } else
        if (x + t > xPos + width()) {
            int dx = x + t - xPos - width();
            xPos = x - width() + t;
            scrollX(-dx);
        }
  } else
    xPos = 0;

  if (y < yPos) {
    int dy = yPos - y;
    yPos = y;
    scrollY(dy);
  } else
    if (y + f > yPos + height()) {
      int dy = y + f - yPos - height();
      yPos = y - height() + f;
      scrollY(-dy);
    }

  updateView(true);
  cursor->setEnabled();
}

void TKTextView::clearFound()
{
  if (found.topLeft() != found.bottomRight()) {
    update.expand(found.normalize().top(), found.normalize().bottom());
    found = QRect(QPoint(0, 0), QPoint(0, 0));
  }

  editor->notifyChange(TKTextEditor::Selection);
  updateView(false);
}

void TKTextView::setFound(int line, int pos, int length)
{
  update.expand(line);
  found = QRect(QPoint(pos, line), QPoint(pos + length, line));

  editor->notifyChange(TKTextEditor::Selection);
}

void TKTextView::setSelection(const QPoint &from, const QPoint &to)
{
  if (selected.topLeft() != selected.bottomRight())
    update.expand(selected.normalize().top(), selected.normalize().bottom());

  if((from.y() < to.y()) || 
          ((from.y() == to.y()) && (from.x() <= to.x())))
    selected = QRect(from, to);
  else
    selected = QRect(to, from);

  update.expand(selected.normalize().top(), selected.normalize().bottom());

  editor->notifyChange(TKTextEditor::Selection);
}

void TKTextView::setCursorPosition(int line, int column, bool select)
{
  if (select) {
    QPoint start(cursor->column(), cursor->line());
    QPoint end(column, line);
    if (selected.topLeft() == selected.bottomRight())
      setSelection(start, end);
    else
      setSelection(selectionStart, end);
  } else
    clearSelection();

  cursor->set(line, column);
  editor->notifyChange(TKTextEditor::CursorPosition);
}

void TKTextView::clearSelection()
{
  if (!selectionFixed && selected.topLeft() != selected.bottomRight()) {
    update.expand(selected.normalize().top(), selected.normalize().bottom());
    selected = QRect(QPoint(0, 0), QPoint(0, 0));
  }
  if (found.topLeft() != found.bottomRight()) {
    update.expand(found.normalize().top(), found.normalize().bottom());
    found = QRect(QPoint(0, 0), QPoint(0, 0));
  }

  editor->notifyChange(TKTextEditor::Selection);
  updateView(false);
}

void TKTextView::paintEvent(QPaintEvent *ev)
{
  cursor->setEnabled(false);
  QRect rect = ev->rect();
  int x = rect.x();
  int w = rect.width();
  int h = document->fontHeight;
  int line = document->lineNumber(yPos + rect.y());
  int y = document->linePosition(line) - yPos;
  int end = rect.y() + rect.height();

  while (y < end && line <= document->lastLine()) {
//    TKTextLine *textLine = document->lineOf(line);
    document->paintLine(this, line);
    bitBlt(this, x, y, document->buffer, x, 0, w, h);

    y += h;
    line++;
  }

  rect.setY(y);
  document->paintEmptyArea(this, rect);
  cursor->setEnabled();
}

void TKTextView::focusInEvent(QFocusEvent *)
{
  cursor->set(cursor->line(), cursor->column());
  updateView(true);
  cursor->activate();
}

void TKTextView::focusOutEvent(QFocusEvent *)
{
  cursor->deactivate();
}

void TKTextView::keyPressEvent(QKeyEvent *ev)
{
  if (mouseScrollMode)
    stopMouseScroll();

  bool cb = ev->state() & Qt::ControlButton;
  bool sb = ev->state() & Qt::ShiftButton;

  switch (ev->key()) {
    case Key_Escape:
      clearSelection();
      break;
    case Key_Left:
      cursor->move(cb ? TKEditorCursor::MoveWordBackward : TKEditorCursor::MoveBackward, sb);
      break;
    case Key_Right:
      cursor->move(cb ? TKEditorCursor::MoveWordForward : TKEditorCursor::MoveForward, sb);
      break;
    case Key_Up:
      cursor->move(cb ? TKEditorCursor::MovePgUp : TKEditorCursor::MoveUp, sb);
      break;
    case Key_Down:
      cursor->move(cb ? TKEditorCursor::MovePgDown : TKEditorCursor::MoveDown, sb);
      break;
    case Key_Home:
      cursor->move(cb ? TKEditorCursor::MoveHome : TKEditorCursor::MoveLineStart, sb);
      break;
    case Key_End:
      cursor->move(cb ? TKEditorCursor::MoveEnd : TKEditorCursor::MoveLineEnd, sb);
      break;
    case Key_Prior:
      cursor->move(TKEditorCursor::MovePgUp, sb);
      break;
    case Key_Next:
      cursor->move(TKEditorCursor::MovePgDown, sb);
      break;

    case Key_Return:
    case Key_Enter:
            document->insertLineBreak(this);
      break;
    case Key_Backspace:
      if (hasSelection())
        document->selectionCommand(this, TKTextDocument::SelectionRemove);
      else
        document->backspace(this);
      break;
    case Key_Delete:
      //if (!hasSelection())
        editor->del();
      break;

    case Key_Insert:
        if (cb && !sb)
          editor->copy();
        else
          if (sb && !cb)
            editor->paste();
          else
            manager->setOverwriteMode(!manager->overwriteMode());
      break;

    case Key_Shift:
        selectionStart = QPoint(cursor->column(), cursor->line());
      break;

    default:
        if (ev->text().length() && (!ev->ascii() || ev->ascii() >= 32) || (ev->text() == "\t" && !(ev->state() & Qt::ControlButton))) {
        document->selectionCommand(this, TKTextDocument::SelectionRemove);
        document->insert(this, ev->text());
        document->qwrap(this);
      } else
        ev->ignore();
      break;
  }
}

QString TKTextView::selectedText()
{
  return QString::null;
}

bool TKTextView::hasSelection(bool withSelection, bool withFound)
{
  bool s = selected.topLeft() != selected.bottomRight();
  bool f = found.topLeft() != found.bottomRight();
  return (withSelection & s) | (withFound & f);
}

bool TKTextView::inSelected(int line, int column)
{
  return inRect(selected, line, column, manager->fVerticalSelection) ||
         inRect(found, line, column, false);
}

bool TKTextView::inRect(const QRect &rect, int line, int column, bool v)
{
  bool result = false;
  if (rect.topLeft() != rect.bottomRight()) {
    QRect sel = rect.normalize();
    QRect linesel = sel.intersect(QRect(0, line, sel.right() + 1, 1));
    if (!linesel.isEmpty()) {
      int start = linesel.left();
      int end = linesel.right();
      if (v || sel.height() == 1) {
        result = column >= start && column < end;
      } else {
        bool f = (rect.width() > 0 && rect.height() > 0) || (rect.width() <= 0 && rect.height() <= 0);
        int start = f ? linesel.left() : linesel.right();
        int end = f ? linesel.right() : linesel.left();
        if (line == sel.top())
          result = column >= start;
        else
          if (line == sel.bottom())
            result = column < end;
          else
            result = true;
      }
    }
  }
  return result;
}

void TKTextView::stopMouseScroll()
{
  delete mouseCenter;
  mouseCenter = 0;
  qApp->restoreOverrideCursor();

  killTimer(mouseScrollTimer);
  mouseScrollMode = false;
  int line = yPos / document->lineHeight();
  yPos = (line + 1) * document->lineHeight();
  updateView(true);
  repaint(false);
  indicatorMargin->repaint(false);
  selectionMargin->repaint(false);
  linuNumberMargin->repaint(false);
  cursor->activate();
  releaseMouse();
}

void TKTextView::mousePressEvent(QMouseEvent *ev)
{
  if (mouseScrollMode) {
    stopMouseScroll();
    return;
  }
  if (ev->button() == LeftButton) {

    int line ;
    int column ;
    translatePos (ev->pos(), line, column) ;
//    bool beyond = false ;
//    int line = document->lineNumber(yPos + ev->y());
//    if (line > document->lastLine())
//    {  line = document->lastLine() ;
//       beyond = true ;
//    }
//
//    int lineLen = document->lineOf(line)->length();
//    int column = document->textLength(line, xPos + ev->x());
//    
//    if (( column > lineLen ) || beyond)
//        column = lineLen;

    if (inSelected(line, column)) {
      setCursorPosition(line, column, ev->state() & Qt::ShiftButton);
    } else {
      setCursorPosition(line, column, ev->state() & Qt::ShiftButton);
      selectionStart = QPoint(column, line);
      mousePressed = true;
    }
  } else
    if (ev->button() == MidButton) {
    /*
      cursor->deactivate();
      mouseScrollMode = true;
      mouseScrollModeDone = false;
      mouseScrollPos = ev->pos();
      mouseScrollDist = QPoint(0, 0);
      updateView(true);

      QCursor centerCursor = mouseCursor(8);
      mouseCenter = new QWidget(editor);
      mouseCenter->setBackgroundColor(colorGroup().dark());
      QPoint pos = mapTo(editor, ev->pos()) - centerCursor.hotSpot();
      mouseCenter->setGeometry(pos.x(), pos.y(), 32, 32);
      mouseCenter->setMask(*centerCursor.bitmap());
      mouseCenter->show();

      qApp->setOverrideCursor(centerCursor);
      mouseScrollTimer = startTimer(10);
    */
	editor->paste();
    }
}

void TKTextView::mouseReleaseEvent(QMouseEvent *ev)
{
  mousePressed = false;

  if (ev->button() == RightButton)
  {
        int line = document->lineNumber(yPos + ev->y());
	emit mouseRightClick (ev->globalPos(), line) ;
	return ;
  }

  if (mouseScrollMode)
    if (mouseScrollModeDone)
      stopMouseScroll();
    else
      grabMouse();
}

void TKTextView::mouseMoveEvent(QMouseEvent *ev)
{
  if (mouseScrollMode) {
    mouseScrollDist = ev->pos();
    mouseScrollDist -= mouseScrollPos;
    return;
  }

  if (!rect().contains(ev->pos())) {
    return;
  }

  int line = document->lineNumber(yPos + ev->y());
  bool out = line > document->lastLine();
  int column = !out ? document->textLength(line, xPos + ev->x()) : 0;
  if (mousePressed && ev->state() == Qt::LeftButton) {
    setCursorPosition(line, column, true);
  } else
    setCursor(out || inSelected(line, column) ? arrowCursor : ibeamCursor);
}

void TKTextView::mouseDoubleClickEvent(QMouseEvent *ev)
{
  clearSelection();
  int line = document->lineNumber(yPos + ev->y());
  if (line > document->lastLine())
    return;

  int column = document->textLength(line, xPos + ev->x());
  int start = 0;
  int end = 0;
  wordIn(line, column, &start, &end);
  setCursorPosition(line, start);
  setCursorPosition(line, end, true);
}

void TKTextView::dragEnterEvent(QDragEnterEvent *ev)
{
  ev->accept(QTextDrag::canDecode(ev) && !QUriDrag::canDecode(ev)) ;
}

void TKTextView::translatePos (QPoint pos, int &line, int &column)
{
  bool beyond = false ;
  line = document->lineNumber(yPos + pos.y());
  if (line > document->lastLine())
  {  line = document->lastLine() ;
     beyond = true ;
  }

  int lineLen = document->lineOf(line)->length();
  column = document->textLength(line, xPos + pos.x());
    
  if (( column > lineLen ) || beyond)
      column = lineLen;
}

void TKTextView::dropEvent (QDropEvent *ev)
{
//fprintf(stderr, "***** DROP [%s][%d]\n", ev->format(), ev->provides("text/plain")) ;
  QString text ;
  if (!QTextDrag::decode (ev, text)) return ;

  int line ;
  int column ;
  translatePos (ev->pos(), line, column) ;
//  bool beyond = false ;
//  int line = document->lineNumber(yPos + ev->pos().y());
//  if (line > document->lastLine())
//  {  line = document->lastLine() ;
//     beyond = true ;
//  }
//
//  int lineLen = document->lineOf(line)->length();
//  int column = document->textLength(line, xPos + ev->pos().x());
//    
//  if (( column > lineLen ) || beyond)
//      column = lineLen;

  setCursorPosition(line, column, false) ;
  document->insert(this, text);
}

QString TKTextView::wordIn(int line, int column, int *start, int *end)
{
  TKTextLine *text = document->lineOf(line);
  if (start)
    *start = column;
  if (end)
    *end = column;

  if (column < 0 || column >= (int)text->length())
    return QString::null;

  QRegExp word("[\\w_]+");
  QRegExp nonword("[^\\w_\\s]+");
  QRegExp regexp;
  if (word.search(*text, column) == column)
    regexp = word;
  else
    if (nonword.search(*text, column) == column)
      regexp = nonword;
    else
      return QString::null;

  int s = column;
  while (s > 0 && regexp.search(*text, s - 1) == s - 1)
    --s;

  int e = column;
  while (e < (int)text->length() && regexp.search(*text, e) == e)
    ++e;

  if (start)
    *start = s;
  if (end)
    *end = e;
  return text->mid(s, e - s);
}

void TKTextView::wheelEvent(QWheelEvent *ev)
{
  QApplication::sendEvent(&vbar, ev);
}

void TKTextView::resizeEvent(QResizeEvent *)
{
  document->resizeBuffer();
  document->rewrap(this);
  updateView(true);
}

void TKTextView::showEvent(QShowEvent *)
{
  document->resizeBuffer();
  document->rewrap(this);
  updateView(true);
}

void TKTextView::timerEvent(QTimerEvent *ev)
{
  if (ev->timerId() == mouseScrollTimer) {
    killTimer(mouseScrollTimer);
    int dx = mouseScrollDist.x() / (3 * document->lineHeight());
    int dy = mouseScrollDist.y() / (2 * document->lineHeight());

    int id = 0;
    if (dx > 0 && dy > 0)
      id = 6;
    else
      if (dx > 0 && dy < 0)
        id = 5;
      else
        if (dx > 0 && dy == 0)
          id = 2;
        else
          if (dx < 0 && dy > 0)
            id = 7;
          else
            if (dx < 0 && dy < 0)
              id = 4;
            else
              if (dx < 0 && dy == 0)
                id = 3;
              else
                if (dx == 0 && dy > 0)
                  id = 1;
                else
                  if (dx == 0 && dy < 0)
                    id = 0;
                  else
                    id = 8;

    qApp->setOverrideCursor(mouseCursor(id), true);

    int speedx = (dx == 0) ? 0 : (dx / 2 + (dx > 0 ? 1 : -1));
    int speedy = (dy == 0) ? 0 : (dy / 2 + (dy > 0 ? 1 : -1));

    if (!mouseScrollModeDone && (speedx != 0 || speedy != 0))
      mouseScrollModeDone = true;

    hbar.setValue(hbar.value() + speedx);
    vbar.setValue(vbar.value() + speedy);

    mouseScrollTimer = startTimer(10);
  }
}

void TKTextView::updateView(bool newDocGeometry)
{
  if (!document->buffer)
    return;

  cursor->setEnabled(false);

  int first = document->lineNumber(yPos);
  int last = document->lineNumber(yPos + height() - 1);

  if (newDocGeometry) {
    int w = width();
    int h = height();
    int l = document->lineHeight();

    int tw = QMAX(document->textWidth(cursor->line(), cursor->column()) + document->tabWidth, document->documentWidth());
    int xMax = tw - w;
    int yMax = document->documentHeight() - h;

    hbar.blockSignals(true);
    if( document->getWrap())
    {
        scroll(-xPos, 0);
        xPos = 0;
        hbar.setEnabled(false);
        hbar.setRange(0, 0);
        hbar.setEnabled(false);
        hbar.hide();
    }
    else
    {
        if(hbar.isHidden())
            hbar.show();
        hbar.setEnabled(true);
        if (xMax > 0){
            hbar.setRange(0, xMax);
            hbar.setValue(xPos);
            hbar.setSteps(document->fontHeight, w);
            hbar.setEnabled(true);
        } else {
            hbar.setRange(0, 0);
            hbar.setEnabled(false);
        }
    }
    hbar.blockSignals(false);

    vbar.blockSignals(true);
    if (yMax > 0) {
      if (mouseScrollMode) {
        vbar.setRange(0, document->documentHeight() - h);
        vbar.setValue(yPos);
        vbar.setSteps(1, 1);
      } else {
        vbar.setRange(0, (document->documentHeight() - h) / l + 1);
        vbar.setValue(yPos / l);
        vbar.setSteps(1, h / l + 1);
      }
      vbar.setEnabled(true);
    } else {
      vbar.setRange(0, 0);
      vbar.setEnabled(false);
    }
    vbar.blockSignals(false);

    if (last > document->lastLine() + 2 && first != 0) {
      int dl = last - document->lastLine() - 2;
      first = QMAX(0, first - dl);
      last = document->lastLine();
      int dy = first == 0 ? yPos : dl * document->fontHeight;
      yPos -= dy;
      scrollY(dy);
      updateView(true);
      cursor->setEnabled();
      return;
    }

    if (xPos > 0 && tw - xPos < w) {
      int dx = xPos - QMAX(0, tw - w);
      xPos = QMAX(0, tw - w);
      scrollX(dx);
      updateView(true);
      cursor->setEnabled();
      return;
    }
  }

  update.expand(document->updateRegion);
  if (update.end > document->lastLine()) {
    repaint(false);
    indicatorMargin->repaint(false);
    selectionMargin->repaint(false);
    linuNumberMargin->repaint(false);
  } else {
    update.intersect(first, last);
    if (!update.isEmpty()) {
      int h = document->fontHeight;
      for (int line = update.start; line <= update.end; line++) {
        int y = document->linePosition(line);
        document->paintLine(this, line);
        bitBlt(this, 0, y - yPos, document->buffer, 0, 0, width(), h);
        indicatorMargin->repaint(0, y - yPos, indicatorMargin->width(), h, false);
        selectionMargin->repaint(0, y - yPos, selectionMargin->width(), h, false);
        linuNumberMargin->repaint(0, y - yPos, linuNumberMargin->width(), h, false);
      }
    }
  }

  update.clear();
  cursor->setEnabled();
}

QString TKTextView::currentWord()
{
  return wordIn(cursor->line(), cursor->column() - 1);
}

void TKTextView::completeWord()
{
  completionWord = wordIn(cursor->line(), cursor->column() - 1);
  if (completionWord.isEmpty())
    return;

  QApplication::setOverrideCursor(waitCursor);

  QRegExp regexp("[\\w_]+");
  QAsciiDict<int> dict;
  int val = 0;
  for (int line = 0; line <= document->lastLine(); line++) {
    TKTextLine *text = document->lineOf(line);
    int pos = regexp.search(*text);
    while (pos != -1) {
      QString word = text->mid(pos, regexp.matchedLength());
      if (!dict.find(word.latin1())) {
        dict.insert(word.latin1(), &val);
        completionWords << word;
      }
      pos = regexp.search(*text, pos + regexp.matchedLength());
    }
  }
  completionGrep = completionWords.grep(completionWord, false);

  QApplication::restoreOverrideCursor();

  if (completionGrep.count() == 0) {
    finishCompletion();
    return;
  }
  cursor->move(TKEditorCursor::MoveBackward, false);
  cursor->move(TKEditorCursor::MoveWordForward, false);
  completionGrep.sort();

  int x = document->textWidth(cursor->line(), cursor->column()) - xPos - 2;
  int y = (cursor->line() + 1) * document->lineHeight() - yPos;

  popupBox = new QListBox(this, 0, Qt::WType_Popup | Qt::WDestructiveClose);

  popupBox->setFixedSize(100, 100);
  popupBox->setFrameStyle(QFrame::Box | QFrame::Raised);
  popupBox->insertStringList(completionGrep);
  popupBox->setCurrentItem(0);

  QPoint p = mapToGlobal(QPoint(x, y));
  popupBox->move(QMIN(p.x(), qApp->desktop()->width() - 101), QMIN(p.y(), qApp->desktop()->height() - 101));
  popupBox->installEventFilter(this);
  connect(popupBox, SIGNAL(returnPressed(QListBoxItem *)), SLOT(applyCompleteWord()));
  connect(popupBox, SIGNAL(clicked(QListBoxItem *)), SLOT(applyCompleteWord()));
  connect(popupBox, SIGNAL(doubleClicked(QListBoxItem *)), SLOT(applyCompleteWord()));

  popupBox->show();
}

void TKTextView::applyCompleteWord()
{
  if (!popupBox) return;

  QString text = popupBox->currentText();
  cursor->move(TKEditorCursor::MoveWordBackward, true);
  document->selectionCommand(this, TKTextDocument::SelectionRemove);
  document->insert(this, text);
  QApplication::postEvent(this, new QCustomEvent(QEvent::User + 1));
}

void TKTextView::finishCompletion()
{
  completionGrep.clear();
  completionWords.clear();
  completionWord = QString::null;
}

void TKTextView::lineHeightChanged()
{
  indicatorMargin->resizeBuffer();
  selectionMargin->resizeBuffer();
  linuNumberMargin->resizeBuffer();
}

QRect TKTextView::selection()
{
  return selected;
}


void TKTextView::repaintMargins()
{
  indicatorMargin->update();
  selectionMargin->update();
  linuNumberMargin->update();
}

void TKTextView::centerCursor(int line, int column)
{
  int dy = document->lineNumber(yPos + height() - 1) - document->lineNumber(yPos);
  setCursorPosition(line - dy / 2, 0);
  setCursorPosition(line + dy / 2, 0);
  setCursorPosition(line, column);
}

void TKTextView::lockSeletion(bool f)
{
  selectionFixed = f;
}

bool TKTextView::isSelectionFixed() const
{
  return selectionFixed;
}

void TKTextView::repaintIndicatorMargin()
{
	indicatorMargin->repaint();
}

void TKTextView::setFocus ()
{
	if (editor->viewSetFocusOK()) QWidget::setFocus() ;
}
