/***************************************************************************
    file	         : kb_editor.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>

#include	<qsplitter.h>

#include	<qfont.h>
#include	<qfile.h>
#include	<qtextstream.h>
#include 	<qtimer.h>


#include	"kb_classes.h"
#include	"kb_location.h"
#include	"kb_dbinfo.h"
#include	"kb_gui.h"
#include	"kb_script.h"
#include	"kb_viewer.h"
#include	"kb_dialog.h"
#include	"kb_options.h"

#ifndef 	_WIN32
#include	"kb_editor.moc"
#else
#include 	"kb_editor.h"
#endif

#include	"tk_icons.h"
#include	"tk_messagebox.h"


/*  KBEditor	: 							*/
/*  KBEditor	: Constructor for simple script editor			*/
/*  parent	: KBObjBase *	: Parent object				*/
/*  embed	: QWidget *	: Embedding widget			*/
/*  (returns)	: KBEditor	:					*/

KBEditor::KBEditor
	(	KBObjBase	*parent,
		QWidget		*embed
	)
	:
	KBViewer   (parent, embed, WDestructiveClose|WStyle_NormalBorder),
	m_splitter (new QSplitter (m_partWidget))
{
	m_splitter->setOrientation (QSplitter::Vertical) ;

	QString	hldir	= locateDir
			  (	"appdata",
				"highlight/global/nohighlight"
			  )
			  + "highlight/"   ;

	QString	font	= KBOptions::getScriptFont() ;
	TKTextEditorManager *manager = _KBDialog::getTextManager (font) ;

	m_document  = new TKTextDocument(manager)	  ;
	m_editWin   = new TKTextEditor  (m_document, m_splitter) ;

	manager->addEditor (m_editWin) ;
	connect	  (m_editWin, SIGNAL(changed(int)), SLOT(modified())) ;

	m_errWin    = new QListBox     (m_splitter)	;
	m_mapper    = new TKCTKEMapper (m_editWin )	; 
	m_gui	    = 0	 ;

	m_splitter->show () ;
	m_editWin ->show () ;
	m_errWin  ->show () ;

	if (getLocation().docExtn.isEmpty())
	{
		KBError::EError
		(	QString (TR("Script file \"%1\" does not have an extension")).arg(getLocation().docName),
			TR("Will not be able to compile"),
			__ERRLOCN
		) ;
		scrIface = 0 ;
	}
	else
	{
		KBError	error	;
		scrIface = LinkKBScript (getLocation().docExtn, error) ;
		if (scrIface == 0) error.DISPLAY () ;
	}

	if (scrIface != 0)
		scrIface->editorInit (m_editWin) ;

	TKConfig *config = TKConfig::getConfig () ;
	config->setGroup ("Editor Options") ;
	m_size	 = config->readSizeEntry ("Geometry" ) ;
	m_ewind	 = config->readNumEntry  ("EditDepth") ;

	if (m_size.isEmpty())
		m_size = 	  QSize (500, 400)  ;
	else	m_size.boundedTo (QSize (300, 200)) ;

	m_topWidget = m_splitter	;

	setGUI (m_gui = new KBaseGUI (this, this, "rekallui.editor")) ;
	if (scrIface == 0) m_gui->setEnabled ("KB_compile", false) ;

#if	__KB_EMBEDDED
	m_partWidget->showMaximized  ()  ;
#else
	m_partWidget->resize	     (m_size.width(), m_size.height(), true, false) ;
	m_partWidget->setMinimumSize (300, 200) ;
#endif
	m_partWidget->setIcon	     (getSmallIcon ("shellscript")) ;
	m_partWidget->show	     ()  ;

	QValueList<int> sizes ;

	if (m_ewind == 0)
	{	sizes.append  (m_size.height() - 45) ;
		sizes.append  (45)    ;
	}
	else
	{	sizes.append  (m_ewind) ;
		sizes.append  (m_size.height() - m_ewind) ;
	}
	m_splitter->setGeometry (0, 0, m_size.width(), m_size.height()) ;
	m_splitter->setSizes    (sizes) ;

	connect	(m_editWin, SIGNAL(modified()), SLOT(modified())) ;
}

/*  KBEditor	: 							*/
/*  ~KBEditor	: Destructor for simple script editor			*/
/*  (returns)	:		:					*/

KBEditor::~KBEditor ()
{
	TKConfig *config = TKConfig::getConfig () ;

	config->setGroup   ("Editor Options"    ) ;
	config->writeEntry ("Geometry",  m_size ) ;
	config->writeEntry ("EditDepth", m_ewind) ;
	config->sync	   () ;

	delete	m_mapper ;
}

/*  KBEditor								*/
/*  startup	: Startup editor					*/
/*  _create	: bool		 : New script creation			*/
/*  (returns)	: bool		 : Success				*/

bool	KBEditor::startup
	(	const QString	&errPat,
		const QString	&errText,
		int		lno,
		bool		_create
	)
{
	/* Compile the line number pattern, and load the script source	*/
	/* and error files.						*/
	loadPattern (errPat) ;
	if (!_create) loadFiles (errText, lno) ;

	setCaption  (getLocation().title()) ;
	return	true	;
}


/*  KBEditor	: 							*/
/*  modified	: Handle scintilla modification notice			*/
/*  (returns)	: void		  :					*/

void	KBEditor::modified ()
{
	/* NOTE: Scintilla issues a modified notification the first	*/
	/* time it is shown, as the styling is set up, so we call back	*/
	/* to find out if the text has really changed. Sheesh!		*/
	m_gui->setEnabled ("KB_saveDoc", m_editWin->isModified()) ;
}

/*  KBEditor	: 							*/
/*  loadPattern	: Load error number pattern				*/
/*  ePat	: const QString & : Error pattern			*/
/*  (returns)	: void		  :					*/

void	KBEditor::loadPattern
	(	const QString	&ePat
	)
{
	/* Pattern will be empty if we are beihg invoked to directly	*/
	/* edit a script.						*/
	if (ePat.isEmpty()) return ;

	/* Disconnect error window, since it may already be connected	*/
	/* and we don't want multiple signals.				*/
	disconnect (m_errWin, SIGNAL(selected(int)), this, SLOT(errSelected(int))) ;

	/* Compile the regular expression used to extract line numbers	*/
	/* from error messages. If this fails, display a fault and	*/
	/* don't bother to wire up the error list box.			*/
	m_errExp = QRegExp (ePat) ;
	connect	   (m_errWin, SIGNAL(selected(int)), this, SLOT(errSelected(int))) ;
}

/*  KBEditor	: 							*/
/*  gotoLine	: Go to line number					*/
/*  lno		: uint		  : Line number				*/
/*  (returns)	: void		  :					*/

void	KBEditor::gotoLine
	(	int	lno
	)
{
	m_editWin->moveCursor (lno, 0) ;
	m_editWin->setFocus   () ;
}

/*  KBEditor	: 							*/
/*  loadFiles	: Load files into edit and error windows		*/
/*  eText	: const QString & : Script errors			*/
/*  lno		: uint		  : Initial line number			*/
/*  (returns)	: void		  :					*/

void	KBEditor::loadFiles
	(	const QString	&eText,
		uint		lno
	)
{
	QString	doc	;
	KBError	error	;

	doc = getLocation().contents (error) ;
	if (doc.isNull())
	{	error.DISPLAY () ;
		return	;
	}
	
	m_editWin->setText (doc) ;

	int	ptr1	= 0 ;
	int	ptr2	= eText.find ('\n') ;

	while (ptr2 > 0)
	{	m_errWin->insertItem (eText.mid (ptr1, ptr2 - ptr1 - 1)) ;
		ptr1	= ptr2 + 1 ;
		ptr2	= eText.find ('\n', ptr1) ;
	}
	if (ptr1 < (int)eText.length())
		m_errWin->insertItem (eText.mid (ptr1)) ;

	gotoLine (lno) ;
	m_document->documentChanged (false) ;
}

/*  KBEditor	: 							*/
/*  errSelected	: User selects an error					*/
/*  idx		: int		: Index into error list box		*/
/*  (returns)	: void		:					*/

void	KBEditor::errSelected
	(	int	
	)
{
	/* This is pretty basic. Get the line number from the error	*/
	/* message and move the cursor to that line in the edit window.	*/
	/* This may not be the right line if the user has moved any	*/
	/* lines about ....						*/
	if (m_errExp.search (m_errWin->text(m_errWin->currentItem())) < 0)
		return	;

	gotoLine (m_errExp.cap(1).toInt()) ;
}

bool	KBEditor::queryClose ()
{
	if (m_editWin->isModified())
		if (TKMessageBox::questionYesNo
		   	(	0,
				QString (TR("Script file \"%1\" has been changed: close anyway?")).arg(getLocation().docName),
				TR("Edit script file")
			)
			!= TKMessageBox::Yes) return false ;

	m_size	= m_partWidget->size () ;
	m_ewind	= m_splitter  ->sizes() [0] ;
	return	true	;
}

/*  KBEditor	: 							*/
/*  saveDocument: Handle file save					*/
/*  (returns)	: void		:					*/

void	KBEditor::saveDocument ()
{
	if (m_objBase->saveDocument())
	{
		m_document->documentChanged (false) ;
		m_gui     ->setEnabled      ("KB_saveDoc", false) ;
		setCaption (getLocation().title()) ;
	}
}

/*  KBEditor	   : 							*/
/*  saveDocumentAs : Handle file save-as				*/
/*  (returns)	   : void	:					*/

void	KBEditor::saveDocumentAs ()
{
	if (m_objBase->saveDocumentAs())
	{
		m_document->documentChanged (false) ;
		m_gui     ->setEnabled      ("KB_saveDoc", false) ;
		setCaption (getLocation().title()) ;
	}
}

QString	KBEditor::def ()
{
	return	m_editWin->text().stripWhiteSpace() + "\n" ;
}

/*  KBEditor	: 							*/
/*  doCompile	: Handle file compile					*/
/*  (returns)	: void		: Success				*/

void	KBEditor::doCompile ()
{
	if (m_editWin->isModified())
		if (m_objBase->saveDocument())
		{
			m_document->documentChanged (false) ;
			m_gui     ->setEnabled     ("KB_saveDoc", false) ;
		}

	if (scrIface == 0) return ;

	QString	errText	;
	QString	errPat  ;
	KBError	error	;

	if (!scrIface->compile (getLocation(), errText, errPat, error))
	{
		error.DISPLAY () ;
		reloadScript  (errText, errPat, 0) ;
		return	;
	}

	m_errWin->clear () ;
}

/*  KBEditor	: 							*/
/*  reloadFile	: Reload files						*/
/*  errText	: const QString & : Script error text			*/
/*  errPat	: const QString & : Error line number pattern		*/
/*  lno		: uint		  : Initial line number			*/
/*  (returns)	: void		  :					*/

void	KBEditor::reloadScript
	(	const QString	&errText,
		const QString	&errPat,
		uint		lno
	)
{
	if (m_editWin->isModified())
		if (TKMessageBox::questionYesNo
		   	(	0,
				QString (TR("Script file \"%1\" has been changed: reload anyway?")).arg(getLocation().docName),
				TR("Edit script file")
			)
			!= TKMessageBox::Yes) return ;

	loadPattern (errPat) ;
	loadFiles   (errText, lno) ;
}

/*  KBEditor	: 							*/
/*  getChanged	: Report whether there are unsaved changes		*/
/*  both	: bool		: True for both data and design change	*/
/*  (returns)	: cchar *	: Changed message or null if none	*/

cchar	*KBEditor::getChanged
	(	bool
	)
{
	return	m_editWin->isModified () ? "text" : 0 ;
}

void	KBEditor::showAs
	(	KB::ShowAs
	)
{
}
