/*
 * Copyright (c) 1998, 1999, 2000 Phil Thompson (phil@river-bank.demon.co.uk)
 *
 * The parse tree transformation module for SIP.
 */


#include <stddef.h>
#include <string.h>
#include <stdlib.h>

#include "sip.h"


static int supportedSimpleType(argDef *);
static int supportedComplexType(argDef *);
static int sameArgType(argDef *,argDef *);
static int sameScopedName(scopedNameDef *,scopedNameDef *);
static void classIsUsed(sipSpec *,moduleDef *,classVersDef *,argDef *,int);
static void scopeDefaultValue(classVersDef *,argDef *);
static void markAsUsed(classDef *);
static void expandClassVersions(sipSpec *,classDef *);
static void getSupersVersions(sipSpec *,classVersDef **,classVersList **,versionAnded *,classVersDef *,int,versionQual *);
static void setHierachy(classVersDef *,classVersList *);
static classVersDef *newClassVersion(classVersDef *,versionAnded *,classVersList *);
static void adjustVersionRange(versionAnded *,versionAnded *);
static void appendToClassVersList(classVersList **,classVersDef *);
static void freeClassVersList(classVersList *);
static void transformVariableList(sipSpec *,classVersDef *,varDef **);
static void getVarTypeVersions(sipSpec *,varDef **,argDef *,versionAnded *,varDef *,int,versionQual *,classVersDef *);
static varDef *newVarVersion(varDef *,versionAnded *,argDef *);
static void transformOverloads(sipSpec *,classVersDef *,overDef **);
static void getOverArgsVersions(sipSpec *,overDef ***,argDef *,versionAnded *,overDef *,int,versionQual *,classVersDef *);
static overDef *newOverVersion(overDef *,versionAnded *,argDef *);
static void transformCtors(sipSpec *,classVersDef *);
static void getCtorArgsVersions(sipSpec *,ctorDef ***,argDef *,versionAnded *,ctorDef *,int,versionQual *,classVersDef *);
static ctorDef *newCtorVersion(ctorDef *,versionAnded *,argDef *);
static void getVisibleMembers(classVersDef *);
static void getVirtuals(classVersDef *,int,versionQual *);
static void getVirtualsVersions(virtOverDef **,classVersList *,versionAnded *);
static void addSignalSignatures(sipSpec *,classVersDef *);
static void addSignalArgs(sipSpec *,funcArgs *,versionAnded *);
static void checkArgumentTypes(overDef *,classVersDef *);
static void checkVariableType(classVersDef *,varDef *);
static void fatalNoDefinedType(sipSpec *,scopedNameDef *,versionAnded *);
static void fatalScopedName(char *,char *);
static void getBaseType(sipSpec *,classVersDef *,argDef *,versionAnded *,argDef *,versionAnded **);
static void searchScope(classVersDef *,char *,versionAnded *,argDef *,versionAnded **);
static void searchTypedefs(typedefDef *,char *,versionAnded *,argDef *,versionAnded **);
static void searchEnums(enumDef *,char *,versionAnded *,argDef *,versionAnded **);
static void searchClasses(classDef *,char *,versionAnded *,argDef *,versionAnded **);


/*
 * Transform the parse tree.
 */

void transform(sipSpec *pt)
{
	int classnr;
	classDef *cd;
	classList *cl;

	if (pt -> module -> name == NULL)
		fatal("No %%Module has been specified for the module\n");

	if (pt -> cppmname == NULL)
		pt -> cppmname = pt -> module -> name -> text;

	/* The simple class checks. */

	classnr = 0;

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		/* Check it has been defined. */

		if (cd -> cvds == NULL)
			fatal("Class %s hasn't been defined\n",cd -> name -> text);

		/* Allocate a class number if it is in the main module. */

		if (cd -> module == pt -> module)
		{
			cd -> classnr = classnr++;

			/*
			 * If we find a class defined in this module called
			 * QObject, assume it's Qt.
			 */

			if (strcmp(cd -> name -> text,"QObject") == 0)
				pt -> qobjclass = cd -> classnr;
		}

		/* For the next loop. */

		resetWalk(cd);
	}

	/*
	 * Create additional class versions dependent on what's in a class's
	 * hierachy.
	 */

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
		expandClassVersions(pt,cd);

	/* Transform global variables and functions. */

	transformOverloads(pt,NULL,&pt -> overs);
	transformVariableList(pt,NULL,&pt -> globals);

	/* Transform class variables and functions. */

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		classVersDef *cvd;

		for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
		{
			transformCtors(pt,cvd);
			transformOverloads(pt,cvd,&cvd -> overs);
			transformVariableList(pt,cvd,&cvd -> vars);
		}
	}

	/* Generate the different class views. */

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		classVersDef *cvd;

		for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
		{
			int v;

			/* Get the list of visible member functions. */

			getVisibleMembers(cvd);

			if (cvd -> cd -> module == pt -> module)
				addSignalSignatures(pt,cvd);

			/* Get the virtual members. */

			cvd -> vmembers = NULL;

			for (v = cvd -> version.lowidx; v < cvd -> version.uppidx; ++v)
				if (cvd -> version.secondary != NULL)
					getVirtuals(cvd,v,cvd -> version.secondary);
				else
				{
					versionQual *vq;

					/* Step through each secondary. */

					for (vq = pt -> versions; vq != NULL; vq = vq -> next)
						if (vq -> order < 0)
							getVirtuals(cvd,v,vq);

					/*
					 * If the module doesn't define any
					 * secondaries just use the primary.
					 */

					if (pt -> nrsecs == 0)
						getVirtuals(cvd,v,NULL);
				}
		}
	}

	/* Work out which class header files need to be generated. */

	for (cl = pt -> used; cl != NULL; cl = cl -> next)
		markAsUsed(cl -> c);

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
		if (cd -> module == pt -> module)
			markAsUsed(cd);
}


/*
 * Mark a class and all the classes used by it and its class hierachy as being
 * used.
 */

static void markAsUsed(classDef *cd)
{
	classVersDef *cvd;

	/* Don't bother if we have already visited this class. */

	if (isUsed(cd))
		return;

	setIsUsed(cd);

	for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
	{
		classVersList *cvl;

		for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
		{
			classList *cl;

			for (cl = cvl -> cv -> cd -> used; cl != NULL; cl = cl -> next)
				markAsUsed(cl -> c);
		}
	}
}


/*
 * Expand the versions of a class and set up the class hierachy for each of
 * those versions.
 */

static void expandClassVersions(sipSpec *pt,classDef *cd)
{
	classVersDef *cvd, *newcvds;

	if (isWalkDone(cd))
		return;

	if (isWalkStarted(cd))
		fatal("Class %s appears in it's own super-class hierachy\n",cd -> name -> text);

	setWalkStarted(cd);

	/* Go through each current version. */

	newcvds = NULL;

	for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
	{
		int v;
		classList *cl;
		classVersDef *newcvd;
		classVersList *prevscvl;
		versionAnded prevva;

		/* Make sure the super-classes are done. */

		for (cl = cvd -> supers; cl != NULL; cl = cl -> next)
			expandClassVersions(pt,cl -> c);

		/* Remember the previous version. */

		prevscvl = NULL;

		/* Step through the timeline. */

		prevva = cvd -> version;

		for (v = cvd -> version.lowidx; v < cvd -> version.uppidx; ++v)
		{
			/*
			 * If the class has a secondary then we are only
			 * interested in super-classes with the same secondary.
			 */

			if (cvd -> version.secondary != NULL)
				getSupersVersions(pt,&newcvds,&prevscvl,&prevva,cvd,v,cvd -> version.secondary);
			else
			{
				versionQual *vq;

				/* Step through each secondary. */

				for (vq = pt -> versions; vq != NULL; vq = vq -> next)
					if (vq -> order < 0)
						getSupersVersions(pt,&newcvds,&prevscvl,&prevva,cvd,v,vq);

				/*
				 * If the module doesn't define any secondaries
				 * just use the primary.
				 */

				if (pt -> nrsecs == 0)
					getSupersVersions(pt,&newcvds,&prevscvl,&prevva,cvd,v,NULL);
			}
		}

		/* The last timeslice still needs doing. */

		newcvd = newClassVersion(cvd,&prevva,prevscvl);

		newcvd -> next = newcvds;
		newcvds = newcvd;
	}

	/* Replace the old list with the new one. */

	cd -> cvds = newcvds;

	setWalkDone(cd);
}


/*
 * Add a unique list of specific class versions for a class's super-class that
 * match the given class.
 */

static void getSupersVersions(sipSpec *pt,classVersDef **newheadp,classVersList **prevscvl,versionAnded *prevva,classVersDef *cvd,int v,versionQual *vq)
{
	int nosec, match;
	versionAnded va;
	classList *cl;
	classVersList *head, *prev;

	/* Set up the specific condition we are interested in. */

	va.lowidx = v;
	va.uppidx = v + 1;
	va.secondary = vq;

	/*
	 * Go through each super-class.  While we do so, see if there is a
	 * dependency on a secondary, and if this is different to the previous
	 * super-classes.
	 */

	nosec = TRUE;
	match = TRUE;

	prev = *prevscvl;
	head = NULL;

	for (cl = cvd -> supers; cl != NULL; cl = cl -> next)
	{
		classVersDef *scvd;

		for (scvd = cl -> c -> cvds; scvd != NULL; scvd = scvd -> next)
			if (versionIsSubset(&va,&scvd -> version))
				break;

		if (scvd == NULL)
		{
			fatal("Super-class %s isn't defined for class %s",cl -> c -> name -> text,classVersName(cvd));
			fatalVersion(pt,&cvd -> version);
			fatal("\n");
		}

		if (scvd -> version.secondary != NULL)
			nosec = FALSE;

		appendToClassVersList(&head,scvd);

		/* Compare with any previous row. */

		if (prev != NULL)
		{
			if (prev -> cv != scvd)
				match = FALSE;

			prev = prev -> next;
		}
	}

	/*
	 * If there was a match with the previous row, extend it to cover this
	 * one.
	 */

	if (match)
	{
		prevva -> uppidx = v + 1;

		/* If there isn't a previous one we will need this later. */

		if (*prevscvl == NULL)
			*prevscvl = head;
		else
			freeClassVersList(head);
	}
	else
	{
		classVersDef *newcvd;

		/* Create a new version based on the previous one. */

		newcvd = newClassVersion(cvd,prevva,*prevscvl);

		newcvd -> next = *newheadp;
		*newheadp = newcvd;

		/*
		 * Make the current one unconditional on any secondary if all
		 * the super-classes are unconditional and the original was
		 * unconditional on a secondary.
		 */

		if (nosec && cvd -> version.secondary == NULL)
			va.secondary = NULL;

		/* Remember the current one as the previous one. */

		*prevva = va;
		*prevscvl = head;
	}
}


/*
 * Given a list of super-class versions, set the complete hierachy for a class
 * version.
 */

static void setHierachy(classVersDef *cvd,classVersList *cvl)
{
	/* The class is first in its own hierachy. */

	cvd -> hierachy = NULL;

	appendToClassVersList(&cvd -> hierachy,cvd);

	while (cvl != NULL)
	{
		classVersList *next, *scvl;

		/* Append the super-classes hierachy. */

		for (scvl = cvl -> cv -> hierachy; scvl != NULL; scvl = scvl -> next)
		{
			appendToClassVersList(&cvd -> hierachy,scvl -> cv);

			/* Ripple through the complex flag. */

			if (isComplex(scvl -> cv))
				setIsComplex(cvd);
		}

		/* Free the list as we go. */

		next = cvl -> next;
		free(cvl);
		cvl = next;
	}
}


/*
 * Create a new version of a class based on an existing version.
 */

static classVersDef *newClassVersion(classVersDef *ocvd,versionAnded *newva,classVersList *scvl)
{
	classVersDef *newcvd;
	enumDef *ed;
	typedefDef *td;
	varDef *vd;
	ctorDef *ct, **ctp;
	overDef *od, **odp;

	/* Create the new class version. */

	newcvd = sipMalloc(sizeof (classVersDef));

	*newcvd = *ocvd;

	/* Add the new version. */

	newcvd -> version = *newva;

	/*
	 * Now that we have the new complete version for this class, for those
	 * parts of it that have versions, only copy the ones relevant to this
	 * one.  Start with the enums.
	 */

	newcvd -> enums = NULL;

	for (ed = ocvd -> enums; ed != NULL; ed = ed -> next)
		if (versionsOverlap(newva,&ed -> version))
		{
			enumDef *ned;

			ned = sipMalloc(sizeof (enumDef));

			*ned = *ed;

			adjustVersionRange(&ned -> version,newva);

			ned -> cvd = newcvd;
			ned -> next = newcvd -> enums;

			newcvd -> enums = ned;
		}

	/* Now do the typedefs. */

	newcvd -> typedefs = NULL;

	for (td = ocvd -> typedefs; td != NULL; td = td -> next)
		if (versionsOverlap(newva,&td -> version))
		{
			typedefDef *ntd;

			ntd = sipMalloc(sizeof (typedefDef));

			*ntd = *td;

			adjustVersionRange(&ntd -> version,newva);

			ntd -> next = newcvd -> typedefs;

			newcvd -> typedefs = ntd;
		}

	/* Now do the variables. */

	newcvd -> vars = NULL;

	for (vd = ocvd -> vars; vd != NULL; vd = vd -> next)
		if (versionsOverlap(newva,&vd -> version))
		{
			varDef *nvd;

			nvd = sipMalloc(sizeof (varDef));

			*nvd = *vd;

			adjustVersionRange(&nvd -> version,newva);

			nvd -> next = newcvd -> vars;

			newcvd -> vars = nvd;
		}

	/* Now do the ctors retaining the order. */

	ctp = &newcvd -> ctors;

	for (ct = ocvd -> ctors; ct != NULL; ct = ct -> next)
		if (versionsOverlap(newva,&ct -> version))
		{
			ctorDef *nct;

			nct = sipMalloc(sizeof (ctorDef));

			*nct = *ct;

			adjustVersionRange(&nct -> version,newva);

			*ctp = nct;
			ctp = &nct -> next;
		}

	*ctp = NULL;

	/* Now do the overloads retaining the order. */

	odp = &newcvd -> overs;

	for (od = ocvd -> overs; od != NULL; od = od -> next)
		if (versionsOverlap(newva,&od -> version))
		{
			overDef *nod;

			nod = sipMalloc(sizeof (overDef));

			*nod = *od;

			adjustVersionRange(&nod -> version,newva);

			*odp = nod;
			odp = &nod -> next;

			/*
			 * The result is on the heap and so needs doing as
			 * well.
			 */

			if (od -> result != NULL)
			{
				nod -> result = sipMalloc(sizeof (argDef));

				*nod -> result = *od -> result;
			}
		}

	*odp = NULL;

	/* Finally set the hierachy. */

	setHierachy(newcvd,scvl);

	return newcvd;
}


/*
 * Adjust the range of a version so that it is a sub-set of another range.
 */

static void adjustVersionRange(versionAnded *va,versionAnded *range)
{
	if (va -> lowidx < range -> lowidx)
		va -> lowidx = range -> lowidx;

	if (va -> uppidx > range -> uppidx)
		va -> uppidx = range -> uppidx;
}


/*
 * Append a class version to a list of them and maintain a pointer to the
 * tail of the list.
 */

static void appendToClassVersList(classVersList **cvlp,classVersDef *cvd)
{
	classVersList *new;

	/* Find the end of the list. */

	while (*cvlp != NULL)
		cvlp = &(*cvlp) -> next;

	new = sipMalloc(sizeof (classVersList));

	new -> cv = cvd;
	new -> next = NULL;

	*cvlp = new;
}


/*
 * Free a list of class versions.
 */

static void freeClassVersList(classVersList *cvl)
{
	while (cvl != NULL)
	{
		classVersList *nxt;

		nxt = cvl -> next;
		free(cvl);
		cvl = nxt;
	}
}


/*
 * Return a list of visible member functions for a class version.
 */

static void getVisibleMembers(classVersDef *cvd)
{
	classVersList *cvl;

	cvd -> visible = NULL;

	for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
	{
		memberDef *md;

		for (md = cvl -> cv -> members; md != NULL; md = md -> next)
		{
			visibleList *vl;

			/*
			 * See if it is already in the list.  This has the
			 * desired side effect of eliminating any functions
			 * that have a version closer to this class in the
			 * hierachy.  This is the only reason to define private
			 * functions.
			 */

			for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
				if (vl -> m -> pyname == md -> pyname)
					break;

			/* See if it is a new member function. */

			if (vl == NULL)
			{
				vl = sipMalloc(sizeof (visibleList));

				vl -> m = md;
				vl -> cv = cvl -> cv;
				vl -> next = cvd -> visible;

				cvd -> visible = vl;
			}
		}
	}
}


/*
 * Get all the virtuals for a particular class version.
 */

static void getVirtuals(classVersDef *cvd,int v,versionQual *vq)
{
	versionAnded va;
	virtOverDef *head;

	/* Set the specific version we are interested in. */

	va.lowidx = v;
	va.uppidx = v + 1;
	va.secondary = vq;

	head = NULL;

	getVirtualsVersions(&head,cvd -> hierachy,&va);

	/* See if there were any for this version. */

	if (head != NULL)
	{
		virtVersDef *vvd;

		vvd = sipMalloc(sizeof (virtVersDef));
 
		vvd -> version = va;
		vvd -> vo = head;
		vvd -> next = cvd -> vmembers;
 
		cvd -> vmembers = vvd;
	}
}


/*
 * Get the list of visible virtual functions for a class.
 */
 
static void getVirtualsVersions(virtOverDef **headp,classVersList *cvl,versionAnded *va)
{
	classVersDef *cvd = cvl -> cv;
	overDef *od;

	/* Got through the next in the super-class hierachy. */

	for (od = cvd -> overs; od != NULL; od = od -> next)
	{
		virtOverDef *vod;

		if (!isVirtual(od) || !versionIsSubset(va,&od -> version))
			continue;

		/*
		 * See if a virtual of this name and signature is already in
		 * the list.
		 */
 
		for (vod = *headp; vod != NULL; vod = vod -> next)
			if (strcmp(vod -> o -> cppname,od -> cppname) == 0 && sameOverload(vod -> o,od))
				break;
 
		if (vod == NULL)
		{
			vod = sipMalloc(sizeof (virtOverDef));
 
			vod -> o = od;
			vod -> next = *headp;
 
			*headp = vod;
		}
 
		/*
		 * Set the far pointer, possibly overwriting one that wasn't so
		 * far.
		 */

		vod -> farc = cvd;
	}
 
	/* Now go through the next in the hierachy. */
 
	if (cvl -> next != NULL)
		getVirtualsVersions(headp,cvl -> next,va);
 
	/*
	 * Look for any members in this class that match any in the virtual
	 * list and set the "near" class.
	 */
 
	for (od = cvd -> overs; od != NULL; od = od -> next)
	{
		virtOverDef *vod;

		if (!versionIsSubset(va,&od -> version))
			continue;

		for (vod = *headp; vod != NULL; vod = vod -> next)
			if (strcmp(vod -> o -> cppname,od -> cppname) == 0 && sameOverload(vod -> o,od))
				break;
 
		if (vod != NULL)
		{
			/*
			 * Use this version of the overload to pick up its
			 * protected/private flags.
			 */

			vod -> o = od;
			vod -> nearc = cvd;
		}
	}
}


/*
 * Transform a list of variables.
 */

static void transformVariableList(sipSpec *pt,classVersDef *cvd,varDef **vdlist)
{
	varDef *vd;

	vd = *vdlist;

	/* Build up a new list. */

	*vdlist = NULL;

	while (vd != NULL)
	{
		int v;
		varDef *newvd;
		argDef prevad;
		versionAnded prevva;

		/* Remember the previous version. */

		prevad.atype = no_type;

		/* Step through the timeline. */

		prevva = vd -> version;

		for (v = vd -> version.lowidx; v < vd -> version.uppidx; ++v)
		{
			/*
			 * If the variable has a secondary then we are only
			 * interested in types with the same secondary.
			 */

			if (vd -> version.secondary != NULL)
				getVarTypeVersions(pt,vdlist,&prevad,&prevva,vd,v,vd -> version.secondary,cvd);
			else
			{
				versionQual *vq;

				/* Step through each secondary. */

				for (vq = pt -> versions; vq != NULL; vq = vq -> next)
					if (vq -> order < 0)
						getVarTypeVersions(pt,vdlist,&prevad,&prevva,vd,v,vq,cvd);

				/*
				 * If the module doesn't define any secondaries
				 * just use the primary.
				 */

				if (pt -> nrsecs == 0)
					getVarTypeVersions(pt,vdlist,&prevad,&prevva,vd,v,NULL,cvd);
			}
		}

		/* The last timeslice still needs doing. */

		newvd = newVarVersion(vd,&prevva,&prevad);

		newvd -> next = *vdlist;
		*vdlist = newvd;

		vd = vd -> next;
	}

	/* Now check the types. */

	for (vd = *vdlist; vd != NULL; vd = vd -> next)
	{
		checkVariableType(cvd,vd);
		classIsUsed(pt,vd -> module,cvd,&vd -> type,FALSE);

		if (vd -> type.atype != class_type && vd -> accessfunc != NULL)
		{
			fatal("Variable %s",vd -> name -> text);
			fatalVersion(pt,&vd -> version);
			fatal(" has %%VariableCode but isn't a class instance\n");
		}
	}
}


/*
 * Add a unique list of specific variable versions that match the given
 * version.
 */

static void getVarTypeVersions(sipSpec *pt,varDef **newheadp,argDef *prevad,versionAnded *prevva,varDef *vd,int v,versionQual *vq,classVersDef *scope)
{
	versionAnded va, *act;
	argDef ad;

	/* Set up the specific condition we are interested in. */

	va.lowidx = v;
	va.uppidx = v + 1;
	va.secondary = vq;

	/* Get the new type. */

	act = &vd -> version;

	getBaseType(pt,scope,&vd -> type,&va,&ad,&act);

	/*
	 * If there was a match with the previous one, extend it to cover this
	 * one.
	 */

	if (sameArgType(prevad,&ad))
		prevva -> uppidx = v + 1;
	else
	{
		/* See if there was a previous one. */

		if (prevad -> atype != no_type)
		{
			varDef *newvd;

			/* Create a new version based on the previous one. */

			newvd = newVarVersion(vd,prevva,prevad);

			newvd -> next = *newheadp;
			*newheadp = newvd;
		}
 
		/*
		 * Make the current one unconditional on any secondary if the
		 * type was unconditional and the original was unconditional on
		 * a secondary.
                 */
 
		if (act -> secondary == NULL && vd -> version.secondary == NULL)
			va.secondary = NULL;

		/* Remember the current one as the previous one. */

		*prevva = va;
		*prevad = ad;
	}
}


/*
 * Create a new version of a variable based on an existing version.
 */

static varDef *newVarVersion(varDef *ovd,versionAnded *newva,argDef *ntype)
{
	varDef *newvd;

	/* Create the new variable version. */

	newvd = sipMalloc(sizeof (varDef));

	*newvd = *ovd;

	newvd -> type = *ntype;

	/* Add the new version. */

	newvd -> version = *newva;

	return newvd;
}


/*
 * Check that the arguments to a member function are of types supported by the
 * generated code.
 */

static void checkArgumentTypes(overDef *od,classVersDef *cvd)
{
	char *cname;
	int a;

	cname = (cvd != NULL ? classVersName(cvd) : NULL);

	/* Results must be simple. */

	if (od -> cppcode == NULL && !isAbstract(od) && od -> result != NULL)
	{
		if (isSignal(od))
		{
			fatalScopedName(cname,od -> cppname);
			fatal("() - signals must return void\n");
		}

		if (!supportedSimpleType(od -> result))
		{
			fatalScopedName(cname,od -> cppname);
			fatal("() - unsupported member function return type - provide %%MemberCode\n");
		}
	}

	for (a = 0; a < od -> args.nrArgs; ++a)
	{
		if (isVirtual(od))
		{
			/* Arguments to virtual functions must be simple. */

			if (!supportedSimpleType(&od -> args.args[a]))
			{
				if (od -> cppcode == NULL && !isAbstract(od))
				{
					fatalScopedName(cname,od -> cppname);
					fatal("() - unsupported argument type for virtual functions - provide %%MemberCode\n");
				}

				if (od -> virtcode == NULL)
				{
					fatalScopedName(cname,od -> cppname);
					fatal("() - unsupported argument type for virtual functions - provide %%VirtualCode\n");
				}
			}
		}
		else if (od -> cppcode == NULL && !isAbstract(od))
		{
			if (isSignal(od) || isSlot(od))
			{
				/* Signal arguments must be simple. */

				if (!supportedSimpleType(&od -> args.args[a]))
				{
					fatalScopedName(cname,od -> cppname);
					fatal("() - unsupported argument type for signals and slots\n");
				}
			}
			else if (!supportedComplexType(&od -> args.args[a]))
			{
				fatalScopedName(cname,od -> cppname);
				fatal("() - unsupported member function argument type - provide %%MemberCode\n");
			}
		}
	}
}


/*
 * Check that a variable is a type supported by the generated code.
 */

static void checkVariableType(classVersDef *cvd,varDef *vd)
{
	argDef *ad = &vd -> type;

	switch (ad -> atype)
	{
	case class_type:
		/* Class, Class & and Class * are supported. */

		if (ad -> nrderefs <= 1)
			return;
		break;

	case ustring_type:
	case string_type:
		/* (unsigned) char, (unsigned) char * are supported. */

		if (!isReference(ad) && ad -> nrderefs <= 1)
			return;
		break;

	case float_type:
	case double_type:
	case enum_type:
	case bool_type:
	case ushort_type:
	case short_type:
	case uint_type:
	case int_type:
	case ulong_type:
	case long_type:
		/* These are supported without pointers or references. */

		if (!isReference(ad) && ad -> nrderefs == 0)
			return;
		break;

	case struct_type:
	case voidptr_type:
		/* A simple pointer is supported. */

		if (!isReference(ad) && ad -> nrderefs == 1)
			return;
		break;
	}

	/* Not supported. */

	fatal("Variable ");

	if (cvd != NULL)
		fatal("%s::",classVersName(cvd));

	fatal("%s has an unsupported type\n",vd -> name -> text);
}


/*
 * Transform a list of constructors.
 */

static void transformCtors(sipSpec *pt,classVersDef *cvd)
{
	int privateCopyCt;
	ctorDef *ct, *copyct, **tailp;
	classVersList *cvl;

	ct = cvd -> ctors;

	/* Build up a new list. */

	tailp = &cvd -> ctors;

	while (ct != NULL)
	{
		int v;
		ctorDef *newct;
		argDef prevad[MAX_NR_ARGS];
		versionAnded prevva;

		/* Remember the previous version. */

		prevad[0].atype = no_type;

		/* Step through the timeline. */

		prevva = ct -> version;

		for (v = ct -> version.lowidx; v < ct -> version.uppidx; ++v)
		{
			/*
			 * If the variable has a secondary then we are only
			 * interested in types with the same secondary.
			 */

			if (ct -> version.secondary != NULL)
				getCtorArgsVersions(pt,&tailp,prevad,&prevva,ct,v,ct -> version.secondary,cvd);
			else
			{
				versionQual *vq;

				/* Step through each secondary. */

				for (vq = pt -> versions; vq != NULL; vq = vq -> next)
					if (vq -> order < 0)
						getCtorArgsVersions(pt,&tailp,prevad,&prevva,ct,v,vq,cvd);

				/*
				 * If the module doesn't define any secondaries
				 * just use the primary.
				 */

				if (pt -> nrsecs == 0)
					getCtorArgsVersions(pt,&tailp,prevad,&prevva,ct,v,NULL,cvd);
			}
		}

		/* The last timeslice still needs doing. */

		newct = newCtorVersion(ct,&prevva,prevad);

		*tailp = newct;
		tailp = &newct -> next;

		ct = ct -> next;
	}

	*tailp = NULL;

	/* Check argument types. */

	for (ct = cvd -> ctors; ct != NULL; ct = ct -> next)
	{
		int a;

		for (a = 0; a < ct -> args.nrArgs; ++a)
		{
			classIsUsed(pt,cvd -> cd -> module,cvd,&ct -> args.args[a],FALSE);
			scopeDefaultValue(cvd,&ct -> args.args[a]);
		}

		if (ct -> cppcode == NULL)
		{
			int a;

			for (a = 0; a < ct -> args.nrArgs; ++a)
				if (!supportedComplexType(&ct -> args.args[a]))
				{
					fatalScopedName(classVersName(cvd),classVersName(cvd));
					fatal("() - unsupported ctor argument type - provide explicit code\n");
				}
		}
	}

	/* See if there is a private copy ctor in the hierachy. */

	copyct = NULL;
	privateCopyCt = FALSE;

	for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
	{
		for (ct = cvl -> cv -> ctors; ct != NULL; ct = ct -> next)
		{
			argDef *ad = &ct -> args.args[0];

			/* See if is a copy ctor. */

			if (ct -> args.nrArgs != 1 || ad -> nrderefs != 0 || !isReference(ad))
				continue;

			/*
			 * We might not have transformed this class yet so
			 * deal with classes and defined types.
			 */

			if (ad -> atype == class_type)
			{
				if (ad -> u.cvd != cvl -> cv)
					continue;
			}
			else if (ad -> atype == defined_type)
			{
				if (strcmp(ad -> u.snd -> name,cvl -> cv -> cd -> name -> text) != 0 || ad -> u.snd -> next != NULL)
					continue;
			}
			else
				continue;

			/*
			 * Remember if it's in the class we are dealing with.
			 */

			if (cvl == cvd -> hierachy)
				copyct = ct;

			/* Remember if it is private. */

			if (isPrivateCtor(ct))
				privateCopyCt = TRUE;

			break;
		}

		/* No need to look any further if we've found a private one. */

		if (privateCopyCt)
			break;
	}

	if (!privateCopyCt && !isOpaque(cvd) && copyct == NULL)
	{
		ctorDef **tailp;

		/* Create a default public copy ctor. */

		copyct = sipMalloc(sizeof (ctorDef));
 
		copyct -> args.nrArgs = 1;
		copyct -> args.args[0].atype = class_type;
		copyct -> args.args[0].u.cvd = cvd;
		copyct -> args.args[0].argflags = (SECT_IS_PUBLIC | ARG_IS_REF | ARG_IS_CONST);
		copyct -> args.args[0].nrderefs = 0;
		copyct -> args.args[0].defval = NULL;
 
		copyct -> cppcode = NULL;
		copyct -> version = cvd -> version;
		copyct -> next = NULL;
 
		/* Append it to the list. */

		for (tailp = &cvd -> ctors; *tailp != NULL; tailp = &(*tailp) -> next)
			;

		*tailp = copyct;
	}
}


/*
 * Add a unique list of specific ctor versions that match the given version.
 */

static void getCtorArgsVersions(sipSpec *pt,ctorDef ***tailpp,argDef *prevargs,versionAnded *prevva,ctorDef *ct,int v,versionQual *vq,classVersDef *scope)
{
	int a, nosec, match;
	versionAnded va;
	argDef args[MAX_NR_ARGS];

	/* Set up the specific condition we are interested in. */

	va.lowidx = v;
	va.uppidx = v + 1;
	va.secondary = vq;

	/*
	 * Go through each argument.  While we do so, see if this is the same
	 * as any previous version.
	 */

	nosec = TRUE;
	match = TRUE;

	for (a = 0; a < ct -> args.nrArgs; ++a)
	{
		versionAnded *act;

		act = &ct -> version;

		getBaseType(pt,scope,&ct -> args.args[a],&va,&args[a],&act);

		if (act -> secondary != NULL)
			nosec = FALSE;

		if (!sameArgType(&prevargs[a],&args[a]))
			match = FALSE;
	}

	/*
	 * If there was a match with the previous one, extend it to cover this
	 * one.
	 */

	if (match)
		prevva -> uppidx = v + 1;
	else
	{
		/* Check there is a previous version. */

		if (prevargs[0].atype != no_type)
		{
			ctorDef *newct;

			/* Create a new version based on the previous one. */

			newct = newCtorVersion(ct,prevva,prevargs);

			**tailpp = newct;
			*tailpp = &newct -> next;
		}

		/*
		 * Make the current one unconditional on any secondary if all
		 * the types were unconditional and the original was
		 * unconditional on a secondary.
                 */
 
		if (nosec && ct -> version.secondary == NULL)
			va.secondary = NULL;

		/* Remember the current one as the previous one. */

		*prevva = va;

		for (a = 0; a < ct -> args.nrArgs; ++a)
			prevargs[a] = args[a];
	}
}


/*
 * Add a new constructor for a particular combination of versioned types.
 */

static ctorDef *newCtorVersion(ctorDef *oct,versionAnded *newva,argDef *newargs)
{
	int a;
	ctorDef *newct;

	/* Create the new ctor version. */

	newct = sipMalloc(sizeof (ctorDef));

	*newct = *oct;

	/* Add the new version. */

	newct -> version = *newva;

	/* Overwrite with the real types. */

	for (a = 0; a < newct -> args.nrArgs; ++a)
		newct -> args.args[a] = *newargs++;

	return newct;
}


/*
 * Transform a list of overloads.
 */

static void transformOverloads(sipSpec *pt,classVersDef *cvd,overDef **overs)
{
	overDef *od, **tailp;

	od = *overs;

	/* Build up a new list. */

	tailp = overs;

	while (od != NULL)
	{
		int v;
		overDef *newod;
		argDef prevad[MAX_NR_ARGS + 1];
		versionAnded prevva;

		/* Remember the previous version. */

		prevad[0].atype = prevad[1].atype = no_type;

		/* Step through the timeline. */

		prevva = od -> version;

		for (v = od -> version.lowidx; v < od -> version.uppidx; ++v)
		{
			/*
			 * If the variable has a secondary then we are only
			 * interested in types with the same secondary.
			 */

			if (od -> version.secondary != NULL)
				getOverArgsVersions(pt,&tailp,prevad,&prevva,od,v,od -> version.secondary,cvd);
			else
			{
				versionQual *vq;

				/* Step through each secondary. */

				for (vq = pt -> versions; vq != NULL; vq = vq -> next)
					if (vq -> order < 0)
						getOverArgsVersions(pt,&tailp,prevad,&prevva,od,v,vq,cvd);

				/*
				 * If the module doesn't define any secondaries
				 * just use the primary.
				 */

				if (pt -> nrsecs == 0)
					getOverArgsVersions(pt,&tailp,prevad,&prevva,od,v,NULL,cvd);
			}
		}

		/* The last timeslice still needs doing. */

		newod = newOverVersion(od,&prevva,prevad);

		*tailp = newod;
		tailp = &newod -> next;

		od = od -> next;
	}

	*tailp = NULL;

	/* Check argument types. */

	for (od = *overs; od != NULL; od = od -> next)
	{
		int a;

		checkArgumentTypes(od,cvd);

		if (od -> result != NULL)
			classIsUsed(pt,od -> common -> module,cvd,od -> result,FALSE);

		for (a = 0; a < od -> args.nrArgs; ++a)
		{
			classIsUsed(pt,od -> common -> module,cvd,&od -> args.args[a],isSignal(od));

			if (cvd != NULL)
				scopeDefaultValue(cvd,&od -> args.args[a]);
		}
	}
}


/*
 * Add any signal signatures defined by a class to the list that the module's
 * proxy needs to deal with.
 */

static void addSignalSignatures(sipSpec *pt,classVersDef *cvd)
{
	visibleList *vl;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			int a;

			if (od -> common != vl -> m)
				continue;

			if (isSignal(od))
				addSignalArgs(pt,&od -> args,&od -> version);

			for (a = 0; a < od -> args.nrArgs; ++a)
				if (od -> args.args[a].atype == slotcon_type)
					addSignalArgs(pt,od -> args.args[a].u.sa,&od -> version);
		}
	}
}


/*
 * Add a set of arguments to the list of signals that might be emitted by this
 * module.
 */

static void addSignalArgs(sipSpec *pt,funcArgs *fa,versionAnded *va)
{
	funcArgsList *fl;

	for (fl = pt -> sigargs; fl != NULL; fl = fl -> next)
	{
		/*
		 * Note that there is a bug here.  If a module has a signal
		 * with the signature (char *,int) and another
		 * (SIP_ARRAY,SIP_ARRAY_LEN) then SIP thinks they are different
		 * but it will produce two signal handlers with the same C++
		 * signature.  There isn't such a case in PyQt or PyKDE (yet)
		 * so we are OK for now.
		 */

		if (sameFuncArgs(fa,fl -> fa))
			break;
	}
 
	/* Add if it is a new signature. */

	if (fl == NULL)
	{
		fl = sipMalloc(sizeof (funcArgsList));
 
		fl -> fa = fa;
		fl -> vol = orVersionQual(pt,NULL,va);
		fl -> next = pt -> sigargs;
 
		pt -> sigargs = fl;
	}
	else if (fl -> vol != NULL)
		fl -> vol = orVersionQual(pt,fl -> vol,va);

	reduceOredVersion(pt,&fl -> vol);
}


/*
 * Add a unique list of specific overload versions that match the given
 * version.
 */

static void getOverArgsVersions(sipSpec *pt,overDef ***tailpp,argDef *prevargs,versionAnded *prevva,overDef *od,int v,versionQual *vq,classVersDef *scope)
{
	int a, nosec, match;
	versionAnded va;
	argDef args[MAX_NR_ARGS + 1];

	/* Set up the specific condition we are interested in. */

	va.lowidx = v;
	va.uppidx = v + 1;
	va.secondary = vq;

	/* Assume that this is the same as the previous set of arguments. */

	nosec = TRUE;
	match = TRUE;

	/* Handle the result. */

	if (od -> result != NULL)
	{
		versionAnded *act;

		act = &od -> version;

		getBaseType(pt,scope,od -> result,&va,&args[0],&act);

		if (act -> secondary != NULL)
			nosec = FALSE;

		if (!sameArgType(&prevargs[0],&args[0]))
			match = FALSE;
	}

	/*
	 * Go through each argument.  While we do so, see if this is the same
	 * as any previous version.
	 */

	for (a = 0; a < od -> args.nrArgs; ++a)
	{
		versionAnded *act;

		act = &od -> version;

		getBaseType(pt,scope,&od -> args.args[a],&va,&args[a + 1],&act);

		if (act -> secondary != NULL)
			nosec = FALSE;

		if (!sameArgType(&prevargs[a + 1],&args[a + 1]))
			match = FALSE;
	}

	/*
	 * If there was a match with the previous one, extend it to cover this
	 * one.
	 */

	if (match)
		prevva -> uppidx = v + 1;
	else
	{
		/* Check there is a previous version. */

		if (prevargs[0].atype != no_type || prevargs[1].atype != no_type)
		{
			overDef *newod;

			/* Create a new version based on the previous one. */

			newod = newOverVersion(od,prevva,prevargs);

			**tailpp = newod;
			*tailpp = &newod -> next;
		}

		/*
		 * Make the current one unconditional on any secondary if all
		 * the types were unconditional and the original was
		 * unconditional on a secondary.
                 */
 
		if (nosec && od -> version.secondary == NULL)
			va.secondary = NULL;

		/* Remember the current one as the previous one. */

		*prevva = va;

		for (a = 0; a <= od -> args.nrArgs; ++a)
			prevargs[a] = args[a];
	}
}


/*
 * Reduce an or'ed list of versions to something that might be a simpler
 * expression.
 */

void reduceOredVersion(sipSpec *pt,versionOrList **volp)
{
#define	mapat(p,s)	map[(p) * nrcols + (s)]

	char *map;
	int p, s, s1, s2, needSec, nrrows, nrcols;
	versionOrList *vol;
	versionQual *vq;

	/* Handle the trivial case. */

	if (*volp == NULL)
		return;

	/* Create the map. */

	nrrows = pt -> latestversion;

	if ((nrcols = pt -> nrsecs) < 1)
		nrcols = 1;

	map = sipMalloc(nrrows * nrcols * sizeof (char));

	/* Clear the map of all possible conditions. */

	for (p = 0; p < nrrows; ++p)
		for (s = 0; s < nrcols; ++s)
			mapat(p,s) = FALSE;

	/* Add each condition to the map. */

	for (vol = *volp; vol != NULL; vol = vol -> next)
	{
		/* Find the range of secondaries. */

		if (vol -> va.secondary == NULL)
		{
			s1 = 0;
			s2 = nrcols;
		}
		else
		{
			s1 = 0;

			for (vq = pt -> versions; vq != NULL; vq = vq -> next)
				if (vq -> order < 0)
				{
					if (vq == vol -> va.secondary)
						break;

					++s1;
				}

			s2 = s1 + 1;
		}

		for (p = vol -> va.lowidx; p < vol -> va.uppidx; ++p)
			for (s = s1; s < s2; ++s)
				mapat(p,s) = TRUE;
	}

	/* See if the secondaries are relevant. */

	needSec = FALSE;

	for (p = 0; p < nrrows; ++p)
		for (s = 0; s < nrcols - 1; ++s)
			if (mapat(p,s) != mapat(p,s + 1))
			{
				needSec = TRUE;
				break;
			}

	s2 = (needSec ? pt -> nrsecs : 1);

	/* Create a new list. */

	*volp = NULL;

	for (s = 0; s < s2; ++s)
	{
		int low;

		/* Find the relevant secondary for this set of primaries. */

		if (needSec)
		{
			int secnr = 0;

			for (vq = pt -> versions; vq != NULL; vq = vq -> next)
				if (vq -> order < 0)
				{
					if (secnr == s)
						break;

					++secnr;
				}
		}
		else
			vq = NULL;

		/* Start looking for contiguous blocks in the map. */

		low = 0;

		do
		{
			/* Find the start. */

			while (low < pt -> latestversion && !mapat(low,s))
				++low;

			if (low < pt -> latestversion)
			{
				int upp;

				/* Find the end. */

				upp = low + 1;

				while (upp < pt -> latestversion && mapat(upp,s))
					++upp;

				/*
				 * Add the new condition to the list if it is
				 * not unconditional.
				 */

				if (low > 0 || upp < pt -> latestversion || vq != NULL)
				{
					vol = sipMalloc(sizeof (versionOrList));

					vol -> va.lowidx = low;
					vol -> va.uppidx = upp;
					vol -> va.secondary = vq;

					vol -> next = *volp;
					*volp = vol;
				}

				/* Start the next block. */

				low = upp + 1;
			}
		}
		while (low < pt -> latestversion);
	}

	free(map);
}


/*
 * Add a version qualifier to an or'ed list of them.
 */

versionOrList *orVersionQual(sipSpec *pt,versionOrList *list,versionAnded *va)
{
	versionOrList *vol;

	/*
	 * If the new version is unconditional then return an empty list, ie.
	 * always true.
	 */

	if (va -> lowidx == 0 && va -> uppidx == pt -> latestversion && va -> secondary == NULL)
		return NULL;

	/* Leave any more optimizations until later. */

	vol = sipMalloc(sizeof (versionOrList));

	vol -> va = *va;
	vol -> next = list;

	return vol;
}


/*
 * Add a new overload for a particular combination of versioned types.
 */

static overDef *newOverVersion(overDef *ood,versionAnded *newva,argDef *newargs)
{
	int a;
	overDef *newod;

	/* Create the new overload version. */

	newod = sipMalloc(sizeof (overDef));

	*newod = *ood;

	/* Add the new version. */

	newod -> version = *newva;

	/* Handle the result. */

	if (newod -> result != NULL)
	{
		newod -> result = sipMalloc(sizeof (argDef));
		*newod -> result = *newargs;
	}

	/* Overwrite with the real types. */

	for (a = 0; a < newod -> args.nrArgs; ++a)
		newod -> args.args[a] = *++newargs;

	return newod;
}


/*
 * See if a simple type is supported by the generated code.  Simple types are
 * typically those that we can deal with when getting them from C++ code.
 */

static int supportedSimpleType(argDef *ad)
{
	/*
	 * In addition to complex types, we can't support references to
	 * anything other than classes.
	 */

	return (supportedComplexType(ad) &&
		(ad -> atype == class_type || !isReference(ad)));
}


/*
 * See if a complex type is supported by the generated code.  Complex types are
 * typically those that we can deal with when passing them to C++ code.
 */

static int supportedComplexType(argDef *ad)
{
	switch (ad -> atype)
	{
	case rxcon_type:
	case rxdis_type:
	case slotcon_type:
	case slotdis_type:
	case array_type:
	case uarray_type:
	case arraysize_type:
	case arrayusize_type:
		/* These can only appear in argument lists without * or &. */

		return TRUE;

	case ustring_type:
	case string_type:
		/* (unsigned) char and (unsigned) char * are supported. */

		if (!isReference(ad) && ad -> nrderefs <= 1)
			return TRUE;
		break;

	case float_type:
	case double_type:
	case enum_type:
	case bool_type:
	case ushort_type:
	case short_type:
	case uint_type:
	case int_type:
	case ulong_type:
	case long_type:
		/* These are supported without pointers or references. */

		if (!isReference(ad) && ad -> nrderefs == 0)
			return TRUE;
		break;

	case class_type:
		/* Class, Class & and Class * are supported. */

		if ((isReference(ad) && ad -> nrderefs == 0) || (!isReference(ad) && ad -> nrderefs <= 1))
			return TRUE;
		break;

	case struct_type:
	case voidptr_type:
		/*
		 * A simple pointer is supported (although we could easily
		 * handle greater indirections.
		 */

		if (!isReference(ad) && ad -> nrderefs == 1)
			return TRUE;
		break;
	}

	/* Not supported. */

	return FALSE;
}


/*
 * Put a scoped name to stderr.
 */

static void fatalScopedName(char *cname,char *fname)
{
	if (cname != NULL)
		fatal("%s::%s",cname,fname);
	else
		fatal("%s",fname);
}


/*
 * Compare two overloads and return TRUE if they are the same as far as C++ is
 * concerned.
 */

int sameOverload(overDef *od1,overDef *od2)
{
	/* They must both be const, or both not. */

	if (isConst(od1) != isConst(od2))
		return FALSE;

	/* The return type must be the same. */

	if (od1 -> result == NULL && od2 -> result != NULL)
		return FALSE;

	if (od1 -> result != NULL && od2 -> result == NULL)
		return FALSE;

	if (od1 -> result != NULL && od2 -> result != NULL && !sameArgType(od1 -> result,od2 -> result))
		return FALSE;

	return sameFuncArgs(&od1 -> args,&od2 -> args);
}


/*
 * Compare two sets of function arguments are return TRUE if they are the same.
 */

int sameFuncArgs(funcArgs *fa1,funcArgs *fa2)
{
	int a;

	/* The number of arguments must be the same. */

	if (fa1 -> nrArgs != fa2 -> nrArgs)
		return FALSE;

	/* The arguments must be the same. */

	for (a = 0; a < fa1 -> nrArgs; ++a)
		if (!sameArgType(&fa1 -> args[a],&fa2 -> args[a]))
			return FALSE;

	/* Must be the same if we've got this far. */

	return TRUE;
}


/*
 * Compare two argument types are return TRUE if they are the same.
 */

static int sameArgType(argDef *a1,argDef *a2)
{
	/* The type and references must be the same. */

	if (a1 -> atype != a2 -> atype || isReference(a1) != isReference(a2) || a1 -> nrderefs != a2 -> nrderefs)
		return FALSE;

	switch (a1 -> atype)
	{
	case class_type:
		if (a1 -> u.cvd != a2 -> u.cvd)
			return FALSE;

		break;

	case enum_type:
		if (a1 -> u.ed != a2 -> u.ed)
			return FALSE;

		break;

	case slotcon_type:
	case slotdis_type:
		if (!sameFuncArgs(a1 -> u.sa,a2 -> u.sa))
			return FALSE;

		break;

	case template_type:
		if (strcmp(a1 -> u.td -> name,a2 -> u.td -> name) != 0 ||
		    !sameArgType(&a1 -> u.td -> type,&a2 -> u.td -> type))
			return FALSE;

		break;

	case struct_type:
		if (strcmp(a1 -> u.sname,a2 -> u.sname) != 0)
			return FALSE;

		break;

	case defined_type:
		if (!sameScopedName(a1 -> u.snd,a2 -> u.snd))
			return FALSE;

		break;
	}

	/* Must be the same if we've got this far. */

	return TRUE;
}


/*
 * Return TRUE if two scoped names are the same.
 */

static int sameScopedName(scopedNameDef *snd1,scopedNameDef *snd2)
{
	while (snd1 != NULL && snd2 != NULL && strcmp(snd1 -> name,snd2 -> name) == 0)
	{
		snd1 = snd1 -> next;
		snd2 = snd2 -> next;
	}

	return (snd1 == NULL && snd2 == NULL);
}


/*
 * Print an optional version to stderr.
 */

void fatalVersion(sipSpec *pt,versionAnded *va)
{
	versionQual *lower, *upper;

	lower = getVersionQual(pt,va -> lowidx);
	upper = getVersionQual(pt,va -> uppidx);

	if (lower != NULL || upper != NULL || va -> secondary != NULL)
	{
		int doneDash = FALSE, doAnd = FALSE;

		fatal(" (Version:");

		if (lower != NULL)
		{
			fatal(" %s -",lower -> name);

			doneDash = TRUE;
			doAnd = TRUE;
		}

		if (upper != NULL)
		{
			fatal(" %s%s",(doneDash ? "" : "- "),upper -> name);

			doAnd = TRUE;
		}

		if (va -> secondary != NULL)
			fatal(" %s%s",(doAnd ? "&& " : ""),va -> secondary -> name);

		fatal(")");
	}
}


/*
 * Add an explicit scope to the default value of an argument if possible.
 */

static void scopeDefaultValue(classVersDef *cvd,argDef *ad)
{
	valueDef *vd, **tailp;

	/*
	 * We do a quick check to see if we need to do anything.  This means
	 * we can limit the times we need to copy the default value.  It needs
	 * to be copied because it will be shared by class versions that have
	 * been created on the fly and it may need to be scoped differently for
	 * each of those versions.
	 */

	for (vd = ad -> defval; vd != NULL; vd = vd -> next)
		if (vd -> vtype == scoped_value && vd -> u.vscp -> next == NULL)
			break;

	if (vd == NULL)
		return;

	/*
	 * It's not certain that we will do anything, but we assume we will and
	 * start copying.
	 */

	tailp = &ad -> defval;

	for (vd = ad -> defval; vd != NULL; vd = vd -> next)
	{
		classVersList *cvl;
		scopedNameDef *origname;

		/* Make the copy. */

		*tailp = sipMalloc(sizeof (valueDef));

		**tailp = *vd;
		vd = *tailp;
		tailp = &vd -> next;

		/*
		 * Skip this part of the expression if it isn't a named value
		 * or it already has a scope.
		 */

		if (vd -> vtype != scoped_value || vd -> u.vscp -> next != NULL)
			continue;

		/*
		 * Search the class hierachy for an enum value with the same
		 * name.  If we don't find one, leave it as it is (the compiler
		 * will find out if this is a problem).
		 */

		origname = vd -> u.vscp;

		for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
		{
			enumDef *ed;

			for (ed = cvl -> cv -> enums; ed != NULL; ed = ed -> next)
			{
				enumValueDef *evd;

				for (evd = ed -> values; evd != NULL; evd = evd -> next)
					if (strcmp(evd -> name -> text,origname -> name) == 0)
					{
						scopedNameDef *snd;

						/*
						 * Take the scope from the
						 * class that the enum was
						 * defined in.
						 */

						snd = sipMalloc(sizeof (scopedNameDef));

						snd -> name = cvl -> cv -> cd -> name -> text;
						snd -> next = origname;

						vd -> u.vscp = snd;

						/* Nothing more to do. */

						break;
					}

				if (evd != NULL)
					break;
			}

			if (ed != NULL)
				break;
		}
	}
}


/*
 * Make sure the version of a type is a base type.
 */

static void getBaseType(sipSpec *pt,classVersDef *defscope,argDef *otype,versionAnded *va,argDef *ntype,versionAnded **act)
{
	*ntype = *otype;

	/* Loop until we've got to a base type. */

	while (ntype -> atype == defined_type)
	{
		scopedNameDef *snd;

		snd = ntype -> u.snd;

		ntype -> atype = no_type;

		if (snd -> next == NULL)
		{
			char *basename;

			basename = snd -> name;

			if (defscope != NULL)
				searchScope(defscope,basename,va,ntype,act);

			if (ntype -> atype == no_type)
				searchTypedefs(pt -> typedefs,basename,va,ntype,act);

			if (ntype -> atype == no_type)
				searchEnums(pt -> enums,basename,va,ntype,act);

			if (ntype -> atype == no_type)
				searchClasses(pt -> classes,basename,va,ntype,act);
		}
		else if (snd -> next -> next == NULL)
		{
			char *basename;
			classDef *cd;
			classVersDef *scope;

			basename = snd -> next -> name;

			/* Find the class that is the scope. */

			scope = NULL;

			for (cd = pt -> classes; cd != NULL; cd = cd -> next)
				if (strcmp(cd -> name -> text,snd -> name) == 0)
				{
					classVersDef *cvd;

					for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
						if (versionIsSubset(va,&cvd -> version))
						{
							scope = cvd;
							break;
						}

					break;
				}

			if (scope == NULL)
				fatalNoDefinedType(pt,snd,va);

			searchScope(scope,basename,va,ntype,act);
		}
		else
		{
			fatalScopedName(snd -> name,snd -> next -> name);
			fatal("...");
			fatalVersion(pt,va);
			fatal(" has too many levels of scope\n");
		}

		if (ntype -> atype == no_type)
			fatalNoDefinedType(pt,snd,va);
	}

	/* Get the base of type of any slot arguments. */

	if (ntype -> atype == slotcon_type || ntype -> atype == slotdis_type)
	{
		int sa;
		funcArgs *fa = ntype -> u.sa;

		for (sa = 0; sa < fa -> nrArgs; ++sa)
		{
			argDef satype;

			getBaseType(pt,defscope,&fa -> args[sa],va,&satype,act);
			fa -> args[sa] = satype;
		}
	}
}


/*
 * Search for a versioned name in a scope and return the corresponding type.
 */

static void searchScope(classVersDef *scope,char *name,versionAnded *va,argDef *ad,versionAnded **act)
{
	classVersList *cvl;

	for (cvl = scope -> hierachy; cvl != NULL; cvl = cvl -> next)
	{
		searchTypedefs(cvl -> cv -> typedefs,name,va,ad,act);

		if (ad -> atype != no_type)
			break;

		searchEnums(cvl -> cv -> enums,name,va,ad,act);

		if (ad -> atype != no_type)
			break;
	}
}


/*
 * Search a list of typedefs for a versioned name and return the type.
 */

static void searchTypedefs(typedefDef *td,char *name,versionAnded *va,argDef *ad,versionAnded **act)
{
	while (td != NULL)
	{
		if (strcmp(td -> name,name) == 0 && versionIsSubset(va,&td -> version))
		{
			/* Copy the type. */

			ad -> atype = td -> type.atype;
			ad -> argflags |= td -> type.argflags;
			ad -> nrderefs += td -> type.nrderefs;
			ad -> u = td -> type.u;

			*act = &td -> version;

			break;
		}

		td = td -> next;
	}
}


/*
 * Search a list of enums for a versioned name and return the type.
 */

static void searchEnums(enumDef *ed,char *name,versionAnded *va,argDef *ad,versionAnded **act)
{
	while (ed != NULL)
	{
		if (ed -> name != NULL && strcmp(ed -> name,name) == 0 && versionIsSubset(va,&ed -> version))
		{
			ad -> atype = enum_type;
			ad -> u.ed = ed;

			*act = &ed -> version;

			break;
		}

		ed = ed -> next;
	}
}


/*
 * Search a list of classes for a versioned name and return the type.
 */

static void searchClasses(classDef *cd,char *name,versionAnded *va,argDef *ad,versionAnded **act)
{
	while (cd != NULL)
	{
		if (strcmp(cd -> name -> text,name) == 0)
		{
			classVersDef *cvd;

			for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
				if (versionIsSubset(va,&cvd -> version))
				{
					ad -> atype = class_type;
					ad -> u.cvd = cvd;

					*act = &cvd -> version;

					return;
				}
		}

		cd = cd -> next;
	}
}


/*
 * Print an error message describing an undefined defined type to stderr and
 * terminate.
 */

static void fatalNoDefinedType(sipSpec *pt,scopedNameDef *snd,versionAnded *va)
{
	char *scope, *name;

	if (snd -> next != NULL)
	{
		scope = snd -> name;
		name = snd -> next -> name;
	}
	else
	{
		scope = NULL;
		name = snd -> name;
	}

	fatalScopedName(scope,name);
	fatalVersion(pt,va);
	fatal(" is undefined\n");
}


/*
 * If a type is a class then add it to the appropriate list of used classes so
 * that the header file is #included in the generated code.
 */

static void classIsUsed(sipSpec *pt,moduleDef *mod,classVersDef *cvd,argDef *ad,int issig)
{
	/* Ignore non-classes. */

	if (ad -> atype != class_type)
		return;

	if (cvd != NULL && cvd != ad -> u.cvd)
		addToUsedList(&cvd -> cd -> used,ad -> u.cvd -> cd);

	/*
	 * It's used by the main module if it is outside a class or it is a
	 * signal.
	 */

	if (mod == pt -> module && (cvd == NULL || issig))
		addToUsedList(&pt -> used,ad -> u.cvd -> cd);
}


/*
 * Return the version qualifier for a particular primary index.
 */

versionQual *getVersionQual(sipSpec *pt,int idx)
{
	versionQual *vq;

	for (vq = pt -> versions; vq != NULL; vq = vq -> next)
		if (vq -> order == idx)
			break;

	return vq;
}
