#include "fifo.h"
#include "../../module_registry.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "../../common.h"

DataSetMap *Fifo::moduleInfo = 0;

Module* fifo_constructor()
{
        return new Fifo();
}

const DataSetMap& Fifo::get_module_info_instance()
{
        if (!moduleInfo) {
                moduleInfo = new DataSetMap();
                moduleInfo->set("description", DataSet("Create and monitor a named pipe (FIFO)"));
                moduleInfo->set("supported_vars", DataSet("fifoname"));
                moduleInfo->set("supported_var_types", DataSet("string"));
                moduleInfo->set("supported_var_defaults", DataSet("fifo1"));
                moduleInfo->set("overridable", DataSet("false"));
                moduleInfo->set("output_variables", DataSet("textline"));
        }
        return *moduleInfo;
}

extern "C" int fifo_plugin_startup()
{
        ModuleRegistry::instance()->add_module("Fifo", &fifo_constructor, Fifo::get_module_info_instance());
        return 0;
}

extern "C" void fifo_plugin_shutdown()
{
        ModuleRegistry::instance()->remove_module("Fifo");
        Fifo::destroy_module_info();
}

Fifo::Fifo()
        : file(0),
          fifoIsOpen(false)
{
}

Fifo::~Fifo()
{
        if (fifoIsOpen)
                closeFifo();
        rpdbgmsg(getName() << " module destroyed");
}

void Fifo::service()
{
        Module::service();

        if (!fifoIsOpen)
                openFifo();

        if (fifoIsOpen) {
                // attempt to read a line from the fifo
                string text;
                int bytesRead = 0;
                int res;
                do {
                        char ch;
                        res = read(file, &ch, 1);
                        if (ch == 10 || ch == 13)
                                res = 0;
                        if (res > 0) {
                                text += ch;
                                bytesRead += res;
                        }
                } while (res > 0);
                
                if (text.length() > 0) {
                        DataSet msg;
                        msg.addString(text);
                        updateParent("textline", msg);
                }                
        }
}

void Fifo::openFifo()
{
        assert(!fifoIsOpen);

        if (filename.length() > 0) {
                if (verbose)
                        cout << "opening fifo: \"" << filename << "\"" << endl;
                
                // if fifo doesn't exist, create it
                if (access(filename.c_str(), F_OK) == -1) {
                        if (mkfifo(filename.c_str(), 0777) != 0)
                                cerr << "error creating fifo:\"" << filename << "\"" << endl;
                }
                
                // if a file exists, but isn't a fifo, issue an error        
                
                // open the fifo for read (non-blocking)
                file = open(filename.c_str(), O_RDONLY | O_NONBLOCK);

                // we don't report errors here, since it is non-blocking,
                // so most of the time it is going to give us errors
                if (file != -1) {
                        fifoIsOpen = true;
                }
        }
}

void Fifo::closeFifo()
{
        assert(fifoIsOpen);
        close(file);
        fifoIsOpen = false;
}

void Fifo::updated(const string& keyName, const DataSet& data)
{
        if (keyName == "fifoname") {
                if (fifoIsOpen)
                        closeFifo();
                fifoname = data.getString();
                filename = getBaseDir() + "/" + fifoname;
        }
}
