/*
*  Rubrica
*  file: file.c
*  
*  Copyright (C) 2000-2002 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <gnome.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <time.h>

#include "rubrica_type.h"
#include "rubrica.h"
#include "card.h"
#include "form_int.h"
#include "main.h"
#include "file.h"
#include "export.h"
#include "utils.h"
#include "prefer.h"
#include "dictionary.h"
#include "dialog.h"
#include "ui.h"
#include "log.h"
#include "view.h"


xmlDocPtr 
rubrica_file_open_doc(gchar *filename)
{
  xmlDocPtr doc;

  g_return_val_if_fail(filename != NULL, FALSE);
  
  /* read and parse file 
   */
  doc = xmlParseFile(filename);

  if (!doc)
    {
      rubrica_dialog_error_loading_file(filename, RE_PARSE_XML_DOC);
      
      return NULL;
    }  

  return doc;
}


gboolean 
rubrica_file_is_rubrica_doc(xmlDocPtr doc, gchar *filename)
{
  g_return_val_if_fail(doc != NULL, FALSE);

  if (!doc->children || 
      !doc->children->name || 
      g_strcasecmp(doc->children->name, "Rubrica") != 0)
    {      
      rubrica_dialog_error_loading_file(filename, 
					RE_NO_RUBRICA_HEADER);   
      xmlFreeDoc(doc);

      return FALSE;
    }  

  return TRUE;
}


gint 
rubrica_file_get_fileformat(xmlDocPtr doc)
{
  if(!xmlHasProp(doc->children, "fileformat"))
    return rubrica_utils_str2int(NULL);

  return rubrica_utils_str2int(xmlGetProp(doc->children, "fileformat"));
}


gchar*
rubrica_file_get_version(xmlDocPtr doc)
{
  if(!xmlHasProp(doc->children, "version"))
    return NULL;

  return xmlGetProp(doc->children, "version");
}


gchar*
rubrica_file_get_document_tpye(xmlDocPtr doc)
{
  if(!xmlHasProp(doc->children, "doc_type"))
    return NULL;

  return xmlGetProp(doc->children, "doc_type");
}

gboolean rubrica_file_import_old_file(gchar* fname)
{
  gchar* msg;
  gchar* stylesheet;
  gchar* filename;
  
  g_print("\nImporting old rubrica document. Please wait...");
  g_print("\nThe document %s will be moved to file format #%d",
	  fname, RUBRICA_FILE_FORMAT);
  
  filename = g_strdup_printf("%s%s.tmpfile",
			     rubrica_preferences_get_rubrica_home(),
			     G_DIR_SEPARATOR_S);
  
  stylesheet = g_strdup_printf("%s%sstylesheet%srub2rub.xsl", 
			       RUBRICA_SCRIPT_DIR, 
			       G_DIR_SEPARATOR_S,
			       G_DIR_SEPARATOR_S); 
  
  if (!g_file_exists(stylesheet))
    {
      rubrica_dialog_error(_("Can't find the stylesheet."
			     "\nI can't read the old file."));
      
      g_free(stylesheet);
      g_free(filename);
      
      return FALSE;
    }        
  
  if(!rubrica_export_engine(fname, filename, stylesheet))
    {
      /* error dialog shown in rubrica_export_engine */
      g_free(stylesheet);
      g_free(filename);
      
      return FALSE;
    }
  else
    {
      gchar *original;

      original = g_strdup_printf("%s.SAVE", fname);
      if (-1 == rename(fname, original))
	{
	  msg = g_strdup_printf("\nrubrica_file_real_load_xmlfile ** "
				"renaming of file %s to %s failed", 
				fname, original);
	  perror(msg);	
	  g_free(msg);
	  
	  return FALSE;
	}
      g_free(original);
      
      remove(fname);
      if (-1 == rename(filename, fname))
	{
	  gchar *msg;
	  
	  msg = g_strdup_printf("\nrubrica_file_real_load_xmlfile ** "
				"renaming of file %s to %s failed", 
				filename, fname);
	  
	  perror(msg);
	  g_free(msg);
	  
	  return FALSE;
	}
      g_free(filename);
    }

  return TRUE;
}


/*  return:
    TRUE  if filename has extension .rub
    FALSE otherwise
*/
gboolean rubrica_file_file_has_rub_extension(gchar *filename)
{
  gchar *tmp=NULL, *buf;
  gint i, len;

  i = 0;
  buf = g_strdup(g_basename(filename));
  len = strlen(buf);

  while(*buf != '.')
    {
      buf++;
      i++;
            
      if (i == len)     /* end buffer was reached and dot wasn't found */
	return FALSE;	
    }

  tmp = (gchar*) g_malloc(((len - i)*sizeof(gchar))+1);
  strcpy(tmp, buf);
  
  if ((strcmp(tmp,".rub")) == 0)
    {
      g_free(tmp);
      return TRUE;
    }
  else 
    g_free(tmp);

  g_free(buf);
  return FALSE;  
}

void rubrica_file_do_backup(Rubrica *rubrica, gchar *oldname)
{
  gint err;
  gchar *newname;

  if (backup_is_increasing())
    {
      newname = g_strdup_printf("%s1", oldname);
      
      /* se esiste gi un file con lo stesso nome, crea una copia di backup 
	 if exist a file with same name, then do a backup of old file */
      if (g_file_exists(newname))
	{
	  gchar *path;
	  gchar *ext=NULL, *f=NULL;
	  gchar *bfile;
	  gint num;

	  path = g_dirname(newname);
	  rubrica_utils_file_split(newname, &f, &ext, &num);
	  
	  bfile = g_strdup_printf("%s/%s%d.%s", path, f, num, ext);
	  rubrica_file_do_backup(rubrica, bfile);

	  g_free(bfile);
	}
      else
	{
	  err = rename(oldname, newname);  
	  
	  if (!err)
	    {
	      gchar *str;

	      str = g_strdup_printf("\nBackup log ****"
				    "\n\told file: %s"
				    "\n\thas been saved as: %s", 
				    oldname, newname);
	      
	      rubrica_log_write_log(str);
	      g_free(str);
	    }
	      
	  else
	    {
	      gchar *str;

	      str = g_strdup_printf("\nBackup -- This error occurred "
				    "during backup\nold file name: %s\n"
				    "new file name: %s\nerror: %s", 
				    oldname, newname, strerror(errno)); 
	     
	      rubrica_log_write_log(str);
	      g_free(str);
	    }	      
	}
      
      g_free(newname);
    }
  else
    rubrica_file_do_bak_backup(rubrica, oldname);
}


void rubrica_file_do_bak_backup(Rubrica *rubrica, gchar *fname)
{
  gchar *ext=NULL, *file=NULL, *new;
  gchar *path;
  gint num, err;
  
  path = g_dirname(fname);
  rubrica_utils_file_split(fname, &file, &ext, &num);	  
  
  new = g_strdup_printf("%s/%s.bak", path, file);
  err = rename(fname, new);  
  
  if (!err)
    {
      gchar *str;
      
      str = g_strdup_printf("\nBackup log ****"
			    "\n\told file: %s"
			    "\n\thas been saved as: %s", fname, new);
      
      rubrica_log_write_log(str);
      g_free(str);
    }
  
  g_free(new);  
}



void rubrica_file_save(Rubrica *rubrica)
{
  GtkWidget *app;
  gchar *filename;
  gchar *tmp;

  app = rubrica_get_app();
  filename = rubrica_get_name(rubrica);

  /* 
     se non c' un nome file di default allora bisogna richiamare 
     salva_con_nome 
     if filename is NULL, then call rubrica_file_salva_come (ie save as)
  */
  if (!filename)
    {    
      rubrica_file_save_as(rubrica);
    }
  else 
    {
      gchar *name = NULL;      
      
      if (rubrica_preferences_get_have_default_folder())
	{      
	  gchar *path;
	  
	  path = rubrica_preferences_get_default_folder();
	  if (path)
	    {
	      name = g_strdup_printf("%s/%s", path, filename);
	      
	      g_free(path);
	    }
	  else
	    g_print("\nRubrica error ** Can't read default folder");
	}
      else
	name = g_strdup(filename);

      if (!rubrica_file_write_xmlfile(rubrica, name))  
	{      
	  tmp = g_strdup_printf(_("I can't save the file:\n%s"), name);
	  
	  rubrica_dialog_error(tmp);
	  g_free(tmp);
	}
      else  /* ok, file has been saved */
	{
	  rubrica_set_modified(rubrica, FALSE);
	  // TODO: messaggio nella status bar -- File Salvato con successo
	}
      
      g_free(name);
    }
}


void 
rubrica_file_save_as(Rubrica *rubrica)
{
  GtkWidget *app;
  GtkWidget *fs;
  gchar *filename; 
  
  app = rubrica_get_app();
  filename = rubrica_get_name(rubrica);

  fs = gtk_file_selection_new(_("Rubrica Save as"));
  g_object_set_data(G_OBJECT(fs), "rubrica", rubrica);

  if (rubrica_preferences_get_have_default_folder())
    {      
      gchar *path;
     
      path = rubrica_preferences_get_default_folder();
      if (path)
	{
	  gtk_file_selection_set_filename (GTK_FILE_SELECTION(fs), path);

	  g_free(path);
	}
    }
 
  /* se ho gi un nome di file lo mostro nell'entry */
  if (filename)
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), filename);
  
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked",
		   G_CALLBACK(rubrica_file_write_file), fs);
    
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			   "clicked", G_CALLBACK(gtk_widget_destroy), 
			   (gpointer) fs);
  
  gtk_window_position(GTK_WINDOW(fs), GTK_WIN_POS_CENTER);
  gtk_window_set_transient_for(GTK_WINDOW(fs), GTK_WINDOW(app));
  
  gtk_widget_show(fs);
}


void
rubrica_file_write_file(GtkWidget *ok_button, gpointer data)
{
  Rubrica *rubrica;
  GtkWidget *fs;
  gchar *tmp;
  gchar *fname = NULL;

  fs = (GtkWidget *)data;
  rubrica = g_object_get_data(G_OBJECT(fs), "rubrica");

  tmp = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));

  /* se il file non ha estensione o l'estensione non  .rub 
   */
  if (!rubrica_file_file_has_rub_extension(tmp)) 
    fname = g_strdup_printf("%s.rub",tmp);  
  else 
    fname = g_strdup(tmp);
  
  /* se non  stato selezionato nessun file
     o non  possibile aprire il file segnalare l'errore 
  */  
  if (!fname || !rubrica_file_write_xmlfile(rubrica, fname)) 
    {
      gchar *msg;
      
      if (fname)
	msg = g_strdup_printf(_("I can't save the file:\n%s"), fname); 
      else
	msg = g_strdup_printf(_("I can't save a file without name.")); 
      
      rubrica_dialog_error(msg);

      g_free(msg);
    }
  else /* il file  stato salvato */
    {      
      gchar *path  = NULL;
      gchar *newname = NULL;

      if (rubrica_preferences_get_have_default_folder())  
	path  = rubrica_preferences_get_default_folder();
      else
	path = g_dirname(fname);
      newname = g_strdup_printf("%s/%s", path, g_basename(fname));
          
      /* imposta il file appena salvato come l'ultimo aperto/visualizzato 
      */
      rubrica_preferences_set_last_book(newname);
      rubrica_set_path(rubrica, path);
      rubrica_set_name(rubrica, g_strdup(g_basename(fname)));
      rubrica_set_modified(rubrica, FALSE);
          
      g_free(newname);
      g_free(path);
    }

  gtk_widget_destroy(fs);
  g_free(fname);
}

gboolean
rubrica_file_write_xmlfile(Rubrica *rubrica, gchar *name)
{
  xmlDocPtr doc;     /* puntatore al documento xml */
  gint i, len; 
  gchar *path = NULL, *fname;
  gchar *fileformat;
  gchar *str;
  
  g_return_val_if_fail(name != NULL, FALSE);

  path  = g_dirname(name);
  fname = g_strdup(name);

  /* creazione di un nuovo documento xml 
   */
  doc = xmlNewDoc("1.0");
  
  /* creo un nuovo nodo radice rubrica 
   */
  fileformat = g_strdup_printf("%d", RUBRICA_FILE_FORMAT);

  doc->children = xmlNewDocNode(doc, NULL, "Rubrica", NULL);
  xmlSetProp(doc->children, "version", VERSION);
  xmlSetProp(doc->children, "fileformat", fileformat); 
//   xmlSetProp(doc->children, "doctype", "AddressBook"); 

  g_free(fileformat);

  /* attraverso la rubrica e scrivo i dati 
     nell'albero xml
  */
  len = rubrica_get_items(rubrica);
  for (i = 0; i < len; i++) 
    {
      RubricaItem *item = rubrica_get_nth_item(rubrica, i);
      
      rubrica_file_build_xmlnode(doc->children, item);
    }
  
  /* verifico che non esista un file con lo stesso nome dell'attuale
     se chiesto, faccio il backup del vecchio
     altrimenti cancello il vecchio file e scrivo il nuovo 
  */
  if (g_file_exists(fname))
    {
      if (rubrica_preferences_get_do_backup())
	rubrica_file_do_backup(rubrica, fname);
      else
	{
	  remove(fname);
	}
    }
  
  if (rubrica_preferences_get_compress())
    {
      xmlSetCompressMode(rubrica_preferences_get_compress_rate());
      xmlSetDocCompressMode(doc, rubrica_preferences_get_compress_rate());
    }
  else
    {
      xmlSetDocCompressMode(doc, 0);     
      xmlSetCompressMode(0);
    }
  
  if (xmlSaveFormatFile(fname, doc, 1) == -1) 
    {
      gchar *msg;

      msg = g_strdup_printf(_("An error occurred during file saving.\n"
			      "I can't save the file %s"), fname);
      rubrica_log_write_log(msg);
      rubrica_dialog_error(msg);

      xmlFreeDoc(doc);
      g_free(msg);

      return FALSE;
    }  

  /* if we are here the addressbook is write 
   */
  str = g_strdup_printf("\nSave log ****"
			"\n\taddressbook has been saved as: %s"
			"\n\tin folder: %s", g_basename(fname), path);
  
  rubrica_log_write_log(str);
  g_free(str);  
  
  rubrica_view_set_path(path);

  /* xmlFreeDoc do a segfault, why? */
  //  xmlFreeDoc(doc);    
  g_free(fname);

  return TRUE;
}

void 
rubrica_file_build_xmlnode(xmlNodePtr xmlnode, RubricaItem *item)
{
  GList *alias;
  xmlNodePtr cardxml;

  xmlNodePtr datixml;
  xmlNodePtr addressxml;
  xmlNodePtr webxml;
  xmlNodePtr mailxml;
  xmlNodePtr telxml;
  xmlNodePtr companyxml;
  xmlNodePtr workxml;
  xmlNodePtr notexml;
  gchar *str;

  xmlNodePtr nodo;

  g_return_if_fail(xmlnode != NULL);
  g_return_if_fail(item != NULL);


  /*
    cardxml  figlio di xmlnode (il primo xmlnode  doc->children)
    schedaxml, datixml, ..., sono figli di cardxml
  */    
  cardxml = xmlNewChild(xmlnode, NULL, "Card", NULL);

  /* setto le propriet della scheda 
     nome, editabile, concellabile
  */
  xmlSetProp(cardxml, "name",      item->card   ? item->card : "");
  xmlSetProp(cardxml, "deleting",  item->delete ? "true" : "false");
  xmlSetProp(cardxml, "group",     item->group.label);

  /* data di creazione e ultima modifica della scheda 
   */
  str = g_strdup_printf("%ld", item->created);
  xmlSetProp(cardxml, "created", str);
  g_free(str);
  str = g_strdup_printf("%ld", item->last_change);  
  xmlSetProp(cardxml, "last_change", str);
  g_free(str);

  /*
    xmlNewTextChild 
    sintassi :
         - nodo genitore del nodo creato con la chiamata
	 - namespace
	 - nome del nodo creato
	 - testo (elemento) da inserire nel nodo creato
  */
  
  /* Personal */
  datixml = xmlNewTextChild(cardxml, NULL, "Data", NULL);

  nodo = xmlNewTextChild(datixml, NULL, "FirstName", item->personal.first);
  xmlSetProp(nodo, "know_birthday", (item->personal.birthknow ? 
				     "true" : "false"));
  str = g_strdup_printf("%ld", item->personal.birthdate);
  xmlSetProp(nodo, "birthday", str);
  g_free(str);  
  xmlNewTextChild(datixml, NULL, "MiddleName", item->personal.middle);
  xmlNewTextChild(datixml, NULL, "LastName",   item->personal.last);
  xmlNewTextChild(datixml, NULL, "Profession", item->personal.profession);
  xmlNewTextChild(datixml, NULL, "NamePrefix", item->personal.prefix);
  xmlNewTextChild(datixml, NULL, "Title",      item->personal.title);

  /* Address */
  addressxml = xmlNewTextChild(cardxml, NULL, "Address", NULL);

  xmlNewTextChild(addressxml, NULL, "Street",       item->address.street);
  xmlNewTextChild(addressxml, NULL, "StreetNumber", item->address.number);
  xmlNewTextChild(addressxml, NULL, "ZipCode",      item->address.zip);
  xmlNewTextChild(addressxml, NULL, "City",         item->address.city);
  xmlNewTextChild(addressxml, NULL, "Province",     item->address.province);
  xmlNewTextChild(addressxml, NULL, "State",        item->address.state);
  xmlNewTextChild(addressxml, NULL, "Country",      item->address.country);

  /* Net */
  webxml = xmlNewChild(cardxml, NULL, "WebAddresses", NULL);
  alias = item->net.web;
  while(alias)
    {
      if (alias && alias->data)
	xmlNewTextChild(webxml, NULL, "Web", alias->data);

      alias = alias->next;
    }
  
  mailxml = xmlNewChild(cardxml, NULL, "EmailAddresses", NULL);
  alias = item->net.email;
  while(alias)
    {
      if (alias && alias->data)
	  xmlNewTextChild(mailxml, NULL, "Email", alias->data);

      alias = alias->next;
    }

  
  /* Telephone */
  telxml = xmlNewChild(cardxml, NULL, "TelephoneNumbers", NULL);
  alias = item->phone.telephone;
  while(alias)
    {
      if (alias && alias->data)
	{
	  TelNum *tn = (TelNum*) alias->data;

	  nodo = xmlNewTextChild(telxml, NULL, "Telephone", tn->number);      
	  xmlSetProp(nodo, "type", tn->type);
	}
      
      alias = alias->next;
    }

  /* Company */
  companyxml = xmlNewTextChild(cardxml, NULL, "Company", NULL);

  xmlNewTextChild(companyxml, NULL, "CompanyName",  item->company.name);
  xmlNewTextChild(companyxml, NULL, "Street",       item->company.street);
  xmlNewTextChild(companyxml, NULL, "StreetNumber", item->company.number);
  xmlNewTextChild(companyxml, NULL, "ZipCode",      item->company.zip);
  xmlNewTextChild(companyxml, NULL, "City",         item->company.city);
  xmlNewTextChild(companyxml, NULL, "Province",     item->company.province);
  xmlNewTextChild(companyxml, NULL, "Country",      item->company.country);
  xmlNewTextChild(companyxml, NULL, "Web",          item->company.web);
  xmlNewTextChild(companyxml, NULL, "Email",        item->company.email);
  xmlNewTextChild(companyxml, NULL, "Operator",     item->company.operator);
  xmlNewTextChild(companyxml, NULL, "Fax",          item->company.fax);
  xmlNewTextChild(companyxml, NULL, "Green",        item->company.green);
  xmlNewTextChild(companyxml, NULL, "CustomerCare", 
		  item->company.customer_care);
  xmlNewTextChild(companyxml, NULL, "Notes",        item->company.notes);
  
  
  /* Work */
  workxml = xmlNewTextChild(cardxml, NULL, "Work", NULL);
  xmlNewTextChild(workxml, NULL, "Assigment",      item->work.assigment);
  xmlNewTextChild(workxml, NULL, "Organization",   item->work.organization);
  xmlNewTextChild(workxml, NULL, "Department",     item->work.department);
  xmlNewTextChild(workxml, NULL, "SubDepartment",  item->work.subdep);
  xmlNewTextChild(workxml, NULL, "SecretaryName",  item->work.secretary);
  xmlNewTextChild(workxml, NULL, "SecretaryPhone", item->work.telephone);


  /* Notes */
  notexml = xmlNewTextChild(cardxml, NULL, "Notes", NULL);
  xmlSetProp(notexml, "married", item->notes.is_married ? "true" : "false");
  xmlSetProp(notexml, "children", item->notes.has_children ? "true" : "false");
  xmlSetProp(notexml, "know_birthday", item->notes.birthknow ?
	     "true" : "false");  
  xmlSetProp(notexml, "know_anniversary", item->notes.anniverknow  ? 
	     "true" : "false");

  nodo = xmlNewTextChild(notexml, NULL, "SpouseName", item->notes.spouse);
  str = g_strdup_printf("%ld", item->notes.birthdate);
  xmlSetProp(nodo, "birthday",  str);
  g_free(str);
  str = g_strdup_printf("%ld", item->notes.anniverdate);
  xmlSetProp(nodo, "anniversary",  str);
  g_free(str);  

  xmlNewTextChild(notexml, NULL, "Child",      item->notes.children);
  xmlNewTextChild(notexml, NULL, "Hobbies",    item->notes.hobbies);
  xmlNewTextChild(notexml, NULL, "OtherNotes", item->notes.notes);
  xmlNewTextChild(notexml, NULL, "PublicKey",  item->notes.pubkey);
}



/**
 **
 **  Load Routines
 **
 **/
/*  this function is called when user want start rubrica whit a file
*/
void rubrica_file_start_with_file(gchar *file)
{
  gboolean loaded = FALSE;
  RubricaErrorType err = RE_NO_ERROR;

  rubrica_utils_set_start(TRUE);

  /* load file */
  loaded = rubrica_file_load_xmlfile(file, &err);
  if (!loaded)
    {
      rubrica_dialog_error_loading_file(file, err);      
    }
  
  rubrica_utils_set_start(FALSE);
}

void 
rubrica_file_load_file(void)
{
  GtkWidget *fs;
  GtkWidget *app;

  app = rubrica_get_app();

  fs = gtk_file_selection_new(_("Open file..."));
  
  if (rubrica_preferences_get_have_default_folder())
    {
      gchar *path;
         
      path = rubrica_preferences_get_default_folder();
      if (path)
	{
	  gtk_file_selection_set_filename (GTK_FILE_SELECTION(fs), path);      
	  
	  g_free(path);
	}
    }
 
  /* connetto i bottoni ok e cancel */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked",
		   G_CALLBACK(rubrica_file_open_file), (gpointer) fs);

  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			   "clicked", G_CALLBACK(gtk_widget_destroy), 
			   (gpointer)fs);

  /* visualizzo la finestra in posizione centrale */
  gtk_window_position(GTK_WINDOW(fs), GTK_WIN_POS_CENTER);

  /* il genitore della finestra fileselection  app */
  gtk_window_set_transient_for(GTK_WINDOW(fs), GTK_WINDOW(app));

  gtk_widget_show(GTK_WIDGET(fs));
}

void 
rubrica_file_open_file(GtkWidget *w, gpointer data)
{
  GtkWidget *fs = data;
  gchar *filename;
  gboolean loaded;
  RubricaErrorType err;

  filename = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));

  loaded = rubrica_file_load_xmlfile(filename, &err);

  if (!loaded)
    rubrica_dialog_error_loading_file(filename, err);
  else
    gtk_widget_destroy(fs);
}


gboolean 
rubrica_file_load_xmlfile(gchar *filename, RubricaErrorType* err)
{
  Rubrica *rubrica;
  gchar *fpath, *fname;
  
  if (!g_file_exists(filename))
    {
      *err = RE_FILE_NOT_EXIST;
            
      return FALSE;
    }

  rubrica_card_add_page_view(filename); 
  rubrica = rubrica_get_current_rubrica();

  /* real file load 
   */
  if (!rubrica_file_real_load_xmlfile(rubrica, filename, err))
    return FALSE;

  /* if we are here, the file has been loaded, 
     now we need to put in memory some information 
     about this file and addressbook      
  */
  fpath = g_dirname(filename);
  fname = g_strdup(g_basename(filename));
  
  rubrica_set_path(rubrica, fpath);
  rubrica_set_name(rubrica, fname);
  rubrica_set_new(rubrica, FALSE);
  rubrica_set_empty(rubrica, FALSE);
  rubrica_set_modified(rubrica, FALSE);

  g_free(fname);
  
  *err = RE_NO_ERROR;
  return TRUE;
}


/* 
 *  read the addressbook file 
 */
gboolean 
rubrica_file_real_load_xmlfile(Rubrica *rubrica, gchar *fname, 
			       RubricaErrorType *err)
{
  xmlDocPtr doc;
  gint fileformat = 0;
  gchar* msg;

  if((doc = rubrica_file_open_doc(fname)) == NULL)
    return FALSE;


  if(!rubrica_file_is_rubrica_doc(doc, fname))
    return FALSE;

  /* get the file format
   */
  fileformat = rubrica_file_get_fileformat(doc);
  if (fileformat < RUBRICA_FILE_FORMAT)  /* this is an old rubrica file */
    {
      /* close opened file */
      xmlFreeDoc(doc);
  
      if (!rubrica_file_import_old_file(fname))
	return FALSE;

      if((doc = rubrica_file_open_doc(fname)) == NULL)
	return FALSE;
      
      if(!rubrica_file_is_rubrica_doc(doc, fname))
	return FALSE;     
    }

  /* parse doc, extract address from file and put them in memory 
   */
  rubrica_file_parse_doc(rubrica, doc, err);
  
  msg = g_strdup_printf(_("document %s succesfully opened"), fname);
  g_print("\n%s", msg);  
  g_free(msg);

  /* addressbook has been parsed. Free memory */
  xmlFreeDoc(doc);
  
  return TRUE;  
}


void 
rubrica_file_parse_doc(Rubrica *rubrica, xmlDocPtr doc, RubricaErrorType *err)
{  
  xmlNodePtr nodo;
  gint n = 0;

  /* prendo il nodo Indirizzo dal file e lo inserisco nella (nuova) rubrica 
     il primo nodo  doc->children->children == xmlnew (salvato)
     
     nodo (= doc->children) contiene "Rubrica"
     nodo->children (doc->children->children)  vuoto
     nodo->children->next (doc->children->children->next) 
      il primo nodo utile ("Card")     
  */

  nodo = doc->children;
  if (xmlIsBlankNode(nodo))
    nodo = nodo->next;
  
  if (nodo && nodo->name && g_strcasecmp(nodo->name, "Rubrica") == 0) 
    {
      nodo = nodo->children;
      
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      
      while (nodo)
	{
	  rubrica_file_load_from_xmlnode(rubrica, nodo);  
	  n++;
	  
	  nodo = nodo->next;
 	  if(xmlIsBlankNode(nodo))
	    nodo = nodo->next;
	}
    }
  else
    *err = RE_NO_RUBRICA_HEADER;

  *err = RE_NO_ERROR;

  g_print("\nRubrica reads %d card%s", n, (n > 1 ? "s" : ""));
}

void
rubrica_file_load_from_xmlnode(Rubrica *rubrica, xmlNodePtr xmlnode)
{
  RubricaItem *r;  
  GList *web = NULL, *email = NULL;
  GList *tel = NULL;
  xmlNodePtr figlio = xmlnode;

  /* i nodi xmlnode sono nodi cardxml
     - devo estrarre i figli di cardxml:
  */

  r = g_malloc(sizeof(RubricaItem));

  /* Card */
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;
  if (figlio &&                                  /* se il nodo  valido e */
      figlio->name &&                            /* se il nodo ha nome e  */
      g_strcasecmp(figlio->name, "Card") == 0)   /* il nome  Scheda      */
    {
      gchar *tmp = NULL;  

      r->card = ((tmp = xmlGetProp(xmlnode, "name")) ? g_strdup(tmp) : "");
      if (tmp)
	g_free(tmp);

      r->delete = ((g_strcasecmp(xmlGetProp(xmlnode,"deleting"), "true") == 0)
		   ? TRUE : FALSE);
      if (xmlHasProp(figlio, "group"))
	{
	  r->group.label = ((tmp = xmlGetProp(xmlnode, "group")) ? 
			    g_strdup(tmp) : "");
	  r->group.type  = rubrica_form_get_option_type(r->group.label);
	  if (tmp)
	    g_free(tmp);      
	}
      else
	r->group.label = "";

      r->created = ((tmp = xmlGetProp(xmlnode, "created")) ? 
		    atol(tmp) : time(NULL));
      if (tmp)
	g_free(tmp);
      
      r->last_change = ((tmp = xmlGetProp(xmlnode, "last_change")) ? 
			atol(tmp) : time(NULL));
      if (tmp)
	g_free(tmp);      
    }
  else
    {
      /* TODO 
	 
      segnalare l'errore, far scegliere all'utente se 
      continuare a caricare automaticamente il file 
      o vuole un caricamento assistito
      */	 
    }
  
  /*                          dati personali -- datixml -- 
   */
  figlio = figlio->children;
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;
  if (figlio &&
      figlio->name &&
      g_strcasecmp(figlio->name, "Data") == 0)  
    {
      /* carico i figli di datixml
       */
      xmlNodePtr nodo;
      gchar *tmp;
      
      nodo = figlio->children;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->personal.first = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : ""); 

      if ((tmp = xmlGetProp(nodo, "know_birthday")))
	r->personal.birthknow = ((g_strcasecmp(tmp, "true") == 0) ? 
				 TRUE : FALSE);
      else
	r->personal.birthknow = FALSE;	  
     
      r->personal.birthdate = ((tmp = xmlGetProp(nodo, "birthday")) ? 
			       atol(tmp) : time(NULL));
     
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->personal.middle = ((tmp = xmlNodeGetContent(nodo)) ? 
			    g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;      
      r->personal.last = ((tmp  = xmlNodeGetContent(nodo)) ? 
			  g_strdup(tmp) : "");
      
      nodo = nodo->next;  
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;    
      r->personal.profession = ((tmp = xmlNodeGetContent(nodo)) ? 
				g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->personal.prefix = ((tmp = xmlNodeGetContent(nodo)) ? 
			    g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->personal.title = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : "");
    }
  else 
    {
      g_warning(_("May be this file is corrupted. Can't read %s field"), 
		"Data");
      
      r->personal.first = r->personal.middle = r->personal.last 
	= r->personal.profession = r->personal.prefix 
	= r->personal.title = "";
      r->personal.birthdate = time(NULL);
    }
  
  /*                                Indirizzo -- Address
   */
  figlio = figlio->next;
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;
  if (figlio &&
      figlio->name &&
      g_strcasecmp(figlio->name, "Address") == 0)
    {
      xmlNodePtr nodo;
      gchar *tmp;
      
      nodo = figlio->children;  
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->address.street = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->address.number = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->address.zip = ((tmp = xmlNodeGetContent(nodo)) ? 
			g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->address.city = ((tmp = xmlNodeGetContent(nodo)) ? 
			 g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->address.province = ((tmp = xmlNodeGetContent(nodo)) ? 
			     g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->address.state = ((tmp = xmlNodeGetContent(nodo)) ? 
			  g_strdup(tmp) : "");
      
      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->address.country = ((tmp = xmlNodeGetContent(nodo)) ? 
			    g_strdup(tmp) : "");
    }
  else 
    {
      g_warning(_("May be this file is corrupted. Can't read %s field"), 
		"Address");
      
      r->address.street = r->address.number = r->address.zip = 
	r->address.city = r->address.province = r->address.state = 
	r->address.country = "";
    }
  
  /*                                      web 
   */
  figlio = figlio->next;
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;
  if (figlio &&
      figlio->name &&
      g_strcasecmp(figlio->name, "WebAddresses") == 0)
    {
      gchar *buffer;
      xmlNodePtr nodo;
      
      nodo = figlio->children;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      while(nodo)
	{
	  gchar *tmp;
	  
	  if (xmlIsBlankNode(nodo))
	    nodo = nodo->next;
    	  
	  tmp = xmlNodeGetContent(nodo);
	  if (tmp)
	    {
	      buffer = g_strdup(tmp);
	      
	      web = g_list_append(web, buffer);
	      nodo = nodo->next;
	    }	  
	}
      r->net.web = web;
    }
  else
    {
      g_warning(_("May be this file is corrupted. Can't read %s field"), 
		"WebAddresses");
      
      r->net.web = NULL;
    }

  /*                                         email 
   */
  figlio = figlio->next;
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;

  if (figlio &&
      figlio->name &&
      g_strcasecmp(figlio->name, "EmailAddresses") == 0)
    {
      gchar *buffer;
      xmlNodePtr nodo;
      
      nodo = figlio->children;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      while(nodo)
	{
	  gchar *tmp;
	  
	  if (xmlIsBlankNode(nodo))
	    nodo = nodo->next;
	  
	  tmp = xmlNodeGetContent(nodo);
	  
	  if (tmp)
	    {
	      buffer = g_strdup(tmp);
	      
	      email = g_list_append(email, buffer);
	      nodo = nodo->next;
	    }	
	}
      r->net.email = email;
    }  
  else
    {
      g_warning(_("May be this file is corrupted. Can't read %s field"), 
		"EmailAddresses");
      
      r->net.email = NULL;
    }
  
  /*                                  telephone 
   */
  figlio = figlio->next;
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;
  if (figlio &&
      figlio->name &&
      g_strcasecmp(figlio->name, "TelephoneNumbers") == 0)
    {
      xmlNodePtr nodo;
      TelNum *tn;
      
      nodo = figlio->children;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;

      do
	{
	  gchar *tmp1=NULL, *tmp2=NULL;
	  
	  if (nodo)
	    {
	      tmp1 = xmlNodeGetContent(nodo);
	      
	      if (tmp1)
		{	
		  tn = g_malloc(sizeof(TelNum));

		  if (xmlHasProp(nodo, "type"))
		    tmp2 = xmlGetProp(nodo, "type");
		  else 
		    tmp2 = "???";
		  
		  tn->number = g_strdup(tmp1);
		  tn->type = g_strdup(tmp2);	  	     
		  
		  tel = g_list_append(tel, tn);
		  
		  nodo = nodo->next;
		  if(nodo && xmlIsBlankNode(nodo))
		    nodo = nodo->next;	      
		}
	    }
	} 
      while(nodo);

      r->phone.telephone = tel;
    }
  else
    {
      g_warning(_("May be this file is corrupted. Can't read %s field"), 
		"TelephoneNumbers");
      
      r->phone.telephone = NULL;
    }
  
  /*                                        company 
   */
  figlio = figlio->next; 
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;
  if (figlio &&
      figlio->name && 
      g_strcasecmp(figlio->name, "Company") == 0)
    {
      xmlNodePtr nodo;
      gchar *tmp;
      
      nodo = figlio->children;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.name = ((tmp = xmlNodeGetContent(nodo)) ? 
			 g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.street = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.number = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.zip = ((tmp = xmlNodeGetContent(nodo)) ? 
			g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.city = ((tmp = xmlNodeGetContent(nodo)) ? 
			 g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.province = ((tmp = xmlNodeGetContent(nodo)) ? 
			     g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.country = ((tmp = xmlNodeGetContent(nodo)) ? 
			    g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.web = ((tmp = xmlNodeGetContent(nodo)) ?
			g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.email = ((tmp = xmlNodeGetContent(nodo)) ? 
			  g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.operator = ((tmp = xmlNodeGetContent(nodo)) ? 
			     g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.fax = ((tmp = xmlNodeGetContent(nodo)) ? 
			g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.green = ((tmp = xmlNodeGetContent(nodo)) ? 
			  g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.customer_care =((tmp = xmlNodeGetContent(nodo)) ? 
				 g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->company.notes = ((tmp = xmlNodeGetContent(nodo)) ? 
			  g_strdup(tmp) : "");
    }
  else   
    {
      g_warning(_("May be this file is corrupted. Can't read %s field"), 
		"Company");
      
      r->company.name = r->company.street = r->company.number 
	= r->company.zip = r->company.city  = r->company.province
	= r->company.country = r->company.web = r->company.email
	= r->company.operator = r->company.fax = r->company.green 
	= r->company.customer_care = r->company.notes = "";
    }
  
  
  /*                                      work 
   */
  figlio = figlio->next; 
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;
  if (figlio &&
      figlio->name && 
      g_strcasecmp(figlio->name, "Work") == 0)
    {
      xmlNodePtr nodo;
      gchar *tmp;
      
      nodo = figlio->children;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->work.assigment = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->work.organization = ((tmp = xmlNodeGetContent(nodo)) ? 
			      g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->work.department = ((tmp = xmlNodeGetContent(nodo)) ?
			    g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->work.subdep = ((tmp = xmlNodeGetContent(nodo)) ? 
			g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->work.secretary = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->work.telephone = ((tmp = xmlNodeGetContent(nodo)) ?
			   g_strdup(tmp) : "");
       
    }
  else
    {
      g_warning(_("May be this file is corrupted. Can't read %s field"), 
		"Work");

      r->work.assigment =  r->work.organization = 
	r->work.department = r->work.subdep = 
	r->work.secretary  = r->work.telephone = "";
    }

  /*                                          note 
   */
  figlio = figlio->next;
  if(xmlIsBlankNode(figlio))
    figlio = figlio->next;
  if (figlio &&
      figlio->name &&
      g_strcasecmp(figlio->name, "Notes") == 0)
    {
      xmlNodePtr nodo;
      gchar *tmp;
      
      tmp = xmlGetProp(figlio, "married");
      r->notes.is_married = ((g_strcasecmp(tmp, "true") == 0) ? TRUE : FALSE);

      tmp = xmlGetProp(figlio, "children");
      r->notes.has_children =((g_strcasecmp(tmp, "true") == 0) ? TRUE : FALSE);
      
      tmp = xmlGetProp(figlio, "know_birthday");
      r->notes.birthknow = ((g_strcasecmp(tmp, "true") == 0) ? TRUE : FALSE);

      tmp = xmlGetProp(figlio, "know_anniversary");
      r->notes.anniverknow = ((g_strcasecmp(tmp, "true") == 0) ? TRUE : FALSE);
      
      nodo = figlio->children;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->notes.spouse = ((tmp = xmlNodeGetContent(nodo)) ? g_strdup(tmp) : "");

      if (r->notes.birthknow)
	r->notes.birthdate = ((tmp = xmlGetProp(nodo, "birthday")) ? 
			      atol(tmp) : time(NULL));
      else
	r->notes.birthdate = time(NULL);

      if (r->notes.anniverknow)    
	r->notes.anniverdate = ((tmp= xmlGetProp(nodo, "anniversary")) ? 
				atol(tmp) : time(NULL));
      else
	r->notes.anniverdate = time(NULL);	

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->notes.children = ((tmp = xmlNodeGetContent(nodo)) ? 
			   g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->notes.hobbies = ((tmp = xmlNodeGetContent(nodo)) ? 
			  g_strdup(tmp) : "");

      nodo = nodo->next;
       if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
       r->notes.notes = ((tmp = xmlNodeGetContent(nodo)) ? 
			 g_strdup(tmp) : "");

      nodo = nodo->next;
      if (xmlIsBlankNode(nodo))
	nodo = nodo->next;
      r->notes.pubkey = ((tmp = xmlNodeGetContent(nodo)) ? g_strdup(tmp) : "");
    } 
  else
    {
      g_warning(_("May be this file is corrupted. Can't read %s field"), 
		"Notes");

      r->notes.birthdate = r->notes.anniverdate = time(NULL);
      r->notes.is_married = r->notes.has_children = 
	r->notes.birthknow = r->notes.anniverknow= FALSE;
      r->notes.spouse = r->notes.children = r->notes.hobbies = 
	r->notes.notes = r->notes.pubkey = "";
    }

  rubrica_append_item(rubrica, r);
}


