# Video streaming via Gst::Playbin.

require 'gtk2'
require 'gst0.10'

Gtk.init
Gst.init

if ARGV.length != 1
    $stderr.puts "Usage: #{__FILE__} uri"
    exit 1
end

class VideoWidget < Gtk::DrawingArea
  def initialize(uri)
    super()
    unset_flags(Gtk::Widget::Flags::DOUBLE_BUFFERED)
    
    # setup the playbin
    @playbin = Gst::ElementFactory.make('playbin')
    @playbin.uri = uri
    @playbin.ready
    
    # setup the sinks
    @videosink = Gst::ElementFactory.make('autovideosink')
    @audiosink = Gst::ElementFactory.make('autoaudiosink')
    @videosink.signal_connect('element-added') { | sink, element | 
      video_element_added_cb(sink, element );
    } 

    @playbin.video_sink = @videosink
    @playbin.audio_sink = @audiosink
    @playbin.signal_connect('notify::stream-info') { got_stream_info } 
    @playbin.bus.add_watch { | message | got_bus_message(message) }

    #@state = "STOPPED"

    signal_connect('map') { play }

  end

  def got_stream_info
    @playbin.stream_info_value_array.each { | x | 
      if x.type.nick == "video"
        x.object.signal_connect('notify::caps') { | obj, caps |
          w =  obj.caps.get_structure(0)["width"]
           h =  obj.caps.get_structure(0)["height"]
           set_size_request(w,h)
        }
      end
    }
  end

  def got_bus_message(message)
    case message.get_type
      when Gst::Message::MessageType::ERROR then
        print "An error occured: #{message.parse_error} \n" 
        Gtk.main_quit
      when Gst::Message::MessageType::EOS then
        print "EOF!\n"
      when Gst::Message::MessageType::DURATION then
        print "Got duration update...\n"
      when Gst::Message::MessageType::TAG then
        tag = message.parse_tag
        print "Got tags: #{tag.inspect} \n"
    end
    true
  end

  def video_element_added_cb(sink, element) 
    if element.is_a? Gst::XOverlay
      element.set_xwindow_id_with_buswatch(@playbin.bus, window.xid)
    end
    if element.class.properties.include? 'force-aspect-ratio'
      element.force_aspect_ratio = true 
    end
  end

  def play
    #if @state != "PLAYING"
      @playbin.play
    #  @state = "PLAYING"
    #end
  end

  def pause
    #if @state == "PLAYING":
      @playbin.pause
    #  @state = "PAUSED"
    #end
  end

  def stop
    #if @state != "STOPPED"
      @playbin.stop
    #  @state = "STOPPED"
    #end
  end

  def uri
    @playbin.uri
  end

  def duration
   @playbin.query_duration 
  end

  def position
   @playbin.query_position 
  end

  def seek(time)
    @playbin.seek(1.0, Gst::Format::Type::TIME,
                  Gst::Seek::FLAG_FLUSH.to_i | 
                  Gst::Seek::FLAG_KEY_UNIT.to_i,
                  Gst::Seek::TYPE_SET, time,
                  Gst::Seek::TYPE_NONE, -1)
    # Wait at most 100 miliseconds for a state changes, this throttles the seek
    # events to ensure the playbin can keep up
    @playbin.get_state(100 * Gst::MSECOND)
  end
end

class Slider < Gtk::HScale
  def initialize
    super
    self.draw_value = false
    self.sensitive = false
    @value_change_block = nil 
    @duration = -1
    @seeking = false
    signal_connect('button_press_event') { @seeking = true ; false }
    signal_connect('button_release_event') { @seeking = false ; false }
    signal_connect('value_changed') {  
      if @seeking and @value_change_block
        @value_change_block.call((value * Gst::MSECOND).to_i)
      end
      false
    }
  end

  def duration=(duration) 
    if @duration != duration 
      if duration < 1
        self.sensitive = false
        self.value = 0
      else 
        self.sensitive = true
        set_range(0, duration.to_f/Gst::MSECOND)
      end
    end
  end

  def position=(position)
    set_value position/Gst::MSECOND if !@seeking
  end

  def on_value_change (&block) 
    @value_change_block = block
  end
end

window = Gtk::Window.new

video = VideoWidget.new(ARGV.first)

slider = Slider.new 
Gtk.timeout_add(200) { 
  duration = video.duration[1]
  if duration > 0
    slider.duration = duration
    slider.position = video.position[1]
  end
  true
} 

slider.on_value_change { | position |  video.seek(position) }

buttonbox = Gtk::HButtonBox.new() 
buttonbox.layout_style = Gtk::ButtonBox::START

button = Gtk::Button.new(Gtk::Stock::MEDIA_PLAY)
button.signal_connect("clicked") { video.play }
buttonbox.add(button)

button = Gtk::Button.new(Gtk::Stock::MEDIA_PAUSE)
button.signal_connect("clicked") { video.pause }
buttonbox.add(button)

button = Gtk::Button.new(Gtk::Stock::MEDIA_STOP)
button.signal_connect("clicked") { 
   slider.duration = 0
   video.stop 
}
buttonbox.add(button)


hbox = Gtk::HBox.new(false, 3)
hbox.pack_start(buttonbox, false, false)
hbox.pack_start(slider, true, true)

vbox = Gtk::VBox.new(false, 3)
window.add(vbox)
vbox.pack_start(video)
vbox.pack_start(hbox, false, false)

window.signal_connect('destroy') { 
  # First stop the pipeline, so the sink window is nicely destroyed before it's
  # parent widget goes away
  video.stop
  Gtk.main_quit 
}


puts "Streaming #{video.uri}..."
begin 
  window.set_default_size(410, 325)
  window.show_all
  Gtk.main
rescue Interrupt 
  puts "Got an interrupt, exiting"
end
video.stop
