/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file Helper.cxx
    \author Romain BONDUE
    \date 04/01/2006 */
#include <iostream>
#include <memory>
#include <map>
#include <exception>
#include <stdexcept>
#include <cstdlib> // strtol(), strtoul()
#include <cstring> // strncmp(), strcmp()

extern "C"{
#include <shadow.h> // ::getspnam()
#include <crypt.h> // ::crypt()
#include <unistd.h> // ::sleep()
}

#include "MsgHandlerFactory.h"
#include "SystemTools.h"
#include "Msg.h"
#include "IMsgHandler.h"
#include "Exceptions.h"
#include "StaticSettings.h"
#include "ErrorsCode.h"



namespace
{
    typedef std::map<unsigned long, nsRoot::IMsgHandler*> pMsgHandlersMap_t;


    class CFormattedPipe : protected nsSystem::CPipe
    {
      public :
        CFormattedPipe () throw (nsErrors::CSystemExc) : CPipe() {}

        CFormattedPipe (int Fd1, int Fd2) throw (nsErrors::CSystemExc)
            : CPipe (Fd1, Fd2) {}


        bool ReadIntOrStop (int& Buffer) throw (nsErrors::CException)
        {
            const unsigned NbByteRead (Read (reinterpret_cast<char*> (&Buffer),
                                       sizeof (Buffer)));
            if (NbByteRead)
            {
                if (NbByteRead != sizeof (Buffer))
                    throw nsErrors::CException ("Invalid data read, integer"
                                                " expected.",
                                                nsErrors::InvalidData);
                return false;
            }
            return true;

        } // ReadIntOrStop()


        int ReadInt () throw (nsErrors::CException)
        {
            int Buffer;
            if (ReadIntOrStop (Buffer))
                throw nsErrors::CException ("Unexpected end of stream.",
                                            nsErrors::UnexpectedStreamEnd);
            return Buffer;

        } // ReadInt()


        void Write (int Value) throw (nsErrors::CSystemExc)
        {
            CPipe::Write (reinterpret_cast<const char*> (&Value),
                          sizeof (Value));

        } // Write()


        std::string ReadString (unsigned Size) throw (nsErrors::CException,
                                                      std::bad_alloc)
        {
            std::auto_ptr<char> Buffer (new char [Size]);
            if (Read (Buffer.get(), Size) != Size)
                throw nsErrors::CException ("Invalid data read.",
                                            nsErrors::InvalidData);
            return std::string (Buffer.get(), 0, Size);

        } // ReadString()


        void Write (const std::string& Str) throw (nsErrors::CSystemExc)
        {
            Write (Str.size());
            CPipe::Write (Str.data(), Str.size());

        } // Write()


        void Write (const nsRoot::CMsg& Msg) throw (nsErrors::CSystemExc)
        {
            Write (Msg.GetText());
            Write (Msg.GetCode());

        } // Write()

    }; // CFormattedPipe

    
    unsigned long ParseULong (const std::string& Str)
                                                throw (nsErrors::CException)
    {
        char* pEnd;
        const unsigned long Num (strtoul (Str.c_str(), &pEnd, 10));
        if (*pEnd)
            throw nsErrors::CException ("Invalid data read.",
                                        nsErrors::InvalidData);
        return Num;

    } // ParseULong()


    bool IsRootPasswd (const std::string& Password)
                                                throw (nsErrors::CSystemExc)
    {
        const ::spwd* const pShadow (::getspnam ("root"));
        if (!pShadow)
            throw nsErrors::CSystemExc ("Can't check root password.");
        const unsigned CstSaltMaxSize (13);
        char Salt [CstSaltMaxSize];
        if (!std::strncmp (pShadow->sp_pwdp, "$1$", 3)) // MD5
        {
            unsigned i (0);
            for (unsigned Cpt (0) ; Cpt < 3 && i < CstSaltMaxSize - 1 ; ++i)
            {
                Salt [i] = pShadow->sp_pwdp [i];
                if (Salt [i] == '$') ++Cpt;
            }
            Salt [i] = '\0';
        }
        else // DES
        {
            Salt [0] = pShadow->sp_pwdp [0];
            Salt [1] = pShadow->sp_pwdp [1];
            Salt [2] = '\0';
        }
        const char* const CryptedPassord (::crypt (Password.c_str(), Salt));
        if (!std::strcmp (CryptedPassord, pShadow->sp_pwdp))
            return true;
        else
        {
            ::sleep (3); // For security reasons.
            return false;
        }

    } // IsRootPasswd()

} // anonymous namespace



int main (int argc, char* argv [])
{
    try
    {
        if (argc != 4)
            throw nsErrors::CException ("Invalid number of arguments.",
                                        nsErrors::InvalidArguments);
        CFormattedPipe Pipe (ParseULong (argv [1]), ParseULong (argv [2]));

#ifdef NOROOTPASSCHECK
        if (Pipe.ReadInt() != nsRoot::IsAliveCmd)
            throw nsErrors::CException ("Invalid command, alive check"
                                        " expected.", nsErrors::InvalidData);
        if (Pipe.ReadInt()) // The text size, which should be 0.
            throw nsErrors::CException ("Invalid data, empty string expected.",
                                        nsErrors::InvalidData);
#else
            // Check root password :
        if (Pipe.ReadInt() != nsRoot::CheckRootPassword)
            throw nsErrors::CException ("Invalid command, password check"
                                        " expected.", nsErrors::InvalidData);
        if (!IsRootPasswd (Pipe.ReadString (Pipe.ReadInt())))
        {
            Pipe.Write (nsErrors::InvalidRootPassword);
            return nsErrors::InvalidRootPassword;
        }
#endif // NOROOTPASSCHECK
        Pipe.Write (0);

        nsRoot::IMsgHandler* pCurrentMsgHandler (0);
        unsigned long InstanceCounter (ParseULong (argv [3]));
        pMsgHandlersMap_t MsgHandlersMap;
        for ( ; ; )
            try
            {
                int Value;
                if (Pipe.ReadIntOrStop (Value))
                    break;
                if (Value < 0) // Internal command.
                {
                    const unsigned Size (Pipe.ReadInt());
                    const std::string Data (Pipe.ReadString (Size));
                    switch (Value)
                    {
                      case nsRoot::CreateRemoteHandlerCmd :
                        MsgHandlersMap [InstanceCounter] =
                                                   nsRoot::MakeHandler (Data);
                        ++InstanceCounter;
                      break;

                      case nsRoot::ChangeRemoteHandlerCmd :
                        pCurrentMsgHandler = MsgHandlersMap
                                                          [ParseULong (Data)];
                        if (!pCurrentMsgHandler)
                            throw nsErrors::CException
                                              ("Invalid instance number.",
                                               nsErrors::InvalidInstanceNum);
                      break;

                      case nsRoot::DeleteRemoteHandlerCmd :
                      {
                        const pMsgHandlersMap_t::iterator Iter
                                (MsgHandlersMap.find (ParseULong (Data)));
                        if (Iter != MsgHandlersMap.end())
                        {
                            delete Iter->second;
                            MsgHandlersMap.erase (Iter);
                        }
                      }
                      break;

                      default :
                        throw nsErrors::CException ("Invalid command.",
                                                    nsErrors::InvalidCommand);
                    }
                    Pipe.Write (0); // Everything is ok.
                }
                else // CMsg
                {
                    const std::string Data (Pipe.ReadString (Value));
                    if (pCurrentMsgHandler)
                        Pipe.Write ((*pCurrentMsgHandler) (nsRoot::CMsg (Data,
                                                           Pipe.ReadInt())));
                    else throw nsErrors::CException ("No handler defined.",
                                                nsErrors::NoHandlerDefined);
                }
            }
            catch (const nsErrors::CException& Exc)
            {
                if (Exc.GetCode() > 127) // Not a "standard" error.
                    Pipe.Write (nsRoot::CMsg (Exc.GetMsg(), Exc.GetCode()));
                throw;
            }
            catch (const std::exception& Exc)
            {
                Pipe.Write (nsRoot::CMsg (Exc.what(), nsErrors::UnknownExc));
                throw;
            }
        for (std::map<unsigned long, nsRoot::IMsgHandler*>::iterator Iter
            (MsgHandlersMap.begin()) ; Iter != MsgHandlersMap.end() ; ++Iter)
            delete Iter->second;
    }
    catch (const std::bad_alloc& Exc)
    {
        std::cerr << "Helper : " << Exc.what() << std::endl;
        return nsErrors::OutOfMemory;
    }
    catch (const nsErrors::CException& Exc) // Handle CSystemExc too.
    {
        std::cerr << "Helper : " << Exc.GetMsg() << "\nCode : "
                  << Exc.GetCode() << std::endl;
        return Exc.GetCode();
    }
    catch (const std::exception& Exc)
    {
        std::cerr << "Helper : exception : " << Exc.what() << std::endl;
        return nsErrors::UnknownExc;
    }
    catch (...)
    {
        std::cerr << "Helper : unknown execption." << std::endl;
        return nsErrors::UnknownExc;
    }

} // main()
