/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file Su.cxx
    \author Romain BONDUE
    \date 19/12/2005 */
#include <cstdlib> // abort()
#include <sstream>

#include "Su.h"
#include "Msg.h"
#include "StaticSettings.h"
#include "ErrorsCode.h"



namespace
{
    template <typename T>
    std::string ToString (const T& Value) throw()
    {
        std::ostringstream Os;
        Os << Value;
        return Os.str();

    } // ToString()

} // anonymous namespace



std::auto_ptr<nsSystem::CPipe> nsRoot::CSu::pPipe (0);
unsigned long nsRoot::CSu::CurrentRemoteHandlerNumber (0);
unsigned long nsRoot::CSu::InstanceCounter (1);


bool nsRoot::CSu::Init (const std::string& RootPassword)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
    pPipe.reset (new nsSystem::CPipe);
    std::auto_ptr<nsSystem::CPipe> pSecondPipe (new nsSystem::CPipe);
        // Swap the read file descriptor.
    const int Mem (pPipe->GetReadFD());
    pPipe->SetReadFD (pSecondPipe->GetReadFD());
    pSecondPipe->SetReadFD (Mem);
        // pPipe is to be used by this process and pSecondPipe by it child.

    if (!nsSystem::Fork())
    {
        std::ostringstream Arg1;
        Arg1 << pSecondPipe->GetReadFD();
        std::ostringstream Arg2;
        Arg2 << pSecondPipe->GetWriteFD();
        std::ostringstream Arg3;
        Arg3 << InstanceCounter;
        pPipe.reset(); // Unsure the unused fd get closed.
        pSecondPipe.release(); // So its fd don't get closed by the destructor.
#ifdef EXTERNALLAUNCHER
        const char* Argv [6] = {nsCore::HelperLauncherPath.c_str(),
                                (nsCore::HelperPrefix +
                                                nsCore::HelperName).c_str(),
                                Arg1.str().c_str(), Arg2.str().c_str(),
                                Arg3.str().c_str(), 0};
#else
        const char* Argv [5] = {(nsCore::HelperPrefix +
                                                nsCore::HelperName).c_str(),
                                Arg1.str().c_str(), Arg2.str().c_str(),
                                Arg3.str().c_str(), 0};
#endif // EXTERNALLAUNCHER
        nsSystem::MandatoryExec (Argv);
    }
    pSecondPipe.reset(); // No more use.
#ifdef NOROOTPASSCHECK
    return !SendInternalCommand (IsAliveCmd, "");
#else
    return !SendInternalCommand (CheckRootPassword, RootPassword);
#endif // NOROOTPASSCHECK

} // Init()


nsRoot::CSu::~CSu () throw()
{
    try
    {
        if (pPipe.get())
            SendInternalCommand (DeleteRemoteHandlerCmd,
                                 ToString (m_InstanceNumber));
    }
    catch (...) {} // We just ignore errors.

} // ~CSu()


void nsRoot::CSu::CreateRemoteHandler
        (const std::string& RemoteHandlerTypeName) throw (nsErrors::CSystemExc)
{
    int Value (SendInternalCommand (CreateRemoteHandlerCmd,
                                    RemoteHandlerTypeName));
    if (Value)
    {
        std::auto_ptr<char> Buffer (0);
        try{Buffer.reset (new char [Value + 1]);}
        catch (const std::bad_alloc& Exc)
        {
            throw nsErrors::CSystemExc (Exc.what(), nsErrors::OutOfMemory);
        }
        Buffer.get() [pPipe->Read (Buffer.get(), Value)] = '\0';
        pPipe->Read (reinterpret_cast<char*> (&Value), sizeof (Value));
        throw nsErrors::CSystemExc
            (std::string ("Can't create remote handler : ") + Buffer.get(),
             Value);
    }

} // CreateRemoteHandler()


void nsRoot::CSu::WriteString (const std::string& Str)
                                                throw (nsErrors::CSystemExc)
{
    const int Size (Str.size());
    pPipe->Write (reinterpret_cast<const char*> (&Size), sizeof (Size));
    if (Size) pPipe->Write (Str.data(), Size);

} // WriteString()


    /* The main difference with Do() is the order on which fields are written :
           Size, Data, Code for Do(), then the same sequence of data (value may
           differ of course) is expected back.
           Code, Size, Data for this function, an int is expected to be sent
           back.
       That's why commands must be negatives (a size can't be negative) so
       the remote process can identify the data type. */
int nsRoot::CSu::SendInternalCommand (int Code, const std::string& Text)
                                                throw (nsErrors::CSystemExc)
{
    pPipe->Write (reinterpret_cast<const char*> (&Code), sizeof (Code));
    WriteString (Text);
    int ACK (nsErrors::InvalidData);
    pPipe->Read (reinterpret_cast<char*> (&ACK), sizeof (ACK));

    return ACK;

} // SendInternalCommand()


void nsRoot::CSu::SetRemoteHandler () throw (nsErrors::CException)
{
    if (CurrentRemoteHandlerNumber != m_InstanceNumber)
    {
        int Value (SendInternalCommand (ChangeRemoteHandlerCmd,
                                        ToString (m_InstanceNumber)));
        if (Value)
        {
            std::string ErrorMsg ("Can't set remote handler.");
            if (Value > 0)
                try
                {
                    char Buffer [Value + 1];
                    Buffer [pPipe->Read (Buffer, Value)] = '\0';
                    ErrorMsg += '\n';
                    ErrorMsg += Buffer;
                    pPipe->Read (reinterpret_cast<char*> (&Value),
                                 sizeof (Value));
                }
                catch (...) {Value = nsErrors::CodeNotTransmitted;}
            else
                Value = nsErrors::CodeNotTransmitted;
            throw nsErrors::CException (ErrorMsg, Value);
        }
        CurrentRemoteHandlerNumber = m_InstanceNumber;
    }

} // SetRemoteHandler()


nsRoot::CMsg nsRoot::CSu::Do (const CMsg& Msg) throw (nsErrors::CException,
                                                      std::bad_alloc)
{
    SetRemoteHandler();
    WriteString (Msg.GetText());
    int Code (Msg.GetCode());
    pPipe->Write (reinterpret_cast<const char*> (&Code), sizeof (Code));

    std::string::size_type Size (0);
    pPipe->Read (reinterpret_cast<char*> (&Size), sizeof (Size));
    std::auto_ptr<char> Buffer (0);
    if (Size)
    {
        Buffer.reset (new char [Size]);
        Size = pPipe->Read (Buffer.get(), Size);
    }
    pPipe->Read (reinterpret_cast<char*> (&Code), sizeof (Code));

    return Size ? CMsg (std::string (Buffer.get(), 0, Size), Code)
                : CMsg (std::string(), Code);

} // Do()
