/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file SystemTools.cxx
    \author Romain BONDUE
    \date 20/12/2005 */
#include <csignal>
#include <cstring> // std::memset()
#include <exception>

extern "C"{
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h> // ::pid_t
#include <unistd.h> // ::close(), ::fork(), ::execl()
#include <linux/version.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION (2, 6, 0)
#include <sys/wait.h> // ::wait()
#endif
}

#include "SystemTools.h"



namespace
{
    inline void Sigaction (int Signum, const struct sigaction& NewSettings,
                           struct sigaction* OldSettings = 0)
                                                throw (nsErrors::CSystemExc)
    {
        if (::sigaction (Signum, &NewSettings, OldSettings))
            throw nsErrors::CSystemExc ("::sigaction() failed.");

    } // Sigaction()


#if LINUX_VERSION_CODE < KERNEL_VERSION (2, 6, 0)
    void ZombieCleaner (int) throw() {::wait (0);}
#endif
        // Set up things when RutilT starts.
    struct SInitializer
    {
      public :
        const int Sd;


        SInitializer () throw (nsErrors::CSystemExc)
            : Sd (::socket (PF_INET, SOCK_DGRAM, 0))
        {
            if (Sd == -1)
                throw nsErrors::CSystemExc ("Can't create socket.");
            IgnoreSignals();

        } // SInitializer()


        ~SInitializer () throw() {::close (Sd);}


        void IgnoreSignals () throw (nsErrors::CSystemExc)
        {
            struct sigaction Settings;
            std::memset (&Settings, 0, sizeof (struct sigaction));
            Settings.sa_handler = SIG_IGN;
            Sigaction (SIGPIPE, Settings);

#if LINUX_VERSION_CODE < KERNEL_VERSION (2, 6, 0)
            Settings.sa_flags = SA_NOCLDSTOP;
            Settings.sa_handler = ZombieCleaner;
#else
            Settings.sa_flags = SA_NOCLDWAIT;
#endif
            Sigaction (SIGCHLD, Settings);

        } // IgnoreSignals()


        void RestoreSignals () throw (nsErrors::CSystemExc)
        {
            struct sigaction Settings;
            std::memset (&Settings, 0, sizeof (struct sigaction));
            Settings.sa_handler = SIG_DFL;
            Sigaction (SIGPIPE, Settings);
            Sigaction (SIGCHLD, Settings);
                /* !! UGLY !! They are modified in GtkGUI.cxx
                   They have not been modified here because this code is linked
                   with the helper, and it's not linked with gtk. */
            Sigaction (SIGINT, Settings);
            Sigaction (SIGTERM, Settings);

        } // RestoreSignals()

    } Data;


    inline void Close (int FD) throw (nsErrors::CSystemExc)
    {
        if (::close (FD))
            throw nsErrors::CSystemExc ("Can't close file descriptor.");

    } // Close()

} // anonymous namespace



void nsSystem::Ioctl (int Request, void* pData, const std::string& ErrorMsg)
                                                throw (nsErrors::CSystemExc)
{
    if (::ioctl (Data.Sd, Request, pData) < 0)
        throw nsErrors::CSystemExc (ErrorMsg);

} // Ioctl()


nsSystem::CPipe::CPipe () throw (nsErrors::CSystemExc)
{
    if (::pipe (m_FileDes))
        throw nsErrors::CSystemExc ("Can't create pipe.");

} // CPipe()


nsSystem::CPipe::~CPipe () throw (nsErrors::CSystemExc)
{
    Close (m_FileDes [0]);
    Close (m_FileDes [1]);

} // ~CPipe()


void nsSystem::CPipe::Write (const char* Data, ssize_t Size)
                                                throw (nsErrors::CSystemExc)
{
    if (::write (m_FileDes [1], Data, Size) != Size)
        throw nsErrors::CSystemExc ("Can't write in pipe.");

} // Write()


size_t nsSystem::CPipe::Read (char* Buffer, size_t Size)
                                                throw (nsErrors::CSystemExc)
{
    const ssize_t ByteRead (::read (m_FileDes [0], Buffer, Size));
    if (ByteRead < 0)
        throw nsErrors::CSystemExc ("Can't read in pipe.");
    return ByteRead;

} // Read()


bool nsSystem::Fork () throw (nsErrors::CSystemExc)
{
    const ::pid_t Pid (::fork());
    if (Pid == -1)
        throw nsErrors::CSystemExc ("Can't create sub-process.");
    return Pid;

} // Fork()


void nsSystem::Exec (const char* Argv []) throw (nsErrors::CSystemExc)
{
    Data.RestoreSignals();
    ::execvp (Argv [0], const_cast<char**> (Argv));
    throw nsErrors::CSystemExc (std::string ("Can't execute file : ")
                                                            + Argv [0] + '.');

} // Exec()
