/* preferences.c - handle the global preferences
 *
 * Copyright (C) 2001  Jochen Voss.  */

static const  char  rcsid[] = "$Id: preferences.c,v 1.7 2002/01/02 02:36:02 voss Rel $";

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>

#include "support.h"

#include "sandcommon.h"
#include "sand-window.h"
#include "locate.h"


enum sand_type { st_RANDOM, st_FIXED };

enum alarm_type { at_BEEP, at_SOUND, at_COMMAND };

struct preferences {
  /* page 0 */
  enum alarm_type  alarm_type;
  char *cmd, *snd;

  /* page 1 */
  enum sand_type  sand_type;
  guint8  r, g, b;

  /* page 2 */
  gboolean  decorations;
  GnomeWinLayer layer;
};

static  struct preferences  defaults;


static void
load_preferences (struct preferences *p)
{
  gnome_config_push_prefix ("/SandUhr/preferences/");
  p->alarm_type = gnome_config_get_int ("alarm_type");
  p->cmd = gnome_config_get_string ("alarm_cmd");
  p->snd = gnome_config_get_string ("alarm_snd");

  p->sand_type = gnome_config_get_int ("sand_type");
  p->r = gnome_config_get_int ("sand_color_r=139");
  p->g = gnome_config_get_int ("sand_color_g=128");
  p->b = gnome_config_get_int ("sand_color_b=38");

  p->decorations
    = gnome_config_get_bool ("decorations");
  p->layer = gnome_config_get_int ("layer=-1");
  if (p->layer == (GnomeWinLayer)-1)  p->layer = WIN_LAYER_NORMAL;
  gnome_config_pop_prefix ();
}

static void
save_preferences (const struct preferences *p)
{
  gnome_config_clean_section ("/SandUhr/preferences");
  gnome_config_push_prefix ("/SandUhr/preferences/");

  if (p->alarm_type)  gnome_config_set_int ("alarm_type", p->alarm_type);
  if (p->cmd)  gnome_config_set_string ("alarm_cmd", p->cmd);
  if (p->snd)  gnome_config_set_string ("alarm_snd", p->snd);

  if (p->sand_type) gnome_config_set_int ("sand_type", p->sand_type);
  if (p->r) gnome_config_set_int ("sand_color_r", p->r);
  if (p->g) gnome_config_set_int ("sand_color_g", p->g);
  if (p->b) gnome_config_set_int ("sand_color_b", p->b);

  if (p->decorations) gnome_config_set_bool ("decorations", p->decorations);
  if (p->layer != WIN_LAYER_NORMAL) gnome_config_set_int ("layer", p->layer);

  gnome_config_pop_prefix ();
  gnome_config_sync ();
}

void
load_defaults (void)
{
  load_preferences (&defaults);
}

void
apply_defaults (struct timer *timer)
/* Fill the appropriate fields of TIMER with the default values.  */
{
  CORBA_Environment  ev;
  void *alarm = NULL;
  guint8  r, g, b;

  CORBA_exception_init (&ev);
  switch (defaults.alarm_type) {
  case at_BEEP:
    alarm = create_alarm_beep (timer->poa, 3, &ev);
    break;
  case at_SOUND:
    alarm = create_alarm_sound (timer->poa, defaults.snd, &ev);
    break;
  case at_COMMAND:
    alarm = create_alarm_command (timer->poa, defaults.cmd, &ev);
    break;
  }
  timer->alarm
    = PortableServer_POA_servant_to_reference (timer->poa, alarm, &ev);
  check_corba_error (&ev, NULL);
  
  if (defaults.sand_type == st_RANDOM) {
    r = rand() * 128.0 / RAND_MAX;
    g = rand() * 128.0 / RAND_MAX;
    b = rand() * 128.0 / RAND_MAX;
  } else {
    r = defaults.r;
    g = defaults.g;
    b = defaults.b;
  }
  sand_window_set_color (SAND_WINDOW (timer->window), r, g, b);
  gtk_object_set (GTK_OBJECT (timer->window),
		  SAND_WINDOW_ARG_DECORATIONS(defaults.decorations),
		  NULL);
  sand_window_set_layer (SAND_WINDOW (timer->window), defaults.layer);
}

static void
prepare_page0 (GtkWidget *preferences, const struct preferences *p)
/* Initialize the dialog's alarm action at page 0.  */
{
  if (p->snd) {
    GtkWidget *pref_file = lookup_widget (preferences, "pref_file");
    gtk_entry_set_text (GTK_ENTRY (pref_file), p->snd);
  }
  {
    GtkWidget *snd_entry = lookup_widget (preferences, "pref_file_entry");
    char *sound_dir = get_sound_dir ();

    gnome_file_entry_set_default_path (GNOME_FILE_ENTRY (snd_entry),
				       sound_dir);
    g_free (sound_dir);
  }
  if (p->cmd) {
    GtkWidget *cmd_entry = lookup_widget (preferences, "pref_cmdstring");
    gtk_entry_set_text (GTK_ENTRY (cmd_entry), p->cmd);
  }

  /* This must be last, because settings the GtkEntry widgets above
   * activates the corresponding toggle buttons.  */
  switch (p->alarm_type) {
  case at_BEEP:
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (preferences, "bell_pref")), TRUE);
    break;
  case at_SOUND:
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (preferences, "snd_pref")), TRUE);
    break;
  case at_COMMAND:
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (preferences, "cmd_pref")), TRUE);
    break;
  }
}

static void
prepare_page1 (GtkWidget *preferences, const struct preferences *p)
/* Initialize the dialog's sand settings at page 1.  */
{
  GtkWidget *color_picker;

  switch (p->sand_type) {
  case st_RANDOM:
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (preferences, "random_color_pref")),
       TRUE);
    break;
  case st_FIXED:
    gtk_toggle_button_set_active
      (GTK_TOGGLE_BUTTON (lookup_widget (preferences, "fixed_color_pref")),
       TRUE);
    break;
  }

  color_picker = lookup_widget (preferences, "color_pref");
  gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (color_picker),
			     p->r, p->g, p->b, 255);
}

static void
prepare_page2 (GtkWidget *preferences, const struct preferences *p)
/* Initialize the dialog's window attributes at page 2.  */
{
  GtkWidget *pref_deco = lookup_widget (preferences, "pref_deco");
  GtkWidget *layer_button;

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref_deco),
				p->decorations);
  switch (p->layer) {
  case WIN_LAYER_DESKTOP:
    layer_button = lookup_widget (preferences, "pref_desktop");
    break;
  case WIN_LAYER_BELOW:
    layer_button = lookup_widget (preferences, "pref_below");
    break;
  case WIN_LAYER_NORMAL:
    layer_button = lookup_widget (preferences, "pref_normal");
    break;
  case WIN_LAYER_ONTOP:
    layer_button = lookup_widget (preferences, "pref_ontop");
    break;
  default:
    layer_button = NULL;
  }
  if (layer_button) {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(layer_button), TRUE);
  }
}

void
prepare_preferences_dialog (GtkWidget *preferences)
{
  prepare_page0 (preferences, &defaults);
  prepare_page1 (preferences, &defaults);
  prepare_page2 (preferences, &defaults);
}

static void
decode_page0 (GtkWidget *dialog, struct preferences *p)
/* Set the default alarm action from page 0 of the dialog.  */
{
  GtkWidget *snd_entry, *cmd_entry;
  const char *str;

  if (gtk_toggle_button_get_active
      (GTK_TOGGLE_BUTTON (lookup_widget (dialog, "bell_pref")))) {
    p->alarm_type = at_BEEP;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (dialog, "snd_pref")))) {
    p->alarm_type = at_SOUND;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (dialog, "cmd_pref")))) {
    p->alarm_type = at_COMMAND;
  }

  snd_entry = lookup_widget (dialog, "pref_file");
  str = gtk_entry_get_text (GTK_ENTRY (snd_entry));
  g_free (p->snd);
  p->snd = (str && *str) ? g_strdup (str) : NULL;

  cmd_entry = lookup_widget (dialog, "pref_cmdstring");
  str = gtk_entry_get_text (GTK_ENTRY (cmd_entry));
  g_free (p->cmd);
  p->cmd = (str && *str) ? g_strdup (str) : NULL;
}

static void
decode_page1 (GtkWidget *dialog, struct preferences *p)
/* Set the default sand settings from page 1 of the dialog.  */
{
  GtkWidget *color_picker;
  guint8  a;

  if (gtk_toggle_button_get_active
      (GTK_TOGGLE_BUTTON (lookup_widget (dialog, "random_color_pref")))) {
    p->sand_type = st_RANDOM;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (dialog,"fixed_color_pref")))) {
    p->sand_type = st_FIXED;
  } else {
    g_assert_not_reached ();
  }

  color_picker = lookup_widget (dialog, "color_pref");
  gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (color_picker),
			     &p->r, &p->g, &p->b, &a);
}

static void
decode_page2 (GtkWidget *dialog, struct preferences *p)
/* Set the default window attributes from page 2 of the dialog.  */
{
  GtkWidget *pref_deco = lookup_widget (dialog, "pref_deco");

  p->decorations
    = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref_deco));

  if (gtk_toggle_button_get_active
      (GTK_TOGGLE_BUTTON (lookup_widget (dialog, "pref_desktop")))) {
    p->layer = WIN_LAYER_DESKTOP;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (dialog,
						"pref_below")))) {
    p->layer = WIN_LAYER_BELOW;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (dialog,
						"pref_normal")))) {
    p->layer = WIN_LAYER_NORMAL;
  } else if (gtk_toggle_button_get_active
	     (GTK_TOGGLE_BUTTON (lookup_widget (dialog,
						"pref_ontop")))) {
    p->layer = WIN_LAYER_ONTOP;
  } else {
    g_assert_not_reached ();
  }
}

void
decode_preferences (GtkWidget *dialog, int page)
{
  switch (page) {
  case 0:
    decode_page0 (dialog, &defaults);
    break;
  case 1:
    decode_page1 (dialog, &defaults);
    break;
  case 2:
    decode_page2 (dialog, &defaults);
    break;
  default:
    g_assert (page == -1);
    save_preferences (&defaults);
    break;
  }
}
