#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# Modal Dialog Support Routines

proc DialogCreate {top title args} {
    global ds9

    eval {toplevel $top -colormap $ds9(main)} $args
    wm title $top $title
    wm iconname $top $title
}

proc DialogCenter {w} {
    # center dialog
    wm withdraw $w

    global debug
    if {$debug(tcl,idletasks)} {
	puts "DialogCenter"
    }
    update idletasks

    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	       - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	       - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w

    # some window managers need a hint
    raise $w
}

proc DialogWait {top varname {focus {}}} {
    upvar $varname var
    global errorInfo

    if {[string length $focus] == 0} {
	set focus $top
    }
    set old [focus -displayof $top]
    focus $focus
    catch {tkwait visibility $top}
    catch {grab $top}
    tkwait variable $varname
    catch {grab release $top}
    focus $old

    # reset errorInfo
    set errorInfo {}
}

proc DialogDismiss {w} {
    destroy $w
}

# Simple List Box

proc SLBDialog {varname title width} {
    upvar #0 $varname var
    global $varname

    set w ".slb"

    set var(ok) 0
    set var(item) {}
    set var(value) {}

    DialogCreate $w $title -borderwidth 2

    frame $w.ed -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2

    pack $w.buttons -side bottom -fill x -ipadx 4 -ipady 4
    pack $w.ed -side top -fill both -expand true

    # lists
    scrollbar $w.ed.scroll -command "$w.ed.list yview"
    set var(listbox) [listbox $w.ed.list \
			  -yscroll "$w.ed.scroll set" \
			  -width $width \
			  -height 15 \
			  -setgrid 1 \
			  -selectmode single]
    grid $w.ed.list $w.ed.scroll -sticky news
    grid rowconfigure $w.ed 0 -weight 1
    grid columnconfigure $w.ed 0 -weight 1

    # buttons
    button $w.buttons.ok -text "OK"  -command "set ${varname}(ok) 1"
    button $w.buttons.cancel -text "Cancel" -command "set ${varname}(ok) 0"
    pack $w.buttons.ok $w.buttons.cancel -side left -expand true -padx 10

    # init
    for {set i 1} {$i <= $var(count)} {incr i} {
	$w.ed.list insert end $var($i,item)
    }
    $w.ed.list selection set 0

    bind $w <Double-1> "set ${varname}(ok) 1"
    bind $w <Return> "set ${varname}(ok) 1"

    bind $w <Up> "SLBArrow $varname -1"
    bind $w <Down> "SLBArrow $varname 1"

    DialogCenter $w
    DialogWait $w ${varname}(ok) $w.buttons.ok

    if {$var(ok)} {
	set i [expr [$var(listbox) curselection]+1]
	if {$i > 0 && $i <= $var(count)} {
	    set var(item) $var($i,item)
	    set var(value) $var($i,value)
	} else {
	    set var(item) {}
	    set var(value) {}
	}
    }

    DialogDismiss $w
}

proc SLBArrow {varname dir} {
    upvar #0 $varname var
    global $varname

    set which [$var(listbox) curselection]
    set end [$var(listbox) index end]

    $var(listbox) selection clear 0 end
    incr which $dir
    if {$which < 0} {
	set which 0
    }
    if {$which >= $end} {
	set which [expr $end -1]
    }
    $var(listbox) selection set $which
}

# Entry Dialog

proc EntryDialog {title message size varname} {
    upvar $varname var
    global ed

    set w ".ed"

    set ed(ok) 0
    set ed(value) $var

    DialogCreate $w $title -borderwidth 2
    frame $w.ed  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ed $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ed.title -text $message
    entry $w.ed.value -textvariable ed(value) -width $size
    pack $w.ed.title $w.ed.value -side top -padx 4 -pady 4
    pack $w.ed.title -anchor w
    pack $w.ed.value -ipadx 2 -ipady 2

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ed.value select range 0 end
    DialogWait $w ed(ok) $w.ed.value
    DialogDismiss $w

    if {$ed(ok)} {
	set var $ed(value)
    }
    
    set r $ed(ok)
    unset ed
    return $r
}

# Status Dialog

proc DisplayStatusMsg {w msg} {
    global ds9

    toplevel $w -colormap $ds9(main)
    wm protocol $w WM_DELETE_WINDOW DestroyStatusMsg

    frame $w.f -relief groove -borderwidth 2
    pack $w.f -fill x -padx 2 -pady 2 -ipadx 4 -ipady 4

    label $w.f.bitmap -bitmap "info"
    pack $w.f.bitmap -side left -padx 3m -pady 3m

    label $w.f.msg -justify left -text "$msg"
    catch {$w.f.msg configure -font \
	       -Adobe-Times-Medium-R-Normal--*-180-*-*-*-*-*-*
    }
    pack $w.f.msg -side right -expand 1 -fill both -padx 3m -pady 3m

    DialogCenter $w

    global debug
    if {$debug(tcl,idletasks)} {
	puts "DisplayStatusMsg"
    }
    update idletasks
}

proc DestroyStatusMsg {w} {
    destroy $w
}

# Simple Text Dialog

proc SimpleTextDialog {tt title width height action pos txt} {
    global ds9
    global st

    set st($tt,top) ".${tt}"
    set st($tt,mb) ".${tt}mb"

    # see if we already have a window visible

    if {[winfo exist $st($tt,top)]} {
	raise $st($tt,top)
    } else {
	# create window

	toplevel $st($tt,top) -colormap $ds9(main)
	wm title $st($tt,top) $title
	wm iconname $st($tt,top) $title
	wm group $st($tt,top) $ds9(top)
	wm protocol $st($tt,top) WM_DELETE_WINDOW "SimpleTextDestroy $tt"

	$st($tt,top) configure -menu $st($tt,mb)

	# simple menus

	menu $st($tt,mb) -tearoff 0
	$st($tt,mb) add cascade -label File -menu $st($tt,mb).file
	$st($tt,mb) add cascade -label Edit -menu $st($tt,mb).edit

	menu $st($tt,mb).file -tearoff 0
	$st($tt,mb).file add command -label "Save..." \
	    -command "SimpleTextSave $tt"
	$st($tt,mb).file add separator
	$st($tt,mb).file add command -label "Print..." \
	    -command "SimpleTextPrint $tt"
	$st($tt,mb).file add separator
	$st($tt,mb).file add command -label "Close" \
	    -command "SimpleTextDestroy $tt"

	menu $st($tt,mb).edit -tearoff 0
	$st($tt,mb).edit add command -label "Cut" -state disabled
	$st($tt,mb).edit add command -label "Copy" -state disabled \
	    -command "SimpleTextCopy $tt"
	$st($tt,mb).edit add command -label "Paste" -state disabled
	$st($tt,mb).edit add command -label "Clear" -state disabled \
	    -command "SimpleTextClear $tt"
	$st($tt,mb).edit add separator
	$st($tt,mb).edit add command -label "Select All" \
	    -command "SimpleTextSelectAll $tt"
	$st($tt,mb).edit add command -label "Select None" \
	    -command "SimpleTextSelectNone $tt"
	$st($tt,mb).edit add separator
	$st($tt,mb).edit add command -label Find... \
	    -command "SimpleTextFind $tt"
	$st($tt,mb).edit add command -label "Find Next" -state disabled \
	    -command "SimpleTextFindNext $tt"

	# create the text and scroll widgets
	
	set st($tt,text) [text $st($tt,top).text -height $height \
			      -width $width -wrap none \
			      -font {courier 12} \
			      -yscrollcommand "$st($tt,top).yscroll set" \
			      -xscrollcommand "$st($tt,top).xscroll set"]

	bind $st($tt,top).text <ButtonRelease-1> "+ SimpleTextUpdateMenu $tt"
	
	scrollbar $st($tt,top).yscroll -command [list $st($tt,text) yview] \
	    -orient vertical
	scrollbar $st($tt,top).xscroll -command [list $st($tt,text) xview] \
	    -orient horizontal

	grid $st($tt,text) $st($tt,top).yscroll -sticky news
	grid $st($tt,top).xscroll -stick news
	grid rowconfigure $st($tt,top) 0 -weight 1
	grid columnconfigure $st($tt,top) 0 -weight 1

	# some window managers need a hint
	raise $st($tt,top)
    }

    $st($tt,text) configure -state normal
    if {$action != "append"} {
	$st($tt,text) delete 1.0 end
    }
    $st($tt,text) insert end "$txt"
    switch -- $pos {
	top {$st($tt,text) see 1.0}
	bottom {$st($tt,text) see end}
    }

    global tcl_platform
    if {$tcl_platform(platform)!= "windows"} {
	$st($tt,text) configure -state disabled
    }
}

proc SimpleTextDestroy {tt} {
    global st

    destroy $st($tt,top)
    destroy $st($tt,mb)

    unset st($tt,top)
    unset st($tt,mb)
    unset st($tt,text)
}

proc SimpleTextUpdateMenu {tt} {
    global st

    if {[catch {selection get -displayof $st($tt,top)}]} {
	$st($tt,mb).edit entryconfig "Copy" -state disabled
	$st($tt,mb).edit entryconfig "Clear" -state disabled
	$st($tt,mb).edit entryconfig "Find Next" -state disabled
    } else {
	$st($tt,mb).edit entryconfig "Copy" -state normal
	$st($tt,mb).edit entryconfig "Clear" -state normal
	$st($tt,mb).edit entryconfig "Find Next" -state normal
    }
}

proc SimpleTextCopy {tt} {
    global st

    tk_textCopy $st($tt,text)
}

proc SimpleTextClear {tt} {
    global st
    
    $st($tt,text) configure -state normal
    $st($tt,text) delete sel.first sel.last
    SimpleTextUpdateMenu $tt
    $st($tt,text) configure -state disabled
}

proc SimpleTextSelectAll {tt} {
    global st

    $st($tt,text) tag add sel 1.0 end
    SimpleTextUpdateMenu $tt
}

proc SimpleTextSelectNone {tt} {
    global st

    $st($tt,text) tag remove sel 1.0 end
    SimpleTextUpdateMenu $tt
}

proc SimpleTextFind {tt} {
    global st
    global text

    $st($tt,text) tag remove sel 1.0 end
    set result "$text(search)"
    if {[EntryDialog "Search" "Enter Search Expression:" 40 result]} {
	set text(search) "$result"
	set start [$st($tt,text) search -nocase -count cnt \
		       -regexp -- $result 1.0 end]
	if {$start != ""} {
	    $st($tt,text) tag add sel $start "$start + $cnt chars"
	    $st($tt,text) see $start
	}
    }
    SimpleTextUpdateMenu $tt
}

proc SimpleTextFindNext {tt} {
    global st
    global text

    set result "$text(search)"
    set start [$st($tt,text) search -nocase -count cnt \
		   -regexp -- $result sel.last end]
    if {$start != ""} {
	$st($tt,text) tag remove sel 1.0 end
	$st($tt,text) tag add sel $start "$start + $cnt chars"
	$st($tt,text) see $start
    }
    SimpleTextUpdateMenu $tt
}

proc SimpleTextPrint {tt} {
    global st
    global ps
    global message

    if {[PrintDialog txt]} { 
	if {$ps(dest) == "file"} {
	    catch {set ch [open "| cat > $ps(filename)" w]}
	} else {
	    catch {set ch [open "| $ps(cmd)" w]}
	}

	if {$ch != ""} {
	    puts -nonewline $ch [$st($tt,text) get 1.0 end]
	    close $ch
	} else {
	    Error "$message(error,dialog,print)"
	    return
	}
    }
}

proc SimpleTextSave {tt} {
    global st
    global message

    set filename [SaveFileDialog textfbox]
    if {$filename != {}} {
	if {[catch {set ch [open "| cat > \"$filename\"" w]}]} {
	    Error "$message(error,dialog,save)"
	    return
	}
	puts -nonewline $ch [$st($tt,text) get 1.0 end]
	close $ch
    }
}

