#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# capture general errors
# this only captures gui errors, not xpa errors
proc bgerror {err} {
    global cursor

    tk_messageBox -message "An internal error has been detected: $err" \
	-type ok -icon error

    # reset cursor
    set cursor(count) 0
    UnsetWatchCursor
}

# unwind xpa errors
# requires catch {} to allow a check to take place
proc InitXPAError {xpa} {
    global ds9
    global errorInfo

    set ds9(xpa,ptr) $xpa
    set errorInfo {}
}

proc CatchXPAError {xpa} {
    global ds9
    global errorInfo

    set ds9(xpa,ptr) 0
    if {$errorInfo != {}} {
	# reset cursor
	set cursor(count) 0
	UnsetWatchCursor

	xpaerror $xpa "$errorInfo"
	set errorInfo {}
    }
}

# force capture xpa/interactive errors
proc Error {message} {
    global ds9

    if {$ds9(xpa,ptr) == 0} {
	tk_messageBox -message $message -type ok -icon error
    } else {
	xpaerror $ds9(xpa,ptr) $message
    }

    # reset cursor
    set cursor(count) 0
    UnsetWatchCursor
}

proc Message {message} {
    global ds9

    tk_messageBox -message $message -type ok -icon info
}

proc tmpnam {base ext} {
    global ds9

    for {set ii 0} {$ii<10} {incr ii} {
	set fn "$ds9(tmpdir)/$base[clock clicks]$ext"
	if {![file exists $fn]} {
	    return $fn
	}
    }

    # give up
    return $ds9(tmpdir)/$base$ext
}

proc ToYesNo {value} {
    if {$value == 1} {
	return "yes\n"
    } else {
	return "no\n"
    }
}

proc FromYesNo {value} {
    set v [string tolower $value]

    if {$v == "no" || $v == "false" || $v == "off" || $v == 0} {
	return 0
    } else {
	return 1
    }
}

proc RealizeDS9 {} {
    # this has to come first, to realize the canvas
    global debug
    if {$debug(tcl,idletasks)} {
	puts "RealizeDS9"
    }

# idletasks fails for windows. we need to process all events to make
# sure all windows are realized
#    update idletasks
    update

    # now, update all frames
    global ds9
    foreach f $ds9(frames) {
	$f update now
    }
}

proc Sex2H {str} {
    scan $str "%d:%d:%f" h m s
    return [expr $h+($m/60.)+($s/(60.*60.))]
}

proc Sex2D {str} {
    set d 0
    set arcm 0
    set arcs 0
    scan $str "%d:%f:%f" d arcm arcs

    if {$d >= 0} {
	return [expr $d+($arcm/60.)+($arcs/(60.*60.))]
    } else {
	return [expr $d-($arcm/60.)-($arcs/(60.*60.))]
    }
}

proc SetCursor {cursor} {
    global ds9
    global iis
    global current

    # if init phase, don't change cursor
    if {$ds9(init)} {
	return
    }

    # if iis cursor mode, don't change cursor
    if {$iis(state)} {
	return
    }

    if {$ds9(cursor) && ($current(cursor) != $cursor)} {
	set current(cursor) $cursor

	if {$cursor != ""} {
	    $ds9(canvas) configure -cursor $cursor
	} else {
	    $ds9(canvas) configure -cursor {}
	}
    }
}

# NOTE
# This is where everything gets done. In particular, <Configure> events are
# processed here. Also, any screen updates are done. In short, if it changes
# the geometry or changes the contents, you need to call Set/UnsetWatchCursor
proc SetWatchCursor {} {
    global ds9
    global cursor

    # if init phase, don't change cursor

    if {$ds9(init)} {
	return
    }

    # if not xpa update, don't change cursor/update
    if {!$ds9(xpa,idletasks)} {
	return
    }

    global debug
    if {$debug(tcl,watch)} {
	puts "SetWatchCursor Start $cursor(count)"
    }

    if {$cursor(count) == 0} {
	set cursor(save) [$ds9(canvas) cget -cursor]
	$ds9(canvas) configure -cursor {}
	$ds9(main) configure -cursor watch

	global debug
	if {$debug(tcl,idletasks)} {
	    puts "SetWatchCursor"
	}
	update idletasks
    }
    incr cursor(count)

    if {$debug(tcl,watch)} {
	puts "SetWatchCursor End $cursor(count)"
    }
}

proc UnsetWatchCursor {} {
    global ds9
    global cursor

    # if init phase, don't change cursor

    if {$ds9(init)} {
	return
    }

    # if not xpa update, don't change cursor/update
    if {!$ds9(xpa,idletasks)} {
	return
    }

    global debug
    if {$debug(tcl,watch)} {
	puts "UnsetWatchCursor Start $cursor(count)"
    }

    if {$cursor(count)>0} {
	incr cursor(count) -1
    }

    if {$cursor(count) == 0} {
	$ds9(main) configure -cursor {}
	$ds9(canvas) configure -cursor $cursor(save)

	global debug
	if {$debug(tcl,idletasks)} {
	    puts "UnsetWatchCursor"
	}
	update idletasks
    }    

    if {$debug(tcl,watch)} {
	puts "UnsetWatchCursor End $cursor(count)"
    }
}

proc CursorTimer {} {
    global ds9
    global cursor

    switch -- $cursor(timer) {
	0 {
	    set cursor(timer,abort) 0
	    set cursor(timer) 0
	    set cursor(id) 0
	    $ds9(canvas) configure -cursor {}
	}
	1 {
	    $ds9(canvas) configure -cursor circle
	    set cursor(timer) 2
	    set cursor(id) [after 1000 CursorTimer]
	}
	2 {
	    $ds9(canvas) configure -cursor dot
	    set cursor(timer) 1
	    set cursor(id) [after 1000 CursorTimer]
	}
    }
}

proc AboutBox {} {
    global help
    global ed

    set w ".about"
    set width 340
    set height 400

    set ed(ok) 0

    DialogCreate $w "About SAOImage DS9" -borderwidth 2
    frame $w.c -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.c $w.buttons -fill x -ipadx 4 -ipady 4

    canvas $w.c.c -background white -height $height -width $width
    pack $w.c.c
    
    set ed(sun) [image create photo -format gif -file doc/sun.gif]

    $w.c.c create image 0 0 -image $ed(sun) -anchor nw
    $w.c.c create text 140 20 -text $help(about) -anchor nw

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    grid $w.buttons.ok -padx 10

    bind $w <Return> {set ed(ok) 1}

    DialogCenter $w
    DialogWait $w ed(ok)
    DialogDismiss $w

    if {$ed(ok)} {
	image delete $ed(sun)
    }

    unset ed
}

proc QuitDS9 {} {
    catch {IISClose}
    focus {}
    exit
}

proc OpenSource {} {
    set filename [OpenFileDialog tclfbox]
    if {$filename != {}} {
	uplevel #0 "source \{$filename\}"
    }
}

proc ToggleFreeze {} {
    global ds9

    if {$ds9(freeze)} {
	set ds9(freeze) 0
	UnFreezeCanvas
	UnFreezePanner
    } else {
	set ds9(freeze) 1
	FreezeCanvas
	FreezePanner
    }
}

proc ChangeMode {} {
    global ds9

    bind $ds9(canvas) <Button-1> {}
    bind $ds9(canvas) <B1-Motion> {}
    bind $ds9(canvas) <ButtonRelease-1> {}

    foreach f $ds9(frames) {
	$f crosshair off
    }
    UpdateMarkerMenu

    foreach f $ds9(frames) {
	$f marker unselect all
    }

    ClearInfoBox
    ClearPixelTable
    ClearGraphData

    switch -- $ds9(mode) {
	none -
	pointer {SetCursor {}}
	crosshair {
	    foreach f $ds9(frames) {
		$f crosshair on
	    }
	    ShowGraphData
	    SetCursor crosshair
	}
	colorbar {
	    bind $ds9(canvas) <Button-1> {ButtonColormap %x %y}
	    bind $ds9(canvas) <B1-Motion> {MotionColormap %x %y}
	    bind $ds9(canvas) <ButtonRelease-1> {ReleaseColormap %x %y}
	    SetCursor center_ptr
	}
	zoom {SetCursor sizing}
	pan {SetCursor fleur}
	rotate {SetCursor exchange}
	examine {SetCursor target}
	illustrate {SetCursor arrow}
	pointer {SetCursor {}}
	imexam {}
    }
}

proc DisplayLog {item} {
    SimpleTextDialog ftplog "Mesage Log" 80 40 append bottom $item
}

proc ParseURL {url varname} {
    upvar $varname r

    set r(scheme) {}
    set r(authority) {}
    set r(path) {}
    set r(query) {}
    set r(fragment) {}
    set exp {^(([^:/?#]+):)?(//([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?}

    if {![regexp -nocase $exp $url x a r(scheme) c r(authority) r(path) f r(query) h r(fragment)]} {
	return 0
    }

    # check for windows disk drives
    global tcl_platform
    if {$tcl_platform(platform) == "windows"} {
	switch -- $r(scheme) {
	    {} -
	    ftp -
	    http -
	    file {}
	    default {
		set r(path) "$r(scheme):$r(path)"
		set r(scheme) {}
	    }
	}
    }

    return 1
}

proc DumpURL {varname} {
    upvar $varname r

    puts "r(scheme)=$r(scheme)"
    puts "r(authority)=$r(authority)"
    puts "r(path)=$r(path)"
    puts "r(query)=$r(query)"
    puts "r(fragment)=$r(fragment)"
}

proc FTPLog {s msg state} {
    global debug

    if {$debug(tcl,ftp)} {
	DisplayLog "$s $msg $state\n"
    }
}

proc HTTPLog {token} {
    global http
    global debug

    if {$debug(tcl,http)} {
	upvar #0 $token t

	DisplayLog "url: $t(url)\n"
	DisplayLog "http: $t(http)\n"
	DisplayLog "type: $t(type)\n"
	DisplayLog "currentsize: $t(currentsize)\n"
	DisplayLog "totalsize: $t(totalsize)\n"
	DisplayLog "status: $t(status)\n"
	if [info exists t(error)] {
	    DisplayLog "error: $t(error)\n"
	}
	DisplayLog "meta: [BreakUp $t(meta)]\n"
    }
}

proc BreakUp {str} {
    set r ""
    set l [string length $str]
    for {set i 0} {$i < $l} {incr i} {
	set c [string index $str $i]
	append r $c
	if {$c=="\}"} {
	    append r "\n"
	}
    }
    return $r
}

proc InPath {which} {
    global env
    global tcl_platform

    if {$tcl_platform(platform) != "windows"} {
	set target ${which}
	set paths [split $env(PATH) :]
    } else {
	set target ${which}.exe
	set paths [split $env(PATH) \;]
    }

    foreach p $paths {
	if {[file executable [file join $p $target]]} {
	    return 1
	}
    }
    return 0
}

proc HTTPDefaultDialog {} {
    global http
    global menu
    global ed

    set w ".http"

    set ed(ok) 0
    set ed(proxy) $http(proxy)
    set ed(proxy,host) $http(proxy,host)
    set ed(proxy,port) $http(proxy,port)
    set ed(auth) $http(auth)
    set ed(auth,user) $http(auth,user)
    set ed(auth,passwd) $http(auth,passwd)

    DialogCreate $w "Configure HTTP" -borderwidth 2
    frame $w.ed -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ed $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ed.tproxy -text "Use Proxy"
    checkbutton $w.ed.proxy -selectcolor $menu(selectcolor) \
	-variable ed(proxy)

    label $w.ed.thost -text "Proxy Host"
    entry $w.ed.host -textvariable ed(proxy,host) -width 30

    label $w.ed.tport -text "Proxy Port"
    entry $w.ed.port -textvariable ed(proxy,port) -width 10
    
    label $w.ed.tauth -text "Use Authentication"
    checkbutton $w.ed.auth -selectcolor $menu(selectcolor) \
	-variable ed(auth)

    label $w.ed.tuser -text "Auth Username"
    entry $w.ed.user -textvariable ed(auth,user) -width 30

    label $w.ed.tpasswd -text "Auth Password"
    entry $w.ed.passwd -textvariable ed(auth,passwd) -show "*" -width 10

    grid rowconfigure $w.ed 0 -pad 4
    grid rowconfigure $w.ed 1 -pad 4
    grid rowconfigure $w.ed 2 -pad 4

    grid rowconfigure $w.ed 3 -pad 4
    grid rowconfigure $w.ed 4 -pad 4
    grid rowconfigure $w.ed 5 -pad 4

    grid $w.ed.tproxy  $w.ed.proxy  -padx 4 -sticky w
    grid $w.ed.thost $w.ed.host -padx 4 -sticky w
    grid $w.ed.tport $w.ed.port -padx 4 -sticky w

    grid $w.ed.tauth  $w.ed.auth  -padx 4 -sticky w
    grid $w.ed.tuser $w.ed.user -padx 4 -sticky w
    grid $w.ed.tpasswd $w.ed.passwd -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ed.host select range 0 end
    DialogWait $w ed(ok) $w.ed.host
    DialogDismiss $w

    if {$ed(ok)} {
	set http(proxy) $ed(proxy)
	set http(proxy,host) $ed(proxy,host)
	set http(proxy,port) $ed(proxy,port)
	set http(auth) $ed(auth)
	set http(auth,user) $ed(auth,user)
	set http(auth,passwd) $ed(auth,passwd)

	ConfigHTTP
    }

    unset ed
}

proc ConfigHTTP {} {
    global http

    # set the User-Agent
    http::config -useragent ds9

    # set the proxy if requested
    if {$http(proxy)} {
	http::config -proxyhost $http(proxy,host) -proxyport $http(proxy,port)
    }
}

proc ProxyHTTP {} {
    global http

    set auth {}
    if {$http(proxy) && $http(auth)} {
	set auth [list "Proxy-Authorization" [concat "Basic" [base64::encode $http(auth,user):$http(auth,passwd)]]]
    } 

    return $auth
}

proc TmpDirDialog {} {
    global ed
    global ds9

    set w ".tmpdir"

    set size 60

    set ed(ok) 0
    set ed(value) "$ds9(tmpdir)"

    DialogCreate $w "Temporary File Directory" -borderwidth 2
    frame $w.tmp  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.tmp $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.tmp.title -text "Select Directory to be used for temporary files:"
    frame $w.tmp.a
    entry $w.tmp.a.value -textvariable ed(value) -width $size
    button $w.tmp.a.browse -text "Browse" -command "TmpDirOpen ed(value)"

    pack $w.tmp.a.value -ipadx 2 -ipady 2
    pack $w.tmp.a.value $w.tmp.a.browse -side left -padx 4 -pady 4
    pack $w.tmp.title $w.tmp.a -side top -padx 4 -pady 4
    pack $w.tmp.title -anchor w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.clear -text "Clear" -command [list set ed(value) {}]
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok $w.buttons.clear $w.buttons.cancel \
	-side left -expand 1 -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.tmp.a.value select range 0 end
    DialogWait $w ed(ok) $w.tmp.a.value
    DialogDismiss $w

    if {$ed(ok)} {
	if {$ed(value) == {}} {
	    set ed(value) {.}
	}
	set ds9(tmpdir) $ed(value)
    }
    
    set r $ed(ok)
    unset ed
    return $r
}

proc TmpDirOpen {varname} {
    upvar $varname var

    FileLast analysisfbox $var
    set var [file dirname [OpenFileDialog tmpdirfbox]]
}

proc SetLanguage {} {
    global ds9
    global menu

    # set English for defaults
    LanguageEnglish

    switch -- $ds9(language) {
	english {}
	francais {LanguageFrancais}
	danish {LanguageDanish}
    }
}

proc DumpCallStack {} {
    for {set x [expr [info level]-1]} {$x>0} {incr x -1} {
	puts "$x: [info level $x]"
    }
}

proc DumpArray {varname} {
    upvar $varname var
    global $varname
    foreach f [array names $varname] {
	puts "${varname}($f) = $var($f)"
    }
}

proc ProcessSourceCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    set fn [file normalize [lindex $var $i]]
    uplevel #0 "source $fn"
}

proc ProcessIconifyCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9

    if {[FromYesNo [lindex $var $i]]} {
	wm iconify $ds9(top)
    } else {
	wm deiconify $ds9(top)
    }
}

proc ProcessLowerCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    lower $ds9(top)
}

proc ProcessRaiseCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    raise $ds9(top)
}

