// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "colorbarbase.h"
#include "util.h"
#include "tkpostscript.h"

// Parser Stuff

#undef yyFlexLexer
#define yyFlexLexer cbFlexLexer
#include <FlexLexer.h>

static cbFlexLexer* lex = NULL;    // used by cblex
static ColorbarBase* cb = NULL;        // used by cberror
extern int cbparse(void*);

// Non-Member Functions

int cblex()
{
  return (lex ? lex->yylex() : 0);
}

void cberror(const char* m)
{
  if (cb) {
    cb->error(m);
    const char* cmd = lex ? lex->YYText() : (const char*)NULL;
    if (cmd && cmd[0] != '\n') {
      cb->error(": ");
      cb->error(cmd);
    }
  }
}

// Public Member Functions

ColorbarBase::ColorbarBase(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item) 
  : Widget(i, c, item)
{
  colorCells = NULL;
  colorCount = 0;

  borderGC = NULL;
}

ColorbarBase::~ColorbarBase()
{
  if (colorCells)
    delete [] colorCells;

  if (borderGC)
    XFreeGC(display, borderGC);

}

// Required Virtual Functions

int ColorbarBase::parse(istringstream& istr)
{
  result = TCL_OK;
  lex = new cbFlexLexer(&istr);
  cb = this;
  cbparse(this);

  delete lex;
  return result;
}

int ColorbarBase::configure(int argc, const char* argv[], int flags)
{
  if (Widget::configure(argc, argv, flags) == TCL_ERROR)
    return TCL_ERROR;

  // only valid for the initial configuration call

  if (flags != TK_CONFIG_ARGV_ONLY)
    return initColormap();
  else
    return TCL_OK;
}

void ColorbarBase::renderBorder()
{
  if (!borderGC) {
    borderGC = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
    XSetForeground(display, borderGC, greyColor->pixel);
    XSetLineAttributes(display, borderGC, 1, LineSolid, CapButt, JoinMiter);
  }

  XDRAWRECTANGLE(display, pixmap, borderGC, 0, 0, 
		 options->width-1, options->height-1);
}

int ColorbarBase::postscriptProc(int prepass)
{
  TkPostscriptInfo* psInfo = (TkPostscriptInfo*)(((TkCanvas*)canvas)->psInfo);

  if (!visible)
    return TCL_OK;

  if (prepass)
    return TCL_OK;

  // image
  Tcl_AppendResult(interp, "gsave\n", NULL);

  Vector org = psOrigin();
  float scale = psInfo->scale * psResolution / 72.;
  int width = (int)(options->width * scale);
  int height = (int)(options->height * scale);

  ostringstream str;
  str << org[0] << ' ' << org[1] << " translate" << endl
      << 1/scale << ' ' << 1/scale << " scale" << endl
      << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);

  switch (psLevel) {
  case 1:
    switch (psColorSpace) {
    case BW:
    case GRAY:
      psLevel1(GRAY, width, height, scale);
      break;
    case RGB:
    case CMYK:
      psLevel1(RGB, width, height, scale);
      break;
    }
    break;
  case 2:
    psLevel2(psColorSpace, width, height, scale);
    break;
  }
  Tcl_AppendResult(interp, "grestore\n", NULL);

  // border
  Tcl_AppendResult(interp, "gsave\n", NULL);

  Vector ll(0,0);
  Vector lr(options->width-1,0);
  Vector ur(options->width-1,options->height-1);
  Vector ul(0,options->height-1);

  {
    ostringstream str;
    str << org[0] << ' ' << org[1] << " translate" << endl;
    str << ".5 setlinewidth" << endl;

    str << "newpath" 
	<< ll << "moveto" 
	<< lr << "lineto" 
	<< ur << "lineto" 
	<< ul << "lineto" 
	<< " closepath stroke" << endl << ends;

    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
  Tcl_AppendResult(interp, "grestore\n", NULL);

  return TCL_OK;
}

