// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>

#include "colorbarrgb.h"
#include "util.h"

ColorbarRGB::ColorbarRGB(Tcl_Interp* i,Tk_Canvas c,Tk_Item* item) 
  : ColorbarBase(i, c, item)
{
  channel = 0;

  for (int i=0; i<3; i++) {
    bias[i] = .5;
    contrast[i] = 1.0;
  }
}

ColorbarRGB::~ColorbarRGB()
{
}

// Public Member Functions

void ColorbarRGB::adjustCmd(float c, float b)
{
  contrast[channel] = c;
  bias[channel] = b;

  updateColors();
}

void ColorbarRGB::getBiasCmd()
{
  ostringstream str;
  str << bias[channel] << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void ColorbarRGB::getColormapCmd()
{
  // use fixed so that the frame parser will not be confused with an int
  // as the first number
  ostringstream str;
  str << "rgb " << setiosflags(ios::fixed);
  for (int i=0; i<3; i++)
    str << bias[i] << ' ';
  for (int i=0; i<3; i++)
    str << contrast[i] << ' ';
  str << (unsigned short*)colorCells << ' ' << colorCount << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void ColorbarRGB::getColormapNameCmd(int id)
{
  Tcl_AppendResult(interp, "rgb", NULL);
}

void ColorbarRGB::getContrastCmd()
{
  ostringstream str;
  str << contrast[channel] << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}

void ColorbarRGB::getCurrentNameCmd()
{
  Tcl_AppendResult(interp, "rgb", NULL);
}

void ColorbarRGB::getCurrentIDCmd()
{
  Tcl_AppendResult(interp, "rgb", NULL);
}

void ColorbarRGB::getCurrentFileNameCmd()
{
  Tcl_AppendResult(interp, "rgb", NULL);
}

void ColorbarRGB::getRGBChannelCmd()
{
  switch (channel) {
  case 0:
    Tcl_AppendResult(interp, "red", NULL);
    return;
  case 1:
    Tcl_AppendResult(interp, "green", NULL);
    return;
  case 2:
    Tcl_AppendResult(interp, "blue", NULL);
    return;
  }
}

void ColorbarRGB::getInvertCmd()
{
  Tcl_AppendResult(interp, "1", NULL);
}

void ColorbarRGB::setColormapCmd(float rb, float gb, float bb, 
				 float rc, float gc, float bc)

{
  bias[0] = rb;
  bias[1] = gb;
  bias[2] = bb;
  
  contrast[0] = rc;
  contrast[1] = gc;
  contrast[2] = bc;

  updateColors();
}

void ColorbarRGB::setRGBChannelCmd(const char* c)
{
  if (!strncmp(c,"red",3))
    channel = 0;
  else if (!strncmp(c,"gre",3))
    channel = 1;
  else if (!strncmp(c,"blu",3))
    channel = 2;
  else
    channel = 0;
}

// Private Member Functions

void ColorbarRGB::reset()
{
  for (int i=0; i<3; i++) {
    bias[i] = .5;
    contrast[i] = 1.0;
  }

  updateColors();
}

int ColorbarRGB::calcContrastBias(int i, float bias, float constrast)
{
  if (fabs(bias - 0.5) < 0.0001 && fabs(constrast - 1.0) < 0.0001)
    return i;
  
  // map i to range of 0 to 1.0
  // shift by bias (if invert, bias = 1-bias)
  // multiply by contrast
  // shift to center of region
  // expand back to number of dynamic colors

  int r = 
    (int)(((((float)i / colorCount) - bias) * constrast + .5 ) * colorCount);

  // clip to bounds if out of range

  if (r < 0)
    return 0;
  else if (r >= colorCount)
    return colorCount-1;
  else
    return r;
}
