// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#include "file.h"
#include "util.h"
#include "outfile.h"
#include "outchannel.h"
#include "outsocket.h"

// Parser Stuff

#undef yyFlexLexer
#define yyFlexLexer ffFlexLexer
#include <FlexLexer.h>

ffFlexLexer* fflexx = NULL;    // used by frlex
static FitsFile* ff = NULL;    // used by frerror
extern int ffparse(void*);

int fflex()
{
  return (fflexx ? fflexx->yylex() : 0);
}

void fferror(const char* m)
{
  if (ff) {
    ff->error(m);
    const char* cmd = fflexx ? fflexx->YYText() : (const char*)NULL;
    if (cmd && cmd[0] != '\n') {
      ff->error(": ");
      ff->error(cmd);
    }
  }
}

FitsFile::FitsFile()
{
  primary_ = NULL;
  managePrimary_ = 0;

  head_ = NULL;
  manageHead_ = 1;

  data_ = NULL;

  ext_ = 0;
  inherit_ = 0;
  byteswap_ = lsb();
  orgFits_ = 1;
  valid_ = 0;

  pName_ = NULL;
  pExt_ = NULL;
  pIndex_ = 0;
  pFilter_ = NULL;
  pSliceFilter_ = NULL;
  pBinX_ = NULL;
  pBinY_ = NULL;
  pBinZ_ = NULL;
  validZlim_ = 0;

  pWidth_ = 0;
  pHeight_ = 0;
  pDepth_ = 1;
  pBitpix_ = 0;
  pSkip_ = 0;
  pArch_ = NATIVE;
}

FitsFile::~FitsFile()
{
  if (manageHead_ && head_)
    delete head_;

  if (managePrimary_ && primary_)
    delete primary_;

  if (pName_)
    delete [] pName_;

  if (pExt_)
    delete [] pExt_;

  if (pFilter_)
    delete [] pFilter_;

  if (pSliceFilter_)
    delete [] pSliceFilter_;

  if (pBinX_)
    delete [] pBinX_;

  if (pBinY_)
    delete [] pBinY_;

  if (pBinZ_)
    delete [] pBinZ_;
}

void FitsFile::parse(const char* fn)
{
  if (fn) {
    string x(fn);
    istringstream str(x);
    fflexx = new ffFlexLexer(&str);
    ff = this;
    ffparse(this);

    delete fflexx;
    fflexx = NULL;
  }

  if (!pBinX_ && !pBinY_) {
    char *env;
    if ((env = getenv("DS9_BINKEY"))) {
      string x(env);
      istringstream str(x);
      fflexx = new ffFlexLexer(&str);
      ff = this;
      ffparse(this);

      delete fflexx;
      fflexx = NULL;
    }
  }

  if (!pWidth_ && !pHeight_ && !pBitpix_) {
    char *env;
    if ((env = getenv("DS9_ARRAY"))) {
      string x(env);
      istringstream str(x);
      fflexx = new ffFlexLexer(&str);
      ff = this;
      ffparse(this);

      delete fflexx;
      fflexx = NULL;
    }
  }
}

void FitsFile::error(const char* m)
{
  // do nothing for now
  // cerr << m << endl;
}

int FitsFile::findEnd(const char* blk)
{
  for (int j=0; j<FTY_BLOCK; j+=FTY_CARDLEN)
    // only check for 4 chars
    if (!strncmp("END ", blk+j,4))
      return 1;

  return 0;
}

void FitsFile::setpName(const char* n)
{
  if (pName_)
    delete [] pName_;

  pName_ = dupstr(n);
}

void FitsFile::setpExt(const char* n)
{
  if (pExt_)
    delete [] pExt_;

  pExt_ = dupstr(n);
}

void FitsFile::setpFilter(const char* filt)
{
  if (pFilter_)
    delete [] pFilter_;

  pFilter_ = dupstr(filt);
}

void FitsFile::setpSliceFilter(const char* filt)
{
  if (pSliceFilter_)
    delete [] pSliceFilter_;

  pSliceFilter_ = dupstr(filt);
}

void FitsFile::setpBinX(const char* x)
{
  if (pBinX_)
    delete [] pBinX_;

  pBinX_ = dupstr(x);
}

void FitsFile::setpBinY(const char* y)
{
  if (pBinY_)
    delete [] pBinY_;

  pBinY_ = dupstr(y);
}

void FitsFile::setpBinZ(const char* z)
{
  if (pBinZ_)
    delete [] pBinZ_;

  pBinZ_ = dupstr(z);
}

Vector FitsFile::getBinZlim()
{
  if (!validZlim_ && pBinZ_) {
    binZlim_ = getColMinMax(pBinZ_);
    validZlim_ = 1;
  }
  return binZlim_;
}

Vector FitsFile::getColMinMax(const char* name)
{
  if (isTable()) {
    FitsTableHDU* hdu = (FitsTableHDU*)(head()->hdu());
    FitsColumn* col = hdu->find(name);
    if (col) {
      if (!col->hasMinMax()) {
	double zmin =  DBL_MAX;
	double zmax = -DBL_MIN;
	int rowlen = hdu->width();
	int numrow = hdu->rows();

	char* ptr = (char*)data();
	for (int i=0; i<numrow; i++, ptr+=rowlen) {
	  // for memory models that support internal paging
	  ptr = page(ptr, rowlen);

	  register double z = col->value(ptr);
	  if (z < zmin)
	    zmin = z;
	  if (z > zmax)
	    zmax = z;
	}
	// for memory models that support internal paging
	resetpage();

	col->setMin(zmin);
	col->setMax(zmax);
	return Vector(zmin,zmax);
      }
      else
	return Vector(col->getMin(), col->getMax());
    }
  }
  return Vector();
}

int FitsFile::validArrayParams()
{
  if (!pWidth_ || !pHeight_ || !pBitpix_)
    return 0;

  // check for valid bitpix
  switch (pBitpix_) {
  case 8:
  case 16:
  case -16:
  case 32:
  case -32:
  case -64:
    break;
  default:
    return 0;
  }

  return 1;
}

void FitsFile::setArrayByteSwap()
{
  switch (pArch_) {
  case NATIVE:
    byteswap_ = 0;
    break;
  case BIGENDIAN:
    byteswap_ = lsb();
    break;
  case LITTLEENDIAN:
    byteswap_ = !lsb();
    break;
  }
}

int FitsFile::find(const char* name)
{
  if (head_)
    if (head_->find(name))
      return 1;
    else
      if (primary_ && inherit_)
	if (primary_->find(name))
	  return 1;

  return 0;
}

int FitsFile::getLogical(const char* name, int def)
{
  if (head_) {
    int r = head_->getLogical(name,def);
    if (r != def)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getLogical(name,def);
  }

  return def;
}

int FitsFile::getInteger(const char* name, int def)
{
  if (head_) {
    int r = head_->getInteger(name,def);
    if (r != def)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getInteger(name,def);
  }

  return def;
}

double FitsFile::getReal(const char* name, double def)
{
  if (head_) {
    double r = head_->getReal(name,def);
    if (r != def)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getReal(name,def);
  }

  return def;
}

void FitsFile::getComplex(const char* name, double* real, double* img,
			  double rdef, double idef)
{
  if (head_) {
    head_->getComplex(name, real, img, rdef, idef);
    if (*real != rdef || *img != idef)
      return;
    else
      if (primary_ && inherit_) {
	primary_->getComplex(name, real, img, rdef, idef);
	return;
      }
  }

  *real = rdef;
  *img = idef;
}

char* FitsFile::getString(const char* name)
{
  if (head_) {
    char* r = head_->getString(name);
    if (r)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getString(name);
  }

  return NULL;
}

char* FitsFile::getComment(const char* name)
{
  if (head_) {
    char* r = head_->getComment(name);
    if (r)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getComment(name);
  }

  return NULL;
}

char* FitsFile::getKeyword(const char* name)
{
  if (head_) {
    char* r = head_->getKeyword(name);
    if (r)
      return r;
    else
      if (primary_ && inherit_)
	return primary_->getKeyword(name);
  }

  return NULL;
}

// save fits image -- as a primary hdu image

int FitsFile::saveFitsImageFile(const char* fn, int compress)
{
  if (!valid_ || !isImage())
    return 0;

  if (!compress) {
    OutFitsFile str(fn);
    if (str.valid())
      return saveFitsImage(str);
  }
  else {
    OutFitsFileGZ str(fn);
    if (str.valid())
      return saveFitsImage(str);
  }
  return 0;
}

int FitsFile::saveFitsImageChannel(Tcl_Interp* interp, const char* ch, 
				   int compress)
{
  if (!valid_ || !isImage())
    return 0;

  OutFitsChannel str(interp, ch);
  if (str.valid())
    return saveFitsImage(str);
  return 0;
}

int FitsFile::saveFitsImageSocket(int s, int compress)
{
  if (!valid_ || !isImage())
    return 0;

  if (!compress) {
    OutFitsSocket str(s);
    if (str.valid())
      return saveFitsImage(str);
  }
  else {
    OutFitsSocketGZ str(s);
    if (str.valid())
      return saveFitsImage(str);
  }
  return 0;
}

int FitsFile::saveFitsImage(OutFitsStream& str)
{
  // write header
  // we have a problem here. the header may be an xtension, so lets force a
  // first line of 'SIMPLE' and then write the rest of the header
  char simple[80];
  memset(simple,' ',80);
  memcpy(simple,"SIMPLE  =                    T / Fits standard",46);
  str.write(simple, 80);
  str.write(head()->cards()+80, head()->headbytes()-80);

  // write valid data
  // our data may be short (mmap or bad fits), so write valid data
  // then write the pad, becareful with arch, if array

  if (orgFits_)
    str.write((char*)data(), head()->realbytes());
  else {
    switch (pArch_) {
    case NATIVE:
      if (!lsb())
	str.write((char*)data(), head()->realbytes());
      else
	str.writeSwap((char*)data(), head()->realbytes(), pBitpix_);
      break;
    case BIGENDIAN:
      str.write((char*)data(), head()->realbytes());
      break;
    case LITTLEENDIAN:
      str.writeSwap((char*)data(), head()->realbytes(), pBitpix_);
      break;
    }
  }

  // we may need to add a buffer to round out to block size
  int diff = head()->databytes() - head()->realbytes();
  if (diff>0) {
    char* buf = new char[diff];
    memset(buf,'\0',diff);
    str.write(buf, diff);
    delete buf;
  }
  return 1;
}

int FitsFile::saveArrayFile(const char* fn)
{
  if (!valid_ || !isImage())
    return 0;

  OutFitsFile str(fn);
  if (str.valid())
    return saveArray(str);

  return 0;
}

int FitsFile::saveArrayChannel(Tcl_Interp* interp, const char* ch)
{
  if (!valid_ || !isImage())
    return 0;

  OutFitsChannel str(interp, ch);
  if (str.valid())
    return saveArray(str);

  return 0;
}

int FitsFile::saveArraySocket(int s)
{
  if (!valid_ || !isImage())
    return 0;

  OutFitsSocket str(s);
  if (str.valid())
    return saveArray(str);

  return 0;
}

int FitsFile::saveArray(OutFitsStream& str)
{
  // write valid data
  // if fits, dump BIGENDIAN, if ARRAY, dump based on how read
  if (orgFits_)
    str.write((char*)data(), head()->realbytes());
  else {
    switch (pArch_) {
    case NATIVE:
      if (!lsb())
	str.write((char*)data(), head()->realbytes());
      else
	str.writeSwap((char*)data(), head()->realbytes(), pBitpix_);
      break;
    case BIGENDIAN:
      str.write((char*)data(), head()->realbytes());
      break;
    case LITTLEENDIAN:
      str.writeSwap((char*)data(), head()->realbytes(), pBitpix_);
      break;
    }
  }

  return 1;
}

// save Fits Table -- save priminary hdu, then extension hdu, then table

int FitsFile::saveFitsTableFile(const char* fn, int compress)
{
  if (!valid_ || !isTable())
    return 0;

  if (!compress) {
    OutFitsFile str(fn);
    if (str.valid())
      return saveFitsTable(str);
  }
  else {
    OutFitsFileGZ str(fn);
    if (str.valid())
      return saveFitsTable(str);
  }
  return 0;
}

int FitsFile::saveFitsTableChannel(Tcl_Interp* interp, const char* ch, 
				   int compress)
{
  if (!valid_ || !isTable())
    return 0;

  OutFitsChannel str(interp, ch);
  if (str.valid())
    return saveFitsTable(str);
  return 0;
}

int FitsFile::saveFitsTableSocket(int s, int compress)
{
  if (!valid_ || !isTable())
    return 0;

  if (!compress) {
    OutFitsSocket str(s);
    if (str.valid())
      return saveFitsTable(str);
  }
  else {
    OutFitsSocketGZ str(s);
    if (str.valid())
      return saveFitsTable(str);
  }
  return 0;
}

int FitsFile::saveFitsTable(OutFitsStream& str)
{
  // primary header
  str.write(primary()->cards(), primary()->headbytes());

  // now, ext header
  str.write(head()->cards(), head()->headbytes());

  // write valid data
  // our data may be short (mmap or bad fits), so write valid data
  // then write the pad, becareful with arch, if array

  if (orgFits_)
    str.write((char*)data(), head()->realbytes());
  else {
    switch (pArch_) {
    case NATIVE:
      if (!lsb())
	str.write((char*)data(), head()->realbytes());
      else
	str.writeSwap((char*)data(), head()->realbytes(), pBitpix_);
      break;
    case BIGENDIAN:
      str.write((char*)data(), head()->realbytes());
      break;
    case LITTLEENDIAN:
      str.writeSwap((char*)data(), head()->realbytes(), pBitpix_);
      break;
    }
  }

  // we may need to add a buffer to round out to block size
  int diff = head()->databytes() - head()->realbytes();
  if (diff>0) {
    char* buf = new char[diff];
    memset(buf,'\0',diff);
    str.write(buf, diff);
    delete buf;
  }
  return 1;
}


