// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "box.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Box::Box(const Box& a) : BaseBox(a) {}

Box::Box(FrameBase* p, const Vector& ctr, 
	 const Vector& seg, 
	 double ang,
	 const char* clr, int wth, const char* fnt, const char* txt,
	 unsigned short prop, const char* cmt, 
	 const List<Tag>& tg, const List<CallBack>& cb)
  : BaseBox(p, ctr, ang, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = 1;
  annuli_ = new Vector[1];
  annuli_[0] = seg;

  strcpy(type,"box");
  numHandle = 4;

  updateBBox();
}

void Box::editBegin(int h)
{
  switch (h) {
  case 1:
    return;
  case 2:
    annuli_[0] = Vector(-annuli_[0][0],annuli_[0][1]);
    return;
  case 3:
    annuli_[0] = -annuli_[0];
    return;
  case 4:
    annuli_[0] = Vector(annuli_[0][0],-annuli_[0][1]);
    return;
  }

  doCallBack(CallBack::EDITBEGINCB);
}

void Box::edit(const Vector& v, int h)
{
  Matrix mm = bckCanvasMatrix();
  Matrix nn = mm.invert();

  // This annuli_s about the opposite node
  Vector ov = annuli_[0]/2 * nn;
  annuli_[0] = (annuli_[0]/2) - (v*mm);
  Vector nv = annuli_[0]/2 * nn;
  center -= nv-ov;

  updateBBox();
  doCallBack(CallBack::EDITCB);
  doCallBack(CallBack::MOVECB);
}

void Box::editEnd()
{
  annuli_[0] = annuli_[0].abs();

  updateBBox();
  doCallBack(CallBack::EDITENDCB);
}

// list

void Box::list(ostream& str, CoordSystem sys, SkyFrame sky,
		   SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);
  listPre(str, sys, sky, ptr, strip, 0);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      Vector r = ptr->mapLenFromRef(annuli_[0],sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	  << r[0] << ',' << r[1] << ',' 
	  << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    Vector r = ptr->mapLenFromRef(annuli_[0],sys,ARCSEC);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
		<< r[0] << "\"" << ',' << r[1] << "\"" << ',' 
		<< radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;

	    Vector r = ptr->mapLenFromRef(annuli_[0],sys,ARCSEC);
	    str << type << '(' << ra << ',' << dec << ',' 
		<< r[0] << "\""<< ',' << r[1] << "\""<< ',' 
		<< radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	Vector r = ptr->mapLenFromRef(annuli_[0],sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << r[0] << ',' << r[1] << ',' 
	    << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Box::listCiao(ostream& str, CoordSystem sys, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listCiaoPre(str);

  // radius is always in image coords
  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,PHYSICAL);
      Vector s = ptr->mapLenFromRef(annuli_[0],PHYSICAL);
      str << "rotbox(" << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << s[0] << ',' << s[1] << ',' << radToDeg(angle) << ')';
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      Vector s = ptr->mapLenFromRef(annuli_[0],sys,ARCMIN);
      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
      string x(buf);
      istringstream wcs(x);
      wcs >> ra >> dec;
      str << "rotbox(" << ra << ',' << dec << ',' 
	  << s[0] << '\'' << ',' << s[1] << '\'' << ',' 
	  << radToDeg(angle) << ')';
    }
  }

  listCiaoPost(str, conj, strip);
}

void Box::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		     SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listSAOtngPre(str, strip);

  // radius is always in image coords
  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,IMAGE);
      Vector s = ptr->mapLenFromRef(annuli_[0],IMAGE);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << s[0] << ',' << s[1] << ',' << radToDeg(angle) << ')';
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  Vector s = ptr->mapLenFromRef(annuli_[0],IMAGE);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	      << s[0] << ',' << s[1] << ',' << radToDeg(angle) << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  Vector s = ptr->mapLenFromRef(annuli_[0],IMAGE);
	  char ra[16];
	  char dec[16];
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  str << type << '(' << ra << ',' << dec << ',' 
	      << s[0] << ',' << s[1] << ',' << radToDeg(angle) << ')';
	}
	break;
      }
    }
  }

  listSAOtngPost(str,strip);
}

void Box::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      Vector v = ptr->mapFromRef(center,sys);
      Vector r = ptr->mapLenFromRef(annuli_[0],IMAGE);
      listProsCoordSystem(str,sys,sky);
      str << "; "<< type << ' ' << setprecision(8) << v << r 
	  << radToDeg(angle);
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  Vector r = ptr->mapLenFromRef(annuli_[0],sys,ARCSEC);
	  listProsCoordSystem(str,sys,sky);
	  str << "; " << type << ' ' << setprecision(8) 
	      << v[0] << "d " << v[1] << "d "
	      << r[0] << "\" " << r[1] << "\" " << radToDeg(angle);
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char decc[16];
	  char *dec = decc;
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    dec++;

	  Vector r = ptr->mapLenFromRef(annuli_[0],sys,ARCSEC);
	  listProsCoordSystem(str,sys,sky);
	  str << "; " << type << ' ' << ra << ' ' << dec << ' ' 
	      << r[0] << "\" " << r[1] << "\" " << radToDeg(angle);
	}
	break;
      }
    }
  }

  listProsPost(str, strip);
}

void Box::listSAOimage(ostream& str, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listSAOimagePre(str);

  Vector v = ptr->mapFromRef(center,IMAGE);
  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
      << annuli_[0][0] << ',' << annuli_[0][1] << ',' 
      << radToDeg(angle) << ')';

  listSAOimagePost(str, strip);
}

