// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "cpanda.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Cpanda::Cpanda(const Cpanda& a) : BasePanda(a), BaseEllipse(a) {}

Cpanda::Cpanda(FrameBase* p, const Vector& ctr, 
	     double a1, double a2, int an,
	     double r1, double r2, int rn,
	     const char* clr, int wth, const char* fnt, const char* txt, 
	     unsigned short prop, const char* cmt, 
	     const List<Tag>& tg, const List<CallBack>& cb)
  : BasePanda(a1, a2, an), 
    BaseEllipse(p, ctr, 0, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = rn+1;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++) {
    double r = i*(r2-r1)/rn+r1;
    annuli_[i] = Vector(r,r);
  }

  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  strcpy(type, "panda");
  numHandle = 4 + numAnnuli_ + numAngles_;

  updateBBox();
}

Cpanda::Cpanda(FrameBase* p, const Vector& ctr, 
	     int an, double* a,
	     int rn, double* r,
	     const char* clr, int wth, const char* fnt, const char* txt, 
	     unsigned short prop, const char* cmt, 
	     const List<Tag>& tg, const List<CallBack>& cb)
  : BasePanda(an, a),
    BaseEllipse(p, ctr, 0, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = rn;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = Vector(r[i],r[i]);
  sortAnnuli();

  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  strcpy(type, "panda");
  numHandle = 4 + numAnnuli_ + numAngles_;

  updateBBox();
}

void Cpanda::updateHandles()
{
  // handles are in canvas coords
  // we can't garantee that the annuli_ have been sorted yet
  if (handle)
    delete [] handle;
  handle = new Vector[numHandle];

  Vector max;
  for(int i=0; i<numAnnuli_; i++)
    if (max[0]<annuli_[i][0])
      max = annuli_[i];
  Vector r = max;

  Matrix mm = fwdCanvasMatrix();

  handle[0] = Vector(-r[0],-r[1]) * mm;
  handle[1] = Vector( r[0],-r[1]) * mm;
  handle[2] = Vector( r[0], r[1]) * mm;
  handle[3] = Vector(-r[0], r[1]) * mm;

  // annuli_
  for (int i=0; i<numAnnuli_; i++)
    handle[i+4] = Vector(annuli_[i][0],0) * mm;

  // angles_
  Vector rr = annuli_[numAnnuli_-1];
  for (int i=0; i<numAngles_; i++)
    handle[4+numAnnuli_+i] = 
      Vector(rr[0]*cos(-angles_[i]),rr[1]*sin(-angles_[i])) * mm;
}

void Cpanda::edit(const Vector& v, int h)
{
  Matrix mm = bckCanvasMatrix();

  if (h<5) {
    // calc dist between edge of circle and handle
    double d = annuli_[numAnnuli_-1].length() - annuli_[numAnnuli_-1][0];

    for (int i=0; i<numAnnuli_; i++) {
      double r = ((v * mm).length() - d)/annuli_[numAnnuli_-1][0];
      annuli_[i] *= Vector(r,r);
    }
  }
  else if (h<(5+numAnnuli_)) {
    double d = (v * mm).length();
    annuli_[h-5] = Vector(d,d);
  }
  else {
    angles_[h-5-numAnnuli_] = -((v * mm).angle());
    sortAngles();
    startAng_ = radToDeg(angles_[0]);
    stopAng_ = radToDeg(angles_[numAngles_-1]);
  }
  
  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Cpanda::editEnd()
{
  sortAnnuli();
  sortAngles();
  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  updateBBox();
  doCallBack(CallBack::EDITENDCB);
}

int Cpanda::addAnnuli(const Vector& v)
{
  Matrix mm = bckCanvasMatrix();
  double l = (v * mm).length();

  // we need to insert into the next to the last location
  // new size array
  Vector* old = annuli_;
  annuli_ = new Vector[numAnnuli_+1];

  // copy old values
  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = old[i];

  // save last
  annuli_[numAnnuli_] = old[numAnnuli_-1];

  // delete old
  if (old)
    delete [] old;

  // new size on end
  annuli_[numAnnuli_-1] = Vector(l,l);

  numAnnuli_++;
  numHandle++;

  updateBBox();

  // return handle number
  return 4+numAnnuli_-1;
}

int Cpanda::addAngles(const Vector& v)
{
  Matrix mm = bckCanvasMatrix();
  addAngle(-((v * mm).angle()));
  numHandle++;

  updateBBox();

  // return handle number
  return 4+numAnnuli_+numAngles_-1;
}

void Cpanda::setAnglesAnnuli(double a1, double a2, int an, 
			    Vector r1, Vector r2, int rn)
{
  numAnnuli_ = rn+1;
  if (annuli_)
    delete [] annuli_;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = ((r2-r1)/rn)*i+r1;
  sortAnnuli();

  setAngles(a1,a2,an);
  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  numHandle = 4 + numAnnuli_ + numAngles_;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Cpanda::setAnglesAnnuli(const double* a, int an, const Vector* r, int rn)
{
  numAnnuli_ = rn;
  if (annuli_)
    delete [] annuli_;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = r[i];
  sortAnnuli();

  setAngles(an,a);
  startAng_ = radToDeg(angles_[0]);
  stopAng_ = radToDeg(angles_[numAngles_-1]);

  numHandle = 4 + numAnnuli_ + numAngles_;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Cpanda::deleteAnglesAnnuli(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (numAnnuli_>2 && hh<numAnnuli_) {
      // new annuli_ array
      Vector* old = annuli_;
      annuli_ = new Vector[numAnnuli_-1];

      // copy up to annuli_ in question
      for (int i=0; i<hh; i++)
	annuli_[i] = old[i];

      // copy remainder
      for (int i=hh; i<numAnnuli_-1; i++)
	annuli_[i] = old[i+1];

      if (old)
	delete [] old;
      numAnnuli_--;
    }
    else if (numAngles_>2 && hh<(numAnnuli_+numAngles_)) {
      hh -= numAnnuli_;
      deleteAngle(hh);
    }

    numHandle = 4 + numAnnuli_ + numAngles_;

    startAng_ = radToDeg(angles_[0]);
    stopAng_ = radToDeg(angles_[numAngles_-1]);

    updateBBox();
    doCallBack(CallBack::EDITCB);
  }
}

// private

void Cpanda::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  BaseEllipse::render(drawable, mx, mode);

  // angles
  Matrix mm = fwdRefMatrix() * mx;
  Vector rr0 = annuli_[0];
  Vector rr1 = annuli_[numAnnuli_-1];

  for (int i=0; i<numAngles_; i++) {
    Vector r0 = Vector(rr0[0]*cos(-angles_[i]),rr0[1]*sin(-angles_[i])) * mm;
    Vector r1 = Vector(rr1[0]*cos(-angles_[i]),rr1[1]*sin(-angles_[i])) * mm;

    switch (mode) {
    case SRC:
      if (selected) {
	if (i == 0)
	  XSetForeground(display, gc, parent->getRedColor());
	else if (i == numAngles_-1)
	  XSetForeground(display, gc, parent->getBlueColor());
	else
	  XSetForeground(display, gc, color);
      }
      else
	  XSetForeground(display, gc, color);
      break;
    case XOR:
      break;
    }

    XDRAWLINE(display, drawable, gc, (int)r0[0], (int)r0[1], 
	      (int)r1[0], (int)r1[1]);    
  }
}

void Cpanda::ps(int mode)
{
  BaseEllipse::ps(mode);

  // annuli_
  Marker::ps(mode);

  Matrix mm = fwdCanvasMatrix();
  Vector rr0 = annuli_[0];
  Vector rr1 = annuli_[numAnnuli_-1];

  for (int i=0; i<numAngles_; i++) {
    Vector r0 = Vector(rr0[0]*cos(-angles_[i]),rr0[1]*sin(-angles_[i])) * mm;
    Vector r1 = Vector(rr1[0]*cos(-angles_[i]),rr1[1]*sin(-angles_[i])) * mm;

    ostringstream str;
    str << "newpath " 
    	<< r0.TkCanvasPs(parent->canvas) << "moveto"
    	<< r1.TkCanvasPs(parent->canvas) << "lineto"
    	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}

// list

void Cpanda::list(ostream& str, CoordSystem sys, SkyFrame sky,
		 SkyFormat format, int conj, int strip)
{
  int regular = 1;
  if (numAngles_>2) {
    double delta;
    if (angles_[1] > angles_[0])
      delta = angles_[1]-angles_[0];
    else
      delta = angles_[1]+2*M_PI-angles_[0];

    for (int i=2; i<numAngles_; i++) {
      double diff;
      if (angles_[i] > angles_[i-1])
	diff = angles_[i]-angles_[i-1];
      else
	diff = angles_[i]+2*M_PI-angles_[i-1];

      if (diff < delta-FLT_EPSILON || diff > delta+FLT_EPSILON) {
	regular = 0;
	break;
      }
    }
  }

  if (numAnnuli_>2) {
    double delta = annuli_[1][0]-annuli_[0][0];
    for (int i=2; i<numAnnuli_; i++) {
      double diff = annuli_[i][0]-annuli_[i-1][0];
      if (diff < delta-FLT_EPSILON || diff > delta+FLT_EPSILON) {
	regular = 0;
	break;
      }
    }
  }

  if (regular)
    listA(str, sys, sky, format, conj, strip);
  else
    listB(str, sys, sky, format, conj, strip);
}

void Cpanda::listA(ostream& str, CoordSystem sys, SkyFrame sky,
		  SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);
  listPre(str, sys, sky, ptr, strip, 0);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	  << radToDeg(parent->mapAngleFromReff(angles_[0],sys)) << ',' 
	  << radToDeg(parent->mapAngleFromReff(angles_[numAngles_-1],sys)) << ','
	  << numAngles_-1 << ','
	  << ptr->mapLenFromRef(annuli_[0][0],sys) << ','
	  << ptr->mapLenFromRef(annuli_[numAnnuli_-1][0],sys) << ','
	  << numAnnuli_-1 << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
		<< radToDeg(parent->mapAngleFromReff(angles_[0],sys,sky))<< ',' 
		<< radToDeg(parent->mapAngleFromReff(angles_[numAngles_-1],sys,sky))
		<<','
		<< numAngles_-1 << ','
		<< ptr->mapLenFromRef(annuli_[0][0],sys,ARCSEC) << '"' << ','
		<< ptr->mapLenFromRef(annuli_[numAnnuli_-1][0],sys,ARCSEC)<< '"'<<','
		<< numAnnuli_-1 << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;
	    str << type << '(' << ra << ',' << dec << ','
		<< radToDeg(parent->mapAngleFromReff(angles_[0],sys,sky))<< ',' 
		<< radToDeg(parent->mapAngleFromReff(angles_[numAngles_-1],sys,sky))
		<<','
		<< numAngles_-1 << ','
		<< ptr->mapLenFromRef(annuli_[0][0],sys,ARCSEC) << '"' << ','
		<< ptr->mapLenFromRef(annuli_[numAnnuli_-1][0],sys,ARCSEC)<< '"'<<','
		<< numAnnuli_-1 << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	    << radToDeg(parent->mapAngleFromReff(angles_[0],sys)) << ',' 
	    << radToDeg(parent->mapAngleFromReff(angles_[numAngles_-1],sys))<<','
	    << numAngles_-1 << ','
	    << ptr->mapLenFromRef(annuli_[0][0],sys) << ','
	    << ptr->mapLenFromRef(annuli_[numAnnuli_-1][0],sys)<< ','
	    << numAnnuli_-1 << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Cpanda::listB(ostream& str, CoordSystem sys, SkyFrame sky,
		  SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      for (int j=1; j<numAngles_; j++)
	for (int i=1; i<numAnnuli_; i++) {

	  listPre(str, sys, sky, ptr, strip, 0);

	  str << type << '(' << setprecision(8) 
	      << v[0] << ',' << v[1] << ','
	      << radToDeg(parent->mapAngleFromReff(angles_[j-1],sys)) << ',' 
	      << radToDeg(parent->mapAngleFromReff(angles_[j],sys)) << ",1,"
	      << ptr->mapLenFromRef(annuli_[i-1][0],sys) << ',' 
	      << ptr->mapLenFromRef(annuli_[i][0],sys) << ",1)";

	  if (!strip) {
	    if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
	      str << " ||";

	    str << " # panda=";
	    if (i==1 && j==1 && !strip) {
	      str << '(';
	      for (int k=0; k<numAngles_; k++)
		str << radToDeg(parent->mapAngleFromReff(angles_[k],sys))
		    << ((k<numAngles_-1) ? ' ' : ')');
	      str << '(';
	      for (int k=0; k<numAnnuli_; k++)
		str << ptr->mapLenFromRef(annuli_[k][0],sys) 
		    << ((k<numAnnuli_-1) ? ' ' : ')');
	      
	      listProps(str);
	    }
	    else
	      str << "ignore";

	    str << (strip ? ';' : '\n');
	  }
	  else {
	    if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
	      str << "||";
	    else
	      str << ";";
	  }
	}
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    for (int j=1; j<numAngles_; j++)
	      for (int i=1; i<numAnnuli_; i++) {

		listPre(str, sys, sky, ptr, strip, 0);

		str << type << '(' << setprecision(8) 
		    << v[0] << ',' << v[1] << ','
		    << radToDeg(parent->mapAngleFromReff(angles_[j-1],sys,sky))
		    <<',' 
		    << radToDeg(parent->mapAngleFromReff(angles_[j],sys,sky))
		    << ",1,"
		    << ptr->mapLenFromRef(annuli_[i-1][0],sys,ARCSEC) << '"' <<',' 
		    << ptr->mapLenFromRef(annuli_[i][0],sys,ARCSEC) << '"' <<",1)";
		
		if (!strip) {
		  if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		    str << " ||";

		  str << " # panda=";
		  if (i==1 && j==1 && !strip) {
		    str << '(';
		    for (int k=0; k<numAngles_; k++)
		      str << 
			radToDeg(parent->mapAngleFromReff(angles_[k],sys,sky))
			  << ((k<numAngles_-1) ? ' ' : ')');
		    str << '(';
		    for (int k=0; k<numAnnuli_; k++)
		      str << ptr->mapLenFromRef(annuli_[k][0],sys,ARCSEC) << '"'
			  << ((k<numAnnuli_-1) ? ' ' : ')');

		    listProps(str);
		  }
		  else
		    str << "ignore";

		  str << (strip ? ';' : '\n');
		}
		else {
		  if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		    str << "||";
		  else
		    str << ";";
		}
	      }
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;

	    for (int j=1; j<numAngles_; j++)
	      for (int i=1; i<numAnnuli_; i++) {

		listPre(str, sys, sky, ptr, strip, 0);

		str << type << '(' << ra << ',' << dec << ','
		    << radToDeg(parent->mapAngleFromReff(angles_[j-1],sys,sky))
		    <<',' 
		    << radToDeg(parent->mapAngleFromReff(angles_[j],sys,sky))
		    << ",1,"
		    << ptr->mapLenFromRef(annuli_[i-1][0],sys,ARCSEC) << '"' <<',' 
		    << ptr->mapLenFromRef(annuli_[i][0],sys,ARCSEC) << '"' <<",1)";

		if (!strip) {
		  if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		    str << " ||";

		  str << " # panda=";
		  if (i==1 && j==1 && !strip) {
		    str << '(';
		    for (int k=0; k<numAngles_; k++)
		      str << 
			radToDeg(parent->mapAngleFromReff(angles_[k],sys,sky))
			  << ((k<numAngles_-1) ? ' ' : ')');
		    str << '(';
		    for (int k=0; k<numAnnuli_; k++)
		      str << ptr->mapLenFromRef(annuli_[k][0],sys,ARCSEC) << '"'
			  << ((k<numAnnuli_-1) ? ' ' : ')');

		    listProps(str);
		  }
		  else
		    str << "ignore";

		  str << (strip ? ';' : '\n');
		}
		else {
		  if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		    str << "||";
		  else
		    str << ";";
		}
	      }
	  }
	  break;
	}
	break;
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	for (int j=1; j<numAngles_; j++)
	  for (int i=1; i<numAnnuli_; i++) {

	    listPre(str, sys, sky, ptr, strip, 0);

	    str << type << '(' << setprecision(8) 
		<< v[0] << ',' << v[1] << ','
		<< radToDeg(parent->mapAngleFromReff(angles_[j-1],sys))<<',' 
		<< radToDeg(parent->mapAngleFromReff(angles_[j],sys))<< ",1,"
		<< ptr->mapLenFromRef(annuli_[i-1][0],sys) <<',' 
		<< ptr->mapLenFromRef(annuli_[i][0],sys) <<",1)";

	    if (!strip) {
	      if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		str << " ||";

	      str << " # panda=";
	      if (i==1 && j==1 && !strip) {
		str << '(';
		for (int k=0; k<numAngles_; k++)
		  str << radToDeg(parent->mapAngleFromReff(angles_[k],sys))
		      << ((k<numAngles_-1) ? ' ' : ')');
		str << '(';
		for (int k=0; k<numAnnuli_; k++)
		  str << ptr->mapLenFromRef(annuli_[k][0],sys) 
		      << ((k<numAnnuli_-1) ? ' ' : ')');

		listProps(str);
	      }
	      else
		str << "ignore";

	      str << (strip ? ';' : '\n');
	    }
	    else {
	      if (conj || !(j==numAngles_-1 && i==numAnnuli_-1))
		str << "||";
	      else
		str << ";";
	    }
	  }
      }
    }
  }
}

void Cpanda::listCiao(ostream& str, CoordSystem sys, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,PHYSICAL);
      for (int i=0; i<numAnnuli_-1; i++)
	for (int j=0; j<numAngles_-1; j++) {
	  listCiaoPre(str);

	  double a1 = radToDeg(angles_[j]);
	  double a2 = radToDeg(angles_[j+1]);
	  if (a1>=a2)
	    a2+=360;

	  str << "pie(" << setprecision(8) << v[0] << ',' << v[1] << ','
	      << ptr->mapLenFromRef(annuli_[i][0],PHYSICAL) << ','
	      << ptr->mapLenFromRef(annuli_[i+1][0],PHYSICAL) << ','
	      << a1 << ',' << a2 << ')';

	  listCiaoPost(str, conj, strip);
	}
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {

      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL, buf, 64);
      char ra[16];
      char dec[16];
      string x(buf);
      istringstream wcs(x);
      wcs >> ra >> dec;

      for (int i=0; i<numAnnuli_-1; i++)
	for (int j=0; j<numAngles_-1; j++) {
	  listCiaoPre(str);

	  double a1 = radToDeg(angles_[j]);
	  double a2 = radToDeg(angles_[j+1]);
	  if (a1>=a2)
	    a2+=360;

	  str << "pie(" << ra << ',' << dec << ','
	      << ptr->mapLenFromRef(annuli_[i][0],sys,ARCMIN) << '\'' << ','
	      << ptr->mapLenFromRef(annuli_[i+1][0],sys,ARCMIN) << '\'' << ','
	      << a1 << ',' << a2 << ')';

	  listCiaoPost(str, conj, strip);
	}
    }
  }
}

// special composite funtionallity

void Cpanda::setComposite(const Matrix& mx, double aa)
{
  center *= mx;
  updateBBox();
}
