// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __fitsdata_h__
#define __fitsdata_h__

#include "framebase.h"
#include "file.h"

class ColorScale;
class FitsFile;

class FitsData {
protected:
  FrameBase* parent;

  int width;
  int height;
  long size;
  char buf[32];

  int byteswap;

  double bscale;
  double bzero;
  int hasScaling;

  int hasBlank;
  int blank;
  float nanf;          // generated NaN
  double nand;         // generated NaN

  double high;         // bscale applied
  double low;          // bscale applied

  float zHigh;         // bscale applied
  float zLow;          // bscale applied
  double aLow;	       // bscale applied		    
  double aHigh;	       // bscale applied		    
  double uLow;	       // bscale applied		    
  double uHigh;	       // bscale applied		    

  int scanValid;
  int incr;

  float zContrast;
  int zSampleSize;
  int zSampleLine;
  int zscaleValid;

  double* autoCutHist;
  int autoCutTotal;
  int autoCutValid;
  float autoCutPer;

  FrScale::ClipMode clipMode;
  FrScale::MinMaxMode mmMode;

  double datamin;      // bscale applied
  double datamax;      // bscale applied
  int hasdatamin;

  double irafmin;      // bscale applied
  double irafmax;      // bscale applied
  int hasirafmin;

  FrScale::ScanMode scanMode;

  // zscale constants

  enum PixelType {GOOD_PIXEL, BAD_PIXEL, REJECT_PIXEL};

  int zSubSample(float*, float*, int, int);
  int zFitLine(float*, int, float*, float*, float, int, int);
  void zFlattenData (float*, float*, float*, int npix, float, float dz);
  int zComputeSigma (float*, short*, int npix, float*, float*);
  int zRejectPixels (float*, float*, float*, short*, int, double*, 
		     double*, double*, double*, float, int);

 protected:
  void autoCut(int*);

public:
  FitsData(FitsFile*, FrameBase*);
  virtual ~FitsData();

  virtual const char* getValue(const Vector&) =0;
  virtual float getValueFloat(const Vector&) =0;
  virtual float getValueFloat(int) =0;
  virtual double getValueDouble(const Vector&) =0;
  virtual double getValueDouble(int) =0;

  virtual const char* getMin() =0;
  virtual const char* getMax() =0;
  virtual double getMinDouble() =0;
  virtual double getMaxDouble() =0;

  const char* getLow();
  const char* getHigh();
  double getLowDouble() {return low;}
  double getHighDouble() {return high;}

  void setClip(double l, double h) {low = l; high = h;}

  void resetMinMax() {scanMode = FrScale::RESET;}

  virtual void updateClip(FrScale*, int*) =0;
  int hasDATAMIN() {return hasdatamin;}
  int hasIRAFMIN() {return hasirafmin;}

  virtual void bin(double*,int,double,double,int*) =0;
};

template<class T>
class FitsDatam : public FitsData {
 private:
  T* data;
  T min;               // bscale not applied
  T max;               // bscale not applied

  void updateMinMax(int*);
  void scan(int*, int);
  void output(ostringstream&, T);
  void zscale(int*);
  int zSampleImage(float**,int*);

 public:
  FitsDatam(FitsFile*, FrameBase*);

  T swap(T*);

  const char* getValue(const Vector&);

  float getValueFloat(int i);
  float getValueFloat(const Vector&);

  double getValueDouble(int i);
  double getValueDouble(const Vector&);

  const char* getMin();
  const char* getMax();

  double getMinDouble();
  double getMaxDouble();

  void updateClip(FrScale*, int*);
  void bin(double*, int, double, double, int*);
};

#endif
