// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <float.h>
#include <limits.h>

#include "fitsimage.h"
#include "framebase.h"
#include "util.h"

#include "mmap.h"
#include "smmap.h"
#include "mmapincr.h"
#include "alloc.h"
#include "allocgz.h"
#include "channel.h"
#include "share.h"
#include "sshare.h"
#include "socket.h"
#include "socketgz.h"
#include "var.h"
#include "iis.h"
#include "hist.h"
#include "analysis.h"

// wcs coordinate system strings (for use with wcssubs)

// this is kluge to speed up doug minks wcssubs 'ksearch' routine
extern "C" {
  FitsHead* wcshead = NULL;
  char* ksearchh(char*,char*);

  char* findit(char* cards, char* key)
  {
    if (wcshead)
      return wcshead->find(key);
    else
      return ksearchh(cards, key);
  }
};

FitsImage::FitsImage(FrameBase* p)
{
  parent = p;
  objectName = NULL;
  rootBaseFileName = NULL;
  fullBaseFileName = NULL;
  rootFileName = NULL;
  fullFileName = NULL;
  iisFileName = NULL;

  fits_ = NULL;
  hist_ = NULL;
  smooth_ = NULL;

  fitsdata_ = NULL;
  smoothdata_ = NULL;

  image_ = NULL;
  data_ = NULL;

  nextMosaic_ = NULL;
  nextSlice_ = NULL;

  width_ = 0;
  height_ = 0;
  depth_ = 0;
  bitpix_ = 0;

  rotation_ = 0;
  zoom_ = Vector(1,1);
  orientation_ = NORMAL;

  binFunction_ = FitsHist::SUM;
  binFactor_ = Vector(1,1);
  binBufferSize_ = 1024;
  binDepth_ = 1;

  keyLTMV = 0;
  keyATMV = 0;
  keyDTMV = 0;
  keyDATASEC = 0;

  imageToData = Translate(-.5, -.5);
  dataToImage = Translate( .5,  .5);

  memset(iparams,0,5*sizeof(int));
  memset(dparams,0,5*sizeof(int));
  memset(imparams,0,5*sizeof(int));
  memset(irparams,0,5*sizeof(int));
  memset(dmparams,0,5*sizeof(int));
  memset(drparams,0,5*sizeof(int));

  wcs = NULL;
  wcstan = NULL;

  iisMode = 0;
  iiszt = 0;

  mWidgetToData = widgetToData.mm();
  mPannerToData = pannerToData.mm();
  mMagnifierToData = magnifierToData.mm();
  mPSToData = PSToData.mm();
}

FitsImage::~FitsImage()
{
  if (objectName)
    delete [] objectName;

  if (rootBaseFileName)
    delete [] rootBaseFileName;

  if (fullBaseFileName)
    delete [] fullBaseFileName;

  if (rootFileName)
    delete [] rootFileName;

  if (fullFileName)
    delete [] fullFileName;

  if (iisFileName)
    delete [] iisFileName;

  if (hist_)
    delete hist_;

  if (smooth_)
    delete smooth_;

  if (fitsdata_)
    delete fitsdata_;

  if (smoothdata_)
    delete smoothdata_;

  if (wcs) {
    for (int i=0; i<MULTWCS; i++)
      if (wcs[i])
	wcsfree(wcs[i]);
    delete [] wcs;
  }  

  if (wcstan)
    delete [] wcstan;
}

FitsImageMaster::~FitsImageMaster()
{
  if (fits_)
    delete fits_;
}

FitsImageSlave::FitsImageSlave(FrameBase* p, const char* fn, FitsFile* f) 
  : FitsImage(p)
{
  fits_ = f;
  process(fn);
}

// Fits

FitsImageFitsAlloc::FitsImageFitsAlloc(FrameBase* p, const char* fn, 
				       FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsAlloc(fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsAllocGZ::FitsImageFitsAllocGZ(FrameBase* p, const char* fn, 
					   FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsAllocGZ(fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsChannel::FitsImageFitsChannel(FrameBase* p, Tcl_Interp* interp, 
					   const char* ch, const char* fn, 
					   FitsFile::FlushMode flush) 
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsChannel(interp, ch, fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsMMap::FitsImageFitsMMap(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsMMap(fn, FitsFile::RELAX);
  process(fn);
}

FitsImageFitsSMMap::FitsImageFitsSMMap(FrameBase* p, const char* hdr, 
				       const char* fn) : FitsImageMaster(p)
{
  fits_ = new FitsFitsSMMap(hdr, fn);
  process(fn);
}

FitsImageFitsMMapIncr::FitsImageFitsMMapIncr(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsMMapIncr(fn, FitsFile::RELAX);
  process(fn);
}

FitsImageFitsShare::FitsImageFitsShare(FrameBase* p, FrameBase::ShmType type, 
				       int id, const char* fn)
  : FitsImageMaster(p)
{
  switch (type) {
  case FrameBase::SHMID:
    fits_ = new FitsFitsShareID(id, fn, FitsFile::RELAX);
    break;
  case FrameBase::KEY:
    fits_ = new FitsFitsShareKey(id, fn, FitsFile::RELAX);
    break;
  }
  process(fn);
}

FitsImageFitsSShare::FitsImageFitsSShare(FrameBase* p, FrameBase::ShmType type,
					 int hdr, int id, const char* fn)
  : FitsImageMaster(p)
{
  switch (type) {
  case FrameBase::SHMID:
    fits_ = new FitsFitsSShareID(hdr, id, fn);
    break;
  case FrameBase::KEY:
    fits_ = new FitsFitsSShareKey(hdr, id, fn);
    break;
  }
  process(fn);
}

FitsImageFitsSocket::FitsImageFitsSocket(FrameBase* p, int s, const char* fn, 
					 FitsFile::FlushMode flush) 
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsSocket(s, fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsSocketGZ::FitsImageFitsSocketGZ(FrameBase* p, int s, 
					     const char* fn, 
					     FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsSocketGZ(s, fn, FitsFile::RELAX, flush);
  process(fn);
}

FitsImageFitsVar::FitsImageFitsVar(FrameBase* p, Tcl_Interp* interp, 
				   const char* var, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsVar(interp, var, fn, FitsFile::RELAX);
  process(fn);
}

// Fits Next

FitsImageFitsNextAlloc::FitsImageFitsNextAlloc(FrameBase* p, 
					       const char* fn,
					       FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextAlloc(prev);
  process(fn);
}

FitsImageFitsNextAllocGZ::FitsImageFitsNextAllocGZ(FrameBase* p, 
						   const char* fn,
						   FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextAllocGZ(prev);
  process(fn);
}

FitsImageFitsNextChannel::FitsImageFitsNextChannel(FrameBase* p, 
						   const char* fn,
						   FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextChannel(prev);
  process(fn);
}

FitsImageFitsNextMMap::FitsImageFitsNextMMap(FrameBase* p, 
					     const char* fn,
					     FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextMMap(prev);
  process(fn);
}

FitsImageFitsNextSMMap::FitsImageFitsNextSMMap(FrameBase* p, 
					       const char* fn,
					       FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextSMMap(prev);
  process(fn);
}

FitsImageFitsNextMMapIncr::FitsImageFitsNextMMapIncr(FrameBase* p, 
						     const char* fn,
						     FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextMMapIncr(prev);
  process(fn);
}

FitsImageFitsNextShare::FitsImageFitsNextShare(FrameBase* p, 
					       const char* fn,
					       FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextShare(prev);
  process(fn);
}

FitsImageFitsNextSShare::FitsImageFitsNextSShare(FrameBase* p, 
						 const char* fn,
						 FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextSShare(prev);
  process(fn);
}

FitsImageFitsNextSocket::FitsImageFitsNextSocket(FrameBase* p, 
						 const char* fn,
						 FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextSocket(prev);
  process(fn);
}

FitsImageFitsNextSocketGZ::FitsImageFitsNextSocketGZ(FrameBase* p, 
						     const char* fn, 
						     FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextSocketGZ(prev);
  process(fn);
}

FitsImageFitsNextVar::FitsImageFitsNextVar(FrameBase* p, 
					   const char* fn,
					   FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsFitsNextVar(prev);
  process(fn);
}

FitsImageFitsNextHist::FitsImageFitsNextHist(FrameBase* p, 
					     FitsImage* fi,
					     FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsHistNext(prev);
  process(NULL);

  fits_->setpFilter(fi->getHistFilter());
  rootBaseFileName = dupstr(fi->getRootBaseFileName());
  fullBaseFileName = dupstr(fi->getFullBaseFileName());
  iisFileName = dupstr(fi->getFullBaseFileName());
}

// Array

FitsImageArrAlloc::FitsImageArrAlloc(FrameBase* p, const char* fn,
				     FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsArrAlloc(fn, flush);
  process(fn);
}

FitsImageArrAllocGZ::FitsImageArrAllocGZ(FrameBase* p, const char* fn,
					 FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsArrAllocGZ(fn, flush);
  process(fn);
}

FitsImageArrChannel::FitsImageArrChannel(FrameBase* p, Tcl_Interp* interp, 
					 const char* ch, const char* fn,
					 FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsArrChannel(interp, ch, fn, flush);
  process(fn);
}

FitsImageArrMMap::FitsImageArrMMap(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsArrMMap(fn);
  process(fn);
}

FitsImageArrMMapIncr::FitsImageArrMMapIncr(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsArrMMapIncr(fn);
  process(fn);
}

FitsImageArrShare::FitsImageArrShare(FrameBase* p, FrameBase::ShmType type, 
				     int id, const char* fn) 
  : FitsImageMaster(p)
{
  switch (type) {
  case FrameBase::SHMID:
    fits_ = new FitsArrShareID(id, fn);
    break;
  case FrameBase::KEY:
    fits_ = new FitsArrShareKey(id, fn);
    break;
  }
  process(fn);
}

FitsImageArrSocket::FitsImageArrSocket(FrameBase* p, int s, const char* fn,
				       FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsArrSocket(s, fn, flush);
  process(fn);
}

FitsImageArrSocketGZ::FitsImageArrSocketGZ(FrameBase* p, int s, const char* fn,
					   FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsArrSocketGZ(s, fn, flush);
  process(fn);
}

FitsImageArrVar::FitsImageArrVar(FrameBase* p, Tcl_Interp* interp, 
				 const char* var, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsArrVar(interp, var, fn);
  process(fn);
}

// Mosaic

FitsImageMosaicAlloc::FitsImageMosaicAlloc(FrameBase* p, const char* fn,
					   FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicAlloc(fn, flush);
  process(fn);
}

FitsImageMosaicAllocGZ::FitsImageMosaicAllocGZ(FrameBase* p, const char* fn,
					       FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicAllocGZ(fn, flush);
  process(fn);
}

FitsImageMosaicChannel::FitsImageMosaicChannel(FrameBase* p, 
					       Tcl_Interp* interp,
					       const char* ch, 
					       const char* fn,
					       FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicChannel(interp, ch, flush);
  process(fn);
}

FitsImageMosaicMMap::FitsImageMosaicMMap(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicMMap(fn);
  process(fn);
}

FitsImageMosaicMMapIncr::FitsImageMosaicMMapIncr(FrameBase* p, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicMMapIncr(fn);
  process(fn);
}

FitsImageMosaicShare::FitsImageMosaicShare(FrameBase* p, 
					   FrameBase::ShmType type,
					   int id, const char* fn)
  : FitsImageMaster(p)
{
  switch (type) {
  case FrameBase::SHMID:
    fits_ = new FitsMosaicShareID(id);
    break;
  case FrameBase::KEY:
    fits_ = new FitsMosaicShareKey(id);
    break;
  }
  process(fn);
}

FitsImageMosaicSocket::FitsImageMosaicSocket(FrameBase* p, 
					     int s, const char* fn,
					     FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicSocket(s, flush);
  process(fn);
}

FitsImageMosaicSocketGZ::FitsImageMosaicSocketGZ(FrameBase* p, 
						 int s, const char* fn,
						 FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicSocketGZ(s, flush);
  process(fn);
}

FitsImageMosaicVar::FitsImageMosaicVar(FrameBase* p, Tcl_Interp* interp,
				       const char* var, const char* fn)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicVar(interp, var, fn);
  process(fn);
}

// Mosaic Next

FitsImageMosaicNextAlloc::FitsImageMosaicNextAlloc(FrameBase* p, 
						   const char* fn,
						   FitsFile* prev,
						   FitsFile::FlushMode flush)

  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextAlloc(prev, flush);
  process(fn);
}

FitsImageMosaicNextAllocGZ::FitsImageMosaicNextAllocGZ(FrameBase* p,
						       const char* fn,
						       FitsFile* prev,
						     FitsFile::FlushMode flush)

  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextAllocGZ(prev, flush);
  process(fn);
}

FitsImageMosaicNextChannel::FitsImageMosaicNextChannel(FrameBase* p, 
						       const char* fn, 
						       FitsFile* prev,
						     FitsFile::FlushMode flush)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextChannel(prev, flush);
  process(fn);
}

FitsImageMosaicNextMMap::FitsImageMosaicNextMMap(FrameBase* p, const char* fn, 
						 FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextMMap(prev);
  process(fn);
}

FitsImageMosaicNextMMapIncr::FitsImageMosaicNextMMapIncr(FrameBase* p,
							 const char* fn, 
							 FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextMMapIncr(prev);
  process(fn);
}

FitsImageMosaicNextShare::FitsImageMosaicNextShare(FrameBase* p, 
						   const char* fn,
						   FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextShare(prev);
  process(fn);
}

FitsImageMosaicNextSocket::FitsImageMosaicNextSocket(FrameBase* p, 
						     const char* fn, 
						     FitsFile* prev,
						     FitsFile::FlushMode flush)

  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextSocket(prev, flush);
  process(fn);
}

FitsImageMosaicNextSocketGZ::FitsImageMosaicNextSocketGZ(FrameBase* p,
							 const char* fn, 
							 FitsFile* prev,
						     FitsFile::FlushMode flush)

  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextSocketGZ(prev, flush);
  process(fn);
}

FitsImageMosaicNextVar::FitsImageMosaicNextVar(FrameBase* p, 
					       const char* fn, 
					       FitsFile* prev)
  : FitsImageMaster(p)
{
  fits_ = new FitsMosaicNextVar(prev);
  process(fn);
}

// IIS

FitsImageIIS::FitsImageIIS(FrameBase* p, int w, int h) : FitsImageMaster(p)
{
  fits_ = new FitsIIS(w, h);
  process("");
  iisMode = 1;
}

void FitsImageIIS::iisErase()
{
  if (fits_)
    ((FitsIIS*)fits_)->erase();
}

const char* FitsImageIIS::iisGet(int x, int y)
{
  if (fits_)
    return ((FitsIIS*)fits_)->get(x,y);
}

void FitsImageIIS::iisSet(int x, int y, const char* d, int l)
{
  if (fits_)
    ((FitsIIS*)fits_)->set(x,y,d,l);
}

void FitsImageIIS::iisWCS(const Matrix& mx, const Vector& z, int zt)
{
  dataToImage = Translate(-.5,-.5) * mx;
  imageToData = dataToImage.invert();
  iisz = z;
  iiszt = zt;
}

void FitsImage::reset()
{
  if (fits_)
    delete fits_;
  fits_ = NULL;
  if (hist_)
    delete hist_;
  hist_ = NULL;
  if (smooth_)
    delete smooth_;
  smooth_ = NULL;

  if (fitsdata_)
    delete fitsdata_;
  fitsdata_ = NULL;
  if (smoothdata_)
    delete smoothdata_;
  smoothdata_ = NULL;

  image_ = NULL;
  data_ = NULL;
}

void FitsImage::process(const char* fn)
{
  if (!fits_->isValid()) {
    reset();
    return;
  }

  setFileName(fn);

  if (fits_->isImage())
    load();
  else if (fits_->isTable()) {
    // make sure we have cols to bin on
    if (!fits_->pBinX())
      fits_->setpBinX("X");
    if (!fits_->pBinY())
      fits_->setpBinY("Y");
    if (!fits_->pBinZ())
      fits_->setpBinZ("TIME");

    // update default values from parent
    binFunction_ = parent->binFunction_;
    binBufferSize_ = parent->binBufferSize_;
    binDepth_ = parent->binDepth_;
    binFactor_ = parent->binFactor_;

    nextHist(getHistCenter());
  }
}

void FitsImage::load()
{
  if (hist_)
    image_ = hist_;
  else
    image_ = fits_;

  // create a valid FitsData
  if (fitsdata_)
    delete fitsdata_;

  switch (image_->head()->bitpix()) {
  case 8: 
    fitsdata_ = new FitsDatam<unsigned char>(image_, parent);
    break;
  case 16:
    fitsdata_ = new FitsDatam<short>(image_, parent);
    break;
  case -16:
    fitsdata_ = new FitsDatam<unsigned short>(image_, parent);
    break;
  case 32:
    fitsdata_ = new FitsDatam<int>(image_, parent);
    break;
  case -32:
    fitsdata_ = new FitsDatam<float>(image_, parent);
    break;
  case -64:
    fitsdata_ = new FitsDatam<double>(image_, parent);
    break;
  }

  FitsHead* head = image_->head();
  width_ = head->naxis(1);
  height_ = head->naxis(2);
  depth_ = head->naxis(3);
  bitpix_ = head->bitpix();

  resetWCS();

  memset(iparams,0,5*sizeof(int));
  memset(dparams,0,5*sizeof(int));
  memset(imparams,0,5*sizeof(int));
  memset(irparams,0,5*sizeof(int));
  memset(dmparams,0,5*sizeof(int));
  memset(drparams,0,5*sizeof(int));

  if (!processKeywords()) {
    reset();
    return;
  }

  analysis();
}

void FitsImage::analysis()
{
  if (hist_)
    image_ = hist_;
  else
    image_ = fits_;
  data_ = fitsdata_;

  if (smooth_)
    delete smooth_;
  smooth_ = NULL;
  if (smoothdata_)
    delete smoothdata_;
  smoothdata_ = NULL;

  if (parent->dosmooth_) {
    smooth_ = new FitsAnalysis(image_);
    if (smooth_->isValid()) {
      smoothdata_ = new FitsDatam<float>(smooth_, parent);

      if (smooth()) {
	image_ = smooth_;
	data_ = smoothdata_;

	FitsHead* head = image_->head();
	depth_ = head->naxis(3);
	bitpix_ = head->bitpix();
      }
      else {
	delete smooth_;
	smooth_ = NULL;
	delete smoothdata_;
	smoothdata_ = NULL;
      }
    }
    else {
      delete smooth_;
      smooth_ = NULL;
    }
  }
}

void FitsImage::setFileName(const char* fn)
{
  if (fullBaseFileName)
    delete [] fullBaseFileName;
  fullBaseFileName = NULL;

  if (rootBaseFileName)
    delete [] rootBaseFileName;
  rootBaseFileName = NULL;

  if (iisFileName)
    delete [] iisFileName;
  iisFileName = NULL;

  // no filename to set
  if (!fn)
    return;

  // strip any '[]'
  char* ffn = strip(fn);

  if(fits_->find("EXTNAME")) {
    char* c = fits_->getString("EXTNAME");
    {
      ostringstream str;
      str << ffn << '[' << c << ']' << ends;
      fullBaseFileName = dupstr(str.str().c_str());
    }
    {
      char* m = root(ffn);
      ostringstream str;
      str << m << '[' << c << ']' << ends;
      rootBaseFileName = dupstr(str.str().c_str());
      delete [] m;
    }
    delete [] c;
  }
  else if (fits_->ext()) {
    {
      ostringstream str;
      str << ffn << '[' << fits_->ext() << ']' << ends;
      fullBaseFileName = dupstr(str.str().c_str());
    }
    {
      char* m = root(ffn);
      ostringstream str;
      str << m << '[' << fits_->ext() << ']' << ends;
      rootBaseFileName = dupstr(str.str().c_str());
      delete [] m;
    }
  }
  else {
    fullBaseFileName = dupstr(ffn);
    rootBaseFileName = root(ffn);
  }
  
  // by default, iisFileName is fullBaseFileName
  if (fullBaseFileName)
    iisFileName = dupstr(fullBaseFileName);

  delete [] ffn;
  updateFileName();
}

void FitsImage::updateFileName()
{
  if (fullFileName)
    delete [] fullFileName;
  fullFileName = NULL;

  if (rootFileName)
    delete [] rootFileName;
  rootFileName = NULL;

  char* filter = (char*)fits_->pFilter();
  char* slice = (char*)fits_->pSliceFilter();
  int doFilter = (filter && *filter);
  int doSlice = (slice && *slice);

  if (fullBaseFileName) {
    ostringstream str;
    if (doFilter && doSlice)
      str << fullBaseFileName << '[' << filter << ',' << slice << ']' <<ends;
    else if (doFilter && !doSlice)
      str << fullBaseFileName << '[' << filter << ']' << ends;
    else if (!doFilter && doSlice)
      str << fullBaseFileName << '[' << slice << ']' << ends;
    else
      str << fullBaseFileName << ends;

    fullFileName = dupstr(str.str().c_str());
  }

  if (rootBaseFileName) {
    ostringstream str;
    if (doFilter && doSlice)
      str << rootBaseFileName << '[' << filter << ',' << slice << ']' <<ends;
    else if (doFilter && !doSlice)
      str << rootBaseFileName << '[' << filter << ']' << ends;
    else if (!doFilter && doSlice)
      str << rootBaseFileName << '[' << slice << ']' << ends;
    else
      str << rootBaseFileName << ends;

    rootFileName = dupstr(str.str().c_str());
  }
}

void FitsImage::iisSetFileName(const char* fn)
{
  if (iisFileName)
    delete [] iisFileName;
  iisFileName = dupstr(fn);
}

char* FitsImage::strip(const char* fn)
{
  if (fn) {
    char* r = dupstr(fn);           // dup the string
    char* ptr = r;                  // init the ptr
    while(*ptr != '[' && *ptr)      // walk it forward til '[' or end
      ptr++;
    *ptr = '\0';                    // zero out rest

    return r;                       // got it!
  }
  else
    return NULL;
}

char* FitsImage::root(const char* fn)
{
  if (fn) {
    const char* ptr = fn;           // init the ptr
    while(*ptr++);                  // walk it forward to end of string
    ptr--;                          // backup one
    while(*ptr != '/' && ptr != fn) // walk it backward til last / or beginning
      ptr--;
    if (*ptr == '/')                // step it over the last '/'
      ptr++;
    return dupstr(ptr);             // got it!
  }
  else
    return NULL;
}

void FitsImage::updateMatrices(FrameBase::MosaicType mt, 
			       Matrix& rgbToRef, Matrix& refToUser,
			       Matrix& userToWidget, Matrix& widgetToCanvas,
			       Matrix& userToPanner)
{
  // basic matrices

  // if in mosaic mode, we need to flip the image, based on orientation
  // which is set via the ATM/V keywords when the file is parsed.

  switch (mt) {
  case FrameBase::NOMOSAIC:
    dataToRef = rgbToRef;
    refToData = dataToRef.invert();
    break;
  case FrameBase::IRAF:
    {
      // this is kluge, put it works ;-)
      // if the segment is flipped, we can have a discontinuity at
      // the edges, due to round off errors, so we 'nudge' it

      Vector oo = origin_;
      Matrix flip;
      switch (orientation_) {
      case XX:
	flip = FlipX();
	oo -= Vector(FLT_EPSILON,0);
	break;
      case YY:
	flip = FlipY();
	oo -= Vector(0,FLT_EPSILON);
	break;
      case XY:
	flip = FlipXY();
	oo -= Vector(FLT_EPSILON,FLT_EPSILON);
	break;
      }

      Vector cc = datasec.center() * imageToData;
      dataToRef =
	Translate(-cc) *
	flip *
	Scale(zoom_) *
	Rotate(rotation_) *
	Translate(cc) *
	Translate(oo) *
	rgbToRef;
      refToData = dataToRef.invert();
    }
    break;
  case FrameBase::WCSMOSAIC:
  case FrameBase::WFPC2:
    {
      // this is kluge, put it works ;-)
      // if the segment is flipped, we can have a discontinuity at
      // the edges, due to round off errors, so we 'nudge' it

      Vector oo = origin_;
      Matrix flip;
      double r = degToRad(180);
      if ((rotation_ > r-FLT_EPSILON && rotation_ < r+FLT_EPSILON) || 
	  (-rotation_ > r-FLT_EPSILON && -rotation_ < r+FLT_EPSILON)) {
	switch (orientation_) {
	case NORMAL:
	  oo -= Vector(0,FLT_EPSILON);
	  break;
	case XX:
	  flip = FlipX();
	  oo -= Vector(FLT_EPSILON,FLT_EPSILON);
	  break;
	case YY:
	  flip = FlipY();
	  break;
	case XY:
	  flip = FlipXY();
	  oo -= Vector(FLT_EPSILON,0);
	  break;
	}
      }
      else {
	switch (orientation_) {
	case NORMAL:
	  break;
	case XX:
	  flip = FlipX();
	  oo -= Vector(FLT_EPSILON,0);
	  break;
	case YY:
	  flip = FlipY();
	  oo -= Vector(0,FLT_EPSILON);
	  break;
	case XY:
	  flip = FlipXY();
	  oo -= Vector(FLT_EPSILON,FLT_EPSILON);
	  break;
	}
      }

      Vector cc = getWCScrpix(WCS) * imageToData;
      dataToRef =
	Translate(-cc) *
	flip *
	Scale(zoom_) *
	Rotate(rotation_) *
	Translate(cc) *
	Translate(oo) *
	rgbToRef;
      refToData = dataToRef.invert();
    }
    break;
  }

  // derived matrices

  refToCanvas = refToUser * userToWidget * widgetToCanvas;
  canvasToRef = refToCanvas.invert();

  physicalToData = physicalToImage * imageToData;
  dataToPhysical = physicalToData.invert();

  physicalToRef = physicalToImage * imageToData * dataToRef;
  refToPhysical = physicalToRef.invert();

  physicalToUser = physicalToImage * imageToData * dataToRef * refToUser;
  userToPhysical = physicalToUser.invert();

  physicalToCanvas = physicalToImage * imageToData * dataToRef * refToUser * 
    userToWidget * widgetToCanvas;
  canvasToPhysical = physicalToCanvas.invert();

  physicalToPanner = physicalToImage * imageToData * dataToRef * refToUser * 
    userToPanner;
  pannerToPhysical = physicalToPanner.invert();

  imageToRef = imageToData * dataToRef;
  refToImage = imageToRef.invert();

  imageToUser = imageToData * dataToRef * refToUser;
  userToImage = imageToUser.invert();

  imageToWidget = imageToData * dataToRef *  refToUser * userToWidget;
  widgetToImage = imageToWidget.invert();

  imageToCanvas = imageToData * dataToRef *  refToUser *
    userToWidget *  widgetToCanvas;
  canvasToImage = imageToCanvas.invert();

  imageToPanner = imageToData * dataToRef * refToUser * userToPanner;
  pannerToImage = imageToPanner.invert();

  dataToUser = dataToRef * refToUser;
  userToData = dataToUser.invert();

  dataToWidget = dataToRef *  refToUser *  userToWidget;
  widgetToData = dataToWidget.invert();

  dataToCanvas = dataToRef *  refToUser *  userToWidget * widgetToCanvas;
  canvasToData = dataToCanvas.invert();

  dataToPanner = dataToRef * refToUser * userToPanner;
  pannerToData = dataToPanner.invert();

  amplifierToRef = 
    amplifierToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef;
  refToAmplifier = amplifierToRef.invert();

  amplifierToUser = 
    amplifierToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser;
  userToAmplifier = amplifierToUser.invert();

  amplifierToCanvas = 
    amplifierToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser *
    userToWidget * 
    widgetToCanvas;
  canvasToAmplifier = amplifierToCanvas.invert();

  amplifierToPanner = 
    amplifierToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser * 
    userToPanner;
  pannerToAmplifier = amplifierToPanner.invert();

  amplifierToImage =
    amplifierToPhysical *
    physicalToImage;
  imageToAmplifier = amplifierToImage.invert();

  detectorToRef = 
    detectorToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef;
  refToDetector = detectorToRef.invert();

  detectorToUser = 
    detectorToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser;
  userToDetector = detectorToUser.invert();

  detectorToCanvas = 
    detectorToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser *
    userToWidget * 
    widgetToCanvas;
  canvasToDetector = detectorToCanvas.invert();

  detectorToPanner = 
    detectorToPhysical * 
    physicalToImage * 
    imageToData * 
    dataToRef * 
    refToUser * 
    userToPanner;
  pannerToDetector = detectorToPanner.invert();

  detectorToImage =
    detectorToPhysical *
    physicalToImage;
  imageToDetector = detectorToImage.invert();

}

const char* FitsImage::getValue(const Vector& v)
{
  if (!isIIS() || (iiszt != 1))
    return data_->getValue(v);
  else {
    double value = data_->getValueDouble(v);

    ostringstream str;
    if (value <= 1)
      str << '<' << iisz[0] << ends;
    else if (value >= 200)
      str << '>' << iisz[1] << ends;
    else
      str << ((value-1) * (iisz[1]-iisz[0]))/199 + iisz[0] << ends;
    memcpy(buf,str.str().c_str(), str.str().length());
    return buf;
  }
}

void FitsImage::updatePS(Matrix ps)
{
  dataToPS = dataToRef * ps;
  PSToData = dataToPS.invert();
}

void FitsImage::updateMagnifierMatrices(Matrix& refToMagnifier)
{
  dataToMagnifier = dataToRef * refToMagnifier;
  magnifierToData = dataToMagnifier.invert();
}

BBox FitsImage::getIRAFbb()
{
  // return in IMAGE coords
  Vector ll = detsize.ll * imageToData * Scale(ccdsum) * dataToImage;
  Vector ur = detsize.ur * imageToData * Scale(ccdsum) * dataToImage;
  return BBox(ll,ur);
}

int* FitsImage::getDataParams(FrScale::ScanMode which)
{
  // return in 0 to width_, 0 to height_
  switch (which) {
  case FrScale::NODATASEC:
    return iparams;
  case FrScale::DATASEC:
    return dparams;
  case FrScale::UNODATASEC:
    return irparams;
  case FrScale::UDATASEC:
    return drparams;
  }
}

int* FitsImage::getMinMaxParams(FrScale::ScanMode which)
{
  // return in IMAGE coords
  switch (which) {
  case FrScale::NODATASEC:
    return iparams;
  case FrScale::DATASEC:
    return dparams;
  case FrScale::UNODATASEC:
    return imparams;
  case FrScale::UDATASEC:
    return dmparams;
  }
}

void FitsImage::setDataParams(int x0, int y0, int x1, int y1)
{
  irparams[0] = drparams[0] = x1;
  irparams[1] = drparams[1] = x0;
  irparams[2] = drparams[2] = x1;
  irparams[3] = drparams[3] = y0;
  irparams[4] = drparams[4] = y1;

  // now clip by DATASEC

  if (dparams[0] < drparams[0])
    drparams[0] = dparams[0];
  if (dparams[1] > drparams[1])
    drparams[1] = dparams[1];
  if (dparams[2] < drparams[2])
    drparams[2] = dparams[2];
  if (dparams[3] > drparams[3])
    drparams[3] = dparams[3];
  if (dparams[4] < drparams[4])
    drparams[4] = dparams[4];
}

void FitsImage::setMinMaxParams(int x0, int y0, int x1, int y1)
{
  imparams[0] = dmparams[0] = x1;
  imparams[1] = dmparams[1] = x0;
  imparams[2] = dmparams[2] = x1;
  imparams[3] = dmparams[3] = y0;
  imparams[4] = dmparams[4] = y1;

  // now clip by DATASEC

  if (dparams[0] < dmparams[0])
    dmparams[0] = dparams[0];
  if (dparams[1] > dmparams[1])
    dmparams[1] = dparams[1];
  if (dparams[2] < dmparams[2])
    dmparams[2] = dparams[2];
  if (dparams[3] > dmparams[3])
    dmparams[3] = dparams[3];
  if (dparams[4] < dmparams[4])
    dmparams[4] = dparams[4];

  if (data_)
    data_->resetMinMax();
}

void FitsImage::resetWCS()
{
  // Process OBJECT keyword
  if (objectName)
    delete [] objectName;
  objectName = image_->getString("OBJECT");

  // Process WCS keywords
  initWCS(image_->head());
}

void FitsImage::replaceWCS(FitsHead* h)
{
  // Process OBJECT keyword
  if (objectName)
    delete [] objectName;
  objectName = h->getString("OBJECT");

  // Process WCS keywords
  initWCS(h);
}

void FitsImage::appendWCS(FitsHead* app)
{
  // process OBJECT keyword

  char* obj = app->getString("OBJECT");
  if (obj) {
    if (objectName)
      delete [] objectName;
    objectName = obj;
  }

  // Process WCS keywords
  FitsHead* head = image_->head();

  // append wcs keywords to the end of the header
  int ll = head->headbytes()+app->headbytes();
  char* cards = new char[ll];

  // copy default wcs
  memcpy(cards, head->cards(), head->headbytes());

  // find first END and zero out
  for (int i=0; i<head->headbytes(); i+=80)
    if (!strncmp(cards+i,"END",3)) {
      memcpy(cards+i, "   ",3);
      break;
    }

  // copy appended wcs
  memcpy(cards+head->headbytes(), app->cards(), app->headbytes());


  FitsHead* hhead = new FitsHead(cards,ll,FitsHead::EXTERNAL);
  initWCS(hhead);
  delete hhead;
  delete [] cards;
}

void FitsImage::initWCS(FitsHead* head)
{
  if (wcs) {
    for (int i=0; i<MULTWCS; i++)
      if (wcs[i])
	wcsfree(wcs[i]);
    delete [] wcs;
  }

  if (wcstan)
    delete [] wcstan;

  // wcsinit is sloooowwww! so try to figure it out first
  wcs = new WorldCoor*[MULTWCS];
  for (int i=0; i<MULTWCS; i++)
    wcs[i] = NULL;
  
  wcstan = new Vector[MULTWCS];

  // look first for default WCS. Let wcsinit figure it out since there can
  // be many different non-standard wcs's present
  wcshead = head;
  wcs[0] = wcsinit(head->cards());
  wcshead  = NULL;

  // now look for WCSA - WCSZ
  // we can take a short cut here, since only valid FITS wcs's are available
  for (int i=1; i<MULTWCS; i++) {
    char str[] = "CTYPE1 ";
    str[6] = '@'+i;
    if (head->find(str)) {
      wcshead = head;
      wcs[i] = wcsinitc(head->cards(),(char)('@'+i));
      wcshead  = NULL;
    }
  }

  // determine what point to use in determining rotation and orientation
  // if projection is a Zenithal projection (section 5.1 WCS FITS II)
  // then use the tangent point, otherwise, use the center of the image
  for (int i=0; i<MULTWCS; i++) {
    if (wcs[i])
      wcstan[i] = (wcs[i]->prjcode>=WCS_AZP && wcs[i]->prjcode<=WCS_AIR) ?
	Vector(wcs[i]->crpix[0],wcs[i]->crpix[1]) : Vector(width_,height_)/2;
  }

  // now see if we have a 'physical' wcs, if so, set LTMV keywords
  for (int i=1; i<MULTWCS; i++) {
    if (wcs[i] && wcs[i]->wcsname && !strcmp(wcs[i]->wcsname,"PHYSICAL")) {
      keyLTMV = 1;

      double ltm11 = wcs[i]->cd[0] != 0 ? 1/wcs[i]->cd[0] : 0;
      double ltm12 = wcs[i]->cd[1] != 0 ? 1/wcs[i]->cd[1] : 0;
      double ltm21 = wcs[i]->cd[2] != 0 ? 1/wcs[i]->cd[2] : 0;
      double ltm22 = wcs[i]->cd[3] != 0 ? 1/wcs[i]->cd[3] : 0;

      double ltv1 = wcs[i]->crpix[0] -
	wcs[i]->crval[0]*ltm11 - wcs[i]->crval[1]*ltm21;
      double ltv2 = wcs[i]->crpix[1] -
	wcs[i]->crval[0]*ltm12 - wcs[i]->crval[1]*ltm22;

      physicalToImage = Matrix(ltm11, ltm12, ltm21, ltm22, ltv1, ltv2);
      imageToPhysical = physicalToImage.invert();
    }
  }

  if (DebugWCS) {
    for (int i=0; i<MULTWCS; i++) {
      if (wcs[i]) {
	cerr << "wcs " << (char)(!i ? ' ' : '@'+i) << endl;
	cerr << "wcs->wcsname " << (wcs[i]->wcsname ? wcs[i]->wcsname : "")
	     << endl;
	cerr << "wcs->rot=" << wcs[i]->rot << endl;
	cerr << "wcs->equinox=" << wcs[i]->equinox << endl;
	cerr << "wcs->epoch=" << wcs[i]->epoch << endl;
	cerr << "wcs->longpole=" << wcs[i]->longpole << endl;
	cerr << "wcs->latpole=" << wcs[i]->latpole << endl;

	for (int j=0; j<2; j++)
	  cerr << "wcs->crpix[" << j << "]=" << wcs[i]->crpix[j] << endl;

	for (int j=0; j<2; j++)
	  cerr << "wcs->crval[" << j << "]=" << wcs[i]->crval[j] << endl;

	for (int j=0; j<2; j++)
	  cerr << "wcs->cdelt[" << j << "]=" << wcs[i]->cdelt[j] << endl;

	for (int j=0; j<4; j++)
	  cerr << "wcs->cd[" << j << "]=" << wcs[i]->cd[j] << endl;

	for (int j=0; j<4; j++)
	  cerr << "wcs->pc[" << j << "]=" << wcs[i]->pc[j] << endl;

	cerr << "wcs->imrot=" << wcs[i]->imrot << endl;
	cerr << "wcs->pa_north=" << wcs[i]->pa_north << endl;
	cerr << "wcs->pa_east=" << wcs[i]->pa_east << endl;
	cerr << "wcs->imflip=" << wcs[i]->imflip << endl;
	cerr << "wcs->prjcode=" << wcs[i]->prjcode << endl;

	cerr << "wcs->coorflip=" << wcs[i]->coorflip << endl;

	for (int j=0; j<2; j++)
	  cerr << "wcs->ctype[" << j << "]=" << wcs[i]->ctype[j] << endl;

	cerr << "wcs->c1type=" << wcs[i]->c1type << endl;
	cerr << "wcs->c2type=" << wcs[i]->c2type << endl;
	cerr << "wcs->ptype=" << wcs[i]->ptype << endl;
	cerr << "wcs->radecsys=" << wcs[i]->radecsys << endl;
	cerr << "wcs->radecout=" << wcs[i]->radecout << endl;
	cerr << "wcs->radecin=" << wcs[i]->radecin << endl;
	cerr << "wcs->syswcs=" << wcs[i]->syswcs << endl;
	cerr << "wcs->wcsproj=" << wcs[i]->wcsproj << endl;

	cerr << "wcs->wcstan=" << wcstan[i] << endl;
      }
    }
  }
}

void FitsImage::listWCS(ostream& str, CoordSystem sys)
{
  if (wcs[sys-WCS]) {
    WorldCoor* wcss = wcs[sys-WCS];

    // required keywords
    str << "# WCSAXES =" << wcss->naxes << endl;
    for (int i=0; i<wcss->naxes; i++)
      str << "# CRVAL" << i+1 << "  =" << wcss->crval[i] << endl;
    for (int i=0; i<wcss->naxes; i++)
      str << "# CRPIX" << i+1 << "  =" << wcss->crpix[i] << endl;
    for (int i=0; i<wcss->naxes; i++)
      str << "# CDELT" << i+1 << "  =" << wcss->cdelt[i] << endl;
    for (int i=0; i<wcss->naxes; i++)
      str << "# CTYPE" << i+1 << "  =" << wcss->ctype[i] << endl;
    for (int i=0; i<wcss->naxes; i++)
      if (wcss->units[i])
	str << "# CUNIT" << i+1 << "  =" << wcss->units[i] << endl;
    str << "# CD1_1   =" << wcss->cd[0] << endl;
    str << "# CD1_2   =" << wcss->cd[1] << endl;
    str << "# CD2_1   =" << wcss->cd[2] << endl;
    str << "# CD2_2   =" << wcss->cd[3] << endl;

    int ilat = 1;
    if (!strncmp(wcss->ctype[1],"LAT",3) || !strncmp(wcss->ctype[1],"DEC",3))
      ilat = 2;
    for (int i=0; i<10; i++)
      if (wcss->prj.p[i])
	str << "# PV" << ilat << "_" << i << "  ="<< wcss->prj.p[i] << endl;

    if (wcss->wcsname)
      str << "# WCSNAME =" << wcss->wcsname << endl;

    // not required, but needed anyways
    if (!strncmp("RA",wcss->ctype[0],2) || !strncmp("RA",wcss->ctype[1],2))
      if (!strncmp("FK4",wcss->radecsys,3) ||
	  !strncmp("FK5",wcss->radecsys,3) ||
	  !strncmp("ICRS",wcss->radecsys,4))
	str << "# RADESYS =" << wcss->radecsys << endl;

    str << "# MJD-OBS = " 
	<< (wcss->epoch-1900)*365.242198781+15019.81352 << endl;

    str << "# EQUINOX =" << wcss->equinox << endl;
    if (wcss->latpole != 999)
      str << "# LATPOLE =" << wcss->latpole << endl;
    if (wcss->longpole != 999)
      str << "# LONPOLE =" << wcss->longpole << endl;
  }
}

void FitsImage::initWCS0(const Vector& pix)
{
  if (wcs[0]) {
    Vector cc = mapFromRef(pix, IMAGE, FK5);
    WorldCoor* ww = wcs[0];
    wcs[WCS0-WCS] = wcskinit(ww->nxpix, ww->nypix, "RA---TAN", "DEC--TAN", cc[0], cc[1], 0, 0, ww->cd, 0, 0, 0, 2000, 0);
  }
}

void FitsImage::resetWCS0()
{
  if (wcs[WCS0-WCS])
    wcsfree(wcs[WCS0-WCS]);
  wcs[WCS0-WCS] = NULL;
}

FitsHead* FitsImage::getHead()
{
  if (image_)
    return image_->head();
  else    
    return NULL;
}

char* FitsImage::getCards()
{
  if (image_) {
    FitsHead* head = image_->head();
    if (head)
      return head->cards();
  }
  return NULL;
}

int FitsImage::getNCards()
{
  if (image_) {
    FitsHead* head = image_->head();
    if (head)
      return head->ncard();
  }
  return 0;
}

char* FitsImage::getKeyword(const char* key)
{
  return fits_->getKeyword(key);
}

int FitsImage::findKeyword(const char* key)
{
  return fits_->find(key);
}

char* FitsImage::displayHeader()
{
  // always return the header to the native fits ext, not the image
  if (DebugBin)
    return display(image_->head());
  else
    return display(fits_->head());
}

char* FitsImage::displayPrimary()
{
  // always return the header to the native fits ext, not the image
  if (DebugBin)
    return display(image_->primary());
  else
    return display(fits_->primary());
}

char* FitsImage::display(FitsHead* head)
{
  int size = head->ncard() * (FTY_CARDLEN+1);
  char* lbuf = new char[size+1];

  char* lptr = lbuf;
  char* cptr = head->cards();
  for (int i=0; i<head->ncard(); i++,cptr+=FTY_CARDLEN) {
    memcpy(lptr, cptr, FTY_CARDLEN);
    lptr+=FTY_CARDLEN;
    *(lptr++) = '\n';
  }

  lbuf[size] = '\0';
  return lbuf;
}

void FitsImage::parseSection(char* lbuf, Vector* v1, Vector* v2)
{
  double x1, y1, x2, y2;
  char d; // dummy char
  string x(lbuf);
  istringstream str(x);
  str >> d >> (*v1)[0] >> d >> (*v2)[0] >> d >> (*v1)[1] >> d >> (*v2)[1] >> d;
}

int FitsImage::processKeywords()
{
  // Physical to Image (LTM/LTV keywords) (with no wcsname already located)

  if (!keyLTMV) {
    if (image_->find("LTM1_1") ||
	image_->find("LTM1_2") ||
	image_->find("LTM2_1") ||
	image_->find("LTM2_2") ||
	image_->find("LTV1") ||
	image_->find("LTV2"))
      keyLTMV = 1;

    double ltm11 = image_->getReal("LTM1_1", 1);
    double ltm12 = image_->getReal("LTM1_2", 0);
    double ltm21 = image_->getReal("LTM2_1", 0);
    double ltm22 = image_->getReal("LTM2_2", 1);

    double ltv1 = image_->getReal("LTV1", 0);
    double ltv2 = image_->getReal("LTV2", 0);

    physicalToImage = Matrix(ltm11, ltm12, ltm21, ltm22, ltv1, ltv2);
    imageToPhysical = physicalToImage.invert();
  }

  // CDD to Detector (DTM/DTV keywords)

  if (image_->find("DTM1_1") ||
      image_->find("DTM1_2") ||
      image_->find("DTM2_1") ||
      image_->find("DTM2_2") ||
      image_->find("DTV1") ||
      image_->find("DTV2"))
    keyDTMV = 1;

  double dtm11 = image_->getReal("DTM1_1", 1);
  double dtm12 = image_->getReal("DTM1_2", 0);
  double dtm21 = image_->getReal("DTM2_1", 0);
  double dtm22 = image_->getReal("DTM2_2", 1);

  double dtv1 = image_->getReal("DTV1", 0);
  double dtv2 = image_->getReal("DTV2", 0);

  physicalToDetector = Matrix(dtm11, dtm12, dtm21, dtm22, dtv1, dtv2);
  detectorToPhysical = physicalToDetector.invert();

  // Physical to Amplifier (ATM/ATV keywords)

  if (image_->find("ATM1_1") ||
      image_->find("ATM1_2") ||
      image_->find("ATM2_1") ||
      image_->find("ATM2_2") ||
      image_->find("ATV1") ||
      image_->find("ATV2"))
    keyATMV = 1;

  double atm11 = image_->getReal("ATM1_1", 1);
  double atm12 = image_->getReal("ATM1_2", 0);
  double atm21 = image_->getReal("ATM2_1", 0);
  double atm22 = image_->getReal("ATM2_2", 1);

  double atv1 = image_->getReal("ATV1", 0);
  double atv2 = image_->getReal("ATV2", 0);

  physicalToAmplifier = Matrix(atm11, atm12, atm21, atm22, atv1, atv2);
  amplifierToPhysical = physicalToAmplifier.invert();

  if (DebugMosaic) {
    cerr << endl;
    cerr << rootBaseFileName << endl;
    cerr << "ATM/V: " << physicalToAmplifier << endl;
    cerr << "ATM/V-1: " << amplifierToPhysical << endl;
    cerr << "DTM/V: " << physicalToDetector << endl;
    cerr << "DTM/V-1: " << detectorToPhysical << endl;
    cerr << "LTM/V: " << physicalToImage << endl;
    cerr << "LTM/V-1: " << imageToPhysical << endl;
  }

  // DATASEC

  iparams[0] = width_;
  iparams[1] = 0;
  iparams[2] = width_;
  iparams[3] = 0;
  iparams[4] = height_;

  char* datstr = image_->getString("DATASEC");
  if (datstr && *datstr) {
    Vector v1,v2;
    parseSection(datstr,&v1,&v2);

    if (v1[0] < 1)
      v1[0] = 1;
    if (v1[1] < 1)
      v1[1] = 1;
    if (v2[0] > width_)
      v2[0] = width_;
    if (v2[1] > height_)
      v2[1] = height_;

    keyDATASEC = 1;
    datasec = BBox(v1,v2);

    // DATASEC is in IMAGE, dparams go from 0 to width
    v1 -= Vector(1,1);
    dparams[0] = width_;
    dparams[1] = v1[0];
    dparams[2] = v2[0];
    dparams[3] = v1[1];
    dparams[4] = v2[1];
  }
  else {
    keyDATASEC = 0;
    datasec = BBox(1,1,width_,height_);

    dparams[0] = width_;
    dparams[1] = 0;
    dparams[2] = width_;
    dparams[3] = 0;
    dparams[4] = height_;
  }
  if (datstr)
    delete [] datstr;

  if (DebugMosaic) {
    cerr << "iparams " 
	 << iparams[0] << ' ' << iparams[1] << ' ' <<  iparams[2] << ' ' 
	 << iparams[3] << ' ' << iparams[4] << endl;

    cerr << "dparams " 
	 << dparams[0] << ' ' << dparams[1] << ' ' <<  dparams[2] << ' ' 
	 << dparams[3] << ' ' << dparams[4] << endl;
  }

  switch (*(parent->currentMosaic)) {
  case FrameBase::NOMOSAIC:
    return 1;
  case FrameBase::IRAF:
    return processKeywordsIRAF();
  case FrameBase::WCSMOSAIC:
  case FrameBase::WFPC2:
    return processKeywordsWCS();
  }
}

int FitsImage::processKeywordsIRAF() 
{
  // reset
  rotation_ = 0;
  zoom_ = Vector(1,1);
  orientation_ = NORMAL;
  origin_ = Vector();

  // DETSIZE
  {
    char* sizestr = image_->getString("DETSIZE");
    if (!(sizestr && *sizestr)) {
      if (sizestr)
	delete [] sizestr;
      return 0;
    }
    
    Vector v1,v2;
    parseSection(sizestr,&v1,&v2);
    delete [] sizestr;

    detsize = BBox(v1,v2);
  }

  // DETSEC
  {
    char* detstr =  image_->getString("DETSEC");
    if (!(detstr && *detstr)) {
      if (detstr)
	delete [] detstr;
      return 0;
    }

    Vector v1,v2;
    parseSection(detstr,&v1,&v2);
    delete [] detstr;

    int xx = (v1[0] < v2[0]);
    int yy = (v1[1] < v2[1]);

    if (xx && yy)
      orientation_ = NORMAL;
    else if (!xx && yy)
      orientation_ = XX;
    else if (!xx && !yy)
      orientation_ = XY;
    else if (xx && !yy)
      orientation_ = YY;

    detsec = BBox(v1,v2);
  }

  // CCDSUM
  ccdsum = Vector(1,1);
  {
    char* ccdstr = image_->getString("CCDSUM");
    if (ccdstr && *ccdstr) {
      double Ns, Np, Ns1, Np1;
      string x(ccdstr);
      istringstream str(x);

      str >> Ns >> Np >> Ns1 >> Np1;
      ccdsum = Vector(1/Ns, 1/Np);
    }

    if (ccdstr)
      delete [] ccdstr;
  }

  origin_ = detsec.ll * Scale(ccdsum) * Translate(-datasec.ll);

  if (*(parent->currentMosaicCount) == 0) {
    Vector cc = datasec.center() * imageToData;
    Matrix flip;
    switch (orientation_) {
    case XX:
      flip = FlipX();
      break;
    case YY:
      flip = FlipY();
      break;
    case XY:
      flip = FlipXY();
      break;
    }

    Matrix m =
      Translate(-cc) *
      flip *
      Scale(zoom_) *
      Rotate(rotation_) *
      Translate(cc) *
      Translate(origin_);
    *(parent->currentWCSmatrix) = m.invert();
  }

  if (DebugMosaic) {
    cerr << "ProcessKeywordsIRAF" << endl
	 << " datasec: " << datasec << endl
	 << " ccdsum : " << ccdsum << endl
	 << " detsize: " << detsize << endl
	 << " detsec : " << detsec << endl
	 << " orientation: " << orientation_ << endl
	 << " origin: " << origin_ << endl
	 << " matrix " << *(parent->currentWCSmatrix) << endl;
  }

  return 1;
}

int FitsImage::processKeywordsWCS()
{
  CoordSystem sys =  *(parent->currentMosaicSystem);

  int* params = getDataParams(parent->currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  if (DebugMosaic)
    cerr << "ProcessKeywordsWCS " << coordSystemStr_[sys-IMAGE] << endl
	 << " crpix " << getWCScrpix(sys) << endl
	 << " cdelt " << getWCScdelt(sys) << endl
	 << " rotation " << radToDeg(getWCSRotation(sys,FK5)) << endl
	 << " orientation " << getWCSOrientation(sys,FK5) << endl
	 << " xmin xmax " << xmin << ' ' << xmax << endl
	 << " ymin ymax " << ymin << ' ' << ymax << endl;

  if (*(parent->currentMosaicCount) == 0) {
    // orientation
    Matrix flip;
    orientation_ = NORMAL;

    // rotation
    rotation_ = 0;

    // zoom
    zoom_ = Vector(1,1);

    // origin
    origin_ = Vector();

    // WCS bbox
    Vector cc = getWCScrpix(sys) * imageToData;

    Matrix m =
      Translate(-cc) *
      flip *
      Scale(zoom_) *
      Rotate(rotation_) *
      Translate(cc) *
      Translate(origin_);

    Vector aa = Vector(xmin,ymin) * Translate(.5,.5) * m * dataToImage;
    BBox bb(aa,aa);
    bb.bound(Vector(xmax-1,ymin) * Translate(.5,.5) * m * dataToImage);
    bb.bound(Vector(xmax-1,ymax-1) * Translate(.5,.5) * m * dataToImage);
    bb.bound(Vector(xmin,ymax-1) * Translate(.5,.5) * m * dataToImage);

    // save for next
    *(parent->currentWCScdelt) = getWCScdelt(sys);
    *(parent->currentWCSrot)   = getWCSRotation(sys,FK5);
    *(parent->currentWCSorientation) = getWCSOrientation(sys,FK5);
    *(parent->currentWCSmatrix) = m;
    *(parent->currentWCSbb) = bb;

    if (DebugMosaic)
      cerr << " WCS 1 matrix " << endl
	   << "  center " << cc << endl
	   << "  rotation " << radToDeg(rotation_) << endl
	   << "  orientation " << orientation_ << endl
	   << "  zoom " << zoom_ << endl
	   << "  origin " << origin_ << endl
	   << "  matrix " << m << endl
	   << "  WCSbb " << *(parent->currentWCSbb) << ' ' 
	   << (parent->currentWCSbb)->size() << endl;
  }
  else {
    // orientation
    Orientation o1 = getWCSOrientation(sys,FK5);
    Orientation o2 = *(parent->currentWCSorientation);
    Matrix flip;
    switch (o1) {
    case NORMAL:
      {
	switch (o2) {
	case NORMAL:
	  orientation_ = NORMAL;
	  break;
	case XX:
	  orientation_ = XX;
	  flip = FlipX();
	  break;
	}
      }
      break;
    case XX:
      {
	switch (o2) {
	case NORMAL:
	  orientation_ = XX;
	  flip = FlipX();
	  break;
	case XX:
	  orientation_ = NORMAL;
	  break;
	}
      }
      break;
    }

    // rotation
    double r1 = getWCSRotation(sys,FK5);
    double r2 = *(parent->currentWCSrot);
    switch (o1) {
    case NORMAL:
      {
	switch (o2) {
	case NORMAL:
	  rotation_ = r2-r1;
	  break;
	case XX:
	  rotation_ = r2+r1;
	  break;
	}
      }
      break;
    case XX:
      {
	switch (o2) {
	case NORMAL:
	  rotation_ = -r2-r1;
	  break;
	case XX:
	  rotation_ = -r2+r1;
	  break;
	}
      }
      break;
    }

    // zoom
    double zz = fabs(getWCScdelt(sys)[0]/(*(parent->currentWCScdelt))[0]);
    zoom_ = Vector(zz,zz);

    // origin
    Vector aa = getWCScrpix(sys);
    Vector bb = pix2wcs(aa,sys,FK5);
    Vector org = (*(parent->channelFits))->wcs2pix(bb,sys,FK5) * imageToData;

    Vector cc = getWCScrpix(sys) * imageToData;

    Matrix mm =
      Translate(-cc) *
      flip *
      Scale(zoom_) *
      Rotate(rotation_) *
      Translate(cc);
    origin_ = org*(*(parent->currentWCSmatrix)) - cc*mm;

    // WCS bbox
    Matrix m =
      Translate(-cc) *
      flip *
      Scale(zoom_) *
      Rotate(rotation_) *
      Translate(cc) *
      Translate(origin_);

    (parent->currentWCSbb)->bound(
       Vector(xmin,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
       Vector(xmax-1,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
       Vector(xmax-1,ymax-1) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
       Vector(xmin,ymax-1) * Translate(.5,.5) * m * dataToImage);

    if (DebugMosaic)
      cerr << " WCS 1+ matrix " << endl
	   << "  center " << cc << endl
	   << "  rotation " << radToDeg(rotation_) << endl
	   << "  orientation " << orientation_ << endl
	   << "  zoom " << zoom_ << endl
	   << "  origin " << origin_ << endl
	   << "  matrix " << m << endl
	   << "  WCSbb " << *(parent->currentWCSbb) << ' ' 
	   << (parent->currentWCSbb)->size() << endl;
  }

  return 1;
}

void FitsImage::updateWCSbb(CoordSystem sys, int first)
{
  int* params = getDataParams(parent->currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  if (first) {
    Vector cc = getWCScrpix(sys) * imageToData;
    Matrix flip;
    switch (orientation_) {
    case XX:
      flip = FlipX();
    }

    // WCS bbox
    Matrix m =
      Translate(-cc) *
      flip *
      Scale(zoom_) *
      Rotate(rotation_) *
      Translate(cc) *
      Translate(origin_);

    Vector aa = Vector(xmin,ymin) * Translate(.5,.5) * m * dataToImage;
    *(parent->currentWCSbb) = BBox(aa,aa);
    (parent->currentWCSbb)->bound(
      Vector(xmax-1,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmax-1,ymax-1) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmin,ymax-1) * Translate(.5,.5) * m * dataToImage);

    if (DebugMosaic)
      cerr << "updateWCSbb " << endl
	   << " WCS 1 matrix " << endl
	   << "  center " << cc << endl
	   << "  rotation " << radToDeg(rotation_) << endl
	   << "  orientation " << orientation_ << endl
	   << "  zoom " << zoom_ << endl
	   << "  origin " << origin_ << endl
	   << "  matrix " << m << endl
	   << "  WCSbb " << *(parent->currentWCSbb) << ' ' 
	   << (parent->currentWCSbb)->size() << endl;
  }
  else {
    Vector cc = getWCScrpix(sys) * imageToData;
    Matrix flip;
    switch (orientation_) {
    case XX:
      flip = FlipX();
    }

    // WCS bbox
    Matrix m =
      Translate(-cc) *
      flip *
      Scale(zoom_) *
      Rotate(rotation_) *
      Translate(cc) *
      Translate(origin_);

    (parent->currentWCSbb)->bound(
      Vector(xmin,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmax-1,ymin) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmax-1,ymax-1) * Translate(.5,.5) * m * dataToImage);
    (parent->currentWCSbb)->bound(
      Vector(xmin,ymax-1) * Translate(.5,.5) * m * dataToImage);

    if (DebugMosaic)
      cerr << "updateWCSbb " << endl
	   << " WCS 1+ matrix " << endl
	   << "  center " << cc << endl
	   << "  rotation " << radToDeg(rotation_) << endl
	   << "  orientation " << orientation_ << endl
	   << "  zoom " << zoom_ << endl
	   << "  origin " << origin_ << endl
	   << "  matrix " << m << endl
	   << "  WCSbb " << *(parent->currentWCSbb) << ' ' 
	   << (parent->currentWCSbb)->size() << endl;
  }
}

int FitsImage::hasWCS(CoordSystem sys)
{
  return (sys>=WCS && wcs && wcs[sys-WCS]) ? 1 : 0;
}

int FitsImage::hasWCSEqu(CoordSystem sys)
{
  if (hasWCS(sys)) {
    const char* str = ::getradecsys(wcs[sys-WCS]);

    if (!strncmp("FK4", str, 3))
      return 1;
    else if (!strncmp("FK5", str, 3))
      return 1;
    else if (!strncmp("ICR", str, 3))
      return 1;
    else if (!strncmp("GAL", str, 3))
      return 1;
    else if (!strncmp("ECL", str, 3))
      return 1;
    else if (!strncmp("NPOLE", str, 4)) // special north polar angle
      return 1;
    else
      return 0;
  }
  else
    return 0;
}

int FitsImage::hasWCSLinear(CoordSystem sys)
{
  
  if (hasWCS(sys))
    return hasWCSEqu(sys) ? 0 : 1;
  else
    return 0;
}

char* FitsImage::pix2wcs(Vector in, CoordSystem sys, SkyFrame sky,
			 SkyFormat format, char* lbuf, int len)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (hasWCSEqu(sys)) {
      if (sky == NATIVEWCS)
	wcsoutinit(wcs[i], ::getradecsys(wcs[i]));
      else
	wcsoutinit(wcs[i], skyFrameStr_[sky-FK4]);

      switch (format) {
      case ARCMIN:
      case ARCSEC:
      case DEGREES:
	setwcsdeg(wcs[i],1);
	wcs[i]->ndec = 5;
	break;
      case SEXAGESIMAL:
      case HMS:
	setwcsdeg(wcs[i],0);
	wcs[i]->ndec = 3;
	break;
      }
    }
    else {
      wcsoutinit(wcs[i], ::getradecsys(wcs[i]));
      setwcslin(wcs[i],2);
    }

    ::pix2wcst(wcs[i], in[0], in[1], lbuf, len);
  }

  return lbuf;
}

Vector FitsImage::pix2wcs(Vector in, CoordSystem sys)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    wcsoutinit(wcs[i], ::getradecsys(wcs[i]));
    double x,y;
    ::pix2wcs(wcs[i], in[0], in[1], &x, &y);

    return Vector(x,y);
  }
  else
    return in;
}

Vector FitsImage::pix2wcs(Vector in, CoordSystem sys, SkyFrame sky)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (hasWCSEqu(sys)) {
      if (sky == NATIVEWCS)
	wcsoutinit(wcs[i], ::getradecsys(wcs[i]));
      else
	wcsoutinit(wcs[i], skyFrameStr_[sky-FK4]);
    }
    else
      wcsoutinit(wcs[i], ::getradecsys(wcs[i]));

    double x,y;
    ::pix2wcs(wcs[i], in[0], in[1], &x, &y);

    return Vector(x,y);
  }
  else
    return in;
}

Vector FitsImage::wcs2pix(Vector in, CoordSystem sys)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;
    wcsininit(wcs[i], ::getradecsys(wcs[i]));

    double x,y;
    int off;
    ::wcs2pix(wcs[i], in[0], in[1], &x, &y, &off);

    return Vector(x,y);
  }
  else
    return in;
}

Vector FitsImage::wcs2pix(Vector in, CoordSystem sys, SkyFrame sky)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (hasWCSEqu(sys)) {
      if (sky == NATIVEWCS)
	wcsininit(wcs[i], ::getradecsys(wcs[i]));
      else
	wcsininit(wcs[i], skyFrameStr_[sky-FK4]);
    }
    else
      wcsininit(wcs[i], ::getradecsys(wcs[i]));

    double x,y;
    int off;
    ::wcs2pix(wcs[i], in[0], in[1], &x, &y, &off);

    return Vector(x,y);
  }
  else
    return in;
}

Vector FitsImage::pix2wcsDist(Vector in, CoordSystem sys)
{
  if (hasWCS(sys)) {
    Vector cd = getWCScdelt(sys);
    return Vector(in[0]*cd[0], in[1]*cd[1]).abs();
  }
  else
    return in;
}

Vector FitsImage::wcs2pixDist(Vector in, CoordSystem sys)
{
  if (hasWCS(sys)) {
    Vector cd = getWCScdelt(sys);
    return Vector(in[0]/cd[0], in[1]/cd[1]).abs();
  }
  else
    return in;
}

double FitsImage::wcsdist(Vector a, Vector b, CoordSystem sys)
{
  if (hasWCS(sys))
    return ::wcsdist(a[0],a[1],b[0],b[1]);
  else
    return 0;
}

Vector FitsImage::getWCScrval(CoordSystem sys)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (!wcs[i]->coorflip)
      return Vector(wcs[i]->crval[0], wcs[i]->crval[1]);
    else
      return Vector(wcs[i]->crval[1], wcs[i]->crval[0]);
  }
  else
    return Vector();
}

Vector FitsImage::getWCScrpix(CoordSystem sys)
{
  if (hasWCS(sys))
      return Vector(wcs[sys-WCS]->crpix[0], wcs[sys-WCS]->crpix[1]);
  else
    return Vector();
}

Vector FitsImage::getWCScdelt(CoordSystem sys)
{
  if (hasWCS(sys)) {
    int i = sys-WCS;

    if (!wcs[i]->coorflip)
      return Vector(wcs[i]->cdelt[0], wcs[i]->cdelt[1]);
    else
      return Vector(wcs[i]->cdelt[1], wcs[i]->cdelt[0]);
  }
  else
    return Vector();
}

Vector FitsImage::getWCStan(CoordSystem sys)
{
  if (wcstan)
    return wcstan[sys-WCS];
  else
    return Vector();
}

double FitsImage::getWCSRotation(CoordSystem sys, SkyFrame sky)
{
  if (hasWCS(sys)) {
    Vector orval = pix2wcs(getWCStan(sys), sys, sky);
    Vector orpix = wcs2pix(orval, sys, sky);
    Vector delta = getWCScdelt(sys).abs();
    Vector npix = wcs2pix(Vector(orval[0],orval[1]+delta[1]), sys, sky);
    Vector north = npix - orpix;

    return -(north.angle()-M_PI/2);
  }

  return 0;
}

Orientation FitsImage::getWCSOrientation(CoordSystem sys, SkyFrame sky)
{
  if (hasWCS(sys)) {
    Vector orval = pix2wcs(getWCStan(sys), sys, sky);
    Vector orpix = wcs2pix(orval, sys, sky);
    Vector delta = getWCScdelt(sys).abs();
    Vector npix = wcs2pix(Vector(orval[0],orval[1]+delta[1]), sys, sky);
    Vector north = npix - orpix;
    Vector epix = wcs2pix(Vector(orval[0]+delta[0],orval[1]), sys, sky);
    Vector east = epix - orpix;

    // take the cross product and see which way the 3rd axis is pointing
    double w = cross(east, north)[2];

    if (!hasWCSEqu(sys))
      return w>0 ? NORMAL : XX;
    else
      return w<0 ? NORMAL : XX;
  }

  return NORMAL;
}

