// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "fitsimage.h"
#include "util.h"

// Map Point

Vector FitsImage::mapFromRef(const Vector& v, CoordSystem out, SkyFrame sky)
{
  switch (out) {
  case IMAGE:
    return v * refToImage;
  case PHYSICAL:
    return v * refToPhysical;
  case AMPLIFIER:
    return v * refToAmplifier;
  case DETECTOR:
    return v * refToDetector;
  default:
    if (hasWCS(out))
      return pix2wcs(v * refToImage, out, sky);
  }

  cerr << "FitsImage Internal Error: mapFromRef()" << endl;
  return v;
}      

void FitsImage::mapFromRef(const Vector& v, CoordSystem out, SkyFrame sky,
			    SkyFormat format, char* buf, int length)
{
  if (hasWCS(out)) {
    pix2wcs(v * refToImage, out, sky, format, buf, length);
    return;
  }

  strcpy(buf,"");
  cerr << "FitsImage Internal Error: mapFromRef()" << endl;
}

Vector FitsImage::mapToRef(const Vector& v, CoordSystem in, SkyFrame sky)
{
  switch (in) {
  case IMAGE:
    return v * imageToRef;
  case PHYSICAL:
    return v * physicalToRef;
  case AMPLIFIER:
    return v * amplifierToRef;
  case DETECTOR:
    return v * detectorToRef;
  default:
    if (hasWCS(in))
      return wcs2pix(v, in, sky) * imageToRef;
  }

  cerr << "FitsImage Internal Error: mapToRef()" << endl;
  return v;
}

// Map Length

Vector FitsImage::mapLen(const Vector& v, const Matrix& mx)
{
  // remove translation
  Vector t = Vector() * mx;
  Matrix sr = mx * Translate(-t);

  // remove rotation
  Vector r = Vector(1,0) * sr;
  Matrix s = sr * Rotate(r.angle());

  // all that is left is Scaling
  return (v*s).abs();
}

double FitsImage::mapLenFromRef(double d, CoordSystem sys, SkyFormat format)
{
  Vector r = mapLenFromRef(Vector(d,0),sys,format);
  return r[0];
}

Vector FitsImage::mapLenFromRef(const Vector& v, CoordSystem sys, 
				SkyFormat format)
{
  // really from image coords

  switch (sys) {
  case IMAGE:
    return mapLen(v,refToImage);
  case PHYSICAL:
    return mapLen(v,refToPhysical);
  case AMPLIFIER:
    return mapLen(v,refToPhysical * physicalToAmplifier);
  case DETECTOR:
    return mapLen(v,refToPhysical * physicalToDetector);
  default:
    if (hasWCS(sys)) {
      if (hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	case SEXAGESIMAL:
	case HMS:
	  return pix2wcsDist(v, sys);
	case ARCMIN:
	  return pix2wcsDist(v*60, sys);
	case ARCSEC:
	  return pix2wcsDist(v*60*60, sys);
	}
      }
      else {
	return pix2wcsDist(v, sys);
      }
    }
    break;
  }

  cerr << "FitsImage Internal Error: mapLenFromRef()" << endl;
  return v;
}

double FitsImage::mapLenToRef(double d, CoordSystem sys, SkyFormat format)
{
  Vector r = mapLenToRef(Vector(d,0), sys, format);
  return r[0];
}

Vector FitsImage::mapLenToRef(const Vector& v,CoordSystem sys,SkyFormat format)
{
  switch (sys) {
  case IMAGE:
    return mapLen(v,imageToRef);
  case PHYSICAL:
    return mapLen(v,physicalToRef);
  case AMPLIFIER:
    return mapLen(v,amplifierToPhysical * physicalToRef);
  case DETECTOR:
    return mapLen(v,detectorToPhysical * physicalToRef);
  default:
    if (hasWCS(sys)) {
      if (hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	case SEXAGESIMAL:
	case HMS:
	  return wcs2pixDist(v, sys);
	case ARCMIN:
	  return wcs2pixDist(v/60, sys);
	case ARCSEC:
	  return wcs2pixDist(v/(60*60), sys);
	}
      }
      else
	return wcs2pixDist(v, sys);
    }
  }

  cerr << "FitsImage Internal Error: mapLenToRef()" << endl;
  return v;
}

// Map Distance

double FitsImage::mapDistFromRef(const Vector& vv1, const Vector& vv2, 
			    CoordSystem sys, SkyFormat format)
{
  switch (sys) {
  case IMAGE:
    {
      Vector v1 = vv1 * refToImage;
      Vector v2 = vv2 * refToImage;
      return (v2-v1).length();
    }
  case PHYSICAL:
    {
      Vector v1 = vv1 * refToPhysical;
      Vector v2 = vv2 * refToPhysical;
      return (v2-v1).length();
    }
  case AMPLIFIER:
    {
      Vector v1 = vv1 * refToPhysical * physicalToAmplifier;
      Vector v2 = vv2 * refToPhysical * physicalToAmplifier;
      return (v2-v1).length();
    }
  case DETECTOR:
    {
      Vector v1 = vv1 * refToPhysical * physicalToDetector;
      Vector v2 = vv2 * refToPhysical * physicalToDetector;
      return (v2-v1).length();
    }
  default:
    if (hasWCS(sys)) {
      if (hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	case SEXAGESIMAL:
	case HMS:
	  {
	    Vector v1 = pix2wcs(vv1 * refToImage, sys);
	    Vector v2 = pix2wcs(vv2 * refToImage, sys);
	    return wcsdist(v1,v2, sys);
	  }
	case ARCMIN:
	  {
	    Vector v1 = pix2wcs(vv1 * refToImage, sys);
	    Vector v2 = pix2wcs(vv2 * refToImage, sys);
	    return wcsdist(v1,v2, sys)*60;
	  }
	case ARCSEC:
	  {
	    Vector v1 = pix2wcs(vv1 * refToImage, sys);
	    Vector v2 = pix2wcs(vv2 * refToImage, sys);
	    return wcsdist(v1,v2, sys)*60*60;
	  }
	}
      }
      else {
	Vector v1 = pix2wcs(vv1 * refToImage, sys);
	Vector v2 = pix2wcs(vv2 * refToImage, sys);
	return (v2-v1).length();
      }
    }
  }

  cerr << "FitsImage Internal Error: mapLenFromRef" << endl;
  return 0;
}

