// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "tcl.h"
#include <Xlib.h>
#include <Xutil.h>

#include "framergbtrue.h"
#include "fitsimage.h"
#include "util.h"
#include "ps.h"

#include "NaN.h"

FrameRGBTrueColor::FrameRGBTrueColor(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item)
  : FrameRGB(i, c, item)
{
  skipUpdate = 0;

  colormapXM = NULL;
  colormapPM = 0;
  colormapGCXOR = 0;

  for (int i=0; i<3; i++)
    colormapData[i] = NULL;

  memset(bgTrueColor_,255,4);
}

FrameRGBTrueColor::~FrameRGBTrueColor()
{
  if (colormapXM)
    XDestroyImage(colormapXM);

  if (colormapPM)
    Tk_FreePixmap(display, colormapPM);

  if (colormapGCXOR)
    XFreeGC(display, colormapGCXOR);

  for (int i=0; i<3; i++)
    if (colormapData[i])
      delete [] colormapData[i];
}

void FrameRGBTrueColor::bgColorCmd(const char* color)
{
  if (bgColorName)
    delete [] bgColorName;
  bgColorName = dupstr(color);
  bgColor = getXColor(bgColorName);
  encodeTrueColor(bgColor, bgTrueColor_);
  update(BASE);
}

// fillRGBImage

unsigned char* FrameRGBTrueColor::fillRGBImage(int width, int height,
					       int x0, int y0, int x1, int y1,
					     double* (FitsImage::*getMatrix)())
{
  // create 3 byte img
  unsigned char* img = new unsigned char[width*height*3];
  memset(img, 0, width*height*3);

  // special case
  if ((!view[0] || !fits[0]) &&
      (!view[1] || !fits[1]) &&
      (!view[2] || !fits[2])) {

    unsigned char* ptr = img;
    for (int jj=0; jj<height; jj++)
      for (int ii=0; ii<width; ii++) {
	*img++ = (unsigned char)bgColor->red;
	*img++ = (unsigned char)bgColor->green;
	*img++ = (unsigned char)bgColor->blue;
      }	
    return img;
  }

  char* mask = new char[width*height];
  memset(mask, 0, width*height);

  // ok, create an RGB image
  for (int ii=0; ii<3; ii++) {
    if (!view[ii] || !fits[ii])
      continue;

    if (!(mosaicCount[ii]>1))
      fillRGBImageSingle(ii, img, mask, x0, y0, x1, y1, getMatrix);
    else {
      if (mosaicFast && mosaicType[ii] == IRAF)
	fillRGBImageMosaicFast(ii, img, mask, x0, y0, x1, y1, getMatrix);
      else
	fillRGBImageMosaic(ii, img, mask, x0, y0, x1, y1, getMatrix);
    }
  }

  // now, apply mask
  unsigned char* ptr = img;
  char* mk = mask;
  for (int jj=y0; jj<y1; jj++)
    for (int ii=x0; ii<x1; ii++, ptr+=3, mk++)
      if (!*mk) {
	*(ptr  ) = (unsigned char)bgColor->red;
	*(ptr+1) = (unsigned char)bgColor->green;
	*(ptr+2) = (unsigned char)bgColor->blue;
      }

  // clean up
  if (mask)
    delete [] mask;

  return img;
}

void FrameRGBTrueColor::fillRGBImageSingle(int ii,
					   unsigned char* img, char* mask,
					   int x0, int y0, int x1, int y1,
					   double* (FitsImage::*getMatrix)())
{
  double* m = (cfits[ii]->*getMatrix)();
  int* params = cfits[ii]->getDataParams(frScale[ii].scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale[ii]->size() - 1;
  const unsigned char* table = colorScale[ii]->psColors();

  double ll = cfits[ii]->getLowDouble();
  double hh = cfits[ii]->getHighDouble();
  double diff = hh - ll;

  unsigned char* ptr = img;
  char* mk = mask;
  for (int j=y0; j<y1; j++) {
    for (int i=x0; i<x1; i++, ptr+=3, mk++) {

      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = cfits[ii]->getValueDouble(((int)y)*srcw + (int)x);
	  
	if (!isnand(value)) {
	  if (value <= ll)
	    *(ptr+ii) = *table;
	  else if (value >= hh)
	    *(ptr+ii) = *(table+length);
	  else
	    *(ptr+ii) = *(table+((int)(((value - ll)/diff * length) +.5)));

	  // set mask
	  *mk = 1;
	}
      }
    }
  }
}

void FrameRGBTrueColor::fillRGBImageMosaic(int ii, 
					   unsigned char* img, char* mask,
					   int x0, int y0, int x1, int y1,
					   double* (FitsImage::*getMatrix)())
{
  int length = colorScale[ii]->size() - 1;
  const unsigned char* table = colorScale[ii]->psColors();

  unsigned char* dest = img;
  char* mk = mask;
  for (int j=y0; j<y1; j++) {
    for (int i=x0; i<x1; i++, dest+=3, mk++) {
      FitsImage* sptr = cfits[ii];
      while (sptr) {
	double* m = (sptr->*getMatrix)();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = sptr->getDataParams(frScale[ii].scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = sptr->getValueDouble(((int)y)*srcw + (int)x);
	  
	  double ll = sptr->getLowDouble();
	  double hh = sptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isnand(value)) {
	    if (value <= ll)
	      *(dest+ii) = *table;
	    else if (value >= hh)
	      *(dest+ii) = *(table+length);
	    else
	      *(dest+ii) = *(table+((int)(((value - ll)/diff * length) +.5)));

	    // set mask
	    *mk = 1;
	  }
	  break;
	}
	else 
	  sptr = sptr->nextMosaic();
      }
    }
  }
}

void FrameRGBTrueColor::fillRGBImageMosaicFast(int ii, 
					       unsigned char* img, char* mask,
					       int x0, int y0, int x1, int y1,
					     double* (FitsImage::*getMatrix)())
{
  int length = colorScale[ii]->size() - 1;
  const unsigned char* table = colorScale[ii]->psColors();

  FitsImage* sptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  unsigned char* dest = img;
  char* mk = mask;
  for (int j=y0; j<y1; j++) {
    for (int i=x0; i<x1; i++, dest+=3, mk++) {
      double x,y;

      if (sptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = sptr->getValueDouble(((int)y)*srcw + (int)x);
	  
	  if (!isnand(value)) {
	    if (value <= ll)
	      *(dest+ii) = *table;
	    else if (value >= hh)
	      *(dest+ii) = *(table+length);
	    else
	      *(dest+ii) = *(table+((int)(((value - ll)/diff * length) +.5)));

	    // set mask
	    *mk = 1;
	  }
	}
	else 
	  sptr = NULL;
      }

      if (!sptr) {
	sptr = cfits[ii];
	while (sptr) {
	  m = (sptr->*getMatrix)();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = sptr->getDataParams(frScale[ii].scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = sptr->getValueDouble(((int)y)*srcw + (int)x);
	  
	    ll = sptr->getLowDouble();
	    hh = sptr->getHighDouble();
	    diff = hh - ll;

	    if (!isnand(value)) {
	      if (value <= ll)
		*(dest+ii) = *table;
	      else if (value >= hh)
		*(dest+ii) = *(table+length);
	      else
		*(dest+ii) = *(table+((int)(((value - ll)/diff * length)+.5)));

	      // set mask
	      *mk = 1;
	    }
	    break;
	  }
	  else
	    sptr = sptr->nextMosaic();
	}
      }
    }
  }
}

// colormap

void FrameRGBTrueColor::colormapCmd(float rb, float gb, float bb, 
				    float rc, float gc, float bc,
				    unsigned char* cells, int cnt)
{
  // first check for change
  if (bias[0] == rb && bias[1] == gb && bias[2] == bb && 
      contrast[0] == rc && contrast[1] == gc && contrast[2] == bc &&
      colorCells)
    return;

  // we got a change
  bias[0] = rb;
  bias[1] = gb;
  bias[2] = bb;
  contrast[0] = rc;
  contrast[1] = gc;
  contrast[2] = bc;

  updateColorCells(cells, cnt);
  updateColorScale();

  if (mosaicCount[0] || mosaicCount[1] || mosaicCount[2]) {
    update(BASE); // always update
    skipUpdate = 0;
  }
  else {
    if (!skipUpdate) {
      update(BASE); // update once
      skipUpdate = 1;
    }      
  }
}

void FrameRGBTrueColor::colormapBeginCmd()
{
  colormapBeginCmd(Vector(options->width,options->height)/2*widgetToCanvas);
}

void FrameRGBTrueColor::colormapBeginCmd(const Vector& w)
{
  // we need a colorScale before we can render
  if (!validColorScale())
    return;

  // we need some fits data
  // we assume the colorScale length will not change during motion calls
  if (!fits[0] && !fits[1] && !fits[2])
    return;

  int width = options->width;
  int height = options->height;
  cmapOffset = Vector();
  if (cmapArea) {
    width = height = cmapArea;

    switch (cmapMode) {
    case CENTER:
      cmapOffset = (Vector(options->width,options->height) - 
		    Vector(width,height))/2;
      break;
    case CLICK: 
      {
	Vector where = w * canvasToWidget;
	if (where[0] >= 0 && where[0] < options->width &&
	    where[1] >= 0 && where[1] < options->height)
	  cmapOffset = where - Vector(width,height)/2;
	else
	  cmapOffset = (Vector(options->width,options->height) - 
			Vector(width,height))/2;
      }
      break;
    }

    if (cmapOffset[0] < 0) {
      width += (int)cmapOffset[0];
      cmapOffset[0] = 0;
    }
    if (cmapOffset[0]+width >= options->width)
      width = options->width - (int)cmapOffset[0];

    if (cmapOffset[1] < 0) {
      height += (int)cmapOffset[1];
      cmapOffset[1] = 0;
    }

    if (cmapOffset[1]+height >= options->height)
      height = options->height - (int)cmapOffset[1];
  }

  // Create XImage
  if (!(colormapXM = XGETIMAGE(display, pixmap, 0, 0, 
			       width, height, AllPlanes, ZPixmap))) {
    cerr << "FrameTrueColor Internal Error: Unable to Create Colormap XImage" 
	 << endl;
    exit(1);
  }

  // Create Pixmap
  colormapPM = 
    Tk_GetPixmap(display, Tk_WindowId(tkwin), width, height, depth);
  if (!colormapPM) {
    cerr << "FrameTrueColor Internal Error: Unable to Create Colormap Pixmap" 
	 << endl;
    exit(1);
  }

  // colormapGCXOR
  colormapGCXOR = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);

  // Create table index array
  for (int i=0; i<3; i++) {
    if (colormapData)
      delete [] colormapData[i];
    colormapData[i] = new long[width*height];

    if (!colormapData[i]) {
      cerr << "FrameTrueColor Internal Error: Unable to alloc tmp data array" 
	   << endl;
      exit(1);
    }
  }

  // fill data array
  for (int ii=0; ii<3; ii++) {
    if (!view[ii] || !fits[ii])
      continue;

    if (!(mosaicCount[ii]>1))
      colormapBeginSingle(ii, width, height);
    else {
      if (mosaicFast && mosaicType[ii] == IRAF)
	colormapBeginMosaicFast(ii, width, height);
      else
	colormapBeginMosaic(ii, width, height);
    }
  }
}

void FrameRGBTrueColor::colormapMotionCmd(float rb, float gb, float bb, 
					  float rc, float gc, float bc,
					  unsigned char* cells, int cnt)
{
  // we need a colorScale before we can render
  if (!validColorScale())
    return;

  // first check for change
  if (bias[0] == rb && bias[1] == gb && bias[2] == bb && 
      contrast[0] == rc && contrast[1] == gc && contrast[2] == bc &&
      colorCells)
    return;

  // we got a change
  bias[0] = rb;
  bias[1] = gb;
  bias[2] = bb;
  contrast[0] = rc;
  contrast[1] = gc;
  contrast[2] = bc;

  updateColorCells(cells, cnt);
  updateColorScale();

  // special case
  if ((!view[0] || !fits[0]) &&
      (!view[1] || !fits[1]) &&
      (!view[2] || !fits[2]))
    return;

  int& width = colormapXM->width;
  int& height = colormapXM->height;

  // create img
  unsigned char* img = new unsigned char[width*height*3];
  char* mask = new char[width*height];
  memset(img, 0, width*height*3);
  memset(mask, 0, width*height);

  for (int ii=0; ii<3; ii++) {
    if (!view[ii] || !fits[ii])
      continue;

    const unsigned char* table = colorScale[ii]->psColors();
    long* src = colormapData[ii];
    unsigned char* dest = img;
    char* mk = mask;
    for (int j=0; j<height; j++) {
      for (int i=0; i<width; i++, src++, dest+=3, mk++)
	if (*src >=0) {
	  memcpy(dest+ii, table+(*src), 1);
	  *mk = 1;
	}
    }
  }

  // apply mask
  unsigned char* ptr = img;
  char* mk = mask;
  for (int j=0; j<height; j++)
    for (int i=0; i<width; i++, ptr+=3, mk++)
      if (!*mk) {
	*(ptr  ) = (unsigned char)bgColor->red;
	*(ptr+1) = (unsigned char)bgColor->green;
	*(ptr+2) = (unsigned char)bgColor->blue;
      }

  // clean up
  if (mask)
    delete mask;

  // build colormapXM
  if (img)
    buildColormapXM(img);

  // clean up
  if (img)
    delete img;

  // XImage to Pixmap
  XPutImage(display, colormapPM, Widget::gc, colormapXM, 0, 0, 0, 0, 
	    width, height);

  // Display Pixmap
  Vector d = cmapOffset * widgetToWindow;
  XCopyArea(display, colormapPM, Tk_WindowId(tkwin), colormapGCXOR, 0, 0, 
	    width, height, (int)d[0], (int)d[1]);
}

void FrameRGBTrueColor::colormapEndCmd(float rb, float gb, float bb, 
				       float rc, float gc, float bc,
				       unsigned char* cells, int cnt)
{
  if (colormapXM) {
    XDestroyImage(colormapXM);
    colormapXM = NULL;
  }

  if (colormapPM) {
    Tk_FreePixmap(display, colormapPM);
    colormapPM = 0;
  }

  if (colormapGCXOR) {
    XFreeGC(display, colormapGCXOR);
    colormapGCXOR = 0;
  }

  for (int i=0; i<3; i++)
    if (colormapData[i]) {
      delete [] colormapData[i];
      colormapData[i] = NULL;
    }

  update(BASE); // always update
}

void FrameRGBTrueColor::colormapBeginSingle(int ii, int width, int height)
{
  int xoff = (int)cmapOffset[0];
  int yoff = (int)cmapOffset[1];

  double* m = cfits[ii]->getmWidgetToData();
  int* params = cfits[ii]->getDataParams(frScale[ii].scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale[ii]->size() - 1;

  double ll = cfits[ii]->getLowDouble();
  double hh = cfits[ii]->getHighDouble();
  double diff = hh - ll;

  long* dest = colormapData[ii];
  for (int j=yoff; j<yoff+height; j++) {
    for (int i=xoff; i<xoff+width; i++, dest++) {
      *dest = -1;

      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = cfits[ii]->getValueDouble(((int)y)*srcw + (int)x);
	
	if (!isnand(value))
	  if (value <= ll)
	    *dest = 0;
	  else if (value >= hh)
	    *dest = length;
	  else
	    *dest = (int)(((value - ll)/diff * length) + .5);
      }
    }
  }
}

void FrameRGBTrueColor::colormapBeginMosaic(int ii, int width, int height)
{
  int xoff = (int)cmapOffset[0];
  int yoff = (int)cmapOffset[1];

  int length = colorScale[ii]->size() - 1;
  long* dest = colormapData[ii];

  for (int j=yoff; j<yoff+height; j++) {
    for (int i=xoff; i<xoff+width; i++, dest++) {
      *dest = -1;

      FitsImage* sptr = cfits[ii];
      while (sptr) {
	double* m = sptr->getmWidgetToData();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = sptr->getDataParams(frScale[ii].scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = sptr->getValueDouble(((int)y)*srcw + (int)x);
	
	  double ll = sptr->getLowDouble();
	  double hh = sptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isnand(value))
	    if (value <= ll)
	      *dest = 0;
	    else if (value >= hh)
	      *dest = length;
	    else
	      *dest = (int)(((value - ll)/diff * length) + .5);

	  break;
	}
	else
	  sptr = sptr->nextMosaic();
      }
    }
  }
}

void FrameRGBTrueColor::colormapBeginMosaicFast(int ii, int width, int height)
{
  int xoff = (int)cmapOffset[0];
  int yoff = (int)cmapOffset[1];

  int length = colorScale[ii]->size() - 1;
  long* dest = colormapData[ii];

  FitsImage* sptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  for (int j=yoff; j<yoff+height; j++) {
    for (int i=xoff; i<xoff+width; i++, dest++) {
      *dest = -1;
      double x,y;

      if (sptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = sptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isnand(value))
	    if (value <= ll)
	      *dest = 0;
	    else if (value >= hh)
	      *dest = length;
	    else
	      *dest = (int)(((value - ll)/diff * length) + .5);
	}
	else
	  sptr = NULL;
      }

      if (!sptr) {
	sptr = cfits[ii];
	while (sptr) {
	  m = sptr->getmWidgetToData();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = sptr->getDataParams(frScale[ii].scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = sptr->getValueDouble(((int)y)*srcw + (int)x);
	
	    ll = sptr->getLowDouble();
	    hh = sptr->getHighDouble();
	    diff = hh - ll;
	    
	    if (!isnand(value))
	      if (value <= ll)
		*dest = 0;
	      else if (value >= hh)
		*dest = length;
	      else
		*dest = (int)(((value - ll)/diff * length) + .5);

	    break;
	  }
	  else
	    sptr = sptr->nextMosaic();
	}
      }
    }
  }
}

// rotation

void FrameRGBTrueColor::rotateMotion()
{
  // be sure we have the correct color encoded
  encodeTrueColor(bgColor, bgTrueColor_);

  // Rotate from src to dest
  Vector cc = Vector(options->width,options->height)/2;
  Matrix m = (Translate(-cc) * Rotate(rotation-rotateRotation) * 
	      Translate(cc)).invert();
  double* mm = m.mm();

  int& width = options->width;
  int& height = options->height;
  char* src = rotateSrcXM->data;

  int bytesPerPixel = rotateDestXM->bits_per_pixel/8;

  for (int j=0; j<height; j++) {
    // the line may be padded at the end
    char* dest = rotateDestXM->data + j*rotateDestXM->bytes_per_line;

    for (int i=0; i<width; i++, dest+=bytesPerPixel) {
      double x = i*mm[0] + j*mm[3] + mm[6];
      double y = i*mm[1] + j*mm[4] + mm[7];

      if (x >= 0 && x < width && y >= 0 && y < height)
	memcpy(dest, 
	       src + ((int)y)*rotateDestXM->bytes_per_line+
	       ((int)x)*bytesPerPixel, 
	       bytesPerPixel);
      else
	memcpy(dest, bgTrueColor_, bytesPerPixel);
    }
  }

  // XImage to Pixmap
  XPutImage(display, rotatePM, gc, rotateDestXM,
	    0, 0, 0, 0, options->width, options->height);

  // Display Pixmap
  Vector d = Vector(0,0) * widgetToWindow;
  XCopyArea(display, rotatePM, Tk_WindowId(tkwin), rotateGCXOR, 0, 0, 
	    options->width, options->height, (int)d[0], (int)d[1]);
}

// ps

void FrameRGBTrueColor::psLevel1(PSColorSpace mode, 
				 int width, int height, float scale)
{
  // we need a colorScale before we can render
  if (!validColorScale())
    return;

  psLevel1Head(mode, width, height);

  // special case
  if ((!view[0] || !fits[0]) &&
      (!view[1] || !fits[1]) &&
      (!view[2] || !fits[2]))
    return;

  // update psMatrix
  for (int ii=0; ii<3; ii++)
    if (fits[ii]) {
      FitsImage* sptr = cfits[ii];
      while (sptr) {
	sptr->updatePS(psMatrix(scale, width, height));
	sptr = sptr->nextMosaic();
      }
    }

  // create img
  unsigned char* img = fillRGBImage(width, height, 0, 0, width, height, 
				    &FitsImage::getmPSToData);
  NoCompressAsciiHex filter;
  unsigned char* ptr = img;

  // size = 2 char per byte * 3 bytes per pixel + estimate of endls + 
  // endl + ends
  int size = (int)(width*3*2*(1 + 1./80.) +.5) + 2;
  char* buf = new char[size];

  for (int j=0; j<height; j++) {
    ostringstream str;
    for (int i=0; i<width; i++, ptr+=3) {
      unsigned char red = ptr[0];
      unsigned char green = ptr[1];
      unsigned char blue = ptr[2];
	    
      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
      case CMYK:
	filter << red << green << blue;
	break;
      }
      str << filter;
    }
    str << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);

  if (img)
    delete img;
}

void FrameRGBTrueColor::psLevel2(PSColorSpace mode, 
				 int width, int height, float scale)
{
  // we need a colorScale before we can render
  if (!validColorScale())
    return;

  psLevel2Head(mode, width, height);

  // special case
  if ((!view[0] || !fits[0]) &&
      (!view[1] || !fits[1]) &&
      (!view[2] || !fits[2]))
    return;

  // update psMatrix
  for (int ii=0; ii<3; ii++)
    if (fits[ii]) {
      FitsImage* sptr = cfits[ii];
      while (sptr) {
	sptr->updatePS(psMatrix(scale, width, height));
	sptr = sptr->nextMosaic();
      }
    }

  // create img
  unsigned char* img = fillRGBImage(width, height, 0, 0, width, height, 
				    &FitsImage::getmPSToData);
  RLEAscii85 filter;
  unsigned char* ptr = img;

  // size = 2 char per byte * 4 bytes per pixel + estimate of endls + 
  // endl + ends
  int size = (int)(width*4*2*(1 + 1./80.) +.5) + 2;
  char* buf = new char[size];

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++, ptr+=3) {
      unsigned char red = ptr[0];
      unsigned char green = ptr[1];
      unsigned char blue = ptr[2];

      switch (mode) {
      case BW:
      case GRAY:
	filter << RGB2Gray(red, green, blue);
	break;
      case RGB:
	filter << red << green << blue;
	break;
      case CMYK:
	{
	  unsigned char cyan, magenta, yellow, black;
	  RGB2CMYK(red, green, blue, &cyan, &magenta, &yellow, &black);
	  filter << cyan << magenta << yellow << black;
	}
	break;
      }
      str << filter;
    }
    str << ends;
    psFix(str);
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);

  if (img)
    delete img;
}
