// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framebase.h"
#include "fitsimage.h"
#include "util.h"
#include "outfile.h"
#include "outchannel.h"
#include "outsocket.h"

#include "NaN.h"

void FrameBase::saveFitsImageFileCmd(const char* fn, int compress)
{
  if (currentFits) {
    if (!currentFits->saveFitsImageFile(fn, compress)) {
      Tcl_AppendResult(interp, "Unable to save Fits", NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveFitsImageChannelCmd(const char* ch, int compress)
{
  if (currentFits) {
    if (!currentFits->saveFitsImageChannel(interp, ch, compress)) {
      Tcl_AppendResult(interp, "Unable to save fits to channel ", ch, NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveFitsImageSocketCmd(int s, int compress)
{
  if (currentFits) {
    if (!currentFits->saveFitsImageSocket(s,compress)) {
      Tcl_AppendResult(interp, "Unable to save Fits", NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveFitsTableFileCmd(const char* fn, int compress)
{
  if (currentFits) {
    if (!currentFits->saveFitsTableFile(fn, compress)) {
      Tcl_AppendResult(interp, "Unable to save Fits", NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveFitsTableChannelCmd(const char* ch, int compress)
{
  if (currentFits) {
    if (!currentFits->saveFitsTableChannel(interp, ch, compress)) {
      Tcl_AppendResult(interp, "Unable to save fits to channel ", ch, NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveFitsTableSocketCmd(int s, int compress)
{
  if (currentFits) {
    if (!currentFits->saveFitsTableSocket(s, compress)) {
      Tcl_AppendResult(interp, "Unable to save Fits", NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveArrayFileCmd(const char* fn)
{
  if (currentFits) {
    if (!currentFits->saveArrayFile(fn)) {
      Tcl_AppendResult(interp, "Unable to save Array", NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveArrayChannelCmd(const char* ch)
{
  if (currentFits) {
    if (!currentFits->saveArrayChannel(interp, ch)) {
      Tcl_AppendResult(interp, "Unable to save Array to channel ", ch, NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveArraySocketCmd(int s)
{
  if (currentFits) {
    if (!currentFits->saveArraySocket(s)) {
      Tcl_AppendResult(interp, "Unable to save Array", NULL);
      result = TCL_ERROR;
    }
  }
}

void FrameBase::saveFitsResampleFileCmd(const char* fn, int compress)
{
  if (!currentFits) {
    Tcl_AppendResult(interp, " no data loaded ", NULL);
    result = TCL_ERROR;
    return;
  }
    
  if (!compress) {
    OutFitsFile str(fn);
    if (str.valid())
      return saveFitsResample(str);
  }
  else {
    OutFitsFileGZ str(fn);
    if (str.valid())
      return saveFitsResample(str);
  }

  Tcl_AppendResult(interp, " unable to open fits output ", fn, NULL);
  result = TCL_ERROR;
}

void FrameBase::saveFitsResampleChannelCmd(const char* ch, int compress)
{
  if (!currentFits) {
    Tcl_AppendResult(interp, " no data loaded ", NULL);
    result = TCL_ERROR;
    return;
  }
    
  OutFitsChannel str(interp, ch);
  if (str.valid())
    return saveFitsResample(str);

  Tcl_AppendResult(interp, " unable to output to channel", NULL);
  result = TCL_ERROR;
}

void FrameBase::saveFitsResampleSocketCmd(int s, int compress)
{
  if (!currentFits) {
    Tcl_AppendResult(interp, " no data loaded ", NULL);
    result = TCL_ERROR;
    return;
  }
    
  if (!compress) {
    OutFitsSocket str(s);
    if (str.valid())
      return saveFitsResample(str);
  }
  else {
    OutFitsSocketGZ str(s);
    if (str.valid())
      return saveFitsResample(str);
  }

  Tcl_AppendResult(interp, " unable to output to socket", NULL);
  result = TCL_ERROR;
}

void FrameBase::saveFitsResample(OutFitsStream& str)
{
  int& width = options->width;
  int& height = options->height;

  int bitpix_ = -32;
  int datapixels_ = width*height;
  int realbytes_ = datapixels_ * (abs(bitpix_)/8);
  int datablocks_ = (realbytes_ + (FTY_BLOCK-1))/FTY_BLOCK;
  int databytes_ = datablocks_ * FTY_BLOCK;

  // create header
  FitsHead hd(width, height, 1, bitpix_);

  // write keywords
  saveFitsResampleKeyword(str, hd);

  // write header
  str.write(hd.cards(), hd.headbytes());

  // write data
  if (!isMosaic())
    saveFitsResampleSingle(str);
  else
    saveFitsResampleMosaic(str);

  // pad rest of block
  {
    int diff = databytes_ - realbytes_;
    char buf[diff];
    memset(buf,'\0',diff);
    str.write(buf, diff);
  }
}

void FrameBase::saveFitsResampleKeyword(OutFitsStream& str, FitsHead& dst)
{
  FitsHead* src = currentFits->getHead();
  Vector center = Vector(options->width, options->height)/2.;

  // OBJECT
  char* object = src->getString("OBJECT");
  if (object) {
    dst.appendString("OBJECT", object, NULL);
    delete [] object;
  }

  // DATE-OBS
  char* date = src->getString("DATE");
  if (date) {
    dst.appendString("DATE", date, NULL);
    delete [] date;
  }
  char* dateobs = src->getString("DATE-OBS");
  if (dateobs) {
    dst.appendString("DATE-OBS", dateobs, NULL);
    delete [] dateobs;
  }
  char* timeobs = src->getString("TIME-OBS");
  if (timeobs) {
    dst.appendString("TIME-OBS", timeobs, NULL);
    delete [] timeobs;
  }
  char* dateend = src->getString("DATE-END");
  if (dateend) {
    dst.appendString("DATE-END", dateend, NULL);
    delete [] dateend;
  }
  char* timeend = src->getString("TIME-END");
  if (timeend) {
    dst.appendString("TIME-END", timeend, NULL);
    delete [] timeend;
  }

  // LTMV,DTMV
  if (!isMosaic()) {
    if (currentFits->hasLTMV()) {
      Matrix ltmv = 
	currentFits->getPhysicalToUser() *
	Translate(-cursor) *
	orientationMatrix *
	wcsOrientationMatrix *
	Scale(zoom_) *
	Rotate(rotation) *
	Rotate(wcsRotation) *
	FlipY() * 
	Translate(center);

      dst.appendReal("LTM1_1", ltmv[0][0], 9, NULL);
      dst.appendReal("LTM1_2", ltmv[0][1], 9, NULL);
      dst.appendReal("LTM2_1", ltmv[1][0], 9, NULL);
      dst.appendReal("LTM2_2", ltmv[1][1], 9, NULL);
      dst.appendReal("LTV1",   ltmv[2][0], 9, NULL);
      dst.appendReal("LTV2",   ltmv[2][1], 9, NULL);
    }
  }
  else {
    if (currentFits->hasDTMV()) {
      Matrix dtmv = 
	currentFits->getDetectorToUser() *
	Translate(-cursor) *
	orientationMatrix *
	wcsOrientationMatrix *
	Scale(zoom_) *
	Rotate(rotation) *
	Rotate(wcsRotation) *
	FlipY() * 
	Translate(center);

      dst.appendReal("DTM1_1", dtmv[0][0], 9, NULL);
      dst.appendReal("DTM1_2", dtmv[0][1], 9, NULL);
      dst.appendReal("DTM2_1", dtmv[1][0], 9, NULL);
      dst.appendReal("DTM2_2", dtmv[1][1], 9, NULL);
      dst.appendReal("DTV1",   dtmv[2][0], 9, NULL);
      dst.appendReal("DTV2",   dtmv[2][1], 9, NULL);
    }
  }

  // WCS
  if (currentFits->hasWCS(WCS)) {
    WorldCoor* wcs = currentFits->getWCS(WCS);

    // abort if this is a DSS image
    if (!strncmp(wcs->ptype,"DSS",3))
      return;

    dst.appendString("RADECSYS", wcs->radecsys, NULL);
    dst.appendReal("EQUINOX", wcs->equinox, 9, NULL);

    dst.appendString("CTYPE1", wcs->ctype[0], NULL);
    dst.appendString("CTYPE2", wcs->ctype[1], NULL);
    dst.appendReal("CRVAL1", wcs->crval[0], 9, NULL);
    dst.appendReal("CRVAL2", wcs->crval[1], 9, NULL);

    char* cunit1 = src->getString("CUNIT1");
    if (cunit1) {
      dst.appendString("CUNIT1", cunit1, NULL);
      delete [] cunit1;
    }

    char* cunit2 = src->getString("CUNIT2");
    if (cunit2) {
      dst.appendString("CUNIT2", cunit2, NULL);
      delete [] cunit2;
    }

    // crpix
    Vector crpix = Vector(wcs->crpix[0],wcs->crpix[1]) * 
      currentFits->getImageToUser() * 
      Translate(-cursor) *
      orientationMatrix *
      wcsOrientationMatrix *
      Scale(zoom_) *
      Rotate(rotation) *
      Rotate(wcsRotation) *
      FlipY() * 
      Translate(center);

    dst.appendReal("CRPIX1", crpix[0], 9, NULL);
    dst.appendReal("CRPIX2", crpix[1], 9, NULL);

    // cd matrix
    Matrix cd = Matrix(wcs->cd[0],wcs->cd[1],wcs->cd[2],wcs->cd[3],0,0) *
      currentFits->getImageToUser() *
      Translate(-cursor) *
      orientationMatrix *
      wcsOrientationMatrix *
      Scale(zoom_.invert()) *
      Rotate(rotation) *
      Rotate(wcsRotation) *
      FlipY() * 
      Translate(center);

    dst.appendReal("CD1_1", cd.matrix(0,0), 9, NULL);
    dst.appendReal("CD1_2", cd.matrix(0,1), 9, NULL);
    dst.appendReal("CD2_1", cd.matrix(1,0), 9, NULL);
    dst.appendReal("CD2_2", cd.matrix(1,1), 9, NULL);
  }
}

void FrameBase::saveFitsResampleSingle(OutFitsStream& str)
{
  float nanf = getnanf();
  int& width = options->width;
  int& height = options->height;

  double* m = currentFits->getmWidgetToData();
  int* params = currentFits->getDataParams(currentScale->scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  for (int j=height-1; j>=0; j--)
    for (int i=0; i<width; i++) {
      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      float v = nanf;
      if (x>=xmin && x<xmax && y>=ymin && y<ymax)
	v = currentFits->getValueFloat(((int)y)*srcw + (int)x);
      if (lsb())
	str.writeSwap((char*)(&v), 4, -32);
      else
	str.write((char*)(&v), 4);
    }
}

void FrameBase::saveFitsResampleMosaic(OutFitsStream& str)
{
  float nanf = getnanf();
  int& width = options->width;
  int& height = options->height;

  for (int j=height-1; j>=0; j--)
    for (int i=0; i<width; i++) {

      float v = nanf;
      FitsImage* sptr = currentFits;
      while (sptr) {
	double* m = sptr->getmWidgetToData();
	int* params = sptr->getDataParams(currentScale->scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  v = sptr->getValueFloat(((int)y)*srcw + (int)x);
	  break;
	}
	else
	  sptr = sptr->nextMosaic();
      }

      if (lsb())
	str.writeSwap((char*)(&v), 4, -32);
      else
	str.write((char*)(&v), 4);
    }
}
