// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "magnifier.h"
#include "util.h"

// Parser Stuff

#undef yyFlexLexer
#define yyFlexLexer mgFlexLexer
#include <FlexLexer.h>

static mgFlexLexer* lex = NULL;    // used by mglex
static Magnifier* mg = NULL;          // used by mgerror
extern int mgparse(void*);

int mglex()
{
  return (lex ? lex->yylex() : 0);
}

void mgerror(const char* m)
{
  if (mg) {
    mg->error(m);
    const char* cmd = lex ? lex->YYText() : (const char*)NULL;
    if (cmd && cmd[0] != '\n') {
      mg->error(": ");
      mg->error(cmd);
    }
  }
}

// Public Member Functions

Magnifier::Magnifier(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item) : Widget(i, c, item)
{
  thumbnail = 0;
  needsUpdate = 0;
}

int Magnifier::parse(istringstream& istr)
{
  result = TCL_OK;
  lex = new mgFlexLexer(&istr);
  mg = this;
  mgparse(this);

  delete lex;
  lex = NULL;
  return result;
}

void Magnifier::update()
{
  needsUpdate = 1;
  redrawNow();
}

// Required Virtual Functions

// UpdatePixmap. This function is responsable for creating a valid 
// pixmap the size of the current Magnifier

int Magnifier::updatePixmap(const BBox& bb)
{
  // create a valid pixmap if needed
  // bb is in canvas coords

  if (!pixmap)
    if (!(pixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin),
				options->width, options->height, depth))) {
      cerr << "Magnifier Internal Error: Unable to Create Pixmap" << endl;
      exit(1);
    }

  if (needsUpdate) {
    if (thumbnail) {
      XSetClipOrigin(display, gc, 0, 0);
      XCopyArea(display, thumbnail, pixmap, gc, 0, 0, 
		options->width, options->height, 0, 0);
    }
    else
      clearPixmap();

    needsUpdate = 0;
  }

  return TCL_OK;
}

void Magnifier::invalidPixmap()
{
  Widget::invalidPixmap();
  update();
}

// Command Functions

void Magnifier::getBBoxCmd()
{
  ostringstream str;
  str << options->width << " " << options->height << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);  
}

void Magnifier::updateCmd(Pixmap p)
{
  thumbnail = p;
  update();
}

