// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <math.h>

#include "smooth.h"

void boxcar(float* kernel, int r)
{
  int rr = 2*r+1;
  int ksz = rr*rr;
  
  float* ptr = kernel;
  for (int jj=0; jj<rr; jj++)
    for (int ii=0; ii<rr; ii++, ptr++)
      *ptr = 1./ksz;
}

void tophat(float* kernel, int r)
{
  int rr = 2*r+1;
  int ksz = rr*rr;
  
  float kt = 0;
  for (int y=-r; y<=r; y++) {
    for (int x=-r; x<=r; x++) { 
      if ((x*x + y*y) <= r*r) {
	kernel[(y+r)*rr+(x+r)] = 1;
	kt++;
      }
    }
  }

  // normalize kernel
  for (int aa=0; aa<ksz; aa++)
    kernel[aa] /= kt;
}

void gaussian(float* kernel, int r)
{
  int rr = 2*r+1;
  int ksz = rr*rr;
  float sigma = r/2.;
  
  float kt = 0;
  float a = 1./(sigma*sigma);
  float c = 1./(sigma*sigma);
  for (int y=-r; y<=r; y++) {
    for (int x=-r; x<=r; x++) { 
      if ((x*x + y*y) <= r*r) {
	float v = exp(-.5*(a*x*x + c*y*y));
	kernel[(y+r)*rr+(x+r)] = v;
	kt += v;
      }
    }
  }

  // normalize kernel
  for (int a=0; a<ksz; a++)
    kernel[a] /= kt;
}

void convolve(float* kernel, float* src, float* dest, 
	      int width, int height, int r)
{
  int rr = 2*r+1;

  float* dptr = dest;
  for (int jj=0; jj<height; jj++) {
    for (int ii=0; ii<width; ii++, dptr++) {

      for (int nn=jj-r, qq=0; nn<=jj+r; nn++, qq++) {
	if (nn>=0 && nn<height) {
	  register int nd = nn*width;
	  register int qd = qq*rr;
	  for (int mm=ii-r, pp=0; mm<=ii+r; mm++, pp++) {
	    if (mm>=0 && mm<width)
	      *dptr += src[nd+mm]*kernel[qd+pp];
	  }
	}
      }
    }
  }
}

