// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#if __GNUC__ >= 3
#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;
#else
#include <iostream.h>
#include <strstream.h>
#include <iomanip.h>
#endif

#include <tcl.h>
#include <Xlib.h>
#include <tkWinInt.h>

extern "C" {

void XXDrawLines(Display* display, Drawable d, GC gc, XPoint* points, 
		 int npoints, int mode);

void XXDrawLine(Display* display, Drawable d, GC gc, 
	       int x1, int y1, int x2, int y2)
{
    XPoint points[2];

    points[0].x = x1;
    points[0].y = y1;
    points[1].x = x2;
    points[1].y = y2;
    XXDrawLines(display, d, gc, points, 2, CoordModeOrigin);
}

void XDrawString(Display* display, Drawable drawable, GC gc, int x, int y,
		 const char* str, int length)
{
  Tk_DrawChars(display, drawable, gc, NULL, str, length, x, y);
}

void XXPutImage(Display* display, Drawable d, GC gc, XImage* image,
		int src_x, int src_y, int dest_x, int dest_y,
		unsigned int width, unsigned int height);

void XPutImage(Display* display, Drawable d, GC gc, XImage* image,
	       int src_x, int src_y, int dest_x, int dest_y,
	       unsigned int width, unsigned int height)
{
  if (image->bits_per_pixel <= 8)
    XXPutImage(display, d, gc, image, src_x, src_y,
	       dest_x, dest_y, width, height);
  else
    TkPutImage(NULL, 0, display, d, gc, image, src_x, src_y,
	       dest_x, dest_y, width, height);
}

void XClearArea(Display* display, Window w, int x, int y,
		unsigned int width, unsigned int height, Bool exposures)
{
  RECT rect;
  rect.left = x;
  rect.top = y;
  rect.right = x+width;
  rect.bottom = y+height;
  TkWinDCState state;
  HDC dc = TkWinGetDrawableDC(display, w, &state);
  FillRect(dc, &rect, (HBRUSH)(COLOR_WINDOW+1));
  TkWinReleaseDrawableDC(w, dc, &state);
}

void XSetClipRectangles(Display* display, GC gc, int clip_x_origin,
			int  clip_y_origin, XRectangle* r,
			int  n, int  ordering)
{
  // ok, we give up. just don't set a clipping region. Otherwise, the
  // memory leak will get us.
  gc->clip_x_origin = clip_x_origin;
  gc->clip_y_origin = clip_y_origin;
  return;

  // this implementation is not complete.
  // also, this is a memory leak: we have to keep the region around.
  // or clipping will not work....bummer...see TkSetRegion for more info

  TkRegion rgn = (TkRegion)CreateRectRgn(r->x, r->y,
					 r->x+r->width, r->y+r->height);
  TkSetRegion(display, gc, rgn);
  gc->clip_x_origin = clip_x_origin;
  gc->clip_y_origin = clip_y_origin;
}

Status XAllocColorCells(Display* display, Colormap xcmap, Bool contig,
			unsigned long *planes_mask, unsigned int nplanes,
			unsigned long *pixels, unsigned int ncolors)
{
  TkWinColormap *cmap = (TkWinColormap*)xcmap;
  PALETTEENTRY entry;

  if (cmap->size+ncolors > 256)
    return 0;

  for (int i=0; i<ncolors; i++) {
    pixels[i] = cmap->size++;

    entry.peRed   = 0;
    entry.peGreen = 0;
    entry.peBlue  = 0;
    entry.peFlags = PC_RESERVED;

    ResizePalette(cmap->palette, cmap->size);
    SetPaletteEntries(cmap->palette, cmap->size-1, 1, &entry);
  }
  return 1;
}

void XStoreColors(Display* display, Colormap xcmap, XColor* color, int ncolors)
{
  TkWinColormap *cmap = (TkWinColormap *)xcmap;
  PALETTEENTRY entry[256];
  int min = color[0].pixel;

  HDC hdc = GetDC(NULL);
  SelectPalette(hdc, cmap->palette, FALSE);

  // Assume that the color cells are contigious!

  for (int i=0; i<ncolors; i++) {
    entry[i].peRed   = (color[i].red)   >> 8;
    entry[i].peGreen = (color[i].green) >> 8;
    entry[i].peBlue  = (color[i].blue)  >> 8;
    entry[i].peFlags = PC_RESERVED;
  }
  AnimatePalette(cmap->palette, min, ncolors, entry);
  ReleaseDC(NULL, hdc);
}

void XInstallColormap(Display* display, Colormap colormap)
{
}

Pixmap XCreatePixmap(Display* display, Drawable drawable, 
		     unsigned int width, unsigned int height,
		     unsigned int depth)
{
  Tk_GetPixmap(display, drawable, width, height, depth);
}

void XFreePixmap(Display* display, Pixmap pixmap)
{
  Tk_FreePixmap(display, pixmap);
}

Status XGetGCValues(Display* disp, GC gc, unsigned long mask, XGCValues* ret)
{
  if (mask & GCFunction) {ret->function = gc->function;}
  if (mask & GCPlaneMask) {ret->plane_mask = gc->plane_mask;}
  if (mask & GCForeground) {ret->foreground = gc->foreground;}
  if (mask & GCBackground) {ret->background = gc->background;}
  if (mask & GCLineWidth) {ret->line_width = gc->line_width;}	
  if (mask & GCLineStyle) {ret->line_style = gc->line_style;}
  if (mask & GCCapStyle) {ret->cap_style = gc->cap_style;}
  if (mask & GCJoinStyle) {ret->join_style = gc->join_style;}
  if (mask & GCFillStyle) {ret->fill_style = gc->fill_style;}
  if (mask & GCFillRule) {ret->fill_rule = gc->fill_rule;}
  if (mask & GCArcMode) {ret->arc_mode = gc->arc_mode;}
  if (mask & GCTile) {ret->tile = gc->tile;}
  if (mask & GCStipple) {ret->stipple = gc->stipple;}
  if (mask & GCTileStipXOrigin) {ret->ts_x_origin = gc->ts_x_origin;}
  if (mask & GCTileStipYOrigin) {ret->ts_y_origin = gc->ts_y_origin;}
  if (mask & GCFont) {ret->font = gc->font;}
  if (mask & GCSubwindowMode) {ret->subwindow_mode = gc->subwindow_mode;}
  if (mask & GCGraphicsExposures) 
    {ret->graphics_exposures = gc->graphics_exposures;}
  if (mask & GCClipXOrigin) {ret->clip_x_origin = gc->clip_x_origin;}
  if (mask & GCClipYOrigin) {ret->clip_y_origin = gc->clip_y_origin;}
  if (mask & GCClipMask) {;} // not implemented
  if (mask & GCDashOffset) {ret->dash_offset = gc->dash_offset;}
  if (mask & GCDashList) {ret->dashes = gc->dashes;}
}

void XCopyGC(Display* display, GC src, unsigned long mask, GC dest)
{
  if (mask & GCFunction) {dest->function = src->function;}
  if (mask & GCPlaneMask) {dest->plane_mask = src->plane_mask;}
  if (mask & GCForeground) {dest->foreground = src->foreground;}
  if (mask & GCBackground) {dest->background = src->background;}
  if (mask & GCLineWidth) {dest->line_width = src->line_width;}	
  if (mask & GCLineStyle) {dest->line_style = src->line_style;}
  if (mask & GCCapStyle) {dest->cap_style = src->cap_style;}
  if (mask & GCJoinStyle) {dest->join_style = src->join_style;}
  if (mask & GCFillStyle) {dest->fill_style = src->fill_style;}
  if (mask & GCFillRule) {dest->fill_rule = src->fill_rule;}
  if (mask & GCArcMode) {dest->arc_mode = src->arc_mode;}
  if (mask & GCTile) {dest->tile = src->tile;}
  if (mask & GCStipple) {dest->stipple = src->stipple;}
  if (mask & GCTileStipXOrigin) {dest->ts_x_origin = src->ts_x_origin;}
  if (mask & GCTileStipYOrigin) {dest->ts_y_origin = src->ts_y_origin;}
  if (mask & GCFont) {dest->font = src->font;}
  if (mask & GCSubwindowMode) {dest->subwindow_mode = src->subwindow_mode;}
  if (mask & GCGraphicsExposures) 
    {dest->graphics_exposures = src->graphics_exposures;}
  if (mask & GCClipXOrigin) {dest->clip_x_origin = src->clip_x_origin;}
  if (mask & GCClipYOrigin) {dest->clip_y_origin = src->clip_y_origin;}
  if (mask & GCDashOffset) {dest->dash_offset = src->dash_offset;}
  if (mask & GCDashList) {
    dest->dashes = src->dashes;
    (&(dest->dashes))[1] = 0;
  }

  if (mask & GCClipMask) {
    if (dest->clip_mask)
      free((void*)dest->clip_mask);
    dest->clip_mask = (Pixmap)ckalloc(sizeof(TkpClipMask));
    ((TkpClipMask*)dest->clip_mask)->type = TKP_CLIP_PIXMAP;
    ((TkpClipMask*)dest->clip_mask)->value.pixmap = src->clip_mask;
  } else {
    dest->clip_mask = None;
  }
}

// we need these routines because Tk's versions are too simplistic

void XXWarpPointer(Display* display, Window src_w, Window dest_w,
		   int src_x, int src_y,
		   unsigned int src_width, unsigned int src_height,
		   int dest_x, int dest_y)
{
  if (dest_w != None)
    // move to absolute location
    XWarpPointer(display, src_w, dest_w, src_x, src_y, src_width, src_height,
		 dest_x, dest_y);
  else {
    // move to relative location
    POINT p;
    GetCursorPos(&p);
    SetCursorPos(p.x+dest_x, p.y+dest_y);
  }
}

XImage* XXGetImage(Display* display, Drawable drawable, int x, int y,
		   unsigned int width, unsigned int height,
		   unsigned long plane, int format)
{
  int depth;
  TkWinDrawable* d = (TkWinDrawable*)drawable;
  switch (d->type) {
  case TWD_BITMAP:
    depth = d->bitmap.depth;
    break;
  case TWD_WINDOW:
  case TWD_WINDC:
    cerr << "Windows Internal Error: Unable to determine window depth" << endl;
    return NULL;
  }

  XImage* image = XCreateImage(display, NULL, depth, format, 0, NULL, 
			       width, height, 32, 0);

  if (image)
    image->data = ckalloc(image->bytes_per_line * image->height);
  else {
    cerr << "Windows Internal Error: XCreateImage failed" << endl;
    return NULL;
  }

  TkWinDCState hdcState;
  HDC hdc = TkWinGetDrawableDC(display, drawable, &hdcState);

  char infoBuf[sizeof(BITMAPINFO) + 256*sizeof(RGBQUAD)];
  BITMAPINFO*infoPtr = (BITMAPINFO*)infoBuf;
  
  infoPtr->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  infoPtr->bmiHeader.biWidth = width;
  infoPtr->bmiHeader.biHeight = -(LONG)height;
  infoPtr->bmiHeader.biPlanes = 1;
  infoPtr->bmiHeader.biBitCount = depth;
  infoPtr->bmiHeader.biCompression = BI_RGB;
  infoPtr->bmiHeader.biSizeImage = 0;
  infoPtr->bmiHeader.biXPelsPerMeter = 0;
  infoPtr->bmiHeader.biYPelsPerMeter = 0;
  infoPtr->bmiHeader.biClrImportant = 0;
  infoPtr->bmiHeader.biClrUsed = 0;

  GetDIBits(hdc, d->bitmap.handle, 0, height, image->data, infoPtr,
	    DIB_RGB_COLORS);

  TkWinReleaseDrawableDC(drawable, hdc, &hdcState);

  return image;
}

// support routines

void XXPutImage(Display* display, Drawable d, GC gc, XImage* image,
		int src_x, int src_y, int dest_x, int dest_y,
		unsigned int width, unsigned int height)
{
  HDC dc, dcMem;
  TkWinDCState state;
  BITMAPINFO *infoPtr;
  HBITMAP bitmap;
  WORD *pwIndex;

  int tkpWinRopModes[] = {
    R2_BLACK,			/* GXclear */
    R2_MASKPEN,			/* GXand */
    R2_MASKPENNOT,		/* GXandReverse */
    R2_COPYPEN,			/* GXcopy */
    R2_MASKNOTPEN,		/* GXandInverted */
    R2_NOT,			/* GXnoop */
    R2_XORPEN,			/* GXxor */
    R2_MERGEPEN,		/* GXor */
    R2_NOTMERGEPEN,		/* GXnor */
    R2_NOTXORPEN,		/* GXequiv */
    R2_NOT,			/* GXinvert */
    R2_MERGEPENNOT,		/* GXorReverse */
    R2_NOTCOPYPEN,		/* GXcopyInverted */
    R2_MERGENOTPEN,		/* GXorInverted */
    R2_NOTMASKPEN,		/* GXnand */
    R2_WHITE			/* GXset */
  };

  display->request++;

  dc = TkWinGetDrawableDC(display, d, &state);
  SetROP2(dc, tkpWinRopModes[gc->function]);
  dcMem = CreateCompatibleDC(dc);
  int depth = image->bits_per_pixel;

  infoPtr = (BITMAPINFO*) ckalloc(sizeof(BITMAPINFOHEADER)
				  + sizeof(WORD)*256);
  pwIndex = (WORD *) ((char *)(infoPtr) + sizeof(BITMAPINFOHEADER));
	
  infoPtr->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  infoPtr->bmiHeader.biWidth = image->width;
  infoPtr->bmiHeader.biHeight = -image->height; /* Top-down order */
  infoPtr->bmiHeader.biPlanes = 1;
  infoPtr->bmiHeader.biBitCount = depth;
  infoPtr->bmiHeader.biCompression = BI_RGB;
  infoPtr->bmiHeader.biSizeImage = 0;
  infoPtr->bmiHeader.biXPelsPerMeter = 0;
  infoPtr->bmiHeader.biYPelsPerMeter = 0;
  infoPtr->bmiHeader.biClrImportant = 0;
  infoPtr->bmiHeader.biClrUsed = (depth==8 ? 256 : 2);

  for (int i=0; i<256; i++)
    pwIndex[i] = i;

  bitmap = CreateDIBitmap(dc, &infoPtr->bmiHeader, CBM_INIT,
			  image->data, infoPtr, 
			  (depth==8 ? DIB_PAL_COLORS : DIB_RGB_COLORS));
  ckfree((char*)infoPtr);
 
  bitmap = (HBITMAP)SelectObject(dcMem, bitmap);
  BitBlt(dc, dest_x, dest_y, width, height, dcMem, src_x, src_y, SRCCOPY);
  DeleteObject(SelectObject(dcMem, bitmap));
  DeleteDC(dcMem);
  TkWinReleaseDrawableDC(d, dc, &state);
}

}
